
/* Gnutella Network Messages routing */

#include "gnutella.h"

#include <stdarg.h>
#include <assert.h>


struct gnutella_node *fake_node;	/* Our fake node */

GList   *messages[256];				/* The messages lists */
guint32 n_messages[256];			/* Numbers of messages in the lists */
GList   *tail[256];					/* The older message in the lists */

struct message
{
	guchar muid[16];					/* Message UID */
	GSList *nodes;						/* Nodes from which the message came */
	guint8 function;					/* Type of the message */
};

gchar *debug_msg[256];

#define MAX_MESSAGES_PER_LIST	256	/* So we can remember 256 * 256 = about 65536 messages */

/* ------------------------------------------------------------------------------------------------ */

/* Log function */

void routing_log(gchar *fmt, ...)
{
	static gchar t[4096];

	va_list va;

	va_start(va, fmt);

	g_vsnprintf(t, sizeof(t), fmt, va);

	/* XXX put the log somewhere */

	va_end(va);

	//printf("%s", t);
}

gboolean node_sent_message(struct gnutella_node *n, struct message *m) {
  GSList *l = m->nodes;

  while(l) {
	if (l->data == n)
	  return TRUE;
	l = l->next;
  }
  return FALSE;
}

/* Init function */

void routing_init(void)
{
	guint32 i;

	/* make sure it segfaults if we try to access it, but it must be distinct from NULL. */
	fake_node = (struct gnutella_node *) 0x01;

	srand(time((time_t *) NULL));

	for (i = 0; i < 16; i++) guid[i]  = rand() % 256;

	for (i = 0; i < 256; i++) { messages[i] = NULL; n_messages[i] = 0; }

	for (i = 0; i < 256; i++) debug_msg[i] = "UNKNOWN MSG  ";

	debug_msg[GTA_MSG_INIT]                = "Ping Request";
	debug_msg[GTA_MSG_INIT_RESPONSE]       = "Ping Reply  ";
	debug_msg[GTA_MSG_SEARCH]              = "Search Req. ";
	debug_msg[GTA_MSG_SEARCH_RESULTS]      = "Search Reply";
}


void generate_new_muid(guchar *muid) {
  gint i;
  for (i = 0; i < 16; i += 2) (*((guint16 *) (muid + i))) = rand() % 65536;
}

/* Generate a new muid and put it in a message header */

void message_set_muid(struct gnutella_header *header) {
	generate_new_muid(header->muid);
}

/* Erase a node from the routing tables */

void routing_node_remove(struct gnutella_node *node)
{
	GList *l;
	guint32 i;

	for (i = 0; i < 256; i++)
		for (l = messages[i]; l; l = l->next) {
			struct message *m = (struct message *)l->data;
			/* g_slist_remove won't do anything if this node isn't in the list */
			m->nodes = g_slist_remove(m->nodes, node);
		}
}

/* Adds a new message in the routing tables */

void message_add(guchar *muid, guint8 function, struct gnutella_node *node)
{
	static struct message *m;
	guint8 f = muid[0];

	if (!node)
	{
		node = fake_node;	/* We are the sender of the message */

		routing_log("%-21s %s %s %3d\n", "OURSELVES", debug_msg[function], md5dump(muid), my_ttl);
	}

	if (n_messages[f] >= MAX_MESSAGES_PER_LIST) /* Table is full */
	{
		GList *ct = tail[f];							/* ct is the current tail */

		tail[f] = ct->prev;							/* The new tail is the prev message */

		tail[f]->next = (GList *) NULL;			/* The tail next pointer must be NULL */
		ct->prev      = (GList *) NULL;			/* The head prev pointer must be NULL */

		ct->next = messages[f];						/* The head next pointer = current head */
		messages[f]->prev = ct;						/* Current head is no more the real head */

		messages[f] = ct;								/* The list head change */

		m = (struct message *) ct->data;			/* We'll use the oldest message memory place */
		g_slist_free(m->nodes);
	}
	else /* Table is not full, allocate a new structure and prepend it to the table */
	{
		m = (struct message *) g_malloc(sizeof(struct message));

		n_messages[f]++;
	
		messages[f] = g_list_prepend(messages[f], (gpointer) m);
	
		if (!tail[f]) tail[f] = messages[f];	/* This is the first message in the table */
	}

	memcpy(m->muid, muid, 16);
	m->function = function;
	m->nodes = g_slist_append(NULL, node);
}

/* Look for a particular message in the routing tables */

gboolean find_message(guchar *muid, guint8 function, struct message **m)
{
	/* Returns TRUE if the message is found */
	/* Set *node to node if there is a connected node associated with the message found */

	static GList *l;

	for (l = messages[(guint8) muid[0]]; l; l = l->next)
	{
		if (!memcmp(((struct message *) l->data)->muid + 1, muid + 1, 15) && ((struct message *) l->data)->function == function)
		{
			/* We found the message */

			*m = (struct message *) l->data;	/* The node the message came from */
			return TRUE;
		}
	}

	*m = NULL;
	return FALSE; /* Message not found in the tables */
}

/* Main routing function -------------------------------------------------------------------------- */

gboolean route_message(struct gnutella_node **node)
{
	static struct gnutella_node *sender;	/* The node that sent the message */
	struct message *m; /* The copy of the message we've already seen */
	struct gnutella_node *found;		/* The node to have sent us this message earliest of those we're still connected to. */
	static gboolean handle_it;
	guchar *databuf = NULL;

	sender = (*node);

	routing_log("%-21s ", node_ip(sender));
	routing_log("%s ", debug_msg[sender->header.function]);
	routing_log("%s ", md5dump(sender->header.muid));
	routing_log("%3d/%3d : ", sender->header.ttl, sender->header.hops);

	if (sender->header.function & 0x01) /* The message is a ping or search reply */
	{
		/* We'll handle all ping replies we receive, even if they are not destinated to us */
		handle_it = (sender->header.function == GTA_MSG_INIT_RESPONSE);

		/* We'll also handle all search replies if we're doing a passive search */
		handle_it = handle_it  ||  (sender->header.function == GTA_MSG_SEARCH_RESULTS  &&  search_passive);

		if (!find_message(sender->header.muid, sender->header.function & ~(0x01), &m))
		{
			/* We have never seen any request matching this reply ! */

			routing_log("[ ] no request matching the reply !\n");

			sender->dropped++;
			dropped_messages++;
			sender->n_bad++;					/* The node shouldn't have forwarded us this message */

			return handle_it;					/* We don't have to handle the message */
		}

		if (node_sent_message(fake_node, m))				/* We are the target of the reply */
		{
			if (sender->header.function == GTA_MSG_INIT_RESPONSE) ping_stats_add(sender);
			routing_log("[H] we are the target\n");
			return TRUE;	
		}

		if (handle_it) routing_log("[H] "); else routing_log("[ ] ");

		if (m && m->nodes)						/* We only have to forward the message the target node */
		{
			if (sender->header.ttl > max_ttl) /* TTL too large, don't forward */
			{
				routing_log("[Max TTL] ");
/*				sender->dropped++; */
/*				dropped_messages++; */
/*				return handle_it; */
			}

			if (!sender->header.ttl || !--sender->header.ttl)	/* TTL expired, message can't go further */
			{
				routing_log("[TTL expired] ");
/*				sender->dropped++; */
/*				dropped_messages++; */
/*				return handle_it; */
			}

			sender->header.hops++;

			found = (struct gnutella_node *) m->nodes->data;

			routing_log("-> sendto_one(%s)\n", node_ip(found));

			if (sender->pos) databuf = sender->buffer;

			sendto_one(found, (guchar *) &(sender->header), databuf,
							sender->size + sizeof(struct gnutella_header), FALSE); // not a priority packet
		}
		else										/* The target node is no more connected to us */
		{
			routing_log("Target no more connected\n");

			routing_errors++;
			sender->dropped++;
			dropped_messages++;
		}

		return handle_it;
	}
	else /* The message is a request */
	{
		if (find_message(sender->header.muid, sender->header.function, &m))
		{
			/* This is a duplicated message */

			if (node_sent_message(sender, m))	/* The same node has sent us a message twice ! */
			{
				routing_log("[ ] Dup message (from the same node !)\n");

				routing_errors++;

				/* That should be a really good reason to kick the offender immediately */
				/* But it will kick far too many people nowadays... */

/*				node_remove(sender, "Kicked (sent identical messages twice)"); */
/*				(*node) = NULL; */
			} else {
			  routing_log("[ ] duplicated\n");

			  if (node_sent_message(fake_node, m)) {
				/* node sent us our own search, which may be ok if they've
				 * just connected and sent it before they read our reissue
				 * of the search, or if we've sent them the search with a
				 * new muid. */
			  } else {
				/* append so that we route matches to the one that sent it
				 * to us first; ie., presumably the one closest to the
				 * original sender. */
				m->nodes = g_slist_append(m->nodes, sender);
			  }
			}

			return FALSE;
		}
		else	/* Never seen this message before */
		{

			if (sender->header.ttl > max_ttl)	/* TTL too large */
			{
				routing_log("[ ] [NEW] Max TTL reached\n");
				sender->dropped++;
				dropped_messages++;
				return FALSE;
			}

			message_add(sender->header.muid, sender->header.function, sender);

			if (!sender->header.ttl || !--sender->header.ttl)	/* TTL expired, message can't go further */
			{
				routing_log("[H] [NEW] (TTL expired)\n");
				sender->dropped++;
				dropped_messages++;
			}
			else 							/* Forward it to all others nodes */
			{
				sender->header.hops++;

				routing_log("[H] [NEW] -> sendto_all_but_one()\n");

				if (sender->pos) databuf = sender->buffer;

				sendto_all_but_one(sender, (guchar *) &(sender->header), databuf,
					sender->size + sizeof(struct gnutella_header), FALSE); // not a priority packet
			}

			return TRUE;
		}
	}

	return FALSE;
}

/* Sending of messages ---------------------------------------------------------------------------- */

/* Send a message to a specific connected node, priority messages are sent before all others */

void sendto_one(struct gnutella_node *n, guchar *msg, guchar *data, guint32 size, gboolean priority)
{

	struct node_send_packet *nsp; // this is a temp holder for a block of memory

	g_return_if_fail(n);
	g_return_if_fail(msg);
	g_return_if_fail(size > 0);

	if (n->status != GTA_NODE_CONNECTED) return;

	nsp = (struct node_send_packet *) g_malloc0(size + sizeof(struct node_send_packet));
	nsp->size = size;
	memcpy(nsp->packet, msg, size);

	if(data == NULL) {
		memcpy(nsp->packet, msg, size);
		node_enqueue(n, nsp, priority); // this is a header only
		}

	else { // append data part to the packet buffer
		memcpy(nsp->packet, msg, sizeof(struct gnutella_header));
		memcpy(nsp->packet + sizeof(struct gnutella_header), data, size - sizeof(struct gnutella_header));
		node_enqueue(n, nsp, priority);
		}
}

/* Send a message to all connected nodes but one */

void sendto_all_but_one(struct gnutella_node *theone, guchar *msg, guchar *data, guint32 size, gboolean priority)
{
	GSList *l;
	struct gnutella_node *n;

	g_return_if_fail(theone);
	g_return_if_fail(msg);
	g_return_if_fail(size > 0);

	for(l = sl_nodes; l; l = l->next)
	{
		n = (struct gnutella_node *) l->data;
		if (n) {
			if (n != theone) {
				if (n->status == GTA_NODE_CONNECTED) sendto_one(n, msg, data, size, priority);
				}
			}
	}
}

/* Send a message to all connected nodes */

void sendto_all(guchar *msg, guchar *data, guint32 size, gboolean priority)
{
	GSList *l;
	struct gnutella_node *n;

	g_return_if_fail(msg);
	g_return_if_fail(size > 0);

	for(l = sl_nodes; l; l = l->next)
	{
		n = (struct gnutella_node *) l->data;
		if (n) {
			if (n->status == GTA_NODE_CONNECTED) sendto_one(n, msg, data, size, priority);
			}
	}
}

/* vi: set ts=3: */

