//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// CLI アプリケーションのエントリポイント
//

#include "nono.h"
#include "cuimessage.h"
#include "kevent.h"
#include "mainapp.h"
#include "monitor.h"
#include "power.h"

// wxApp と相似にするためのクラス
class CLIApp
{
 public:
	int OnInit();
	bool Parse();
	int OnCreate();
	void OnHalt();
	void OnAppExit();

	int ac {};
	char **av {};

	autofd kq {};
	bool exit_request {};
};

int
main(int ac, char *av[])
{
	std::unique_ptr<CLIApp> app(new CLIApp());

	// wxApp とよく似せるため
	app->ac = ac;
	app->av = av;
	return app->OnInit();
}

// ここがエントリポイント(のつもり)
int
CLIApp::OnInit()
{
	int rv;

	// CLI 側では関係ないが wx 側で必要なので初期化は2ステージに分かれている
	rv = gMainApp.Init1(true, ac, av);
	if (rv != MainApp::PASS) {
		return rv;
	}
	// UIMessage の初期化はなんとなくスレッド開始前のほうがよかろう
	kq = CUIMessage::Init();
	if (kq < 0) {
		return EXIT_FAILURE;
	}

	// モニタ登録締切。CLI ではあまり関係ないけど揃えておく。
	gMonitorManager->Fix();

	// スレッド作成を伴う初期化。
	if (!gMainApp.Init2()) {
		return EXIT_FAILURE;
	}

	// メインウィンドウに相当するところ
	OnCreate();

	// 戻ってきたので、VM を解放。
	gMainApp.Dispose();

	return EXIT_SUCCESS;
}

// メインウィンドウ作成後のイベントに相当するところ
int
CLIApp::OnCreate()
{
	CUIMessage::Connect(UIMessage::APPEXIT,
		[&](const UIMessage&) { OnAppExit(); });
	if (gMainApp.Has(VMCap::M68K)) {
		CUIMessage::Connect(UIMessage::HALT,
			[&](const UIMessage&) { OnHalt(); });
	}
	// UIMessage の処理を開始する。
	UIMessage::Attach(&CUIMessage::Process);

	// 電源投入
	auto power = GetPowerDevice();
	power->PushPowerButton();

	// メッセージを待つ
	while (exit_request == false) {
		struct kevent kev;

		if (kevent_poll(kq, &kev, 1, NULL) < 0) {
			if (errno == EINTR) {
				continue;
			}
			warn("CLIApp::OnCreate: kevent_poll");
			// XXX どうする?
			continue;
		}

		// パイプから1メッセージ分読み出す
		UIMessage m;
		if (read((int)kev.ident, &m, sizeof(m)) < 0) {
			warn("CLIApp::OnCreate: read");
			// XXX どうする?
			continue;
		}

		// ディスパッチ
		CUIMessage::Dispatch(m);
	}

	return 0;
}

// m680x0 ダブルバスフォールト時に呼ばれるコールバック。
void
CLIApp::OnHalt()
{
	// どうする?
	warnx("Double bus fault has occurred in VM");
}

// アプリケーションの終了要求で呼ばれるコールバック。
void
CLIApp::OnAppExit()
{
	exit_request = true;
}
