//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// モニタ管理
//

#pragma once

#include "geometric.h"
#include "mainapp.h"
#include "object.h"
#include "textscreen.h"
#include <array>

// モニタ識別子
//
// テキストモニタの識別子だがメニューID (とウィンドウの識別ID) としても使う
// ので wxmainframe あたりも参照。
enum {
#include "monitor_id.h"
};

// 識別子 (n番目で指定)
static constexpr uint ID_MONITOR_ATC(uint n) {
	return ID_MONITOR_ATC0 + n;
}
static constexpr uint ID_MONITOR_CMMU(uint n) {
	return ID_MONITOR_CMMU0 + n;
}
static constexpr uint ID_MONITOR_HOSTCOM(uint n) {
	return ID_MONITOR_HOSTCOM0 + n;
}
static constexpr uint ID_MONITOR_HOSTNET(uint n) {
	return ID_MONITOR_HOSTNET0 + n;
}
static constexpr uint ID_SUBWIN_CACHE(uint n) {
	return ID_SUBWIN_CACHE0 + n;
}
static constexpr uint ID_MONITOR_MEMDUMP(uint n) {
	return ID_MONITOR_MEMDUMP0 + n;
}
static constexpr uint ID_MONITOR_RTL8019AS(uint n) {
	return ID_MONITOR_RTL8019AS0 + n;
}
static constexpr uint ID_MONITOR_VIRTIO_BLOCK(uint n) {
	return ID_MONITOR_VIRTIO_BLOCK0 + n;
}
static constexpr uint ID_MONITOR_XPMEMDUMP(uint n) {
	return ID_MONITOR_XPMEMDUMP0 + n;
}
// 判定
static constexpr bool IS_MONITOR_MEMDUMP(uint id) {
	return (ID_MONITOR_MEMDUMP0 <= id &&
	        id < ID_MONITOR_MEMDUMP(MAX_MEMDUMP_MONITOR));
}
static constexpr bool IS_MONITOR_XPMEMDUMP(uint id) {
	return (ID_MONITOR_XPMEMDUMP0 <= id &&
	        id < ID_MONITOR_XPMEMDUMP(MAX_XPMEMDUMP_MONITOR));
}

class Monitor;
using MonitorCallback_t = void (Object::*)(Monitor *, TextScreen&);
#define ToMonitorCallback(f) static_cast<MonitorCallback_t>(f)

// モニタ
class Monitor
{
 public:
	Monitor();
	Monitor(uint id_, Object *obj_);
	~Monitor();

	uint GetId() const { return id; }

	// モニターの表示サイズを設定・取得。
	void SetSize(nnSize size_) { size = size_; }
	void SetSize(int col_, int row_) { SetSize(nnSize(col_, row_)); }
	nnSize GetSize() const { return size; }
	int GetCol() const noexcept { return size.width; }
	int GetRow() const noexcept { return size.height; }

	// 行を増やす。
	void AddHeight(int h) { size.height += h; }

	// モニターが縦に可変長の場合の最大表示行数。(固定長なら 0)
	void SetMaxHeight(int maxh_) { maxh = maxh_; }
	int GetMaxHeight() const { return maxh; }

	// ID を表示用文字列にして返す。
	static const char *GetIdText(uint id);

	// このモニターの ID を表示用文字列にして返す。
	const char *GetIdText() const { return GetIdText(id); }

	// モニタを更新する。
	void Update(TextScreen& screen) {
		(obj->*(func))(this, screen);
	}

	Object *obj {};				// オブジェクト

	MonitorCallback_t func {};	// コールバック関数

 private:
	// モニター識別子
	uint id {};

	// モニターサイズ
	nnSize size {};

	// 縦に可変長の場合の最大表示行数 (固定長なら 0)
	int maxh {};

	// ID を表示用のテキストにしたものとの対応表。
	static const std::vector<std::pair<uint, const char *>> idtext;
};

//
// モニターマネージャ
//
class MonitorManager
{
	// モニター情報 (事前定義)
	struct MonitorInfo {
		uint id;
		VMCap vmcap;
		std::vector<std::string> aliases;
	};

 public:
	MonitorManager();
	~MonitorManager();

	// 登録。
	// モニタの登録はコンストラクタおよび Create フェーズまでに行うこと。
	// Create() の後、Init() より前に -M オプションの処理があるため。
	// see wx/wxapp.cpp
	Monitor *Regist(uint id_, Object *parent_);

	// 削除
	void Unregist(Monitor *mon_);

	// 現在登録されているモニターの(穴空きでない)リストを返す。
	const std::vector<Monitor *> GetList() const;

	// 指定された id を持つモニターを返す。
	// こっちは、なければ assert する。
	Monitor *Get(uint id) const;

	// 指定された id を持つモニターを返す。
	// こっちは、なければ NULL を返す。
	Monitor *Find(uint id) const {
		return monitors[id].get();
	}

	// 指定された id に対応する機種ケーパビリティを返す。
	VMCap GetVMCap(uint id) const {
		assertmsg(id < info.size(), "id=%u < %zu", id, info.size());
		return info[id].vmcap;
	}

	// 指定された id に対応する名前のリストを返す。
	// id が範囲外だと assert する。
	const std::vector<std::string>& GetAliases(uint id) const {
		assertmsg(id < info.size(), "id=%u < %zu", id, info.size());
		return info[id].aliases;
	}

	// id リストから一覧表示用の文字列リストを生成して返す
	std::vector<std::string> MakeListString(std::vector<uint> list) const;

	// 以降の追加を禁止する (プログラムミスを避けるため)
	void Fix();

 private:
	// モニターインスタンスの配列。添字はモニター ID なので歯抜け。
	std::array<std::unique_ptr<Monitor>, ID_SUBWIN_MAX> monitors /*{}*/;

	// モニタの登録は -M オプションの処理が始まるまでに終えないといけない。
	bool fixed {};

	// 事前に設定されているモニター情報
	static const std::vector<MonitorInfo> info;
};

extern MonitorManager *gMonitorManager;
