//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// Stopwatch
//

#pragma once

#include "header.h"
#include <chrono>

// .Net の Stopwatch みたいなもの。
// Start() と Restart() が分かりづらいけど、もうそっくり真似しておく
class Stopwatch
{
	using steady_clock = std::chrono::steady_clock;

 public:
	// ストップウォッチを再開する
	void Start() {
		running = true;
		epoch = Now();
	}

	// ストップウォッチを一時停止する
	void Stop() {
		total += Now() - epoch;
		running = false;
	}

	// 経過時間をリセットした上で、ストップウォッチを再開(開始)する
	void Restart() {
		total = steady_clock::duration::zero();
		Start();
	}

	// ストップウォッチを停止して、経過時間をリセットする
	void Reset() {
		Stop();
		total = steady_clock::duration::zero();
	}

	// 経過時間を nsec で返す
	uint64 Elapsed() const {
		using namespace std::chrono;
		if (running) {
			return duration_cast<nanoseconds>(Now() - epoch + total).count();
		} else {
			return duration_cast<nanoseconds>(total).count();
		}
	}

 private:
	// 現在時刻を返す
	steady_clock::time_point Now() const {
		return steady_clock::now();
	}

	// 前回までの経過時間の合計
	steady_clock::duration total {};

	// 今回の Start 時刻
	steady_clock::time_point epoch {};

	// ストップウォッチが動作中なら true
	bool running {};
};
