//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

//
// DIPSW 設定ウィンドウ
//

#include "wxdipswwindow.h"
#include "wxscrollbar.h"
#include "dipsw.h"
#include "mainapp.h"

// 数字は全部 0 始まりで、表示だけ 1 始まり。
enum {
	ID_DIPSW0	= IDGROUP_DIPSW,
	ID_DIPSW15	= ID_DIPSW0 + 15,

	ID_local_end,	// 最後に置く (チェック用)
};
static_assert(ID_local_end - 1 <= (int)IDGROUP_DIPSW_END, "ID exceeded");

#define ID_DIPSW(id)	(ID_DIPSW0 + id)

// ↓topsizer
//   +----------+
//   |  DIPSW1  |
//   +----------+
//   | (中余白) |
//   +----------+
//   | (DIPSW2) |
//   +----------+
//
// →  DIPSW Box (StaticBoxSizer)
// +--+-----------------------+--+
// |  |  vbox       vbox      |  |
// |  |↓+-------+ +-----+ .. |  |
// |  |  |上余白 | |     |    |  |
// |左|  +-------+ +     +    |右|
// |余|  |  No.  | |     |    |余|
// |白|  +-------+ +     +    |白|
// |  |  |SCR Bar| |     |    |  |
// |  |  +-------+ +     +    |  |
// |  |  |下余白 | |     |    |  |
// |  |  +-------+ +-----+ .. |  |
// +--+-----------------------+--+
//
// (GTK3の?) StaticBoxSizer は枠の中も外もマージンがなさすぎて読みづらいので
// こっちで余白を入れる。どうして…

// コンストラクタ
WXDipswWindow::WXDipswWindow(wxWindow *parent)
	: inherited(parent, wxID_ANY, "DIPSW")
{
	int ndipsw;
	wxString name[2];

	swdev = GetDipswDevice();
	ndipsw = swdev->GetSize();

	if (ndipsw > 8) {
		name[0] = "DIPSW 1";
		name[1] = "DIPSW 2";
	}

	auto *topsizer = new wxBoxSizer(wxVERTICAL);

	// DIPSW 用コントロールを個数分用意する (ここではまだ配置はしない)
	for (int i = 0; i < ndipsw; i++) {
		auto *ctl = new WXScrollBar(this, ID_DIPSW(i), wxSB_VERTICAL);
		ctl->SetScrollParam(0, 1, 2, 1);
		swctl.push_back(ctl);

		// スクロールイベントをこっちに接続
		ctl->Bind(NONO_EVT_SCROLL, &WXDipswWindow::OnScroll, this);
	}

	// DIPSW ボックス
	for (int n = 0; n < ndipsw / 8; n++) {
		if (n > 0) {
			topsizer->AddSpacer(3);
		}

		wxBoxSizer *swbox;
		if (ndipsw > 8) {
			// LUNA では 2組あるので StaticBoxSizer で名前を表示
			swbox = new wxStaticBoxSizer(wxHORIZONTAL, this, name[n]);
		} else {
			// 1組で名前を表示するのは冗長なので枠なし
			swbox = new wxBoxSizer(wxHORIZONTAL);
		}
		// デフォルトの StaticBoxSizer の枠と外のマージンがなさすぎる…
		topsizer->Add(swbox, 1, wxEXPAND | wxALL, 3);

		// デフォルトの StaticBoxSizer の枠と内のマージンがなさすぎる…
		swbox->AddSpacer(3);

		for (int i = 0; i < 8; i++) {
			auto *vbox = new wxBoxSizer(wxVERTICAL);
			vbox->AddSpacer(3);
			vbox->Add(new wxStaticText(this, wxID_ANY,
				string_format("%d", i + 1)), 0, wxALIGN_CENTER_HORIZONTAL);
			vbox->Add(swctl[n * 8 + i], 0, wxALIGN_CENTER_HORIZONTAL);
			vbox->AddSpacer(3);

			swbox->Add(vbox, 1, wxALIGN_CENTER);
		}

		swbox->AddSpacer(3);
	}

	SetSizer(topsizer);
	FontChanged();

	// 現在値を反映
	for (int i = 0; i < ndipsw; i++) {
		uint pos = swdev->GetPosition(i);
		swctl[i]->SetThumbPosition(pos);
	}
}

// デストラクタ
WXDipswWindow::~WXDipswWindow()
{
}

// フォントサイズ変更
void
WXDipswWindow::FontChanged()
{
	// 子コントロールに先に反映させる
	inherited::FontChanged();

	// スクロールバーの幅はフォントサイズで変わるので、
	// 高さをそれに連動させる。
	wxSize size = swctl[0]->GetSize();
	size.y = size.x * 5 / 3;
	for (auto *ctl : swctl) {
		ctl->SetSize(size);
		ctl->SetMinSize(size);
	}

	Fit();
}

// スクロール位置変更通知
void
WXDipswWindow::OnScroll(wxScrollEvent& event)
{
	uint n = event.GetId() - ID_DIPSW0;
	uint pos = event.GetPosition();

	swdev->SetPosition(n, pos);
}
