//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// スレッドを持つデバイス
//

#pragma once

#include "device.h"
#include "fixedqueue.h"
#include <mutex>
#include <thread>
#include <vector>

// スレッド情報。
struct ThreadInfo
{
	static constexpr uint Capacity = 60;
	using LoadQueue = FixedQueue<uint32, Capacity>;

	Object *obj {};
	// 表示するスレッド名
	std::string name {};
	// CPU 利用率を取得するための ID
	clockid_t clockid {};

	std::unique_ptr<LoadQueue> load {};
	uint64 last_clock {};

	ThreadInfo(Object *obj_, const std::string& name_, clockid_t clockid_)
		: obj(obj_), name(name_), clockid(clockid_)
	{
		load.reset(new LoadQueue());
	}
};

class ThreadBase
{
 protected:
	enum class AffinityClass {
		Light,	// 処理の軽いスレッド
		Heavy,	// 処理の重いスレッド
	};

	// このスレッドのアフィニティを設定する
	static void SetThreadAffinityHint(AffinityClass hint);

	// 排他実行用
	static std::mutex exlock;
};

class ThreadDevice : public Device, public ThreadBase
{
	using inherited = Device;

 public:
	explicit ThreadDevice(uint objid_);
	~ThreadDevice() override;

	// スレッドを開始する
	virtual bool StartThread();

	// スレッドに終了を要求し、その終了を待つ
	void TerminateThread();

 protected:
	// オブジェクト名とスレッド名を設定する。
	void SetName(const std::string& newname) override;

	// スレッド名を設定する
	void SetThreadName(const std::string& threadname_);

	// 開始されたスレッドでのエントリポイント
	void OnStart();

	// 派生先で用意するエントリポイント
	virtual void ThreadRun() = 0;

	// スレッドに終了を指示する
	virtual void Terminate() = 0;

	std::unique_ptr<std::thread> thread {};

	// スレッド開始同期用
	std::mutex thread_starter {};

	// スレッド名
	std::string threadname {};
};

//
// スレッドマネージャ
//
class ThreadManager : public Object
{
	using inherited = Object;
	friend class WXThreadMonitorPanel;

 public:
	ThreadManager();
	~ThreadManager() override;

	// スレッドリストに登録する。
	void RegistThread(Object *, const std::string& name);
	// スレッドリストから削除する。
	void UnregistThread(Object *);

	// パフォーマンス測定。
	void CalcPerf(uint64 rtime);

	// x86 CPU のアフィニティを調べる。
	static bool DetectCPUAffinity_x86(std::vector<bool>&);

 private:
	// スレッド一覧。
	std::vector<ThreadInfo> threads {};
	std::mutex threads_mutex {};

	// 前回のパフォーマンス測定時刻。
	uint64 last_rtime {};
};
