/*
 * ntesla - a program for designing Tesla Coils
 * 
 * Copyright (C) 1997  Steven A. Falco
 * 
 * Email contact: sfalco@worldnet.att.net
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define EXTERN extern
#include "protos.h"

/* Calculate inductance of spiral primary - attribution? */
double
spiralInd(double avgRad, double width, double numTurns)
{
	return (avgRad * avgRad * numTurns * numTurns)
		/ ((8.0 * avgRad) + (11.0 * width));
}

/* Build the screen image */
void
prigen(int mode)
{
	int iii;
	double induct, previnduct;
	double avg, width, outer;
	double needInduct;
	double tmpval;

	Row = 0;
S "         *Tubing: %s,  *Spacing: %s,  *Inner Diameter: %s\n",
   UnitOut((void *)&Tcparms.primary_thick, Tcparms.units ? TOINCH : TOCM, Ub0),
   UnitOut((void *)&Tcparms.primary_spacing, Tcparms.units ? TOINCH : TOCM, Ub1),
   UnitOut((void *)&Tcparms.primary_inner, Tcparms.units ? TOINCH : TOCM, Ub2));
S "\n");

	needInduct = 1e3 / (2.0 * PI * Tcparms.secondary_lfreq);
	needInduct *= needInduct;
	needInduct /= Tcparms.primary_cap;
S "We need %.3g uH to resonate at *Frequency of %g kHz with\n",
   needInduct, Tcparms.secondary_lfreq);
S "a %.3g uF *Capacitor.\n", Tcparms.primary_cap);
	if(mode) {
S "(Capacitor is determined by your transformer.  Frequency is determined by\n");
S "your secondary.  You can change these here to see what effect they have but\n");
S "you must go back to the other menus to make a permanent change.)\n");
S "\n");
	}

	for(iii = 3; iii <= MAXPRITURN; iii++) {
		width = ((iii - 1) * Tcparms.primary_spacing)
		       + (iii * Tcparms.primary_thick);
		avg = (width + Tcparms.primary_inner) / 2.0;
		previnduct = induct; /* save for 2 column output */
		induct = spiralInd(avg, width, iii);
		if(!(iii & 1)) {
S "   turn %2d, inductance = %8.3f uH      turn %2d, inductance = %8.3f uH\n",
   iii-1, previnduct, iii, induct);
		}

		if(induct >= needInduct) {
			if(iii & 1) {
S "   turn %2d, inductance = %8.3f uH\n", iii, induct);
			}
			break;
		}
	}
	if(iii > MAXPRITURN) iii = MAXPRITURN;
	Tcparms.primary_turns = iii;
	if(induct < needInduct) {
S "WARNING - CANNOT TUNE - you must adjust the secondary or main capacitor\n");
	}
	outer = width * 2.0 + Tcparms.primary_inner;
	avg = (outer + Tcparms.primary_inner) / 2.0;
   tmpval = PI * iii * avg;
S "Note: at turn %d, outer diameter = %s, tubing length = %s\n",
   Tcparms.primary_turns,
   UnitOut((void *)&outer, Tcparms.units ? TOINCH : TOCM, Ub0),
   UnitOut((void *)&tmpval, Tcparms.units ? TOFEET : TOMETER, Ub1));
}

/* Primary menu */
void
primary()
{
	/*
	 * <-------------outer------------>
	 *      <---ave--->
	 * |||||||||||          |||||||||||   <-- windings of spiral
	 * <--width--><--inner->
	 */

	char buf[BS];
	double tmpval;
	int iii;

	while(1) {
		prigen(1);

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up general instructions */
		general_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 't':
			UnitIn(atof(buf + 1), (void *)&Tcparms.primary_thick,
			  Tcparms.units ? TOINCH : TOCM);
		  	break;

		  case 's':
			UnitIn(atof(buf + 1), (void *)&Tcparms.primary_spacing,
			  Tcparms.units ? TOINCH : TOCM);
		  	break;

		  case 'i':
			UnitIn(atof(buf + 1), (void *)&tmpval,
			  Tcparms.units ? TOINCH : TOCM);
			if(tmpval > Tcparms.secondary_diam) {
				Tcparms.primary_inner = tmpval;
			} else {
				E "\007Primary inner diameter must be bigger than secondary diameter ");
			}
		  	break;

		  case 'c':
		  	Tcparms.primary_cap = atof(buf + 1);
		  	break;

		  case 'f':
		  	Tcparms.secondary_lfreq = atof(buf + 1);
		  	break;

		  case 'h':
		  	pri_help();
			break;

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}

void
pri_help()
{
	char buf[BS];
	int iii;

	while(1) {
		
		/* Build the screen image */
		Row = 0;
S "The primary is usually constructed of soft copper tubing wound in a flat\n");
S "spiral.  There is no insulation on the tubing; this allows moving the\n");
S "connections from turn-to-turn to tune the coil to resonance.  The tubing is\n");
S "often mounted to a plexiglass plate or other plexiglass supports.  It is\n");
S "spaced approximately 0.25\" to 0.50\" turn-to-turn to avoid shorts and arcing\n");
S "between the turns.\n");
S "\n");
S "The flat spiral primary is chosen over a helical (cylindrical) primary to\n");
S "avoid arcing from the secondary to the primary.  Sometimes a funnel or cone\n");
S "shaped primary is used for better coupling.\n");
S "\n");
S "The outer diameter of the primary should be roughly the same as the height\n");
S "of the secondary so the field from the primary envelops the secondary fully.\n");
S "You can make the primary smaller if you like; you will just lose some output.\n");
S "\n");

		/* Clear the screen, then display the menu */
		CLEAR_SCREEN; for(iii = 0; iii < Row; iii++) P Display[iii]);

		/* Put up help instructions */
		help_instructions();

		/* Get the users choice */
		if(fgets(buf, BS, stdin) == NULL) {
			/* We are done */
			break;
		}

		switch(tolower(buf[0])) {

		  case 'p':
		  	print_buf();
			break;

		  case 'q':
		  	save_parms();
			EXIT(0);

		  case 'r':
		  	return;

		  case '\n':
		  case '\r':
		  	break;

		  default:
			E "\007Sorry - %c is illegal - try again ", buf[0]);
			break;
		}
	}

	return;
}
