#include "oak.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* create and return a queue with the given action and fire.  it will have
 * an emtpy pending message list
 */
oak_queue *oak_queue_create(char *name, oak_list *actionlist, oak_fire *fire)
{
  oak_queue *out;

  out=malloc(sizeof(oak_queue));
  out->name=strdup(name);
  out->fire=fire;
  out->actionlist=actionlist;
  oak_list_create(&(out->entrylist));
  out->locktime=0;
  out->header=NULL;
  out->prescan=0;
  if (!strcmp(name, "trash")) {
    out->trashqueue=1;
    out->fire=oak_fire_create("*5s");
  } else {
    out->trashqueue=0;
  }
  return(out);
}

char *oak_queue_get_name(oak_queue *q)
{
  return(q->name);
}

int oak_queue_is_prescan(oak_queue *q)
{
  return(q->prescan);
}

void oak_queue_set_prescan(oak_queue *q)
{
  q->prescan=1;
}

/* return the number of unique pending messages */
int oak_queue_get_nummsgs(oak_queue *q)
{
  return(oak_list_get_size(&(q->entrylist)));
}

/* return the nth pendning message */
char *oak_queue_get_msg_n(oak_queue *q, int n)
{
  oak_queue_entry *qe;
  qe=oak_list_get_element(&(q->entrylist), n);
  return(oak_queue_entry_get_msg(qe));
}

/* return the host associcated with the nth pending message */
char *oak_queue_get_host_n(oak_queue *q, int n)
{
  oak_queue_entry *qe;
  qe=oak_list_get_element(&(q->entrylist), n);
  return(oak_queue_entry_get_host(qe));
}

/* get the reptition count for message n */
int oak_queue_get_msgcount_n(oak_queue *q, int n)
{
  oak_queue_entry *qe;
  qe=oak_list_get_element(&(q->entrylist), n);
  return(oak_queue_entry_get_repcount(qe));
}

/* set the header for when the queue is printed */
void oak_queue_set_header(oak_queue *q, char *header)
{
  q->header=strdup(header);
}

char *oak_queue_get_header(oak_queue *q)
{
  return(q->header);
}

int oak_queue_is_locking(oak_queue *q)
{
  if (q->locktime == 0) {
    return(0);
  }
  return(1);
}

void oak_queue_set_locktime_fromstring(oak_queue *q, char *in)
{
  int foo;
  
  foo=atoi(in);
  if (!strcasecmp(in+strlen(in)-1, "h") ||
      !strcasecmp(in+strlen(in)-2, "hr")) {
    foo=foo*3600;
  } else if (!strcasecmp(in+strlen(in)-1, "m") ||
	     !strcasecmp(in+strlen(in)-3, "min")) {
    foo=foo*60;
  }
  q->locktime=foo;
}

int oak_queue_get_locktime(oak_queue *q)
{
  return(q->locktime);
}

/* add 'msg' to the list of pending messages.  If an identical message
 * is already there just up the count.  Remember to extract the host
 * and do the right thing with it.
 */
void oak_queue_add_msg(oak_queue *q, char *msg, char *host)
{
  int i, j, found;
  oak_list *entrylist;
  oak_queue_entry *qe;

  if (q->trashqueue) {
    if (oak_global_is_debug(&g)) printf("DEBUG: trashqueue tossing message\n");
    return;
  }

  entrylist=&(q->entrylist);
  j=oak_list_get_size(entrylist);

  if (oak_global_is_debug(&g)) printf("DEBUG: trying to add line %s to queue %s\n", msg, q->name);

  /* check if we have the msg already, if so just up the count */
  found=0;
  for (i=0; i<j; i++) {
    qe=oak_list_get_element(entrylist, i);
    if (oak_queue_entry_matches(qe, host, msg)) {
      oak_queue_entry_up_repcount(qe);
      if (oak_global_is_debug(&g)) printf("DEBUG: Found duplicate message for queue %s, repcount now %i\n",
					  q->name, oak_queue_entry_get_repcount(qe));
      found=1;
      break;
    }
  }

  /* if we didn't find it, create a new queue_entry for it */
  if (!found) {
    if (oak_global_is_debug(&g)) printf("DEBUG: Found new message for queue %s\n", q->name);
    qe=malloc(sizeof(oak_queue_entry));
    oak_queue_entry_create(qe, host, msg);
    oak_list_append_element(entrylist, qe);
  }
}

/* delete all messages associated with q */
void oak_queue_delete_msgs(oak_queue *q)
{
  oak_list *entrylist;
  oak_queue_entry *qe;
  int i, j;

  if (q->trashqueue) {
    return;
  }

  entrylist=&(q->entrylist);
  j=oak_list_get_size(entrylist);
  for (i=0; i<j; i++) {
    qe=oak_list_get_element(entrylist, i);
    if (qe) {
      oak_queue_entry_free(qe);
      oak_list_remove_element(entrylist, i);
      free(qe);
    }
  }
  oak_list_free_simple(entrylist);
  oak_list_create(entrylist);
}

/* return the fire associcated with the queue */
oak_fire *oak_queue_get_fire(oak_queue *q)
{
  return(q->fire);
}

/* return the action associcated with the queue */
oak_list *oak_queue_get_actionlist(oak_queue *q)
{
  return(q->actionlist);
}

/* the the fire associated with the queue */
void oak_queue_set_fire(oak_queue *q, oak_fire *fire)
{
  if (q->fire) free(q->fire);
  q->fire=fire;
}

/* set the action associcated with the queue */
void oak_queue_set_actionlist(oak_queue *q, oak_list *actionlist)
{
  if (q->actionlist) free(q->actionlist); /* not good enough */
  q->actionlist=actionlist;
}

/* return a string representation fo the queue, where:
 *   The number of lines will be truncated to 'lines'
 *   The length of all lines will be truncated to 'linelen'
 *   The number of hosts will be truncated after 'numhosts'
 *   The number of messages per host will be limited to 'hostents'
 * But if any of these values is '0' then there is no limit.
 * The caller must free the return.
 */
char *oak_queue_to_string(oak_queue *q, int lines, int linelen, int numhosts, int hostents)
{
  int i, j, found, x, y, linecounter;
  oak_list *entrylist;
  oak_queue_entry *qe;
  oak_list hostlist;
  oak_string oakstr;
  char *out, buff[1024], *host;

  entrylist=&(q->entrylist);
  j=oak_list_get_size(entrylist);
  oak_string_create_with_size(&oakstr, 5280);
  linecounter=0;

  /* print the header, if we have one */
  if (q->header) {
    oak_string_strcat(&oakstr, q->header);
    oak_string_strcat(&oakstr, "\n\n");
    linecounter+=2;
  }

  /* first create a list of hostname strings found in the queue */
  oak_list_create(&hostlist);
  for (i=0; i<j; i++) {
    qe=oak_list_get_element(entrylist, i);

    /* is the host for this element already on the list? */
    found=0;
    y=oak_list_get_size(&hostlist);
    for (x=0; x<y; x++) {
      if (!strcasecmp(oak_list_get_element(&hostlist, x), oak_queue_entry_get_host(qe))) {
	found=1;
	break;
      }
    }

    /* if not, add it.  Note that hostlist entries are pointing into
     * the queue_entry data.
     */
    if (!found) {
      oak_list_append_element(&hostlist, oak_queue_entry_get_host(qe));
    }
  }    

  /* now go through each host in the hostlist and add its messages */
  y=oak_list_get_size(&hostlist);
  for (x=0; x<y; x++) {
    /* stop if we've passed the hostlimit */
    if (numhosts && (x>numhosts)) {
      sprintf(buff, "\n ** %i hosts, limiting at %i **\n", y, numhosts);
      oak_string_strcat(&oakstr, buff);
      linecounter+=2;
      break;
    }

    /* print the host header (and save the host name) */
    host=oak_list_get_element(&hostlist, x);
    if (x!=0) {
      oak_string_strcat(&oakstr, "\n");
      linecounter++;
    }
    oak_string_strcat(&oakstr, host);
    oak_string_strcat(&oakstr, ":\n");
    linecounter++;
    
    /* stop if we've passed the line limit */
    if (lines && (linecounter>lines-2)) {
      oak_string_strcat(&oakstr, "\n ** Too many lines, message has been truncated **\n");
      linecounter+=2;
      break;
    }
    
    /* loop through each queue entry */
    for (i=0; i<j; i++) {
      qe=oak_list_get_element(entrylist, i);

      /* if it's not for this host, keep going */
      if (strcasecmp(host, oak_queue_entry_get_host(qe))) continue;

      /* stop if we've passed the limit of entries per host */
      if (hostents && (i>hostents)) {
	sprintf(buff, "   ** %i messages found for host, limiting at %i **\n", j, hostents);
	oak_string_strcat(&oakstr, buff);
	linecounter+=2;
	break;
      }
      
      /* stop if we've passed the line limit */
      if (lines && (linecounter>lines-2)) {
	oak_string_strcat(&oakstr, "\n ** Too many lines, message has been truncated **\n");
	linecounter+=2;
	break;
      }

      /* write the message count */
      if (!q->locktime) {
	sprintf(buff, "   %4i: ", oak_queue_entry_get_repcount(qe));
	oak_string_strcat(&oakstr, buff);
      } else {
	oak_string_strcat(&oakstr, "   ");
      }

      if (linelen && (strlen(oak_queue_entry_get_msg(qe)) > linelen-6)) {
	char *foo;
	foo=strdup(oak_queue_entry_get_msg(qe));
	foo[linelen-6]='$';
	foo[linelen-5]='\0';
	oak_string_strcat(&oakstr, foo);
	free(foo);
      } else {
	oak_string_strcat(&oakstr, oak_queue_entry_get_msg(qe));
      }
      oak_string_strcat(&oakstr, "\n");
      linecounter++;
    }
    
  }
  out=strdup(oak_string_to_str(&oakstr));
  oak_string_free(&oakstr);
  return(out);
}
