/*
	OBPager.h for OBPager, a pager dockapp designed to work with OpenBox or any netwm-compliant window manager.
	
	Copyright (c) 2004 - Roy Wood
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	See the COPYING file for a copy of the GNU General Public License.
*/


#ifndef __OBPAGER__
#define __OBPAGER__


// Include the STL stuff we need

#include <iostream>
#include <vector>
#include <string>


// And include the Xlib stuff we need

#include <Xlib.h>
#include <Xutil.h>
#include <Xos.h>
#include <Xatom.h>
#include <shape.h>


// Also include the custom Xlib "helper" classes we use to manage ownership of X resources, etc.

#include "XHelperClasses.h"


// During development, it's nice to generate a lot of debug output

//#define __DEBUG__



// 
class OBPager
{
	public:
		
		// Our public interface....
		
		
		/// Default constructor
		OBPager(int desktopNum = 0, const char* activeDesktopFont = NULL, const char* inactiveDesktopFont = NULL, int fd = 0, bool forceSync = false);
		
		/// Destructor
		~OBPager();
		
		/// Connect to a specified X server display, or the default display
		void connectToXServer(char *displayName = NULL);
		
		/// Create and show the X window
		void createShowWindow();
		
		/// Run forever....
		void run();
		
		/// Method to get number of desktops
		long numDesktops();
	
	
	private:
		
		// The Xlib error handler we install
		static int errorHandler(Display* display, XErrorEvent* errEvent);
		
		// The OBPager the error handler is to work with
		static OBPager* sActiveOBPager;
		
		
		// Various private methods we use to get our work done....
		
		/// Set the current action description (used during debugging)
		void setCurrentAction(const char* actionDescription, Window window = 0);
		
		/// Get the value of an atom/property as a long
		long getAtomAsLong(Atom xAtom, Atom reqType = XA_CARDINAL, Window window = 0) const;
		
		/// Get the values of an atom/property as a vector<long>
		void getAtomAsLongs(Atom xAtom, std::vector<long> &longs, Atom reqType = XA_CARDINAL, Window window = 0) const;
		
		/// Get the values of an atom/property as a vector<string>
		void getAtomAsStrings(Atom xAtom, std::vector<std::string> &strings, Atom reqType = XA_STRING, Window window = 0) const;
		
		
		
		/// Get an xbm bitmask of a rectangular area, with ones in the center and zeroes around the edges
		unsigned char* getRectangularXBMMask(unsigned int width, unsigned int height, unsigned int indent) const;
		
		
		
		/// Get the pixel value of a named colour, using a specified ColorMap
		unsigned long getColourPixel(const char *colourName, const Colormap colorMap) const;
		
		
		
		/// Redraw the dockapp window
		void redrawWindow();
		
		
		/// Switch to a different virtual desktop
		void gotoDesktop(int newDesktop) const;
		
		
		
		/// The pixel values are all stored in an array, and these are the symbolic indices used to get them out
		enum { 
			kAppletBorderShadowIndex = 0,
			
			kActiveDesktopUnfocusedWindowIndex = 1,
			kActiveDesktopFocusedWindowIndex = 2,
			kActiveDesktopFocusedBorderIndex = 3,
			kActiveDesktopUnfocusedBorderIndex = 4,
			
			kInactiveDesktopWindowIndex = 5,
			kInactiveDesktopWindowBorderIndex = 6,
			
			kInactiveDesktopFontColor = 7,
			kActiveDesktopFontColor = 8,
			
			kNumColours = 9
		};
		
		
		
	private:
		
		/// The virtual desktop number we monitor
		int mDesktopNum;
		
		/// The names of the fonts we use to display the desktop number
		std::string mActiveDesktopFont;
		std::string mInactiveDesktopFont;
		
		
		/// The X Display we are connected to
		XDisplayKeeper mDisplay;
		
		/// The X screen number we are connected to (not the desktop!)
		int mScreenNum;
		
		/// Our X window
		XWindowKeeper mWindow;
		
		// A bunch of atoms/properties we are interested in
		Atom mAtom_UTF8_STRING;
		Atom mAtom_NET_CURRENT_DESKTOP;
		Atom mAtom_NET_NUMBER_OF_DESKTOPS;
		Atom mAtom_NET_SUPPORTED;
		Atom mAtom_NET_CLIENT_LIST_STACKING;
		Atom mAtom_NET_WMDESKTOP;
		Atom mAtom_NET_ACTIVE_WINDOW;
		Atom mAtom_NET_DESKTOP_GEOMETRY;
		Atom mAtom_NET_SUPPORTING_WM_CHECK;
		Atom mAtom_NET_WM_NAME;
		Atom mAtom_WM_NAME;
		
		/// The fonts we use to draw text
		XFontStructKeeper mFontStructSmall;
		XFontStructKeeper mFontStructLarge;
		
		/// The X GC graphics context we use for drawing
		XGCKeeper mGC;
		
		/// The X Pixmap we use to do our offscreen drawing
		XPixmapKeeper mPixmap;
		
		/// The pixel colours we use for drawing
		unsigned long mPixelColours[kNumColours];
		
		/// Force synchronized Xlib activity?  (useful for debugging)
		bool mForceSync;
		
		/// Received first expose event
		bool mExposed;
		
		/// A textual description of our current activity (used by errorHandler)
		std::string mCurrentAction;
		
		/// The file descriptor of the pipe we write when we have received our first expose event
		int mFD;
};

#endif
