/**
 * SPDX-FileCopyrightText: (C) 2010 Dominik Seichter <domseichter@web.de>
 * SPDX-FileCopyrightText: (C) 2021 Francesco Pretto <ceztko@gmail.com>
 * SPDX-License-Identifier: LGPL-2.0-or-later
 */

#include "PdfDeclarationsPrivate.h"
#include "PdfStandard14FontData.h"
#include <podofo/main/PdfFontMetricsStandard14.h>

using namespace std;
using namespace PoDoFo;

constexpr const char* FONT_TIMES_ROMAN_STD = "Times-Roman";
constexpr const char* FONT_TIMES_ITALIC_STD = "Times-Italic";
constexpr const char* FONT_TIMES_BOLD_STD = "Times-Bold";
constexpr const char* FONT_TIMES_BOLD_ITALIC_STD = "Times-BoldItalic";
constexpr const char* FONT_HELVETICA_STD = "Helvetica";
constexpr const char* FONT_HELVETICA_OBLIQUE_STD =  "Helvetica-Oblique";
constexpr const char* FONT_HELVETICA_BOLD_STD = "Helvetica-Bold";
constexpr const char* FONT_HELVETICA_BOLD_OBLIQUE_STD = "Helvetica-BoldOblique";
constexpr const char* FONT_COURIER_STD = "Courier";
constexpr const char* FONT_COURIER_OBLIQUE_STD = "Courier-Oblique";
constexpr const char* FONT_COURIER_BOLD_STD = "Courier-Bold";
constexpr const char* FONT_COURIER_BOLD_OBLIQUE_STD = "Courier-BoldOblique";
constexpr const char* FONT_SYMBOL_STD = "Symbol";
constexpr const char* FONT_ZAPF_DINGBATS_STD = "ZapfDingbats";

constexpr const char* FONT_TIMES_ROMAN_ALT = "TimesNewRoman";
constexpr const char* FONT_TIMES_ITALIC_ALT = "TimesNewRoman,Italic";
constexpr const char* FONT_TIMES_BOLD_ALT = "TimesNewRoman,Bold";
constexpr const char* FONT_TIMES_BOLD_ITALIC_ALT = "TimesNewRoman,BoldItalic";
constexpr const char* FONT_HELVETICA_ALT = "Arial";
constexpr const char* FONT_HELVETICA_OBLIQUE_ALT = "Arial,Italic";
constexpr const char* FONT_HELVETICA_BOLD_ALT = "Arial,Bold";
constexpr const char* FONT_HELVETICA_BOLD_OBLIQUE_ALT = "Arial,BoldItalic";
constexpr const char* FONT_COURIER_ALT = "CourierNew";
constexpr const char* FONT_COURIER_OBLIQUE_ALT = "CourierNew,Italic";
constexpr const char* FONT_COURIER_BOLD_ALT = "CourierNew,Bold";
constexpr const char* FONT_COURIER_BOLD_OBLIQUE_ALT = "CourierNew,BoldItalic";

constexpr const char* TIMES_ROMAN_BASE_NAME = "Times";
constexpr const char* HELVETICA_BASE_NAME = "Helvetica";
constexpr const char* COURIER_BASE_NAME = "Courier";

constexpr const char* TIMES_ROMAN_FAMILY_NAME = "Times New Roman";
constexpr const char* HELVETICA_FAMILY_NAME = "Arial";
constexpr const char* COURIER_FAMILY_NAME = "Courier Std";

enum class PdfStandard14FontFamily
{
    Unknown = 0,
    Times,
    Helvetica,
    Courier,
    Symbol,
    ZapfDingbats,
};

// The widths and character codes data were obtained
// directly from Foxit fonts

static const unsigned short CHAR_DATA_TIMES_ROMAN[] = {
    250,
    250,
    333,
    408,
    500,
    500,
    833,
    778,
    180,
    333,
    333,
    500,
    564,
    250,
    333,
    250,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    278,
    278,
    564,
    564,
    564,
    444,
    921,
    722,
    667,
    667,
    722,
    611,
    556,
    722,
    722,
    333,
    389,
    722,
    611,
    889,
    722,
    722,
    556,
    722,
    667,
    556,
    611,
    722,
    722,
    944,
    722,
    722,
    611,
    333,
    278,
    333,
    469,
    500,
    333,
    444,
    500,
    444,
    500,
    444,
    333,
    500,
    500,
    278,
    278,
    500,
    278,
    778,
    500,
    500,
    500,
    500,
    333,
    389,
    278,
    500,
    500,
    722,
    500,
    500,
    444,
    480,
    200,
    480,
    541,
    333,
    500,
    500,
    500,
    500,
    200,
    500,
    333,
    760,
    276,
    500,
    564,
    760,
    333,
    400,
    564,
    300,
    300,
    333,
    500,
    453,
    250,
    333,
    300,
    310,
    500,
    750,
    750,
    750,
    444,
    722,
    722,
    722,
    722,
    722,
    722,
    889,
    667,
    611,
    611,
    611,
    611,
    333,
    333,
    333,
    333,
    722,
    722,
    722,
    722,
    722,
    722,
    722,
    564,
    722,
    722,
    722,
    722,
    722,
    722,
    556,
    500,
    444,
    444,
    444,
    444,
    444,
    444,
    667,
    444,
    444,
    444,
    444,
    444,
    278,
    278,
    278,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    564,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    278,
    611,
    278,
    889,
    722,
    556,
    389,
    722,
    611,
    444,
    500,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    500,
    1000,
    333,
    333,
    333,
    444,
    444,
    444,
    500,
    500,
    350,
    1000,
    1000,
    333,
    333,
    167,
    722,
    980,
    564,
    556,
    556,
};

static const unsigned short CHAR_DATA_TIMES_ITALIC[] = {
    250,
    250,
    333,
    420,
    500,
    500,
    833,
    778,
    214,
    333,
    333,
    500,
    675,
    250,
    333,
    250,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    333,
    333,
    675,
    675,
    675,
    500,
    920,
    611,
    611,
    667,
    722,
    611,
    611,
    722,
    722,
    333,
    444,
    667,
    556,
    833,
    667,
    722,
    611,
    722,
    611,
    500,
    556,
    722,
    611,
    833,
    611,
    556,
    556,
    389,
    278,
    389,
    422,
    500,
    333,
    500,
    500,
    444,
    500,
    444,
    278,
    500,
    500,
    278,
    278,
    444,
    278,
    722,
    500,
    500,
    500,
    500,
    389,
    389,
    278,
    500,
    444,
    667,
    444,
    444,
    389,
    400,
    275,
    400,
    541,
    389,
    500,
    500,
    500,
    500,
    275,
    500,
    333,
    760,
    276,
    500,
    675,
    760,
    333,
    400,
    675,
    300,
    300,
    333,
    500,
    523,
    250,
    333,
    300,
    310,
    500,
    750,
    750,
    750,
    500,
    611,
    611,
    611,
    611,
    611,
    611,
    889,
    667,
    611,
    611,
    611,
    611,
    333,
    333,
    333,
    333,
    722,
    667,
    722,
    722,
    722,
    722,
    722,
    675,
    722,
    722,
    722,
    722,
    722,
    556,
    611,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    667,
    444,
    444,
    444,
    444,
    444,
    278,
    278,
    278,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    675,
    500,
    500,
    500,
    500,
    500,
    444,
    500,
    444,
    278,
    556,
    278,
    944,
    667,
    500,
    389,
    556,
    556,
    389,
    500,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    500,
    889,
    333,
    333,
    333,
    556,
    556,
    556,
    500,
    500,
    350,
    889,
    1000,
    333,
    333,
    167,
    500,
    980,
    675,
    500,
    500,
};

static const unsigned short CHAR_DATA_TIMES_BOLD[] = {
    250,
    250,
    333,
    555,
    500,
    500,
    1000,
    833,
    278,
    333,
    333,
    500,
    570,
    250,
    333,
    250,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    333,
    333,
    570,
    570,
    570,
    500,
    930,
    722,
    667,
    722,
    722,
    667,
    611,
    778,
    778,
    389,
    500,
    778,
    667,
    944,
    722,
    778,
    611,
    778,
    722,
    556,
    667,
    722,
    722,
    1000,
    722,
    722,
    667,
    333,
    278,
    333,
    581,
    500,
    333,
    500,
    556,
    444,
    556,
    444,
    333,
    500,
    556,
    278,
    333,
    556,
    278,
    833,
    556,
    500,
    556,
    556,
    444,
    389,
    333,
    556,
    500,
    722,
    500,
    500,
    444,
    394,
    220,
    394,
    520,
    333,
    500,
    500,
    500,
    500,
    220,
    500,
    333,
    747,
    300,
    500,
    570,
    747,
    333,
    400,
    570,
    300,
    300,
    333,
    556,
    540,
    250,
    333,
    300,
    330,
    500,
    750,
    750,
    750,
    500,
    722,
    722,
    722,
    722,
    722,
    722,
    1000,
    722,
    667,
    667,
    667,
    667,
    389,
    389,
    389,
    389,
    722,
    722,
    778,
    778,
    778,
    778,
    778,
    570,
    778,
    722,
    722,
    722,
    722,
    722,
    611,
    556,
    500,
    500,
    500,
    500,
    500,
    500,
    722,
    444,
    444,
    444,
    444,
    444,
    278,
    278,
    278,
    278,
    500,
    556,
    500,
    500,
    500,
    500,
    500,
    570,
    500,
    556,
    556,
    556,
    556,
    500,
    556,
    500,
    278,
    667,
    278,
    1000,
    722,
    556,
    389,
    722,
    667,
    444,
    500,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    500,
    1000,
    333,
    333,
    333,
    500,
    500,
    500,
    500,
    500,
    350,
    1000,
    1000,
    333,
    333,
    167,
    722,
    1000,
    570,
    556,
    556,
};

static const unsigned short CHAR_DATA_TIMES_BOLD_ITALIC[] = {
    250,
    250,
    389,
    555,
    500,
    500,
    833,
    778,
    278,
    333,
    333,
    500,
    570,
    250,
    333,
    250,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    333,
    333,
    570,
    570,
    570,
    500,
    832,
    667,
    667,
    667,
    722,
    667,
    667,
    722,
    778,
    389,
    500,
    667,
    611,
    889,
    722,
    722,
    611,
    722,
    667,
    556,
    611,
    722,
    667,
    889,
    667,
    611,
    611,
    333,
    278,
    333,
    570,
    500,
    333,
    500,
    500,
    444,
    500,
    444,
    333,
    500,
    556,
    278,
    278,
    500,
    278,
    778,
    556,
    500,
    500,
    500,
    389,
    389,
    278,
    556,
    444,
    667,
    500,
    444,
    389,
    348,
    220,
    348,
    570,
    389,
    500,
    500,
    500,
    500,
    220,
    500,
    333,
    747,
    266,
    500,
    606,
    747,
    333,
    400,
    570,
    300,
    300,
    333,
    576,
    500,
    250,
    333,
    300,
    300,
    500,
    750,
    750,
    750,
    500,
    667,
    667,
    667,
    667,
    667,
    667,
    944,
    667,
    667,
    667,
    667,
    667,
    389,
    389,
    389,
    389,
    722,
    722,
    722,
    722,
    722,
    722,
    722,
    570,
    722,
    722,
    722,
    722,
    722,
    611,
    611,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    722,
    444,
    444,
    444,
    444,
    444,
    278,
    278,
    278,
    278,
    500,
    556,
    500,
    500,
    500,
    500,
    500,
    570,
    500,
    556,
    556,
    556,
    556,
    444,
    500,
    444,
    278,
    611,
    278,
    944,
    722,
    556,
    389,
    611,
    611,
    389,
    500,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    333,
    500,
    1000,
    333,
    333,
    333,
    500,
    500,
    500,
    500,
    500,
    350,
    1000,
    1000,
    333,
    333,
    167,
    667,
    1000,
    606,
    556,
    556,
};

static const unsigned short CHAR_DATA_HELVETICA[] = {
    278,
    667,
    1000,
    667,
    667,
    667,
    667,
    667,
    667,
    667,
    722,
    722,
    722,
    667,
    667,
    667,
    667,
    667,
    722,
    667,
    611,
    778,
    722,
    278,
    278,
    278,
    278,
    278,
    500,
    667,
    556,
    556,
    833,
    722,
    722,
    778,
    1000,
    778,
    778,
    778,
    778,
    778,
    778,
    667,
    778,
    722,
    667,
    667,
    611,
    667,
    722,
    722,
    722,
    722,
    722,
    667,
    944,
    667,
    667,
    667,
    667,
    611,
    611,
    556,
    556,
    556,
    333,
    556,
    889,
    556,
    667,
    556,
    469,
    584,
    389,
    1015,
    556,
    556,
    278,
    260,
    334,
    334,
    278,
    278,
    333,
    260,
    350,
    500,
    333,
    500,
    333,
    556,
    333,
    278,
    278,
    737,
    556,
    556,
    556,
    556,
    400,
    333,
    584,
    556,
    333,
    278,
    556,
    556,
    556,
    556,
    556,
    556,
    1000,
    1000,
    556,
    584,
    556,
    278,
    333,
    278,
    500,
    556,
    500,
    556,
    556,
    167,
    556,
    611,
    333,
    584,
    612,
    612,
    333,
    366,
    556,
    333,
    333,
    222,
    278,
    278,
    278,
    278,
    222,
    500,
    222,
    584,
    584,
    222,
    833,
    333,
    584,
    556,
    584,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    944,
    333,
    556,
    556,
    834,
    834,
    333,
    370,
    365,
    611,
    556,
    556,
    537,
    333,
    333,
    889,
    278,
    278,
    1000,
    584,
    584,
    556,
    556,
    611,
    355,
    333,
    333,
    333,
    222,
    222,
    222,
    191,
    333,
    737,
    333,
    500,
    500,
    556,
    278,
    556,
    556,
    278,
    278,
    556,
    278,
    556,
    556,
    834,
    333,
    333,
    1000,
    556,
    333,
    556,
    556,
    556,
    556,
    556,
    556,
    500,
    722,
    500,
    500,
    500,
    500,
    556,
    500,
    500,
    556,
};

static const unsigned short CHAR_DATA_HELVETICA_OBLIQUE[] = {
    278,
    667,
    1000,
    667,
    667,
    667,
    667,
    667,
    667,
    667,
    722,
    722,
    722,
    667,
    667,
    667,
    667,
    667,
    722,
    667,
    611,
    778,
    722,
    278,
    278,
    278,
    278,
    278,
    500,
    667,
    556,
    556,
    833,
    722,
    722,
    778,
    1000,
    778,
    778,
    778,
    778,
    778,
    778,
    667,
    778,
    722,
    667,
    667,
    611,
    667,
    722,
    722,
    722,
    722,
    722,
    667,
    944,
    667,
    667,
    667,
    667,
    611,
    611,
    556,
    556,
    556,
    333,
    556,
    889,
    556,
    667,
    556,
    469,
    584,
    389,
    1116,
    556,
    556,
    278,
    260,
    334,
    334,
    278,
    278,
    333,
    260,
    350,
    500,
    333,
    500,
    333,
    556,
    333,
    278,
    278,
    737,
    556,
    556,
    556,
    556,
    400,
    333,
    584,
    556,
    333,
    278,
    556,
    556,
    556,
    556,
    556,
    556,
    1000,
    1000,
    556,
    584,
    556,
    278,
    333,
    278,
    500,
    556,
    500,
    556,
    556,
    167,
    556,
    611,
    333,
    584,
    556,
    556,
    350,
    333,
    556,
    333,
    333,
    222,
    278,
    278,
    278,
    278,
    222,
    500,
    222,
    584,
    584,
    222,
    833,
    333,
    584,
    556,
    584,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    944,
    333,
    556,
    556,
    834,
    834,
    333,
    370,
    365,
    611,
    556,
    556,
    537,
    333,
    333,
    889,
    278,
    278,
    1000,
    584,
    584,
    556,
    556,
    611,
    355,
    333,
    333,
    333,
    222,
    222,
    222,
    191,
    333,
    737,
    333,
    500,
    500,
    556,
    278,
    556,
    556,
    278,
    278,
    556,
    278,
    556,
    556,
    834,
    333,
    333,
    1000,
    556,
    333,
    556,
    556,
    556,
    556,
    556,
    556,
    500,
    722,
    500,
    500,
    500,
    500,
    556,
    500,
    500,
    556,
};

static const unsigned short CHAR_DATA_HELVETICA_BOLD[] = {
    278,
    278,
    333,
    333,
    333,
    333,
    333,
    333,
    278,
    167,
    611,
    611,
    611,
    278,
    611,
    500,
    333,
    474,
    556,
    556,
    889,
    722,
    238,
    333,
    333,
    389,
    584,
    278,
    333,
    278,
    278,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    556,
    333,
    333,
    584,
    584,
    584,
    611,
    1072,
    722,
    722,
    722,
    722,
    667,
    611,
    778,
    722,
    278,
    556,
    722,
    611,
    833,
    722,
    778,
    667,
    778,
    722,
    667,
    611,
    722,
    667,
    944,
    667,
    667,
    611,
    333,
    278,
    333,
    584,
    556,
    333,
    556,
    611,
    556,
    611,
    556,
    333,
    611,
    611,
    278,
    278,
    556,
    278,
    889,
    611,
    611,
    611,
    611,
    389,
    556,
    333,
    611,
    556,
    778,
    556,
    556,
    500,
    389,
    280,
    389,
    584,
    722,
    278,
    556,
    500,
    1000,
    556,
    556,
    333,
    1000,
    667,
    333,
    1000,
    278,
    278,
    500,
    500,
    350,
    556,
    1000,
    333,
    1000,
    556,
    333,
    944,
    667,
    333,
    556,
    556,
    556,
    556,
    280,
    556,
    333,
    737,
    370,
    556,
    584,
    584,
    737,
    333,
    400,
    584,
    333,
    333,
    333,
    611,
    556,
    278,
    333,
    333,
    365,
    556,
    834,
    834,
    834,
    611,
    722,
    722,
    722,
    722,
    722,
    722,
    1000,
    722,
    667,
    667,
    667,
    667,
    278,
    278,
    278,
    278,
    722,
    722,
    778,
    778,
    778,
    778,
    778,
    584,
    778,
    722,
    722,
    722,
    722,
    667,
    667,
    611,
    556,
    556,
    556,
    556,
    556,
    556,
    889,
    556,
    556,
    556,
    556,
    556,
    278,
    278,
    278,
    278,
    611,
    611,
    611,
    611,
    611,
    611,
    611,
    584,
    611,
    611,
    611,
    611,
    611,
    556,
    611,
    556,
};

static const unsigned short CHAR_DATA_HELVETICA_BOLD_OBLIQUE[] = {
    278,
    278,
    333,
    333,
    333,
    333,
    333,
    333,
    278,
    167,
    611,
    611,
    611,
    278,
    611,
    500,
    333,
    474,
    556,
    556,
    889,
    722,
    238,
    333,
    333,
    389,
    584,
    278,
    333,
    278,
    278,
    556,
    556,
    556,
    556,
    556,
    528,
    556,
    556,
    556,
    556,
    333,
    333,
    584,
    584,
    584,
    611,
    1072,
    722,
    722,
    722,
    722,
    667,
    611,
    778,
    722,
    278,
    556,
    722,
    611,
    833,
    722,
    778,
    667,
    778,
    722,
    667,
    611,
    722,
    667,
    944,
    667,
    667,
    611,
    333,
    278,
    333,
    584,
    556,
    333,
    556,
    611,
    556,
    611,
    556,
    333,
    611,
    611,
    278,
    278,
    556,
    278,
    889,
    611,
    611,
    611,
    611,
    389,
    556,
    333,
    611,
    556,
    778,
    556,
    556,
    500,
    389,
    280,
    389,
    584,
    722,
    278,
    556,
    500,
    1000,
    556,
    556,
    333,
    1000,
    667,
    333,
    1000,
    278,
    278,
    500,
    500,
    350,
    556,
    1000,
    333,
    1000,
    556,
    333,
    944,
    667,
    333,
    556,
    556,
    556,
    556,
    280,
    556,
    333,
    737,
    370,
    556,
    584,
    584,
    737,
    333,
    400,
    584,
    333,
    333,
    333,
    611,
    556,
    278,
    333,
    333,
    365,
    556,
    834,
    834,
    834,
    611,
    722,
    722,
    722,
    722,
    722,
    722,
    1000,
    722,
    667,
    667,
    667,
    667,
    278,
    278,
    278,
    278,
    722,
    722,
    778,
    778,
    778,
    778,
    778,
    584,
    778,
    722,
    722,
    722,
    722,
    667,
    667,
    611,
    556,
    556,
    556,
    556,
    556,
    556,
    889,
    556,
    556,
    556,
    556,
    556,
    278,
    278,
    278,
    278,
    611,
    611,
    611,
    611,
    611,
    611,
    611,
    584,
    519,
    611,
    611,
    611,
    611,
    556,
    611,
    556,
};

static const unsigned short CHAR_DATA_COURIER[] = {
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
};

static const unsigned short CHAR_DATA_COURIER_OBLIQUE[] = {
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
};

static const unsigned short CHAR_DATA_COURIER_BOLD[] = {
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
};

static const unsigned short CHAR_DATA_COURIER_BOLD_OBLIQUE[] = {
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
    600,
};

static const unsigned short CHAR_DATA_SYMBOL[] = {
    250,
    250,
    333,
    500,
    833,
    778,
    333,
    333,
    549,
    250,
    250,
    278,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    500,
    278,
    278,
    549,
    549,
    549,
    444,
    333,
    333,
    500,
    480,
    200,
    480,
    713,
    400,
    549,
    576,
    549,
    549,
    500,
    722,
    667,
    603,
    611,
    611,
    722,
    741,
    333,
    722,
    686,
    889,
    722,
    645,
    722,
    768,
    556,
    592,
    611,
    690,
    763,
    722,
    795,
    631,
    549,
    411,
    494,
    439,
    494,
    603,
    521,
    329,
    549,
    549,
    521,
    493,
    549,
    549,
    549,
    439,
    603,
    439,
    576,
    521,
    549,
    686,
    686,
    631,
    620,
    603,
    713,
    460,
    1000,
    247,
    411,
    167,
    750,
    686,
    987,
    795,
    768,
    823,
    987,
    603,
    987,
    603,
    1042,
    658,
    987,
    603,
    987,
    603,
    1042,
    713,
    494,
    549,
    823,
    612,
    713,
    713,
    713,
    439,
    823,
    713,
    549,
    500,
    549,
    713,
    713,
    768,
    603,
    603,
    768,
    768,
    274,
    863,
    549,
    549,
    549,
    549,
    549,
    549,
    549,
    713,
    713,
    713,
    713,
    713,
    768,
    768,
    658,
    250,
    686,
    686,
    329,
    329,
    494,
    753,
    753,
    753,
    753,
    790,
    790,
    890,
    500,
    603,
    1000,
    790,
    790,
    786,
    384,
    384,
    384,
    384,
    384,
    384,
    494,
    494,
    494,
    494,
    686,
    384,
    384,
    384,
    384,
    384,
    384,
    494,
    494,
    494,
    790,
};

static const unsigned short CHAR_DATA_ZAPF_DINGBATS[] = {
    278,
    974,
    692,
    668,
    732,
    544,
    544,
    910,
    911,
    667,
    760,
    760,
    626,
    960,
    694,
    595,
    776,
    690,
    791,
    790,
    939,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    549,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    855,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    911,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    788,
    933,
    894,
    838,
    924,
    1016,
    458,
    924,
    918,
    927,
    928,
    928,
    945,
    834,
    873,
    828,
    924,
    917,
    930,
    931,
    463,
    883,
    836,
    974,
    867,
    696,
    874,
    760,
    946,
    865,
    967,
    831,
    873,
    927,
    755,
    970,
    918,
    748,
    836,
    771,
    888,
    748,
    771,
    888,
    867,
    961,
    846,
    696,
    874,
    974,
    762,
    759,
    509,
    410,
    762,
    761,
    571,
    677,
    763,
    760,
    759,
    754,
    786,
    980,
    788,
    788,
    790,
    793,
    794,
    816,
    823,
    789,
    841,
    823,
    719,
    833,
    816,
    831,
    923,
    744,
    723,
    749,
    790,
    792,
    695,
    789,
    776,
    768,
    792,
    759,
    707,
    708,
    682,
    701,
    826,
    815,
    494,
    789,
    789,
    707,
    687,
    696,
    689,
    786,
    787,
    713,
    791,
    552,
    785,
    791,
    873,
    761,
    762,
    759,
    892,
    892,
    788,
    784,
    537,
    438,
    138,
    277,
    415,
    509,
    410,
    234,
    234,
    390,
    577,
    390,
    276,
    276,
    317,
    317,
    334,
    334,
    392,
    392,
    668,
    278,
};

string_view PoDoFo::GetStandard14FontName(PdfStandard14FontType stdFont)
{
    switch (stdFont)
    {
        case PdfStandard14FontType::TimesRoman:
            return FONT_TIMES_ROMAN_STD;
        case PdfStandard14FontType::TimesItalic:
            return FONT_TIMES_ITALIC_STD;
        case PdfStandard14FontType::TimesBold:
            return FONT_TIMES_BOLD_STD;
        case PdfStandard14FontType::TimesBoldItalic:
            return FONT_TIMES_BOLD_ITALIC_STD;
        case PdfStandard14FontType::Helvetica:
            return FONT_HELVETICA_STD;
        case PdfStandard14FontType::HelveticaOblique:
            return FONT_HELVETICA_OBLIQUE_STD;
        case PdfStandard14FontType::HelveticaBold:
            return FONT_HELVETICA_BOLD_STD;
        case PdfStandard14FontType::HelveticaBoldOblique:
            return FONT_HELVETICA_BOLD_OBLIQUE_STD;
        case PdfStandard14FontType::Courier:
            return FONT_COURIER_STD;
        case PdfStandard14FontType::CourierOblique:
            return FONT_COURIER_OBLIQUE_STD;
        case PdfStandard14FontType::CourierBold:
            return FONT_COURIER_BOLD_STD;
        case PdfStandard14FontType::CourierBoldOblique:
            return FONT_COURIER_BOLD_OBLIQUE_STD;
        case PdfStandard14FontType::Symbol:
            return FONT_SYMBOL_STD;
        case PdfStandard14FontType::ZapfDingbats:
            return FONT_ZAPF_DINGBATS_STD;
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR(PdfErrorCode::InvalidEnumValue);
    }
}

string_view PoDoFo::GetStandard14FontFamilyName(PdfStandard14FontType stdFont)
{
    switch (stdFont)
    {
        case PdfStandard14FontType::TimesRoman:
        case PdfStandard14FontType::TimesItalic:
        case PdfStandard14FontType::TimesBold:
        case PdfStandard14FontType::TimesBoldItalic:
            return TIMES_ROMAN_FAMILY_NAME;
        case PdfStandard14FontType::Helvetica:
        case PdfStandard14FontType::HelveticaOblique:
        case PdfStandard14FontType::HelveticaBold:
        case PdfStandard14FontType::HelveticaBoldOblique:
            return HELVETICA_FAMILY_NAME;
        case PdfStandard14FontType::Courier:
        case PdfStandard14FontType::CourierOblique:
        case PdfStandard14FontType::CourierBold:
        case PdfStandard14FontType::CourierBoldOblique:
            return COURIER_FAMILY_NAME;
        case PdfStandard14FontType::Symbol:
        case PdfStandard14FontType::ZapfDingbats:
            return { }; // There's no font family name for Symbol and ZapfDingbats
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR(PdfErrorCode::InvalidEnumValue);
    }
}

// NOTE: Adobe strictly checks for Standard14 names + alternative
// names. No other combinations are possible
bool PoDoFo::IsStandard14Font(const string_view& fontName, bool useAltNames, PdfStandard14FontType& stdFont)
{
    if (fontName == FONT_TIMES_ROMAN_STD)
    {
        stdFont = PdfStandard14FontType::TimesRoman;
        return true;
    }
    else if (fontName == FONT_TIMES_ITALIC_STD)
    {
        stdFont = PdfStandard14FontType::TimesItalic;
        return true;
    }
    else if (fontName == FONT_TIMES_BOLD_STD)
    {
        stdFont = PdfStandard14FontType::TimesBold;
        return true;
    }
    else if (fontName == FONT_TIMES_BOLD_ITALIC_STD)
    {
        stdFont = PdfStandard14FontType::TimesBoldItalic;
        return true;
    }
    else if (fontName == FONT_HELVETICA_STD)
    {
        stdFont = PdfStandard14FontType::Helvetica;
        return true;
    }
    else if (fontName == FONT_HELVETICA_OBLIQUE_STD)
    {
        stdFont = PdfStandard14FontType::HelveticaOblique;
        return true;
    }
    else if (fontName == FONT_HELVETICA_BOLD_STD)
    {
        stdFont = PdfStandard14FontType::HelveticaBold;
        return true;
    }
    else if (fontName == FONT_HELVETICA_BOLD_OBLIQUE_STD)
    {
        stdFont = PdfStandard14FontType::HelveticaBoldOblique;
        return true;
    }
    else if (fontName == FONT_COURIER_STD)
    {
        stdFont = PdfStandard14FontType::Courier;
        return true;
    }
    else if (fontName == FONT_COURIER_OBLIQUE_STD)
    {
        stdFont = PdfStandard14FontType::CourierOblique;
        return true;
    }
    else if (fontName == FONT_COURIER_BOLD_STD)
    {
        stdFont = PdfStandard14FontType::CourierBold;
        return true;
    }
    else if (fontName == FONT_COURIER_BOLD_OBLIQUE_STD)
    {
        stdFont = PdfStandard14FontType::CourierBoldOblique;
        return true;
    }
    else if (fontName == FONT_SYMBOL_STD)
    {
        stdFont = PdfStandard14FontType::Symbol;
        return true;
    }
    else if (fontName == FONT_ZAPF_DINGBATS_STD)
    {
        stdFont = PdfStandard14FontType::ZapfDingbats;
        return true;
    }

    if (useAltNames)
    {
        if (fontName == FONT_TIMES_ROMAN_ALT)
        {
            stdFont = PdfStandard14FontType::TimesRoman;
            return true;
        }
        else if (fontName == FONT_TIMES_ITALIC_ALT)
        {
            stdFont = PdfStandard14FontType::TimesItalic;
            return true;
        }
        else if (fontName == FONT_TIMES_BOLD_ALT)
        {
            stdFont = PdfStandard14FontType::TimesBold;
            return true;
        }
        else if (fontName == FONT_TIMES_BOLD_ITALIC_ALT)
        {
            stdFont = PdfStandard14FontType::TimesBoldItalic;
            return true;
        }
        else if (fontName == FONT_HELVETICA_ALT)
        {
            stdFont = PdfStandard14FontType::Helvetica;
            return true;
        }
        else if (fontName == FONT_HELVETICA_OBLIQUE_ALT)
        {
            stdFont = PdfStandard14FontType::HelveticaOblique;
            return true;
        }
        else if (fontName == FONT_HELVETICA_BOLD_ALT)
        {
            stdFont = PdfStandard14FontType::HelveticaBold;
            return true;
        }
        else if (fontName == FONT_HELVETICA_BOLD_OBLIQUE_ALT)
        {
            stdFont = PdfStandard14FontType::HelveticaBoldOblique;
            return true;
        }
        else if (fontName == FONT_COURIER_ALT)
        {
            stdFont = PdfStandard14FontType::Courier;
            return true;
        }
        else if (fontName == FONT_COURIER_OBLIQUE_ALT)
        {
            stdFont = PdfStandard14FontType::CourierOblique;
            return true;
        }
        else if (fontName == FONT_COURIER_BOLD_ALT)
        {
            stdFont = PdfStandard14FontType::CourierBold;
            return true;
        }
        else if (fontName == FONT_COURIER_BOLD_OBLIQUE_ALT)
        {
            stdFont = PdfStandard14FontType::CourierBoldOblique;
            return true;
        }
    }

    stdFont = PdfStandard14FontType::Unknown;
    return false;
}

const unsigned short* PoDoFo::GetStd14FontWidths(PdfStandard14FontType stdFont, unsigned& size)
{
    switch (stdFont)
    {
        case PdfStandard14FontType::TimesRoman:
            size = (unsigned)std::size(CHAR_DATA_TIMES_ROMAN);
            return CHAR_DATA_TIMES_ROMAN;
        case PdfStandard14FontType::TimesItalic:
            size = (unsigned)std::size(CHAR_DATA_TIMES_ITALIC);
            return CHAR_DATA_TIMES_ITALIC;
        case PdfStandard14FontType::TimesBold:
            size = (unsigned)std::size(CHAR_DATA_TIMES_BOLD);
            return CHAR_DATA_TIMES_BOLD;
        case PdfStandard14FontType::TimesBoldItalic:
            size = (unsigned)std::size(CHAR_DATA_TIMES_BOLD_ITALIC);
            return CHAR_DATA_TIMES_BOLD_ITALIC;
        case PdfStandard14FontType::Helvetica:
            size = (unsigned)std::size(CHAR_DATA_HELVETICA);
            return CHAR_DATA_HELVETICA;
        case PdfStandard14FontType::HelveticaOblique:
            size = (unsigned)std::size(CHAR_DATA_HELVETICA_OBLIQUE);
            return CHAR_DATA_HELVETICA_OBLIQUE;
        case PdfStandard14FontType::HelveticaBold:
            size = (unsigned)std::size(CHAR_DATA_HELVETICA_BOLD);
            return CHAR_DATA_HELVETICA_BOLD;
        case PdfStandard14FontType::HelveticaBoldOblique:
            size = (unsigned)std::size(CHAR_DATA_HELVETICA_BOLD_OBLIQUE);
            return CHAR_DATA_HELVETICA_BOLD_OBLIQUE;
        case PdfStandard14FontType::Courier:
            size = (unsigned)std::size(CHAR_DATA_COURIER);
            return CHAR_DATA_COURIER;
        case PdfStandard14FontType::CourierOblique:
            size = (unsigned)std::size(CHAR_DATA_COURIER_OBLIQUE);
            return CHAR_DATA_COURIER_OBLIQUE;
        case PdfStandard14FontType::CourierBold:
            size = (unsigned)std::size(CHAR_DATA_COURIER_BOLD);
            return CHAR_DATA_COURIER_BOLD;
        case PdfStandard14FontType::CourierBoldOblique:
            size = (unsigned)std::size(CHAR_DATA_COURIER_BOLD_OBLIQUE);
            return CHAR_DATA_COURIER_BOLD_OBLIQUE;
        case PdfStandard14FontType::Symbol:
            size = (unsigned)std::size(CHAR_DATA_SYMBOL);
            return CHAR_DATA_SYMBOL;
        case PdfStandard14FontType::ZapfDingbats:
            size = (unsigned)std::size(CHAR_DATA_ZAPF_DINGBATS);
            return CHAR_DATA_ZAPF_DINGBATS;
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR(PdfErrorCode::InvalidEnumValue);
    }
}

static const Std14CPToGIDMap& GetTimesRomanMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 159 },
        { 0xE2, 160 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x131, 190 },
        { 0x141, 191 },
        { 0x142, 192 },
        { 0x152, 193 },
        { 0x153, 194 },
        { 0x160, 195 },
        { 0x161, 196 },
        { 0x178, 197 },
        { 0x17D, 198 },
        { 0x17E, 199 },
        { 0x192, 200 },
        { 0x2C6, 201 },
        { 0x2C7, 202 },
        { 0x2C9, 109 },
        { 0x2D8, 203 },
        { 0x2D9, 204 },
        { 0x2DA, 205 },
        { 0x2DB, 206 },
        { 0x2DC, 207 },
        { 0x2DD, 208 },
        { 0x3BC, 115 },
        { 0x2013, 209 },
        { 0x2014, 210 },
        { 0x2018, 211 },
        { 0x2019, 212 },
        { 0x201A, 213 },
        { 0x201C, 214 },
        { 0x201D, 215 },
        { 0x201E, 216 },
        { 0x2020, 217 },
        { 0x2021, 218 },
        { 0x2022, 219 },
        { 0x2026, 220 },
        { 0x2030, 221 },
        { 0x2039, 222 },
        { 0x203A, 223 },
        { 0x2044, 224 },
        { 0x20AC, 225 },
        { 0x2122, 226 },
        { 0x2212, 227 },
        { 0x2215, 224 },
        { 0x2219, 117 },
        { 0xFB01, 228 },
        { 0xFB02, 229 },
    };

    return map;
}

static const Std14CPToGIDMap& GetTimesItalicMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 159 },
        { 0xE2, 160 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x131, 190 },
        { 0x141, 191 },
        { 0x142, 192 },
        { 0x152, 193 },
        { 0x153, 194 },
        { 0x160, 195 },
        { 0x161, 196 },
        { 0x178, 197 },
        { 0x17D, 198 },
        { 0x17E, 199 },
        { 0x192, 200 },
        { 0x2C6, 201 },
        { 0x2C7, 202 },
        { 0x2C9, 109 },
        { 0x2D8, 203 },
        { 0x2D9, 204 },
        { 0x2DA, 205 },
        { 0x2DB, 206 },
        { 0x2DC, 207 },
        { 0x2DD, 208 },
        { 0x3BC, 115 },
        { 0x2013, 209 },
        { 0x2014, 210 },
        { 0x2018, 211 },
        { 0x2019, 212 },
        { 0x201A, 213 },
        { 0x201C, 214 },
        { 0x201D, 215 },
        { 0x201E, 216 },
        { 0x2020, 217 },
        { 0x2021, 218 },
        { 0x2022, 219 },
        { 0x2026, 220 },
        { 0x2030, 221 },
        { 0x2039, 222 },
        { 0x203A, 223 },
        { 0x2044, 224 },
        { 0x20AC, 225 },
        { 0x2122, 226 },
        { 0x2212, 227 },
        { 0x2215, 224 },
        { 0x2219, 117 },
        { 0xFB01, 228 },
        { 0xFB02, 229 },
    };
    return map;
}

static const Std14CPToGIDMap& GetTimesBoldMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 159 },
        { 0xE2, 160 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x131, 190 },
        { 0x141, 191 },
        { 0x142, 192 },
        { 0x152, 193 },
        { 0x153, 194 },
        { 0x160, 195 },
        { 0x161, 196 },
        { 0x178, 197 },
        { 0x17D, 198 },
        { 0x17E, 199 },
        { 0x192, 200 },
        { 0x2C6, 201 },
        { 0x2C7, 202 },
        { 0x2C9, 109 },
        { 0x2D8, 203 },
        { 0x2D9, 204 },
        { 0x2DA, 205 },
        { 0x2DB, 206 },
        { 0x2DC, 207 },
        { 0x2DD, 208 },
        { 0x3BC, 115 },
        { 0x2013, 209 },
        { 0x2014, 210 },
        { 0x2018, 211 },
        { 0x2019, 212 },
        { 0x201A, 213 },
        { 0x201C, 214 },
        { 0x201D, 215 },
        { 0x201E, 216 },
        { 0x2020, 217 },
        { 0x2021, 218 },
        { 0x2022, 219 },
        { 0x2026, 220 },
        { 0x2030, 221 },
        { 0x2039, 222 },
        { 0x203A, 223 },
        { 0x2044, 224 },
        { 0x20AC, 225 },
        { 0x2122, 226 },
        { 0x2212, 227 },
        { 0x2215, 224 },
        { 0x2219, 117 },
        { 0xFB01, 228 },
        { 0xFB02, 229 },
    };
    return map;
}

static const Std14CPToGIDMap& GetTimesBoldItalicMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 159 },
        { 0xE2, 160 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x131, 190 },
        { 0x141, 191 },
        { 0x142, 192 },
        { 0x152, 193 },
        { 0x153, 194 },
        { 0x160, 195 },
        { 0x161, 196 },
        { 0x178, 197 },
        { 0x17D, 198 },
        { 0x17E, 199 },
        { 0x192, 200 },
        { 0x2C6, 201 },
        { 0x2C7, 202 },
        { 0x2C9, 109 },
        { 0x2D8, 203 },
        { 0x2D9, 204 },
        { 0x2DA, 205 },
        { 0x2DB, 206 },
        { 0x2DC, 207 },
        { 0x2DD, 208 },
        { 0x3BC, 115 },
        { 0x2013, 209 },
        { 0x2014, 210 },
        { 0x2018, 211 },
        { 0x2019, 212 },
        { 0x201A, 213 },
        { 0x201C, 214 },
        { 0x201D, 215 },
        { 0x201E, 216 },
        { 0x2020, 217 },
        { 0x2021, 218 },
        { 0x2022, 219 },
        { 0x2026, 220 },
        { 0x2030, 221 },
        { 0x2039, 222 },
        { 0x203A, 223 },
        { 0x2044, 224 },
        { 0x20AC, 225 },
        { 0x2122, 226 },
        { 0x2212, 227 },
        { 0x2215, 224 },
        { 0x2219, 117 },
        { 0xFB01, 228 },
        { 0xFB02, 229 },
    };
    return map;
}

static const Std14CPToGIDMap& GetHelveticaMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 203 },
        { 0x21, 117 },
        { 0x22, 185 },
        { 0x23, 156 },
        { 0x24, 103 },
        { 0x25, 176 },
        { 0x26, 70 },
        { 0x27, 192 },
        { 0x28, 174 },
        { 0x29, 175 },
        { 0x2A, 74 },
        { 0x2B, 180 },
        { 0x2C, 94 },
        { 0x2D, 136 },
        { 0x2E, 177 },
        { 0x2F, 202 },
        { 0x30, 229 },
        { 0x31, 164 },
        { 0x32, 212 },
        { 0x33, 207 },
        { 0x34, 124 },
        { 0x35, 121 },
        { 0x36, 201 },
        { 0x37, 200 },
        { 0x38, 111 },
        { 0x39, 154 },
        { 0x3A, 93 },
        { 0x3B, 199 },
        { 0x3C, 145 },
        { 0x3D, 115 },
        { 0x3E, 129 },
        { 0x3F, 183 },
        { 0x40, 75 },
        { 0x41, 1 },
        { 0x42, 9 },
        { 0x43, 10 },
        { 0x44, 12 },
        { 0x45, 13 },
        { 0x46, 20 },
        { 0x47, 21 },
        { 0x48, 22 },
        { 0x49, 23 },
        { 0x4A, 28 },
        { 0x4B, 29 },
        { 0x4C, 30 },
        { 0x4D, 32 },
        { 0x4E, 33 },
        { 0x4F, 35 },
        { 0x50, 43 },
        { 0x51, 44 },
        { 0x52, 45 },
        { 0x53, 46 },
        { 0x54, 48 },
        { 0x55, 50 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 61 },
        { 0x5B, 82 },
        { 0x5C, 78 },
        { 0x5D, 83 },
        { 0x5E, 72 },
        { 0x5F, 219 },
        { 0x60, 128 },
        { 0x61, 63 },
        { 0x62, 77 },
        { 0x63, 87 },
        { 0x64, 97 },
        { 0x65, 106 },
        { 0x66, 119 },
        { 0x67, 126 },
        { 0x68, 134 },
        { 0x69, 137 },
        { 0x6A, 142 },
        { 0x6B, 143 },
        { 0x6C, 144 },
        { 0x6D, 148 },
        { 0x6E, 153 },
        { 0x6F, 157 },
        { 0x70, 172 },
        { 0x71, 182 },
        { 0x72, 193 },
        { 0x73, 196 },
        { 0x74, 205 },
        { 0x75, 214 },
        { 0x76, 220 },
        { 0x77, 221 },
        { 0x78, 222 },
        { 0x79, 223 },
        { 0x7A, 227 },
        { 0x7B, 80 },
        { 0x7C, 79 },
        { 0x7D, 81 },
        { 0x7E, 73 },
        { 0xA0, 203 },
        { 0xA1, 118 },
        { 0xA2, 91 },
        { 0xA3, 204 },
        { 0xA4, 96 },
        { 0xA5, 226 },
        { 0xA6, 85 },
        { 0xA7, 198 },
        { 0xA8, 101 },
        { 0xA9, 95 },
        { 0xAA, 168 },
        { 0xAB, 130 },
        { 0xAC, 146 },
        { 0xAD, 136 },
        { 0xAE, 194 },
        { 0xAF, 149 },
        { 0xB0, 100 },
        { 0xB1, 181 },
        { 0xB2, 213 },
        { 0xB3, 209 },
        { 0xB4, 66 },
        { 0xB5, 151 },
        { 0xB6, 173 },
        { 0xB7, 178 },
        { 0xB8, 90 },
        { 0xB9, 167 },
        { 0xBA, 169 },
        { 0xBB, 131 },
        { 0xBC, 166 },
        { 0xBD, 165 },
        { 0xBE, 208 },
        { 0xBF, 184 },
        { 0xC0, 6 },
        { 0xC1, 3 },
        { 0xC2, 4 },
        { 0xC3, 8 },
        { 0xC4, 5 },
        { 0xC5, 7 },
        { 0xC6, 2 },
        { 0xC7, 11 },
        { 0xC8, 17 },
        { 0xC9, 14 },
        { 0xCA, 15 },
        { 0xCB, 16 },
        { 0xCC, 27 },
        { 0xCD, 24 },
        { 0xCE, 25 },
        { 0xCF, 26 },
        { 0xD0, 18 },
        { 0xD1, 34 },
        { 0xD2, 40 },
        { 0xD3, 37 },
        { 0xD4, 38 },
        { 0xD5, 42 },
        { 0xD6, 39 },
        { 0xD7, 152 },
        { 0xD8, 41 },
        { 0xD9, 54 },
        { 0xDA, 51 },
        { 0xDB, 52 },
        { 0xDC, 53 },
        { 0xDD, 59 },
        { 0xDE, 49 },
        { 0xDF, 127 },
        { 0xE0, 69 },
        { 0xE1, 64 },
        { 0xE2, 65 },
        { 0xE3, 76 },
        { 0xE4, 67 },
        { 0xE5, 71 },
        { 0xE6, 68 },
        { 0xE7, 89 },
        { 0xE8, 110 },
        { 0xE9, 107 },
        { 0xEA, 108 },
        { 0xEB, 109 },
        { 0xEC, 141 },
        { 0xED, 138 },
        { 0xEE, 139 },
        { 0xEF, 140 },
        { 0xF0, 116 },
        { 0xF1, 155 },
        { 0xF2, 163 },
        { 0xF3, 158 },
        { 0xF4, 159 },
        { 0xF5, 171 },
        { 0xF6, 160 },
        { 0xF7, 102 },
        { 0xF8, 170 },
        { 0xF9, 218 },
        { 0xFA, 215 },
        { 0xFB, 216 },
        { 0xFC, 217 },
        { 0xFD, 224 },
        { 0xFE, 206 },
        { 0xFF, 225 },
        { 0x131, 105 },
        { 0x141, 31 },
        { 0x142, 147 },
        { 0x152, 36 },
        { 0x153, 161 },
        { 0x160, 47 },
        { 0x161, 197 },
        { 0x178, 60 },
        { 0x17D, 62 },
        { 0x17E, 228 },
        { 0x192, 123 },
        { 0x2C6, 92 },
        { 0x2C7, 88 },
        { 0x2C9, 149 },
        { 0x2D8, 84 },
        { 0x2D9, 104 },
        { 0x2DA, 195 },
        { 0x2DB, 162 },
        { 0x2DC, 210 },
        { 0x2DD, 135 },
        { 0x3BC, 151 },
        { 0x2013, 114 },
        { 0x2014, 113 },
        { 0x2018, 189 },
        { 0x2019, 190 },
        { 0x201A, 191 },
        { 0x201C, 187 },
        { 0x201D, 188 },
        { 0x201E, 186 },
        { 0x2020, 98 },
        { 0x2021, 99 },
        { 0x2022, 86 },
        { 0x2026, 112 },
        { 0x2030, 179 },
        { 0x2039, 132 },
        { 0x203A, 133 },
        { 0x2044, 125 },
        { 0x20AC, 19 },
        { 0x2122, 211 },
        { 0x2212, 150 },
        { 0x2215, 125 },
        { 0x2219, 178 },
        { 0xFB01, 120 },
        { 0xFB02, 122 },
    };
    return map;
}

static const Std14CPToGIDMap& GetHelveticaObliqueMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 203 },
        { 0x21, 117 },
        { 0x22, 185 },
        { 0x23, 156 },
        { 0x24, 103 },
        { 0x25, 176 },
        { 0x26, 70 },
        { 0x27, 192 },
        { 0x28, 174 },
        { 0x29, 175 },
        { 0x2A, 74 },
        { 0x2B, 180 },
        { 0x2C, 94 },
        { 0x2D, 136 },
        { 0x2E, 177 },
        { 0x2F, 202 },
        { 0x30, 229 },
        { 0x31, 164 },
        { 0x32, 212 },
        { 0x33, 207 },
        { 0x34, 124 },
        { 0x35, 121 },
        { 0x36, 201 },
        { 0x37, 200 },
        { 0x38, 111 },
        { 0x39, 154 },
        { 0x3A, 93 },
        { 0x3B, 199 },
        { 0x3C, 145 },
        { 0x3D, 115 },
        { 0x3E, 129 },
        { 0x3F, 183 },
        { 0x40, 75 },
        { 0x41, 1 },
        { 0x42, 9 },
        { 0x43, 10 },
        { 0x44, 12 },
        { 0x45, 13 },
        { 0x46, 20 },
        { 0x47, 21 },
        { 0x48, 22 },
        { 0x49, 23 },
        { 0x4A, 28 },
        { 0x4B, 29 },
        { 0x4C, 30 },
        { 0x4D, 32 },
        { 0x4E, 33 },
        { 0x4F, 35 },
        { 0x50, 43 },
        { 0x51, 44 },
        { 0x52, 45 },
        { 0x53, 46 },
        { 0x54, 48 },
        { 0x55, 50 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 61 },
        { 0x5B, 82 },
        { 0x5C, 78 },
        { 0x5D, 83 },
        { 0x5E, 72 },
        { 0x5F, 219 },
        { 0x60, 128 },
        { 0x61, 63 },
        { 0x62, 77 },
        { 0x63, 87 },
        { 0x64, 97 },
        { 0x65, 106 },
        { 0x66, 119 },
        { 0x67, 126 },
        { 0x68, 134 },
        { 0x69, 137 },
        { 0x6A, 142 },
        { 0x6B, 143 },
        { 0x6C, 144 },
        { 0x6D, 148 },
        { 0x6E, 153 },
        { 0x6F, 157 },
        { 0x70, 172 },
        { 0x71, 182 },
        { 0x72, 193 },
        { 0x73, 196 },
        { 0x74, 205 },
        { 0x75, 214 },
        { 0x76, 220 },
        { 0x77, 221 },
        { 0x78, 222 },
        { 0x79, 223 },
        { 0x7A, 227 },
        { 0x7B, 80 },
        { 0x7C, 79 },
        { 0x7D, 81 },
        { 0x7E, 73 },
        { 0xA0, 203 },
        { 0xA1, 118 },
        { 0xA2, 91 },
        { 0xA3, 204 },
        { 0xA4, 96 },
        { 0xA5, 226 },
        { 0xA6, 85 },
        { 0xA7, 198 },
        { 0xA8, 101 },
        { 0xA9, 95 },
        { 0xAA, 168 },
        { 0xAB, 130 },
        { 0xAC, 146 },
        { 0xAD, 136 },
        { 0xAE, 194 },
        { 0xAF, 149 },
        { 0xB0, 100 },
        { 0xB1, 181 },
        { 0xB2, 213 },
        { 0xB3, 209 },
        { 0xB4, 66 },
        { 0xB5, 151 },
        { 0xB6, 173 },
        { 0xB7, 178 },
        { 0xB8, 90 },
        { 0xB9, 167 },
        { 0xBA, 169 },
        { 0xBB, 131 },
        { 0xBC, 166 },
        { 0xBD, 165 },
        { 0xBE, 208 },
        { 0xBF, 184 },
        { 0xC0, 6 },
        { 0xC1, 3 },
        { 0xC2, 4 },
        { 0xC3, 8 },
        { 0xC4, 5 },
        { 0xC5, 7 },
        { 0xC6, 2 },
        { 0xC7, 11 },
        { 0xC8, 17 },
        { 0xC9, 14 },
        { 0xCA, 15 },
        { 0xCB, 16 },
        { 0xCC, 27 },
        { 0xCD, 24 },
        { 0xCE, 25 },
        { 0xCF, 26 },
        { 0xD0, 18 },
        { 0xD1, 34 },
        { 0xD2, 40 },
        { 0xD3, 37 },
        { 0xD4, 38 },
        { 0xD5, 42 },
        { 0xD6, 39 },
        { 0xD7, 152 },
        { 0xD8, 41 },
        { 0xD9, 54 },
        { 0xDA, 51 },
        { 0xDB, 52 },
        { 0xDC, 53 },
        { 0xDD, 59 },
        { 0xDE, 49 },
        { 0xDF, 127 },
        { 0xE0, 69 },
        { 0xE1, 64 },
        { 0xE2, 65 },
        { 0xE3, 76 },
        { 0xE4, 67 },
        { 0xE5, 71 },
        { 0xE6, 68 },
        { 0xE7, 89 },
        { 0xE8, 110 },
        { 0xE9, 107 },
        { 0xEA, 108 },
        { 0xEB, 109 },
        { 0xEC, 141 },
        { 0xED, 138 },
        { 0xEE, 139 },
        { 0xEF, 140 },
        { 0xF0, 116 },
        { 0xF1, 155 },
        { 0xF2, 163 },
        { 0xF3, 158 },
        { 0xF4, 159 },
        { 0xF5, 171 },
        { 0xF6, 160 },
        { 0xF7, 102 },
        { 0xF8, 170 },
        { 0xF9, 218 },
        { 0xFA, 215 },
        { 0xFB, 216 },
        { 0xFC, 217 },
        { 0xFD, 224 },
        { 0xFE, 206 },
        { 0xFF, 225 },
        { 0x131, 105 },
        { 0x141, 31 },
        { 0x142, 147 },
        { 0x152, 36 },
        { 0x153, 161 },
        { 0x160, 47 },
        { 0x161, 197 },
        { 0x178, 60 },
        { 0x17D, 62 },
        { 0x17E, 228 },
        { 0x192, 123 },
        { 0x2C6, 92 },
        { 0x2C7, 88 },
        { 0x2C9, 149 },
        { 0x2D8, 84 },
        { 0x2D9, 104 },
        { 0x2DA, 195 },
        { 0x2DB, 162 },
        { 0x2DC, 210 },
        { 0x2DD, 135 },
        { 0x3BC, 151 },
        { 0x2013, 114 },
        { 0x2014, 113 },
        { 0x2018, 189 },
        { 0x2019, 190 },
        { 0x201A, 191 },
        { 0x201C, 187 },
        { 0x201D, 188 },
        { 0x201E, 186 },
        { 0x2020, 98 },
        { 0x2021, 99 },
        { 0x2022, 86 },
        { 0x2026, 112 },
        { 0x2030, 179 },
        { 0x2039, 132 },
        { 0x203A, 133 },
        { 0x2044, 125 },
        { 0x20AC, 19 },
        { 0x2122, 211 },
        { 0x2212, 150 },
        { 0x2215, 125 },
        { 0x2219, 178 },
        { 0xFB01, 120 },
        { 0xFB02, 122 },
    };
    return map;
}

static const Std14CPToGIDMap& GetHelveticaBoldMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 16 },
        { 0x22, 17 },
        { 0x23, 18 },
        { 0x24, 19 },
        { 0x25, 20 },
        { 0x26, 21 },
        { 0x27, 22 },
        { 0x28, 23 },
        { 0x29, 24 },
        { 0x2A, 25 },
        { 0x2B, 26 },
        { 0x2C, 27 },
        { 0x2D, 28 },
        { 0x2E, 29 },
        { 0x2F, 30 },
        { 0x30, 31 },
        { 0x31, 32 },
        { 0x32, 33 },
        { 0x33, 34 },
        { 0x34, 35 },
        { 0x35, 36 },
        { 0x36, 37 },
        { 0x37, 38 },
        { 0x38, 39 },
        { 0x39, 40 },
        { 0x3A, 41 },
        { 0x3B, 42 },
        { 0x3C, 43 },
        { 0x3D, 44 },
        { 0x3E, 45 },
        { 0x3F, 46 },
        { 0x40, 47 },
        { 0x41, 48 },
        { 0x42, 49 },
        { 0x43, 50 },
        { 0x44, 51 },
        { 0x45, 52 },
        { 0x46, 53 },
        { 0x47, 54 },
        { 0x48, 55 },
        { 0x49, 56 },
        { 0x4A, 57 },
        { 0x4B, 58 },
        { 0x4C, 59 },
        { 0x4D, 60 },
        { 0x4E, 61 },
        { 0x4F, 62 },
        { 0x50, 63 },
        { 0x51, 64 },
        { 0x52, 65 },
        { 0x53, 66 },
        { 0x54, 67 },
        { 0x55, 68 },
        { 0x56, 69 },
        { 0x57, 70 },
        { 0x58, 71 },
        { 0x59, 72 },
        { 0x5A, 73 },
        { 0x5B, 74 },
        { 0x5C, 75 },
        { 0x5D, 76 },
        { 0x5E, 77 },
        { 0x5F, 78 },
        { 0x60, 79 },
        { 0x61, 80 },
        { 0x62, 81 },
        { 0x63, 82 },
        { 0x64, 83 },
        { 0x65, 84 },
        { 0x66, 85 },
        { 0x67, 86 },
        { 0x68, 87 },
        { 0x69, 88 },
        { 0x6A, 89 },
        { 0x6B, 90 },
        { 0x6C, 91 },
        { 0x6D, 92 },
        { 0x6E, 93 },
        { 0x6F, 94 },
        { 0x70, 95 },
        { 0x71, 96 },
        { 0x72, 97 },
        { 0x73, 98 },
        { 0x74, 99 },
        { 0x75, 100 },
        { 0x76, 101 },
        { 0x77, 102 },
        { 0x78, 103 },
        { 0x79, 104 },
        { 0x7A, 105 },
        { 0x7B, 106 },
        { 0x7C, 107 },
        { 0x7D, 108 },
        { 0x7E, 109 },
        { 0xA0, 1 },
        { 0xA1, 135 },
        { 0xA2, 136 },
        { 0xA3, 137 },
        { 0xA4, 138 },
        { 0xA5, 139 },
        { 0xA6, 140 },
        { 0xA7, 141 },
        { 0xA8, 142 },
        { 0xA9, 143 },
        { 0xAA, 144 },
        { 0xAB, 145 },
        { 0xAC, 146 },
        { 0xAD, 28 },
        { 0xAE, 148 },
        { 0xAF, 149 },
        { 0xB0, 150 },
        { 0xB1, 151 },
        { 0xB2, 152 },
        { 0xB3, 153 },
        { 0xB4, 154 },
        { 0xB5, 155 },
        { 0xB6, 156 },
        { 0xB7, 157 },
        { 0xB8, 158 },
        { 0xB9, 159 },
        { 0xBA, 160 },
        { 0xBB, 161 },
        { 0xBC, 162 },
        { 0xBD, 163 },
        { 0xBE, 164 },
        { 0xBF, 165 },
        { 0xC0, 166 },
        { 0xC1, 167 },
        { 0xC2, 168 },
        { 0xC3, 169 },
        { 0xC4, 170 },
        { 0xC5, 171 },
        { 0xC6, 172 },
        { 0xC7, 173 },
        { 0xC8, 174 },
        { 0xC9, 175 },
        { 0xCA, 176 },
        { 0xCB, 177 },
        { 0xCC, 178 },
        { 0xCD, 179 },
        { 0xCE, 180 },
        { 0xCF, 181 },
        { 0xD0, 182 },
        { 0xD1, 183 },
        { 0xD2, 184 },
        { 0xD3, 185 },
        { 0xD4, 186 },
        { 0xD5, 187 },
        { 0xD6, 188 },
        { 0xD7, 189 },
        { 0xD8, 190 },
        { 0xD9, 191 },
        { 0xDA, 192 },
        { 0xDB, 193 },
        { 0xDC, 194 },
        { 0xDD, 195 },
        { 0xDE, 196 },
        { 0xDF, 197 },
        { 0xE0, 198 },
        { 0xE1, 199 },
        { 0xE2, 200 },
        { 0xE3, 201 },
        { 0xE4, 202 },
        { 0xE5, 203 },
        { 0xE6, 204 },
        { 0xE7, 205 },
        { 0xE8, 206 },
        { 0xE9, 207 },
        { 0xEA, 208 },
        { 0xEB, 209 },
        { 0xEC, 210 },
        { 0xED, 211 },
        { 0xEE, 212 },
        { 0xEF, 213 },
        { 0xF0, 214 },
        { 0xF1, 215 },
        { 0xF2, 216 },
        { 0xF3, 217 },
        { 0xF4, 218 },
        { 0xF5, 219 },
        { 0xF6, 220 },
        { 0xF7, 221 },
        { 0xF8, 222 },
        { 0xF9, 223 },
        { 0xFA, 224 },
        { 0xFB, 225 },
        { 0xFC, 226 },
        { 0xFD, 227 },
        { 0xFE, 228 },
        { 0xFF, 229 },
        { 0x131, 8 },
        { 0x141, 12 },
        { 0x142, 13 },
        { 0x152, 121 },
        { 0x153, 133 },
        { 0x160, 119 },
        { 0x161, 131 },
        { 0x178, 134 },
        { 0x17D, 14 },
        { 0x17E, 15 },
        { 0x192, 112 },
        { 0x2C6, 117 },
        { 0x2C7, 7 },
        { 0x2C9, 149 },
        { 0x2D8, 2 },
        { 0x2D9, 3 },
        { 0x2DA, 4 },
        { 0x2DB, 6 },
        { 0x2DC, 129 },
        { 0x2DD, 5 },
        { 0x3BC, 155 },
        { 0x2013, 127 },
        { 0x2014, 128 },
        { 0x2018, 122 },
        { 0x2019, 123 },
        { 0x201A, 111 },
        { 0x201C, 124 },
        { 0x201D, 125 },
        { 0x201E, 113 },
        { 0x2020, 115 },
        { 0x2021, 116 },
        { 0x2022, 126 },
        { 0x2026, 114 },
        { 0x2030, 118 },
        { 0x2039, 120 },
        { 0x203A, 132 },
        { 0x2044, 9 },
        { 0x20AC, 110 },
        { 0x2122, 130 },
        { 0x2212, 147 },
        { 0x2215, 9 },
        { 0x2219, 157 },
        { 0xFB01, 10 },
        { 0xFB02, 11 },
    };
    return map;
}

static const Std14CPToGIDMap& GetHelveticaBoldObliqueMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 16 },
        { 0x22, 17 },
        { 0x23, 18 },
        { 0x24, 19 },
        { 0x25, 20 },
        { 0x26, 21 },
        { 0x27, 22 },
        { 0x28, 23 },
        { 0x29, 24 },
        { 0x2A, 25 },
        { 0x2B, 26 },
        { 0x2C, 27 },
        { 0x2D, 28 },
        { 0x2E, 29 },
        { 0x2F, 30 },
        { 0x30, 31 },
        { 0x31, 32 },
        { 0x32, 33 },
        { 0x33, 34 },
        { 0x34, 35 },
        { 0x35, 36 },
        { 0x36, 37 },
        { 0x37, 38 },
        { 0x38, 39 },
        { 0x39, 40 },
        { 0x3A, 41 },
        { 0x3B, 42 },
        { 0x3C, 43 },
        { 0x3D, 44 },
        { 0x3E, 45 },
        { 0x3F, 46 },
        { 0x40, 47 },
        { 0x41, 48 },
        { 0x42, 49 },
        { 0x43, 50 },
        { 0x44, 51 },
        { 0x45, 52 },
        { 0x46, 53 },
        { 0x47, 54 },
        { 0x48, 55 },
        { 0x49, 56 },
        { 0x4A, 57 },
        { 0x4B, 58 },
        { 0x4C, 59 },
        { 0x4D, 60 },
        { 0x4E, 61 },
        { 0x4F, 62 },
        { 0x50, 63 },
        { 0x51, 64 },
        { 0x52, 65 },
        { 0x53, 66 },
        { 0x54, 67 },
        { 0x55, 68 },
        { 0x56, 69 },
        { 0x57, 70 },
        { 0x58, 71 },
        { 0x59, 72 },
        { 0x5A, 73 },
        { 0x5B, 74 },
        { 0x5C, 75 },
        { 0x5D, 76 },
        { 0x5E, 77 },
        { 0x5F, 78 },
        { 0x60, 79 },
        { 0x61, 80 },
        { 0x62, 81 },
        { 0x63, 82 },
        { 0x64, 83 },
        { 0x65, 84 },
        { 0x66, 85 },
        { 0x67, 86 },
        { 0x68, 87 },
        { 0x69, 88 },
        { 0x6A, 89 },
        { 0x6B, 90 },
        { 0x6C, 91 },
        { 0x6D, 92 },
        { 0x6E, 93 },
        { 0x6F, 94 },
        { 0x70, 95 },
        { 0x71, 96 },
        { 0x72, 97 },
        { 0x73, 98 },
        { 0x74, 99 },
        { 0x75, 100 },
        { 0x76, 101 },
        { 0x77, 102 },
        { 0x78, 103 },
        { 0x79, 104 },
        { 0x7A, 105 },
        { 0x7B, 106 },
        { 0x7C, 107 },
        { 0x7D, 108 },
        { 0x7E, 109 },
        { 0xA0, 1 },
        { 0xA1, 135 },
        { 0xA2, 136 },
        { 0xA3, 137 },
        { 0xA4, 138 },
        { 0xA5, 139 },
        { 0xA6, 140 },
        { 0xA7, 141 },
        { 0xA8, 142 },
        { 0xA9, 143 },
        { 0xAA, 144 },
        { 0xAB, 145 },
        { 0xAC, 146 },
        { 0xAD, 28 },
        { 0xAE, 148 },
        { 0xAF, 149 },
        { 0xB0, 150 },
        { 0xB1, 151 },
        { 0xB2, 152 },
        { 0xB3, 153 },
        { 0xB4, 154 },
        { 0xB5, 155 },
        { 0xB6, 156 },
        { 0xB7, 157 },
        { 0xB8, 158 },
        { 0xB9, 159 },
        { 0xBA, 160 },
        { 0xBB, 161 },
        { 0xBC, 162 },
        { 0xBD, 163 },
        { 0xBE, 164 },
        { 0xBF, 165 },
        { 0xC0, 166 },
        { 0xC1, 167 },
        { 0xC2, 168 },
        { 0xC3, 169 },
        { 0xC4, 170 },
        { 0xC5, 171 },
        { 0xC6, 172 },
        { 0xC7, 173 },
        { 0xC8, 174 },
        { 0xC9, 175 },
        { 0xCA, 176 },
        { 0xCB, 177 },
        { 0xCC, 178 },
        { 0xCD, 179 },
        { 0xCE, 180 },
        { 0xCF, 181 },
        { 0xD0, 182 },
        { 0xD1, 183 },
        { 0xD2, 184 },
        { 0xD3, 185 },
        { 0xD4, 186 },
        { 0xD5, 187 },
        { 0xD6, 188 },
        { 0xD7, 189 },
        { 0xD8, 190 },
        { 0xD9, 191 },
        { 0xDA, 192 },
        { 0xDB, 193 },
        { 0xDC, 194 },
        { 0xDD, 195 },
        { 0xDE, 196 },
        { 0xDF, 197 },
        { 0xE0, 198 },
        { 0xE1, 199 },
        { 0xE2, 200 },
        { 0xE3, 201 },
        { 0xE4, 202 },
        { 0xE5, 203 },
        { 0xE6, 204 },
        { 0xE7, 205 },
        { 0xE8, 206 },
        { 0xE9, 207 },
        { 0xEA, 208 },
        { 0xEB, 209 },
        { 0xEC, 210 },
        { 0xED, 211 },
        { 0xEE, 212 },
        { 0xEF, 213 },
        { 0xF0, 214 },
        { 0xF1, 215 },
        { 0xF2, 216 },
        { 0xF3, 217 },
        { 0xF4, 218 },
        { 0xF5, 219 },
        { 0xF6, 220 },
        { 0xF7, 221 },
        { 0xF8, 222 },
        { 0xF9, 223 },
        { 0xFA, 224 },
        { 0xFB, 225 },
        { 0xFC, 226 },
        { 0xFD, 227 },
        { 0xFE, 228 },
        { 0xFF, 229 },
        { 0x131, 8 },
        { 0x141, 12 },
        { 0x142, 13 },
        { 0x152, 121 },
        { 0x153, 133 },
        { 0x160, 119 },
        { 0x161, 131 },
        { 0x178, 134 },
        { 0x17D, 14 },
        { 0x17E, 15 },
        { 0x192, 112 },
        { 0x2C6, 117 },
        { 0x2C7, 7 },
        { 0x2C9, 149 },
        { 0x2D8, 2 },
        { 0x2D9, 3 },
        { 0x2DA, 4 },
        { 0x2DB, 6 },
        { 0x2DC, 129 },
        { 0x2DD, 5 },
        { 0x3BC, 155 },
        { 0x2013, 127 },
        { 0x2014, 128 },
        { 0x2018, 122 },
        { 0x2019, 123 },
        { 0x201A, 111 },
        { 0x201C, 124 },
        { 0x201D, 125 },
        { 0x201E, 113 },
        { 0x2020, 115 },
        { 0x2021, 116 },
        { 0x2022, 126 },
        { 0x2026, 114 },
        { 0x2030, 118 },
        { 0x2039, 120 },
        { 0x203A, 132 },
        { 0x2044, 9 },
        { 0x20AC, 110 },
        { 0x2122, 130 },
        { 0x2212, 147 },
        { 0x2215, 9 },
        { 0x2219, 157 },
        { 0xFB01, 10 },
        { 0xFB02, 11 },
    };
    return map;
}

static const Std14CPToGIDMap& GetCourierMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 217 },
        { 0x21, 126 },
        { 0x22, 198 },
        { 0x23, 168 },
        { 0x24, 112 },
        { 0x25, 188 },
        { 0x26, 74 },
        { 0x27, 205 },
        { 0x28, 186 },
        { 0x29, 187 },
        { 0x2A, 83 },
        { 0x2B, 192 },
        { 0x2C, 103 },
        { 0x2D, 146 },
        { 0x2E, 189 },
        { 0x2F, 216 },
        { 0x30, 243 },
        { 0x31, 176 },
        { 0x32, 226 },
        { 0x33, 221 },
        { 0x34, 133 },
        { 0x35, 130 },
        { 0x36, 215 },
        { 0x37, 214 },
        { 0x38, 120 },
        { 0x39, 166 },
        { 0x3A, 102 },
        { 0x3B, 213 },
        { 0x3C, 156 },
        { 0x3D, 124 },
        { 0x3E, 139 },
        { 0x3F, 196 },
        { 0x40, 84 },
        { 0x41, 1 },
        { 0x42, 9 },
        { 0x43, 10 },
        { 0x44, 12 },
        { 0x45, 13 },
        { 0x46, 20 },
        { 0x47, 21 },
        { 0x48, 23 },
        { 0x49, 24 },
        { 0x4A, 31 },
        { 0x4B, 32 },
        { 0x4C, 33 },
        { 0x4D, 35 },
        { 0x4E, 36 },
        { 0x4F, 38 },
        { 0x50, 46 },
        { 0x51, 47 },
        { 0x52, 48 },
        { 0x53, 49 },
        { 0x54, 52 },
        { 0x55, 54 },
        { 0x56, 59 },
        { 0x57, 60 },
        { 0x58, 61 },
        { 0x59, 62 },
        { 0x5A, 65 },
        { 0x5B, 91 },
        { 0x5C, 87 },
        { 0x5D, 92 },
        { 0x5E, 81 },
        { 0x5F, 233 },
        { 0x60, 138 },
        { 0x61, 67 },
        { 0x62, 86 },
        { 0x63, 96 },
        { 0x64, 106 },
        { 0x65, 115 },
        { 0x66, 128 },
        { 0x67, 135 },
        { 0x68, 144 },
        { 0x69, 147 },
        { 0x6A, 153 },
        { 0x6B, 154 },
        { 0x6C, 155 },
        { 0x6D, 160 },
        { 0x6E, 165 },
        { 0x6F, 169 },
        { 0x70, 184 },
        { 0x71, 195 },
        { 0x72, 206 },
        { 0x73, 209 },
        { 0x74, 219 },
        { 0x75, 228 },
        { 0x76, 234 },
        { 0x77, 235 },
        { 0x78, 236 },
        { 0x79, 237 },
        { 0x7A, 241 },
        { 0x7B, 89 },
        { 0x7C, 88 },
        { 0x7D, 90 },
        { 0x7E, 82 },
        { 0xA0, 217 },
        { 0xA1, 127 },
        { 0xA2, 100 },
        { 0xA3, 218 },
        { 0xA4, 105 },
        { 0xA5, 240 },
        { 0xA6, 94 },
        { 0xA7, 212 },
        { 0xA8, 110 },
        { 0xA9, 104 },
        { 0xAA, 180 },
        { 0xAB, 140 },
        { 0xAC, 158 },
        { 0xAD, 146 },
        { 0xAE, 207 },
        { 0xAF, 161 },
        { 0xB0, 109 },
        { 0xB1, 193 },
        { 0xB2, 227 },
        { 0xB3, 223 },
        { 0xB4, 70 },
        { 0xB5, 163 },
        { 0xB6, 185 },
        { 0xB7, 190 },
        { 0xB8, 99 },
        { 0xB9, 179 },
        { 0xBA, 181 },
        { 0xBB, 141 },
        { 0xBC, 178 },
        { 0xBD, 177 },
        { 0xBE, 222 },
        { 0xBF, 197 },
        { 0xC0, 6 },
        { 0xC1, 3 },
        { 0xC2, 4 },
        { 0xC3, 8 },
        { 0xC4, 5 },
        { 0xC5, 7 },
        { 0xC6, 2 },
        { 0xC7, 11 },
        { 0xC8, 17 },
        { 0xC9, 14 },
        { 0xCA, 15 },
        { 0xCB, 16 },
        { 0xCC, 30 },
        { 0xCD, 26 },
        { 0xCE, 27 },
        { 0xCF, 28 },
        { 0xD0, 18 },
        { 0xD1, 37 },
        { 0xD2, 43 },
        { 0xD3, 40 },
        { 0xD4, 41 },
        { 0xD5, 45 },
        { 0xD6, 42 },
        { 0xD7, 164 },
        { 0xD8, 44 },
        { 0xD9, 58 },
        { 0xDA, 55 },
        { 0xDB, 56 },
        { 0xDC, 57 },
        { 0xDD, 63 },
        { 0xDE, 53 },
        { 0xDF, 137 },
        { 0xE0, 73 },
        { 0xE1, 68 },
        { 0xE2, 69 },
        { 0xE3, 85 },
        { 0xE4, 71 },
        { 0xE5, 75 },
        { 0xE6, 72 },
        { 0xE7, 98 },
        { 0xE8, 119 },
        { 0xE9, 116 },
        { 0xEA, 117 },
        { 0xEB, 118 },
        { 0xEC, 151 },
        { 0xED, 148 },
        { 0xEE, 149 },
        { 0xEF, 150 },
        { 0xF0, 125 },
        { 0xF1, 167 },
        { 0xF2, 175 },
        { 0xF3, 170 },
        { 0xF4, 171 },
        { 0xF5, 183 },
        { 0xF6, 172 },
        { 0xF7, 111 },
        { 0xF8, 182 },
        { 0xF9, 232 },
        { 0xFA, 229 },
        { 0xFB, 230 },
        { 0xFC, 231 },
        { 0xFD, 238 },
        { 0xFE, 220 },
        { 0xFF, 239 },
        { 0x130, 29 },
        { 0x131, 114 },
        { 0x132, 25 },
        { 0x133, 152 },
        { 0x141, 34 },
        { 0x142, 159 },
        { 0x152, 39 },
        { 0x153, 173 },
        { 0x15E, 51 },
        { 0x15F, 211 },
        { 0x160, 50 },
        { 0x161, 210 },
        { 0x178, 64 },
        { 0x17D, 66 },
        { 0x17E, 242 },
        { 0x192, 132 },
        { 0x1E6, 22 },
        { 0x1E7, 136 },
        { 0x2C6, 101 },
        { 0x2C7, 97 },
        { 0x2C9, 161 },
        { 0x2D8, 93 },
        { 0x2D9, 113 },
        { 0x2DA, 208 },
        { 0x2DB, 174 },
        { 0x2DC, 224 },
        { 0x2DD, 145 },
        { 0x3BC, 163 },
        { 0x2013, 123 },
        { 0x2014, 122 },
        { 0x2018, 202 },
        { 0x2019, 203 },
        { 0x201A, 204 },
        { 0x201C, 200 },
        { 0x201D, 201 },
        { 0x201E, 199 },
        { 0x2020, 107 },
        { 0x2021, 108 },
        { 0x2022, 95 },
        { 0x2026, 121 },
        { 0x2030, 191 },
        { 0x2039, 142 },
        { 0x203A, 143 },
        { 0x2044, 134 },
        { 0x20A4, 157 },
        { 0x20AC, 19 },
        { 0x211E, 194 },
        { 0x2122, 225 },
        { 0x2190, 78 },
        { 0x2191, 80 },
        { 0x2192, 79 },
        { 0x2193, 77 },
        { 0x2194, 76 },
        { 0x2212, 162 },
        { 0x2215, 134 },
        { 0x2219, 190 },
        { 0xFB01, 129 },
        { 0xFB02, 131 },
    };
    return map;
}

static const Std14CPToGIDMap& GetCourierObliqueMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 160 },
        { 0xE2, 159 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x130, 190 },
        { 0x131, 191 },
        { 0x132, 192 },
        { 0x133, 193 },
        { 0x141, 194 },
        { 0x142, 195 },
        { 0x152, 196 },
        { 0x153, 197 },
        { 0x15E, 198 },
        { 0x15F, 199 },
        { 0x160, 200 },
        { 0x161, 201 },
        { 0x178, 202 },
        { 0x17D, 203 },
        { 0x17E, 204 },
        { 0x192, 205 },
        { 0x1E6, 206 },
        { 0x1E7, 207 },
        { 0x2C6, 208 },
        { 0x2C7, 209 },
        { 0x2C9, 109 },
        { 0x2D8, 210 },
        { 0x2D9, 211 },
        { 0x2DA, 212 },
        { 0x2DB, 213 },
        { 0x2DC, 214 },
        { 0x2DD, 215 },
        { 0x3BC, 115 },
        { 0x2013, 216 },
        { 0x2014, 217 },
        { 0x2018, 218 },
        { 0x2019, 219 },
        { 0x201A, 220 },
        { 0x201C, 221 },
        { 0x201D, 222 },
        { 0x201E, 223 },
        { 0x2020, 224 },
        { 0x2021, 225 },
        { 0x2022, 226 },
        { 0x2026, 227 },
        { 0x2030, 228 },
        { 0x2039, 229 },
        { 0x203A, 230 },
        { 0x2044, 231 },
        { 0x20A4, 232 },
        { 0x20AC, 233 },
        { 0x211E, 234 },
        { 0x2122, 235 },
        { 0x2190, 236 },
        { 0x2191, 237 },
        { 0x2192, 238 },
        { 0x2193, 239 },
        { 0x2194, 240 },
        { 0x2212, 241 },
        { 0x2215, 231 },
        { 0x2219, 117 },
        { 0xFB01, 242 },
        { 0xFB02, 243 },
    };
    return map;
}

static const Std14CPToGIDMap& GetCourierBoldMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x22, 3 },
        { 0x23, 4 },
        { 0x24, 5 },
        { 0x25, 6 },
        { 0x26, 7 },
        { 0x27, 8 },
        { 0x28, 9 },
        { 0x29, 10 },
        { 0x2A, 11 },
        { 0x2B, 12 },
        { 0x2C, 13 },
        { 0x2D, 14 },
        { 0x2E, 15 },
        { 0x2F, 16 },
        { 0x30, 17 },
        { 0x31, 18 },
        { 0x32, 19 },
        { 0x33, 20 },
        { 0x34, 21 },
        { 0x35, 22 },
        { 0x36, 23 },
        { 0x37, 24 },
        { 0x38, 25 },
        { 0x39, 26 },
        { 0x3A, 27 },
        { 0x3B, 28 },
        { 0x3C, 29 },
        { 0x3D, 30 },
        { 0x3E, 31 },
        { 0x3F, 32 },
        { 0x40, 33 },
        { 0x41, 34 },
        { 0x42, 35 },
        { 0x43, 36 },
        { 0x44, 37 },
        { 0x45, 38 },
        { 0x46, 39 },
        { 0x47, 40 },
        { 0x48, 41 },
        { 0x49, 42 },
        { 0x4A, 43 },
        { 0x4B, 44 },
        { 0x4C, 45 },
        { 0x4D, 46 },
        { 0x4E, 47 },
        { 0x4F, 48 },
        { 0x50, 49 },
        { 0x51, 50 },
        { 0x52, 51 },
        { 0x53, 52 },
        { 0x54, 53 },
        { 0x55, 54 },
        { 0x56, 55 },
        { 0x57, 56 },
        { 0x58, 57 },
        { 0x59, 58 },
        { 0x5A, 59 },
        { 0x5B, 60 },
        { 0x5C, 61 },
        { 0x5D, 62 },
        { 0x5E, 63 },
        { 0x5F, 64 },
        { 0x60, 65 },
        { 0x61, 66 },
        { 0x62, 67 },
        { 0x63, 68 },
        { 0x64, 69 },
        { 0x65, 70 },
        { 0x66, 71 },
        { 0x67, 72 },
        { 0x68, 73 },
        { 0x69, 74 },
        { 0x6A, 75 },
        { 0x6B, 76 },
        { 0x6C, 77 },
        { 0x6D, 78 },
        { 0x6E, 79 },
        { 0x6F, 80 },
        { 0x70, 81 },
        { 0x71, 82 },
        { 0x72, 83 },
        { 0x73, 84 },
        { 0x74, 85 },
        { 0x75, 86 },
        { 0x76, 87 },
        { 0x77, 88 },
        { 0x78, 89 },
        { 0x79, 90 },
        { 0x7A, 91 },
        { 0x7B, 92 },
        { 0x7C, 93 },
        { 0x7D, 94 },
        { 0x7E, 95 },
        { 0xA0, 1 },
        { 0xA1, 96 },
        { 0xA2, 97 },
        { 0xA3, 98 },
        { 0xA4, 99 },
        { 0xA5, 100 },
        { 0xA6, 101 },
        { 0xA7, 102 },
        { 0xA8, 103 },
        { 0xA9, 104 },
        { 0xAA, 105 },
        { 0xAB, 106 },
        { 0xAC, 107 },
        { 0xAD, 14 },
        { 0xAE, 108 },
        { 0xAF, 109 },
        { 0xB0, 110 },
        { 0xB1, 111 },
        { 0xB2, 112 },
        { 0xB3, 113 },
        { 0xB4, 114 },
        { 0xB5, 115 },
        { 0xB6, 116 },
        { 0xB7, 117 },
        { 0xB8, 118 },
        { 0xB9, 119 },
        { 0xBA, 120 },
        { 0xBB, 121 },
        { 0xBC, 122 },
        { 0xBD, 123 },
        { 0xBE, 124 },
        { 0xBF, 125 },
        { 0xC0, 126 },
        { 0xC1, 127 },
        { 0xC2, 128 },
        { 0xC3, 129 },
        { 0xC4, 130 },
        { 0xC5, 131 },
        { 0xC6, 132 },
        { 0xC7, 133 },
        { 0xC8, 134 },
        { 0xC9, 135 },
        { 0xCA, 136 },
        { 0xCB, 137 },
        { 0xCC, 138 },
        { 0xCD, 139 },
        { 0xCE, 140 },
        { 0xCF, 141 },
        { 0xD0, 142 },
        { 0xD1, 143 },
        { 0xD2, 144 },
        { 0xD3, 145 },
        { 0xD4, 146 },
        { 0xD5, 147 },
        { 0xD6, 148 },
        { 0xD7, 149 },
        { 0xD8, 150 },
        { 0xD9, 151 },
        { 0xDA, 152 },
        { 0xDB, 153 },
        { 0xDC, 154 },
        { 0xDD, 155 },
        { 0xDE, 156 },
        { 0xDF, 157 },
        { 0xE0, 158 },
        { 0xE1, 159 },
        { 0xE2, 160 },
        { 0xE3, 161 },
        { 0xE4, 162 },
        { 0xE5, 163 },
        { 0xE6, 164 },
        { 0xE7, 165 },
        { 0xE8, 166 },
        { 0xE9, 167 },
        { 0xEA, 168 },
        { 0xEB, 169 },
        { 0xEC, 170 },
        { 0xED, 171 },
        { 0xEE, 172 },
        { 0xEF, 173 },
        { 0xF0, 174 },
        { 0xF1, 175 },
        { 0xF2, 176 },
        { 0xF3, 177 },
        { 0xF4, 178 },
        { 0xF5, 179 },
        { 0xF6, 180 },
        { 0xF7, 181 },
        { 0xF8, 182 },
        { 0xF9, 183 },
        { 0xFA, 184 },
        { 0xFB, 185 },
        { 0xFC, 186 },
        { 0xFD, 187 },
        { 0xFE, 188 },
        { 0xFF, 189 },
        { 0x130, 190 },
        { 0x131, 191 },
        { 0x132, 192 },
        { 0x133, 193 },
        { 0x141, 194 },
        { 0x142, 195 },
        { 0x152, 196 },
        { 0x153, 197 },
        { 0x15E, 198 },
        { 0x15F, 199 },
        { 0x160, 200 },
        { 0x161, 201 },
        { 0x178, 202 },
        { 0x17D, 203 },
        { 0x17E, 204 },
        { 0x192, 205 },
        { 0x1E6, 206 },
        { 0x1E7, 207 },
        { 0x2C6, 208 },
        { 0x2C7, 209 },
        { 0x2C9, 109 },
        { 0x2D8, 210 },
        { 0x2D9, 211 },
        { 0x2DA, 212 },
        { 0x2DB, 213 },
        { 0x2DC, 214 },
        { 0x2DD, 215 },
        { 0x3BC, 115 },
        { 0x2013, 216 },
        { 0x2014, 217 },
        { 0x2018, 218 },
        { 0x2019, 219 },
        { 0x201A, 220 },
        { 0x201C, 221 },
        { 0x201D, 222 },
        { 0x201E, 223 },
        { 0x2020, 224 },
        { 0x2021, 225 },
        { 0x2022, 226 },
        { 0x2026, 227 },
        { 0x2030, 228 },
        { 0x2039, 229 },
        { 0x203A, 230 },
        { 0x2044, 231 },
        { 0x20A4, 232 },
        { 0x20AC, 233 },
        { 0x211E, 234 },
        { 0x2122, 235 },
        { 0x2190, 236 },
        { 0x2191, 237 },
        { 0x2192, 238 },
        { 0x2193, 239 },
        { 0x2194, 240 },
        { 0x2212, 241 },
        { 0x2215, 231 },
        { 0x2219, 117 },
        { 0xFB01, 242 },
        { 0xFB02, 243 },
    };
    return map;
}

static const Std14CPToGIDMap& GetCourierBoldObliqueMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 217 },
        { 0x21, 126 },
        { 0x22, 198 },
        { 0x23, 168 },
        { 0x24, 112 },
        { 0x25, 188 },
        { 0x26, 74 },
        { 0x27, 205 },
        { 0x28, 186 },
        { 0x29, 187 },
        { 0x2A, 83 },
        { 0x2B, 192 },
        { 0x2C, 103 },
        { 0x2D, 146 },
        { 0x2E, 189 },
        { 0x2F, 216 },
        { 0x30, 243 },
        { 0x31, 176 },
        { 0x32, 226 },
        { 0x33, 221 },
        { 0x34, 133 },
        { 0x35, 130 },
        { 0x36, 215 },
        { 0x37, 214 },
        { 0x38, 120 },
        { 0x39, 166 },
        { 0x3A, 102 },
        { 0x3B, 213 },
        { 0x3C, 156 },
        { 0x3D, 124 },
        { 0x3E, 139 },
        { 0x3F, 196 },
        { 0x40, 84 },
        { 0x41, 1 },
        { 0x42, 9 },
        { 0x43, 10 },
        { 0x44, 12 },
        { 0x45, 13 },
        { 0x46, 20 },
        { 0x47, 21 },
        { 0x48, 23 },
        { 0x49, 24 },
        { 0x4A, 31 },
        { 0x4B, 32 },
        { 0x4C, 33 },
        { 0x4D, 35 },
        { 0x4E, 36 },
        { 0x4F, 38 },
        { 0x50, 46 },
        { 0x51, 47 },
        { 0x52, 48 },
        { 0x53, 49 },
        { 0x54, 52 },
        { 0x55, 54 },
        { 0x56, 59 },
        { 0x57, 60 },
        { 0x58, 61 },
        { 0x59, 62 },
        { 0x5A, 65 },
        { 0x5B, 91 },
        { 0x5C, 87 },
        { 0x5D, 92 },
        { 0x5E, 81 },
        { 0x5F, 233 },
        { 0x60, 138 },
        { 0x61, 67 },
        { 0x62, 86 },
        { 0x63, 96 },
        { 0x64, 106 },
        { 0x65, 115 },
        { 0x66, 128 },
        { 0x67, 135 },
        { 0x68, 144 },
        { 0x69, 147 },
        { 0x6A, 153 },
        { 0x6B, 154 },
        { 0x6C, 155 },
        { 0x6D, 160 },
        { 0x6E, 165 },
        { 0x6F, 169 },
        { 0x70, 184 },
        { 0x71, 195 },
        { 0x72, 206 },
        { 0x73, 209 },
        { 0x74, 219 },
        { 0x75, 228 },
        { 0x76, 234 },
        { 0x77, 235 },
        { 0x78, 236 },
        { 0x79, 237 },
        { 0x7A, 241 },
        { 0x7B, 89 },
        { 0x7C, 88 },
        { 0x7D, 90 },
        { 0x7E, 82 },
        { 0xA0, 217 },
        { 0xA1, 127 },
        { 0xA2, 100 },
        { 0xA3, 218 },
        { 0xA4, 105 },
        { 0xA5, 240 },
        { 0xA6, 94 },
        { 0xA7, 212 },
        { 0xA8, 110 },
        { 0xA9, 104 },
        { 0xAA, 180 },
        { 0xAB, 140 },
        { 0xAC, 158 },
        { 0xAD, 146 },
        { 0xAE, 207 },
        { 0xAF, 161 },
        { 0xB0, 109 },
        { 0xB1, 193 },
        { 0xB2, 227 },
        { 0xB3, 223 },
        { 0xB4, 70 },
        { 0xB5, 163 },
        { 0xB6, 185 },
        { 0xB7, 190 },
        { 0xB8, 99 },
        { 0xB9, 179 },
        { 0xBA, 181 },
        { 0xBB, 141 },
        { 0xBC, 178 },
        { 0xBD, 177 },
        { 0xBE, 222 },
        { 0xBF, 197 },
        { 0xC0, 6 },
        { 0xC1, 3 },
        { 0xC2, 4 },
        { 0xC3, 8 },
        { 0xC4, 5 },
        { 0xC5, 7 },
        { 0xC6, 2 },
        { 0xC7, 11 },
        { 0xC8, 17 },
        { 0xC9, 14 },
        { 0xCA, 15 },
        { 0xCB, 16 },
        { 0xCC, 30 },
        { 0xCD, 26 },
        { 0xCE, 27 },
        { 0xCF, 28 },
        { 0xD0, 18 },
        { 0xD1, 37 },
        { 0xD2, 43 },
        { 0xD3, 40 },
        { 0xD4, 41 },
        { 0xD5, 45 },
        { 0xD6, 42 },
        { 0xD7, 164 },
        { 0xD8, 44 },
        { 0xD9, 58 },
        { 0xDA, 55 },
        { 0xDB, 56 },
        { 0xDC, 57 },
        { 0xDD, 63 },
        { 0xDE, 53 },
        { 0xDF, 137 },
        { 0xE0, 73 },
        { 0xE1, 68 },
        { 0xE2, 69 },
        { 0xE3, 85 },
        { 0xE4, 71 },
        { 0xE5, 75 },
        { 0xE6, 72 },
        { 0xE7, 98 },
        { 0xE8, 119 },
        { 0xE9, 116 },
        { 0xEA, 117 },
        { 0xEB, 118 },
        { 0xEC, 151 },
        { 0xED, 148 },
        { 0xEE, 149 },
        { 0xEF, 150 },
        { 0xF0, 125 },
        { 0xF1, 167 },
        { 0xF2, 175 },
        { 0xF3, 170 },
        { 0xF4, 171 },
        { 0xF5, 183 },
        { 0xF6, 172 },
        { 0xF7, 111 },
        { 0xF8, 182 },
        { 0xF9, 232 },
        { 0xFA, 229 },
        { 0xFB, 230 },
        { 0xFC, 231 },
        { 0xFD, 238 },
        { 0xFE, 220 },
        { 0xFF, 239 },
        { 0x130, 29 },
        { 0x131, 114 },
        { 0x132, 25 },
        { 0x133, 152 },
        { 0x141, 34 },
        { 0x142, 159 },
        { 0x152, 39 },
        { 0x153, 173 },
        { 0x15E, 51 },
        { 0x15F, 211 },
        { 0x160, 50 },
        { 0x161, 210 },
        { 0x178, 64 },
        { 0x17D, 66 },
        { 0x17E, 242 },
        { 0x192, 132 },
        { 0x1E6, 22 },
        { 0x1E7, 136 },
        { 0x2C6, 101 },
        { 0x2C7, 97 },
        { 0x2C9, 161 },
        { 0x2D8, 93 },
        { 0x2D9, 113 },
        { 0x2DA, 208 },
        { 0x2DB, 174 },
        { 0x2DC, 224 },
        { 0x2DD, 145 },
        { 0x3BC, 163 },
        { 0x2013, 123 },
        { 0x2014, 122 },
        { 0x2018, 202 },
        { 0x2019, 203 },
        { 0x201A, 204 },
        { 0x201C, 200 },
        { 0x201D, 201 },
        { 0x201E, 199 },
        { 0x2020, 107 },
        { 0x2021, 108 },
        { 0x2022, 95 },
        { 0x2026, 121 },
        { 0x2030, 191 },
        { 0x2039, 142 },
        { 0x203A, 143 },
        { 0x2044, 134 },
        { 0x20A4, 157 },
        { 0x20AC, 19 },
        { 0x211E, 194 },
        { 0x2122, 225 },
        { 0x2190, 78 },
        { 0x2191, 80 },
        { 0x2192, 79 },
        { 0x2193, 77 },
        { 0x2194, 76 },
        { 0x2212, 162 },
        { 0x2215, 134 },
        { 0x2219, 190 },
        { 0xFB01, 129 },
        { 0xFB02, 131 },
    };
    return map;
}

static const Std14CPToGIDMap& GetSymbolMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 1 },
        { 0x21, 2 },
        { 0x23, 3 },
        { 0x25, 4 },
        { 0x26, 5 },
        { 0x28, 6 },
        { 0x29, 7 },
        { 0x2B, 8 },
        { 0x2C, 9 },
        { 0x2E, 10 },
        { 0x2F, 11 },
        { 0x30, 12 },
        { 0x31, 13 },
        { 0x32, 14 },
        { 0x33, 15 },
        { 0x34, 16 },
        { 0x35, 17 },
        { 0x36, 18 },
        { 0x37, 19 },
        { 0x38, 20 },
        { 0x39, 21 },
        { 0x3A, 22 },
        { 0x3B, 23 },
        { 0x3C, 24 },
        { 0x3D, 25 },
        { 0x3E, 26 },
        { 0x3F, 27 },
        { 0x5B, 28 },
        { 0x5D, 29 },
        { 0x5F, 30 },
        { 0x7B, 31 },
        { 0x7C, 32 },
        { 0x7D, 33 },
        { 0xA0, 1 },
        { 0xAC, 34 },
        { 0xB0, 35 },
        { 0xB1, 36 },
        { 0xB5, 37 },
        { 0xD7, 38 },
        { 0xF7, 39 },
        { 0x192, 40 },
        { 0x391, 41 },
        { 0x392, 42 },
        { 0x393, 43 },
        { 0x394, 117 },
        { 0x395, 44 },
        { 0x396, 45 },
        { 0x397, 46 },
        { 0x398, 47 },
        { 0x399, 48 },
        { 0x39A, 49 },
        { 0x39B, 50 },
        { 0x39C, 51 },
        { 0x39D, 52 },
        { 0x39E, 53 },
        { 0x39F, 54 },
        { 0x3A0, 55 },
        { 0x3A1, 56 },
        { 0x3A3, 57 },
        { 0x3A4, 58 },
        { 0x3A5, 59 },
        { 0x3A6, 60 },
        { 0x3A7, 61 },
        { 0x3A8, 62 },
        { 0x3A9, 100 },
        { 0x3B1, 63 },
        { 0x3B2, 64 },
        { 0x3B3, 65 },
        { 0x3B4, 66 },
        { 0x3B5, 67 },
        { 0x3B6, 68 },
        { 0x3B7, 69 },
        { 0x3B8, 70 },
        { 0x3B9, 71 },
        { 0x3BA, 72 },
        { 0x3BB, 73 },
        { 0x3BC, 37 },
        { 0x3BD, 74 },
        { 0x3BE, 75 },
        { 0x3BF, 76 },
        { 0x3C0, 77 },
        { 0x3C1, 78 },
        { 0x3C2, 79 },
        { 0x3C3, 80 },
        { 0x3C4, 81 },
        { 0x3C5, 82 },
        { 0x3C6, 83 },
        { 0x3C7, 84 },
        { 0x3C8, 85 },
        { 0x3C9, 86 },
        { 0x3D1, 87 },
        { 0x3D2, 88 },
        { 0x3D5, 89 },
        { 0x3D6, 90 },
        { 0x2022, 91 },
        { 0x2026, 92 },
        { 0x2032, 93 },
        { 0x2033, 94 },
        { 0x2044, 95 },
        { 0x20AC, 96 },
        { 0x2111, 97 },
        { 0x2118, 98 },
        { 0x211C, 99 },
        { 0x2126, 100 },
        { 0x2135, 101 },
        { 0x2190, 102 },
        { 0x2191, 103 },
        { 0x2192, 104 },
        { 0x2193, 105 },
        { 0x2194, 106 },
        { 0x21B5, 107 },
        { 0x21D0, 108 },
        { 0x21D1, 109 },
        { 0x21D2, 110 },
        { 0x21D3, 111 },
        { 0x21D4, 112 },
        { 0x2200, 113 },
        { 0x2202, 114 },
        { 0x2203, 115 },
        { 0x2205, 116 },
        { 0x2206, 117 },
        { 0x2207, 118 },
        { 0x2208, 119 },
        { 0x2209, 120 },
        { 0x220B, 121 },
        { 0x220F, 122 },
        { 0x2211, 123 },
        { 0x2212, 124 },
        { 0x2215, 95 },
        { 0x2217, 125 },
        { 0x221A, 126 },
        { 0x221D, 127 },
        { 0x221E, 128 },
        { 0x2220, 129 },
        { 0x2227, 130 },
        { 0x2228, 131 },
        { 0x2229, 132 },
        { 0x222A, 133 },
        { 0x222B, 134 },
        { 0x2234, 135 },
        { 0x223C, 136 },
        { 0x2245, 137 },
        { 0x2248, 138 },
        { 0x2260, 139 },
        { 0x2261, 140 },
        { 0x2264, 141 },
        { 0x2265, 142 },
        { 0x2282, 143 },
        { 0x2283, 144 },
        { 0x2284, 145 },
        { 0x2286, 146 },
        { 0x2287, 147 },
        { 0x2295, 148 },
        { 0x2297, 149 },
        { 0x22A5, 150 },
        { 0x22C5, 151 },
        { 0x2320, 152 },
        { 0x2321, 153 },
        { 0x2329, 154 },
        { 0x232A, 155 },
        { 0x25CA, 156 },
        { 0x2660, 157 },
        { 0x2663, 158 },
        { 0x2665, 159 },
        { 0x2666, 160 },
        { 0xF6D9, 161 },
        { 0xF6DA, 162 },
        { 0xF6DB, 163 },
        { 0xF8E5, 164 },
        { 0xF8E6, 165 },
        { 0xF8E7, 166 },
        { 0xF8E8, 167 },
        { 0xF8E9, 168 },
        { 0xF8EA, 169 },
        { 0xF8EB, 170 },
        { 0xF8EC, 171 },
        { 0xF8ED, 172 },
        { 0xF8EE, 173 },
        { 0xF8EF, 174 },
        { 0xF8F0, 175 },
        { 0xF8F1, 176 },
        { 0xF8F2, 177 },
        { 0xF8F3, 178 },
        { 0xF8F4, 179 },
        { 0xF8F5, 180 },
        { 0xF8F6, 181 },
        { 0xF8F7, 182 },
        { 0xF8F8, 183 },
        { 0xF8F9, 184 },
        { 0xF8FA, 185 },
        { 0xF8FB, 186 },
        { 0xF8FC, 187 },
        { 0xF8FD, 188 },
        { 0xF8FE, 189 },
        { 0xF8FF, 190 },
    };
    return map;
}

static const Std14CPToGIDMap& GetZapfDingbatsMap()
{
    static Std14CPToGIDMap map = {
        { 0x20, 202 },
        { 0xA0, 202 },
        { 0x2192, 66 },
        { 0x2194, 68 },
        { 0x2195, 69 },
        { 0x2460, 21 },
        { 0x2461, 22 },
        { 0x2462, 23 },
        { 0x2463, 24 },
        { 0x2464, 25 },
        { 0x2465, 26 },
        { 0x2466, 27 },
        { 0x2467, 28 },
        { 0x2468, 29 },
        { 0x2469, 30 },
        { 0x25A0, 174 },
        { 0x25B2, 177 },
        { 0x25BC, 178 },
        { 0x25C6, 179 },
        { 0x25CF, 172 },
        { 0x25D7, 182 },
        { 0x2605, 132 },
        { 0x260E, 137 },
        { 0x261B, 13 },
        { 0x261E, 20 },
        { 0x2660, 12 },
        { 0x2663, 16 },
        { 0x2665, 14 },
        { 0x2666, 15 },
        { 0x2701, 1 },
        { 0x2702, 108 },
        { 0x2703, 112 },
        { 0x2704, 126 },
        { 0x2706, 148 },
        { 0x2707, 19 },
        { 0x2708, 18 },
        { 0x2709, 17 },
        { 0x270C, 31 },
        { 0x270D, 42 },
        { 0x270E, 53 },
        { 0x270F, 64 },
        { 0x2710, 8 },
        { 0x2711, 75 },
        { 0x2712, 86 },
        { 0x2713, 97 },
        { 0x2714, 109 },
        { 0x2715, 117 },
        { 0x2716, 118 },
        { 0x2717, 119 },
        { 0x2718, 120 },
        { 0x2719, 121 },
        { 0x271A, 122 },
        { 0x271B, 123 },
        { 0x271C, 124 },
        { 0x271D, 159 },
        { 0x271E, 170 },
        { 0x271F, 181 },
        { 0x2720, 191 },
        { 0x2721, 2 },
        { 0x2722, 125 },
        { 0x2723, 127 },
        { 0x2724, 128 },
        { 0x2725, 129 },
        { 0x2726, 130 },
        { 0x2727, 131 },
        { 0x2729, 133 },
        { 0x272A, 134 },
        { 0x272B, 135 },
        { 0x272C, 136 },
        { 0x272D, 138 },
        { 0x272E, 139 },
        { 0x272F, 140 },
        { 0x2730, 141 },
        { 0x2731, 142 },
        { 0x2732, 143 },
        { 0x2733, 144 },
        { 0x2734, 145 },
        { 0x2735, 146 },
        { 0x2736, 147 },
        { 0x2737, 149 },
        { 0x2738, 150 },
        { 0x2739, 151 },
        { 0x273A, 152 },
        { 0x273B, 153 },
        { 0x273C, 154 },
        { 0x273D, 155 },
        { 0x273E, 156 },
        { 0x273F, 157 },
        { 0x2740, 158 },
        { 0x2741, 160 },
        { 0x2742, 161 },
        { 0x2743, 162 },
        { 0x2744, 163 },
        { 0x2745, 164 },
        { 0x2746, 165 },
        { 0x2747, 166 },
        { 0x2748, 167 },
        { 0x2749, 168 },
        { 0x274A, 169 },
        { 0x274B, 171 },
        { 0x274D, 173 },
        { 0x274F, 175 },
        { 0x2750, 113 },
        { 0x2751, 176 },
        { 0x2752, 114 },
        { 0x2756, 180 },
        { 0x2758, 183 },
        { 0x2759, 184 },
        { 0x275A, 185 },
        { 0x275B, 199 },
        { 0x275C, 200 },
        { 0x275D, 201 },
        { 0x275E, 3 },
        { 0x2761, 4 },
        { 0x2762, 5 },
        { 0x2763, 6 },
        { 0x2764, 7 },
        { 0x2765, 9 },
        { 0x2766, 10 },
        { 0x2767, 11 },
        { 0x2768, 190 },
        { 0x2769, 192 },
        { 0x276A, 195 },
        { 0x276B, 196 },
        { 0x276C, 193 },
        { 0x276D, 194 },
        { 0x276E, 115 },
        { 0x276F, 186 },
        { 0x2770, 116 },
        { 0x2771, 187 },
        { 0x2772, 188 },
        { 0x2773, 189 },
        { 0x2774, 197 },
        { 0x2775, 198 },
        { 0x2776, 32 },
        { 0x2777, 33 },
        { 0x2778, 34 },
        { 0x2779, 35 },
        { 0x277A, 36 },
        { 0x277B, 37 },
        { 0x277C, 38 },
        { 0x277D, 39 },
        { 0x277E, 40 },
        { 0x277F, 41 },
        { 0x2780, 43 },
        { 0x2781, 44 },
        { 0x2782, 45 },
        { 0x2783, 46 },
        { 0x2784, 47 },
        { 0x2785, 48 },
        { 0x2786, 49 },
        { 0x2787, 50 },
        { 0x2788, 51 },
        { 0x2789, 52 },
        { 0x278A, 54 },
        { 0x278B, 55 },
        { 0x278C, 56 },
        { 0x278D, 57 },
        { 0x278E, 58 },
        { 0x278F, 59 },
        { 0x2790, 60 },
        { 0x2791, 61 },
        { 0x2792, 62 },
        { 0x2793, 63 },
        { 0x2794, 65 },
        { 0x2798, 104 },
        { 0x2799, 70 },
        { 0x279A, 100 },
        { 0x279B, 71 },
        { 0x279C, 72 },
        { 0x279D, 73 },
        { 0x279E, 74 },
        { 0x279F, 76 },
        { 0x27A0, 77 },
        { 0x27A1, 78 },
        { 0x27A2, 79 },
        { 0x27A3, 67 },
        { 0x27A4, 80 },
        { 0x27A5, 81 },
        { 0x27A6, 82 },
        { 0x27A7, 83 },
        { 0x27A8, 84 },
        { 0x27A9, 85 },
        { 0x27AA, 101 },
        { 0x27AB, 87 },
        { 0x27AC, 107 },
        { 0x27AD, 88 },
        { 0x27AE, 110 },
        { 0x27AF, 89 },
        { 0x27B1, 111 },
        { 0x27B2, 90 },
        { 0x27B3, 91 },
        { 0x27B4, 105 },
        { 0x27B5, 92 },
        { 0x27B6, 102 },
        { 0x27B7, 106 },
        { 0x27B8, 93 },
        { 0x27B9, 103 },
        { 0x27BA, 94 },
        { 0x27BB, 95 },
        { 0x27BC, 96 },
        { 0x27BD, 98 },
        { 0x27BE, 99 },
    };
    return map;
}

const Std14CPToGIDMap& PoDoFo::GetStd14CPToGIDMap(PdfStandard14FontType stdFont)
{
    switch (stdFont)
    {
        case PdfStandard14FontType::TimesRoman:
            return GetTimesRomanMap();
        case PdfStandard14FontType::TimesItalic:
            return GetTimesItalicMap();
        case PdfStandard14FontType::TimesBold:
            return GetTimesBoldMap();
        case PdfStandard14FontType::TimesBoldItalic:
            return GetTimesBoldItalicMap();
        case PdfStandard14FontType::Helvetica:
            return GetHelveticaMap();
        case PdfStandard14FontType::HelveticaOblique:
            return GetHelveticaObliqueMap();
        case PdfStandard14FontType::HelveticaBold:
            return GetHelveticaBoldMap();
        case PdfStandard14FontType::HelveticaBoldOblique:
            return GetHelveticaBoldObliqueMap();
        case PdfStandard14FontType::Courier:
            return GetCourierMap();
        case PdfStandard14FontType::CourierOblique:
            return GetCourierObliqueMap();
        case PdfStandard14FontType::CourierBold:
            return GetCourierBoldMap();
        case PdfStandard14FontType::CourierBoldOblique:
            return GetCourierBoldObliqueMap();
        case PdfStandard14FontType::Symbol:
            return GetSymbolMap();
        case PdfStandard14FontType::ZapfDingbats:
            return GetZapfDingbatsMap();
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR(PdfErrorCode::InvalidEnumValue);
    }
}

string_view PoDoFo::GetStandard14FontBaseName(PdfStandard14FontType stdFont)
{
    switch (stdFont)
    {
        case PdfStandard14FontType::TimesRoman:
        case PdfStandard14FontType::TimesItalic:
        case PdfStandard14FontType::TimesBold:
        case PdfStandard14FontType::TimesBoldItalic:
            return TIMES_ROMAN_BASE_NAME;
        case PdfStandard14FontType::Helvetica:
        case PdfStandard14FontType::HelveticaOblique:
        case PdfStandard14FontType::HelveticaBold:
        case PdfStandard14FontType::HelveticaBoldOblique:
            return HELVETICA_BASE_NAME;
        case PdfStandard14FontType::Courier:
        case PdfStandard14FontType::CourierOblique:
        case PdfStandard14FontType::CourierBold:
        case PdfStandard14FontType::CourierBoldOblique:
            return COURIER_BASE_NAME;
        case PdfStandard14FontType::Symbol:
            return FONT_SYMBOL_STD;
        case PdfStandard14FontType::ZapfDingbats:
            return FONT_ZAPF_DINGBATS_STD;
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR(PdfErrorCode::InvalidEnumValue);
    }
}

const Standard14FontData& PoDoFo::GetStandard14FontData(
    PdfStandard14FontType std14Font)
{
    // The following metrics were copied from libharu
    // /StemV, /ItalicAngle, /FontWeight, /Flags, /FontFamily,
    // /FontStretch and default width values were copied from
    // Acrobat Pro by performing font embedding
    static Standard14FontData PODOFO_BUILTIN_STD14FONT_DATA[] = {
        {
            CHAR_DATA_TIMES_ROMAN,
            (unsigned)std::size(CHAR_DATA_TIMES_ROMAN),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif,
            1000,
            PdfFontStretch::Normal,
            727,
            -273,
            450,
            662,
            0,
            400,
            80,
            -1,
            262,
            -100,
            Corners(-168, -218, 1000, 898)
        },
        {
            CHAR_DATA_TIMES_ITALIC,
            (unsigned)std::size(CHAR_DATA_TIMES_ITALIC),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            727,
            -273,
            441,
            653,
            -17,
            400,
            72,
            -1,
            262,
            -100,
            Corners(-169, -217, 1010, 883)
        },
        {
            CHAR_DATA_TIMES_BOLD,
            (unsigned)std::size(CHAR_DATA_TIMES_BOLD),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif,
            1000,
            PdfFontStretch::Normal,
            727,
            -273,
            461,
            676,
            0,
            700,
            136,
            -1,
            262,
            -100,
            Corners(-168, -218, 1000, 935)
        },
        {
            CHAR_DATA_TIMES_BOLD_ITALIC,
            (unsigned)std::size(CHAR_DATA_TIMES_BOLD_ITALIC),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            727,
            -273,
            462,
            669,
            -17,
            700,
            124,
            -1,
            262,
            -100,
            Corners(-200, -218, 996, 921)
        },
        {
            CHAR_DATA_HELVETICA,
            (unsigned)std::size(CHAR_DATA_HELVETICA),
            PdfFontDescriptorFlags::Symbolic,
            1000,
            PdfFontStretch::Normal,
            750,
            -250,
            523,
            718,
            0,
            400,
            88,
            -1,
            290,
            -100,
            Corners(-166, -225, 1000, 931)
        },
        {
            CHAR_DATA_HELVETICA_OBLIQUE,
            (unsigned)std::size(CHAR_DATA_HELVETICA_OBLIQUE),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            750,
            -250,
            532,
            718,
            -12,
            400,
            92,
            -1,
            290,
            -100,
            Corners(-170, -225, 1116, 931)
        },
        {
            CHAR_DATA_HELVETICA_BOLD,
            (unsigned)std::size(CHAR_DATA_HELVETICA_BOLD),
            PdfFontDescriptorFlags::Symbolic,
            1000,
            PdfFontStretch::Normal,
            750,
            -250,
            532,
            718,
            0,
            700,
            136,
            -1,
            290,
            -100,
            Corners(-170, -228, 1003, 962)
        },
        {
            CHAR_DATA_HELVETICA_BOLD_OBLIQUE,
            (unsigned)std::size(CHAR_DATA_HELVETICA_BOLD_OBLIQUE),
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            750,
            -250,
            532,
            718,
            -12,
            700,
            140,
            -1,
            290,
            -100,
            Corners(-174, -228, 1114, 962)
        },
        {
            CHAR_DATA_COURIER,
            315,
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif,
            1000,
            PdfFontStretch::Normal,
            627,
            -373,
            426,
            562,
            0,
            500,
            56,
            -1,
            261,
            -224,
            Corners(-23, -250, 715, 805)
        },
        {
            CHAR_DATA_COURIER_OBLIQUE,
            315,
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            627,
            -373,
            426,
            562,
            -11,
            500,
            56,
            -1,
            261,
            -224,
            Corners(-27, -250, 849, 805)
        },
        {
            CHAR_DATA_COURIER_BOLD,
            315,
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif,
            1000,
            PdfFontStretch::Normal,
            627,
            -373,
            439,
            562,
            0,
            700,
            92,
            -1,
            261,
            -224,
            Corners(-113, -250, 749, 801)
        },
        {
            CHAR_DATA_COURIER_BOLD_OBLIQUE,
            315,
            PdfFontDescriptorFlags::Symbolic | PdfFontDescriptorFlags::Serif | PdfFontDescriptorFlags::Italic,
            1000,
            PdfFontStretch::Normal,
            627,
            -373,
            439,
            562,
            -11,
            700,
            92,
            -1,
            261,
            -224,
            Corners(-57, -250, 869, 801)
        },
        {
            CHAR_DATA_SYMBOL,
            (unsigned)std::size(CHAR_DATA_SYMBOL),
            PdfFontDescriptorFlags::Symbolic,
            0,
            PdfFontStretch::Unknown,
            683,
            -217,
            462,
            669,
            0,
            -1,
            75,
            92,
            341,
            -100,
            Corners(-180, -293, 1090, 1010)
        },
        {
            CHAR_DATA_ZAPF_DINGBATS,
            (unsigned)std::size(CHAR_DATA_ZAPF_DINGBATS),
            PdfFontDescriptorFlags::Symbolic,
            0,
            PdfFontStretch::Unknown,
            683,
            -217,
            462,
            669,
            0,
            -1,
            75,
            50,
            341,
            -100,
            Corners(-1, -143, 981, 820)
        }
    };

    switch (std14Font)
    {
        case PdfStandard14FontType::TimesRoman:
            return PODOFO_BUILTIN_STD14FONT_DATA[0];
        case PdfStandard14FontType::TimesItalic:
            return PODOFO_BUILTIN_STD14FONT_DATA[1];
        case PdfStandard14FontType::TimesBold:
            return PODOFO_BUILTIN_STD14FONT_DATA[2];
        case PdfStandard14FontType::TimesBoldItalic:
            return PODOFO_BUILTIN_STD14FONT_DATA[3];
        case PdfStandard14FontType::Helvetica:
            return PODOFO_BUILTIN_STD14FONT_DATA[4];
        case PdfStandard14FontType::HelveticaOblique:
            return PODOFO_BUILTIN_STD14FONT_DATA[5];
        case PdfStandard14FontType::HelveticaBold:
            return PODOFO_BUILTIN_STD14FONT_DATA[6];
        case PdfStandard14FontType::HelveticaBoldOblique:
            return PODOFO_BUILTIN_STD14FONT_DATA[7];
        case PdfStandard14FontType::Courier:
            return PODOFO_BUILTIN_STD14FONT_DATA[8];
        case PdfStandard14FontType::CourierOblique:
            return PODOFO_BUILTIN_STD14FONT_DATA[9];
        case PdfStandard14FontType::CourierBold:
            return PODOFO_BUILTIN_STD14FONT_DATA[10];
        case PdfStandard14FontType::CourierBoldOblique:
            return PODOFO_BUILTIN_STD14FONT_DATA[11];
        case PdfStandard14FontType::Symbol:
            return PODOFO_BUILTIN_STD14FONT_DATA[12];
        case PdfStandard14FontType::ZapfDingbats:
            return PODOFO_BUILTIN_STD14FONT_DATA[13];
        case PdfStandard14FontType::Unknown:
        default:
            PODOFO_RAISE_ERROR_INFO(PdfErrorCode::InvalidFontData, "Invalid Standard14 font type");
    }
}

