/*
 * Handle reading and interpreting data from the x10.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: read.c,v 1.5 2000/02/06 01:25:42 swbrown Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "x10.h"
#include "error.h"
#include "read.h"
#include "monitor.h"
#include "ppowerd.h"
#include "ppowerd-proto.h"
#include "conf-proto.h"
#include "status.h"
#include "event.h"

/*
 * Address buffer.  The cm11a sometimes sends buffers that have trailing
 * addresses but no functions.  These functions appear in the next buffer. 
 * To deal with this, we keep a buffer of addresses.  I'm figuring that
 * since you can only multiple-address things in the same housecode that the
 * max size should be 16 addresses.
 */
static unsigned char address_buffer[16];
static int address_buffer_count=0;
static unsigned char address_buffer_housecode;


/* Handle reading and handling requests from the x10. */
void read_x10(X10 *x10) {
	unsigned char command;
	char buffer[16];
	
	/* Read the byte sent by the x10 hardware. */
	if(x10_read(x10, &command, 1) != 1) {
		debug(DEBUG_UNEXPECTED, "Could not read command byte.");
		return;
	}
	
	/* Is this a data poll? */
	if(command == 0x5a) {
		debug(DEBUG_STATUS, "Received poll from x10.");
		read_x10_poll(x10);
	}
	
	/* Is this a power-fail time request poll? */
	else if(command == 0xa5) {
		debug(DEBUG_STATUS, "Received power-fail time request poll from x10.");
		
		/* Build a time response to send to the hardware. */
		buffer[0]=(char) 0x9b;
		x10_build_time(&buffer[1], time(NULL), conf_housecode, TIME_TIMER_PURGE);
		
		/* Send this response to the hardware. */
		if(x10_write_message(x10, &buffer, 7) != 0) {
			
			/* 
			 * This shouldn't fail, the cm11a blocks in this
			 * mode until it is answered.  The only way it
			 * should fail is if we are in this mode due to
			 * static and a poll came in to block us.
			 */
			debug(DEBUG_UNEXPECTED, "Timeout trying to send power-fail time request.");
			return;
		}
	}
	
	/* It was an unknown command (probably static or leftovers). */
	else {
		debug(DEBUG_UNEXPECTED, "Unknown command byte from x10: %02x.", command);
		return;
	}
	
	return;
}


/* 
 * Handles a data poll request from the x10. 
 *
 * *** 
 * Need to do better checking that the event is valid.  Make sure things
 * that need at least one device are getting them.  Sometimes after not
 * being listened to for a while, the cm11a will send us a buffer with a
 * function like 'ON' in it but no addresses.
 */
void read_x10_poll(X10 *x10) {
	unsigned char x10_buffer[8];
	unsigned char command;
	unsigned char buffer_size;
	unsigned char function_byte;
	char string_buffer[256];
	int i,j;
	Event event;
	
	/* Acknowledge the x10's poll. */
	command=0xc3;
	if(x10_write(x10, &command, 1) != 1) {
		debug(DEBUG_UNEXPECTED, "Gave up waiting to write an acknowledgement.");
		return;
	}
	debug(DEBUG_STATUS, "Poll acknowledgement sent.");
	
	/* Get the size of the request. */
	if(x10_read(x10, &buffer_size, 1) != 1) {
		
		/* 
		 * Errors here are unexpected since if we didn't get the ack
		 * through, we should at least read another 'poll' byte
		 * here.
		 */
		debug(DEBUG_UNEXPECTED, "Gave up trying to read the buffer size.");
		return;
	}
	debug(DEBUG_STATUS, "Request size: %i.", buffer_size);
	
	/* Must have at least 2 bytes or it's just weird. */
	if(buffer_size < 2) {
		debug(DEBUG_UNEXPECTED, "Short request from x10.");
		return;
	}
	
	/* Read in the function byte. */
	if(x10_read(x10, &function_byte, 1) != 1) {
		debug(DEBUG_UNEXPECTED, "Could not read function byte.");
		return;
	}
	
	/* Read in the buffer from the x10. */
	if(x10_read(x10, &x10_buffer, buffer_size - 1) != buffer_size - 1) {
		debug(DEBUG_UNEXPECTED, "Gave up while reading the buffer.");
		return;
	}
	
	/* Print packet info to debug. */
	sprintf(string_buffer, "X10 packet> size: %i, function %02x, data:", buffer_size, function_byte);
	for(i=0; i < buffer_size - 1; i++) {
		sprintf(string_buffer + strlen(string_buffer), " %02x", x10_buffer[i]);
	}
	debug(DEBUG_STATUS, string_buffer);
	
	/* Decode the packet. */
	for(i=0; i < buffer_size - 1; i++) {
		
		/* Was this byte a function? */
		if(function_byte & 1) {
			
			/* 
			 * If the housecode doesn't match the addresses,
			 * flush the address buffer.
			 */
			if((x10_buffer[i] >> 4) != address_buffer_housecode) {
				debug(DEBUG_EXPECTED, "Function housecode and address housecode mismatch.");
				address_buffer_count=0;
			}
			
			/* Package the event. */
			event.command=x10_buffer[i] & 0x0f;
			event.housecode=x10_buffer[i] >> 4;
			event.devices=address_buffer_count;
			event.time=time(NULL);
			for(j=0; j < address_buffer_count; j++) {
				event.device[j]=address_buffer[j];
			}
			
			/* Was it a function that needs an extra byte? */
			if(event.command == COMMAND_DIM || event.command == COMMAND_BRIGHT) {
				
				/* We'd better have an extra byte. */
				if((i + 1) >= (buffer_size - 1)) {
					debug(DEBUG_UNEXPECTED, "Missing extra byte of function.");
					continue;
				}
				
				/* Save the extra byte and advance 1. */
				event.extended1=x10_buffer[++i];
				function_byte=function_byte >> 1;
			}
			
			/* Was it a function that needs two extra bytes? */
			else if(event.command == COMMAND_EXTENDED_DATA_TRANSFER) {
				
				/* We'd better have two extra bytes. */
				if((i + 2) >= (buffer_size - 1)) {
					debug(DEBUG_UNEXPECTED, "Missing extra 2 bytes of function.");
					continue;
				}
				
				/* Save the extra bytes and advance 2. */
				event.extended1=x10_buffer[++i];
				event.extended2=x10_buffer[++i];
				function_byte=function_byte >> 2;
			}
			
			/* Send the event to all monitoring clients. */
			monitor_broadcast(&event, sizeof(Event));
			
			/* Take action on this event. */
			event_action(&event);
			
			/* Flush the address buffer. */
			address_buffer_count=0;
		}
		
		/* This was an address byte. */
		else {
			
			/* 
			 * If this is the first byte, set the address
			 * housecode.
			 */
			if(address_buffer_count == 0) {
				address_buffer_housecode=x10_buffer[i] >> 4;
			}
			
			/*
			 * If this address doesn't match the housecode of
			 * the addresses we were storing, we start over with
			 * this address.  This should match how the hardware
			 * handles things like A1,B2,function.
			 */
			else if(address_buffer_housecode != x10_buffer[i] >> 4) {
				debug(DEBUG_UNEXPECTED, "Address buffer housecode mismatch.");
				
				address_buffer_housecode=x10_buffer[i] >> 4;
				address_buffer_count=0;
			}
			
			/* The address buffer better not be full. */
			else if(address_buffer_count == 16) {
				debug(DEBUG_UNEXPECTED, "Address buffer overflow.");
				
				/* Just ignore this address. */
				continue;
			}
				
			/* Save it on the address buffer. */
			address_buffer[address_buffer_count++]=x10_buffer[i] & 0x0f;
		}
		
		/* Shift the function byte, we've handled this one. */
		function_byte=function_byte >> 1;
	}
}
