/*
 * Commandline parsing functions for the ppower client.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: parse.c,v 1.7 1999/06/12 19:44:50 kefka Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include "x10.h"
#include "ppower.h"
#include "ppowerd.h"
#include "error.h"
#include "conf-proto.h"
#include "conf.h"

/* Prototypes. */
static void parse_device(unsigned char *device, int *devices, char *name);
static void parse_command_lookup(Client_Command *client_command, char *name);


/*
 * Parse the command given to ppower into a client command.
 * 
 * ***
 * This parser is cheesy and hacky and in bad need of a rewrite.  Functions
 * are strewn all over the place.
 */
void parse_command(char *command) {
	int is_group_target;
	int housecode_index;
	char saved_char;
	char *stringpos;
	char *namepos;
	unsigned char device[256];
	int devices=0;
	int value;
	Client_Command client_command;
	int i,j;
	
	/* If the name is empty, that's bad. */
	if(command[0] == 0) {
		fatal("Unrecognized command.");
	}
	
	/* Initialize the client command. */
	client_command.devices=0;
	
	/* Is this a group target?  Eg, just a single letter. */
	if(command[1] == ':') {
		is_group_target=1;
		
		/* Make sure it's in range. */
		housecode_index=tolower(command[0]) - 'a';
		if(housecode_index >= 16) {
			fatal("Invalid group housecode '%c'.", command[0]);
		}
		
		/* Save this as the housecode. */
		client_command.housecode=housecode_table[housecode_index];
	
		/* Save the position of the name. */
		namepos=&command[1];
		
		debug(DEBUG_STATUS, "Preparing group command on %c.", command[0]);
	}
	
	/* We need to get all the listed devices. */
	else {
		is_group_target=0;
		
		/* Keep looping until we reach the end of the devices. */
		for(namepos=command; (*namepos != ':') && (*namepos != 0); namepos=stringpos) {
		
			/* Find the end of this device name. */
			for(stringpos=namepos; (*stringpos != ':') && (*stringpos != 0) && (*stringpos != ','); stringpos++);
			
			/* Look this device up and add it to the devices. */
			saved_char=*stringpos;
			*stringpos=0;
			parse_device(device, &devices, namepos);
			*stringpos=saved_char;
			
			/* If this is a comma, jump over it. */
			if(*stringpos == ',') stringpos++;
		}
			
		/* We must have at least one device. */
		if(!devices) {
			fatal("No devices specified.");
		}
	}
	
	/* We better have a command. */
	if(*namepos != ':') {
		fatal("Missing command name.");
	}
	
	/* Lowercase the command name and find the end. */
	for(stringpos=namepos + 1; (*stringpos != ':') && (*stringpos != 0); stringpos++) {
		*stringpos=tolower(*stringpos);
	}
	
	/* Look the command name up and save it in the client command. */
	saved_char=*stringpos;
	*stringpos=0;
	parse_command_lookup(&client_command, namepos + 1);
	*stringpos=saved_char;
	namepos=stringpos;
	
	/* Does this command need a value?  Eg, dim and bright. */
	if(client_command.command == COMMAND_DIM || client_command.command == COMMAND_BRIGHT) {
		
		/* We'd better have a value. */
		if(*namepos != ':') {
			fatal("Missing value for dim/bright.");
		}
		namepos++;
		if(*namepos < '0' || *namepos > '9') {
			fatal("Missing value for dim/bright.");
		}
		
		/* Find the end of this value. */
		for(stringpos=namepos; *stringpos != 0; stringpos++);
		
		/* Convert the value to an int. */
		errno=0;
		value=strtol(namepos, &stringpos, 10);
		if(errno) {
			fatal("Invalid dim/bright value.");
		}
		
		/* The value must be 0-22. */
		if(value < 0 || value > 22) {
			fatal("Dim/bright value '%i' out of range, must be 0-22.", value);
		}
		
		/* Save it in the command. */
		client_command.value=value;
	}
	
	/* *** Is this an extended command needing a value? */
	
	/* We had better be at the end of this command. */
	if(*stringpos != 0) {
		fatal("Extra characters on commandline.");
	}
	
	/* We will be sending a request to the daemon. */
	client_command.request=REQUEST_COMMAND;
	
	/*
	 * If this is a group target, just send the command as-is.
	 */
	if(is_group_target) {
		debug(DEBUG_STATUS, "Sending group command.");
		ppower_command(&client_command);
	}
	
	/* 
	 * Send the devices in groups based on housecode.
	 */
	else {
		
		/* Loop through each housecode, ugly huh? */
		for(i=0; i < 16; i++) {
			
			/* Clear the device count and set the housecode. */
			client_command.devices=0;
			client_command.housecode=i;
			
			/* Add each device that matches this housecode. */
			for(j=0; j < devices; j++) {
				if((device[j] >> 4) == i) {
					client_command.device[client_command.devices++]=device[j] & 0x0f;
				}
			}
			
			/* If we have at least one device, send it. */
			if(client_command.devices) {
				debug(DEBUG_STATUS, "Sending command.");
				ppower_command(&client_command);
			}
		}
	}
	
	return;
}


/*
 * Look up the command name specified and set it in the client command.
 */
static void parse_command_lookup(Client_Command *client_command, char *name) {
	
	/* *** Hack for now. */
	if(strcasecmp(name, "all_units_off") == 0) {
		client_command->command=COMMAND_ALL_UNITS_OFF;
	}
	else if(strcasecmp(name, "all_lights_on") == 0) {
		client_command->command=COMMAND_ALL_LIGHTS_ON;
	}
	else if(strcasecmp(name, "on") == 0) {
		client_command->command=COMMAND_ON;
	}
	else if(strcasecmp(name, "off") == 0) {
		client_command->command=COMMAND_OFF;
	}
	else if(strcasecmp(name, "dim") == 0) {
		client_command->command=COMMAND_DIM;
	}
	else if(strcasecmp(name, "bright") == 0) {
		client_command->command=COMMAND_BRIGHT;
	}
	else if(strcasecmp(name, "all_lights_off") == 0) {
		client_command->command=COMMAND_ALL_LIGHTS_OFF;
	}
	else if(strcasecmp(name, "extended_code") == 0) {
		client_command->command=COMMAND_EXTENDED_CODE;
	}
	else if(strcasecmp(name, "hail_request") == 0) {
		client_command->command=COMMAND_HAIL_REQUEST;
	}
	else if(strcasecmp(name, "hail_acknowledge") == 0) {
		client_command->command=COMMAND_HAIL_ACKNOWLEDGE;
	}
	else if(strcasecmp(name, "preset_dim1") == 0) {
		client_command->command=COMMAND_PRESET_DIM1;
	}
	else if(strcasecmp(name, "preset_dim2") == 0) {
		client_command->command=COMMAND_PRESET_DIM2;
	}
	else if(strcasecmp(name, "extended_data_transfer") == 0) {
		client_command->command=COMMAND_EXTENDED_DATA_TRANSFER;
	}
	else if(strcasecmp(name, "status_on") == 0) {
		client_command->command=COMMAND_STATUS_ON;
	}
	else if(strcasecmp(name, "status_off") == 0) {
		client_command->command=COMMAND_STATUS_OFF;
	}
	else if(strcasecmp(name, "status_request") == 0) {
		client_command->command=COMMAND_STATUS_REQUEST;
	}
	else {
		fatal("Invalid command name, '%s'.", name);
	}
	
	return;	
}


/* 
 * Look up a named device specified on the commandline.  This can be an
 * obvious one like 'a2', or it could be a name like 'light1'.  It can
 * expand into multiple devices.
 *
 * This will fill a 'device' buffer with all the devices associated with the
 * device/alias.
 */
static void parse_device(unsigned char *device, int *devices, char *name) {
	Stacklist_Entry *stacklist_entry;
	Alias *alias;
	unsigned char housecode;
	unsigned char devicecode;
	char *endptr;
	int devicecode_index;
	int i;
	
	/* Can we find this name in the list of aliases? */
	for(stacklist_entry=conf_aliaslist.first; stacklist_entry != NULL; stacklist_entry=stacklist_entry->previous) {
		alias=(Alias *) stacklist_entry->info;
		
		/* If the names match, we take its devices. */
		if(strcasecmp(name, alias->name) == 0) {
			
			/* Copy all the devices. */
			for(i=0; i < alias->devices; i++) {
				
				/* Save the device as long as we have space. */
				if(*devices >= 256) {
					fatal("Must not specify more than 256 devices.");
				}
				device[(*devices)++]=alias->device[i]->housecode << 4 | alias->device[i]->devicecode;
				
				debug(DEBUG_STATUS, "Adding device to command: housecode 0x%02X, devicecode: 0x%02X.", alias->device[i]->housecode, alias->device[i]->devicecode);
			}
			
			/* We're done. */
			return;
		}
	}
	
	/* 
	 * We didn't exist in the configuration file, check if this name is
	 * in valid housecode:devicecode form.
	 */
	
	/* Is this an empty string? */
	if(name[0] == 0) {
		fatal("Empty device name.");
	}
	
	/* The first char must be a-p. */
	if(tolower(name[0]) < 'a' || tolower(name[1]) > 'p') {
		fatal("Invalid device name, '%s'.", name);
	}
	housecode=housecode_table[tolower(name[0]) - 'a'];
	
	/* The number must be 1-16. */
	if(name[1] < '0' || name[1] > '9') {
		fatal("Invalid device name, '%s'.", name);
	}
	errno=0;
	devicecode_index=strtol(&name[1], &endptr, 10);
	if(errno || devicecode_index < 1 || devicecode_index > 16) {
		fatal("Invalid device name, '%s'.", name);
	}
	devicecode=devicecode_table[devicecode_index - 1];
	
	/* There must be nothing after it. */
	if(*endptr != 0) {
		fatal("Invalid device name, '%s'.", name);
	}
	
	/* 
	 * Give a warning about this device not being listed in the
	 * configuration file and save it to the device table.
	 */
	warn("Device '%s' was not listed in the configuration file.", name);
	device[(*devices)++]=housecode << 4 | devicecode;
	
	debug(DEBUG_STATUS, "Adding device %c%i to command.", name[0], devicecode_index);
	
	return;
}
