/*
 * Ppower client program.  Handles making requests to the daemon.
 * Copyright (C) 1999  Steven Brown
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Steven Brown <swbrown@ucsd.edu>
 *
 * $Id: ppower.c,v 1.9 1999/08/02 18:58:19 kefka Exp $
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <getopt.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/poll.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include "error.h"
#include "socket.h"
#include "ppower.h"
#include "ppowerd.h"
#include "x10.h"
#include "parse.h"
#include "conf.h"
#include "conf-proto.h"
#include "error-proto.h"
#include "pid.h"
#include "status.h"
#include "snprintf.h"

/* Prototypes. */
static void ppower_show_help(void);
static void ppower_show_version(void);
static void ppower_show_usage(void);
static void ppower_monitor(void);

/* Name of the program. */
char *progname;

/* Name of the conf file. */
char *conf_name=CONFIG_FILE;

/* Mode ppower is to run in. */
int ppower_mode=PPOWER_MODE_DEFAULT;

/* Commandline options. */
static struct option long_options[] = {
	{"version", 0, 0, 'v'},
	{"help", 0, 0, 'h'},
	{"monitor", 0, 0, 'm'},
	{"status", 0, 0, 's'},
	{"info", 0, 0, 'i'},
	{"conf", 1, 0, 'f'},
	{"debug", 1, 0, 'd'},
	{"refresh", 0, 0, 0},
	{"shutdown", 0, 0, 0},
	{0, 0, 0, 0}
};
#define SHORT_OPTIONS "vhmsif:d:"


/* Main... */
int main(int argc, char *argv[]) {
	int longindex;
	int optchar;
	pid_t pid;
	char pid_buffer[PATH_MAX];
	
	/* Save the name of the program. */
	progname=argv[0];
	
	/* Parse the arguments. */
	while((optchar=getopt_long(argc, argv, SHORT_OPTIONS, long_options, &longindex)) != EOF) {
		
		/* Handle each argument. */
		switch(optchar) {
			
			/* Was it a long option? */
			case 0:
				
				/* Was it a refresh request? */
				if(strcmp("refresh", long_options[longindex].name) == 0) {
					
					/* We're in refresh mode now. */
					ppower_mode=PPOWER_MODE_REFRESH;
					
					break;
				}
				
				/* Was it a shutdown request? */
				else if(strcmp("shutdown", long_options[longindex].name) == 0) {
					
					/* We're in shutdown mode now. */
					ppower_mode=PPOWER_MODE_SHUTDOWN;
					
					break;
				}
				
				/* Hrmm, something we don't know about? */
				panic("Unhandled long getopt option '%s'.", long_options[longindex].name);
			
			/* If it was an error, exit right here. */
			case '?':
				exit(1);
			
			/* Was it a version request? */
			case 'v':
				ppower_show_version();
				exit(0);
			
			/* Was it a help request? */
			case 'h':
				ppower_show_help();
				exit(0);
			
			/* Was it a monitor request? */
			case 'm':
				ppower_mode=PPOWER_MODE_MONITOR;
				break;
				
			/* Was it a status request? */
			case 's':
				ppower_mode=PPOWER_MODE_STATUS;
				break;
				
			/* Was it an info request? */
			case 'i':
				ppower_mode=PPOWER_MODE_INFO;
				break;
			
			/* Was it an alternate conf file? */
			case 'f':
				
				/* Save the alternate conf file name. */
				conf_name=strdup(optarg);
				if(!conf_name) {
					fatal("Out of memory.");
				}
				
				break;
			
			/* Was it a debug level set? */
			case 'd':
				
				/* Save the value. */
				debuglvl=strtol(optarg, NULL, 10);
				if((debuglvl == LONG_MIN || debuglvl == LONG_MAX) && errno) {
					fatal("Invalid debug level.");
				}
				if(debuglvl < 0 || debuglvl > DEBUG_MAX) {
					fatal("Invalid debug level.");
				}
				
				break;
				
			/* It was something weird.. */
			default:
				panic("Unhanlded getopt return value %i.", optchar);
		}
	}
	
	/* 
	 * If we are in default mode and don't have one argument, that's
	 * invalid use.
	 */
	if(ppower_mode == PPOWER_MODE_DEFAULT && (optind != argc - 1)){
		ppower_show_usage();
		exit(1);
	}
	
	/* Parse our conf file for settings. */
	conf_parse(conf_name);
	
	/* Create the path of the pid file. */
	if(snprintf(pid_buffer, PATH_MAX, "%s/%s", conf_daemon_dir, DAEMON_PID_FILE) == -1) {
		fatal("Pid file path too long.");
	}
	
	/* Get the pid for the daemon. */
	pid=pid_read(pid_buffer);
	if(pid == -1) {
		fatal("Daemon isn't running.  Run it first.");
	}
		
	/* If we are in default mode, it's a command. */
	if(ppower_mode == PPOWER_MODE_DEFAULT) {
		
		/* 
		 * Parse the command into a client command to feed to the
		 * daemon.
		 */
		parse_command(argv[optind]);
	}
	
	/* Are we in shutdown mode? */
	else if(ppower_mode == PPOWER_MODE_SHUTDOWN) {
		
		/* Kill the daemon. */
		if(kill(pid, SIGTERM) != 0) {
			fatal("Failed to shutdown daemon, could not send SIGTERM to pid %i.", pid);
		}
		
		/* We're done. */
		exit(0);
	}
	
	/* Are we in refresh mode? */
	else if(ppower_mode == PPOWER_MODE_REFRESH) {
		
		/* Refresh the daemon. */
		if(kill(pid, SIGHUP) != 0) {
			fatal("Failed to refresh daemon, could not send SIGHUP to pid %i.", pid);
		}
		
		/* We're done. */
		exit(0);
	}
	
	/* We haven't written status yet.. */
	else if(ppower_mode == PPOWER_MODE_STATUS) {
		error("Status not implemented.");
	}
	
	/* *** We must be some sort of monitor command. */
	else {
		ppower_monitor();
	}
	
	/* We're done. */
	return(0);
}


/*
 * Send a command to the daemon.
 */
void ppower_command(Client_Command *client_command) {
	int daemon_socket;
	struct pollfd pollfd;
	
	/* Open a connection to the daemon. */
	daemon_socket=socket_connect(conf_daemon_socket_path);
	
	/* Save the poll() info for the socket. */
	pollfd.fd=daemon_socket;
	pollfd.events=POLLIN | POLLHUP;
	
	/* Write the command to the daemon. */
	if(write(daemon_socket, client_command, sizeof(Client_Command)) != sizeof(Client_Command)) {
		fatal("Failed to pass command to daemon.");
	}
	
	/* Wait for this socket to be closed indicating the daemon worked. */
	debug(DEBUG_STATUS, "Waiting for daemon to close socket.");
	for(;;) {
		if(poll(&pollfd, 1, -1) == -1) {
			if(errno == EINTR) {
				debug(DEBUG_EXPECTED, "Signal in poll, restarting.");
				continue;
			}
			fatal("Error polling socket.");
		}
		else break;
	}
	debug(DEBUG_STATUS, "Socket activity detected.");
	
	/* Close the socket. */
	if(close(daemon_socket) != 0) {
		fatal("Error closing socket.");
	}
	debug(DEBUG_STATUS, "Command socket closed.");
	
	return;
}


/* 
 * Make a monitor connection with the daemon for getting info, status, or to
 * watch events.
 */
static void ppower_monitor(void) {
	int daemon_monitor_socket;
	int retval;
	Event event;
	
	/* Open a monitor connection to the daemon. */
	daemon_monitor_socket=socket_connect(conf_daemon_monitor_socket_path);
	
	/* Display all the info from the monitor. */
	for(;;) {
		
		/* Get an event. */
		retval=read(daemon_monitor_socket, &event, sizeof(Event));
		if(retval != sizeof(Event)) {
			
			/* We lost our connection to the daemon so exit. */
			fatal("Lost monitor connection to daemon.");
		}
		
		/* Display the event. */
		status_display_event(&event);
	}
	
	return;
}


/* Show usage info for ppower (on error). */
static void ppower_show_usage(void) {
	fprintf(stderr, "%s: You must specify an option or command.\n", progname);
	fprintf(stderr, "Try '%s --help' for more information.\n", progname);
	return;
}


/* Show the help screen for ppower. */
static void ppower_show_help(void) {
	printf("'ppower' controls or monitors the ppowerd daemon to turn x10 switches on,\n");
	printf("off, dim, and bright.\n");
	printf("\n");
	printf("Usage: %s [OPTION]... [COMMAND]\n", progname);
	printf("\n");
	printf("  -v, --version           display program version\n");
	printf("  -h, --help              give help on usage\n");
	printf("  -m, --monitor           monitors the x10 hardware for changes\n");
	printf("  -s, --status            reports status of known x10 devices\n");
	printf("  -f, --conf=FILE	  specify alternate configuration file\n");
	printf("  -d, --debug=LEVEL       set the debug level, 0 is off, the\n");
	printf("                          compiled in default is %i and the max\n", DEBUGLVL);
	printf("                          level allowed is %i\n", DEBUG_MAX);
	printf("      --refresh           force the daemon to reload the\n");
	printf("                          configuration file; this is mainly\n");
	printf("                          for devices, aliases and macros; sockets\n");
	printf("                          and permissions won't be affected\n");
	printf("      --shutdown          stop the currently running daemon\n");
	printf("\n");
	printf("a COMMAND is of the format device,device,...:state<:value>.\n");
	printf("- The device is a housecode and devicecode, like 'A1' or a named\n");
	printf("  representation like 'television'.\n");
	printf("- The status is one of ON, OFF, DIM, or BRIGHT.\n");
	printf("- DIM and BRIGHT require a value from 0 to 22.  22 corresponds to\n");
	printf("  100%% change.\n");
	printf("\n");
	printf("Report bugs to <swbrown@ucsd.edu>.\n");
	return;
}


/* Show the version info for ppower. */
static void ppower_show_version(void) {
	printf("ppower (%s) %s\n", PACKAGE, VERSION);
}
