# Secure transfers with mkcert

_Published on October 14th, 2020, by [Claudio d'Angelis](https://claudiodangelis.com)._


In this tutorial you will learn how to securely transfer files with qrcp by creating a local Certificate Authority using [mkcert](https://github.com/FiloSottile/mkcert) and generating a certificate for `qrcp`. From its README, **mkcert** _is a simple tool for making locally-trusted development certificates. It requires no configuration._

The following values will be used for this tutorial, they refer to a typical Linux system and may differ for you according to your operating system:

| Name  | Value |
| --- | --- |
| IP of the computer/laptop | `192.168.1.107` |
| Root certificate path | `~/.local/share/mkcert/rootCA.pem` |
| Certificate path | `~/certs/192.168.1.107.pem` |
| Certificate key | `~/certs/192.168.1.107-key.pem` | 
| Transferred file | `MyDocument.pdf` |

**Note**: secure transfers are only supported by `qrcp` version `0.7.0` and above.

## Generate certificates

Open your terminal, create a directory called `certs` and change to it
Install **mkcert** _(refer to the [README](https://github.com/FiloSottile/mkcert#installation))_.

Generate the Certificate Authority:
```
mkcert --install
```

If everything worked correctly, you should see the similar output:
```
Created a new local CA at "/home/me/.local/share/mkcert"
Sudo password:
The local CA is now installed in the system trust store! ⚡️
The local CA is now installed in the Firefox and/or Chrome/Chromium trust store (requires browser restart)!
```

Generate a certificate for the IP of your computer/laptop:

```
mkcert 192.168.1.107
```
You should see a similar output:

```
Using the local CA at "/home/me/.local/share/mkcert"

Created a new certificate valid for the following names 
 - "192.168.1.107"

The certificate is at "./192.168.1.107.pem" and the key at "./192.168.1.107-key.pem" 

```

At this point you should securely upload the root certificate generated by mkcert to your mobile phone. You have a few options for this, the simplest is just sending it by email. You can check the location of the root certificate's parent directory by running:

```
mkcert --CAROOT
```

If you are using **iOS**, you should _trust_ the certificate authority, you can find more information here: [Trust manually installed certificate profiles in iOS and iPadOS](https://support.apple.com/en-nz/HT204477).

If you are using **Android**, you must convert the root certificate from PEM format to DER format. Note that you may need to install the `openssl` tool.

```
openssl x509 -inform PEM -outform DER -in $(mkcert --CAROOT)/rootCA.pem -out $(mkcert --CAROOT)/rootCA.der.crt
```

Upload the converted certificate to your Android device, and install it by simply opening the file. When asked to enter the certificate name, you can enter a friendly name, for example "mkcert".

## Transfer something!

You are now ready to securely transfer a file using the certificates generated by mkcert:

```
qrcp --tls-cert ~/certs/192.168.1.107.pem --tls-key ~/certs/192.168.1.107-key.pem screenshot.png
```

The output will be something like this (note that the printed URL starts with "https"):
```
Scan the following URL with a QR reader to start the file transfer:
https://192.168.1.107:40221/send/ljd7
█████████████████████████████████████
█████████████████████████████████████
████ ▄▄▄▄▄ █ █▀▀ █▀ ▄█▄▀▀█ ▄▄▄▄▄ ████
████ █   █ █▀█▄▄▄▄▄▄█▄█▄██ █   █ ████
████ █▄▄▄█ ██▀▄▄▀▄▀█▀█ ▀▄█ █▄▄▄█ ████
████▄▄▄▄▄▄▄█ ▀▄█ ▀ ▀ ▀ █ █▄▄▄▄▄▄▄████
████ ▀▄ ▀▄▄ ▀██▀▄▄▄▄   ▄█▀▄██▄█▄ ████
████▀▀   █▄▀▀ ▄▄█▀ █▀▄█▄▀ ▄▄██▄▄▀████
████▀█ ▀█▄▄█ ▄██ ▀██▀▀  ▄█▀▀█▄ ▀▀████
████  █▀█▀▄ █  ▄ ▀▄   ▀▄██ █▀█▄ ▄████
█████ ▀ ▀▀▄  ▄▀█ ▀██▄█▄▀▄██ ▀ ▀ █████
████▄████▄▄  ▀█ ▀████▄█▄ ▄▀█ █ ██████
█████▄█▄▄▄▄█▀ ▀█  ▀ ▀▀   ▄▄▄   ▀▀████
████ ▄▄▄▄▄ █  █▄▄▄█ █▀█  █▄█  █▄▀████
████ █   █ █▀ ▀ ▄▀ █ █▀ ▄   ▄▀▄▄▄████
████ █▄▄▄█ █▄█▄█ ▀▀▀▄▀ █▄▄ █▀█▄▀▄████
████▄▄▄▄▄▄▄█▄▄▄██▄▄▄█▄▄██▄█▄█▄▄▄█████
█████████████████████████████████████
▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀
```

Scan the QR and the file will be securely transferred to your mobile. Congratulations!

## Configure qrcp to default to those value

If you want to make this setup persistent so that all transfers will be secured by default, run the wizard with `qrcp config` and make sure the choose the right values to the following questions:

_(**Note**: you must enter the **absolute path** of the certificates)_

- Should files be securely transferred with HTTPS?  
  **Yes**
- Choose TLS certificate path. Empty if not using HTTPS  
  **/home/user/certs/192.168.1.107.pem**
- Choose TLS certificate key. Empty if not using HTTPS  
  **/home/user/certs/192.168.1.107-key.pem**

After configuring qrcp, all transfers will be always secured.
If you'll want to temporarily disable security, just add the `--secure=false` flag.

Secure transfer:
```
qrcp MyDocument.pdf
```

Insecure transfer:
```
qrcp --secure=false MyDocument.pdf
```

## Conclusion

In this tutorial we have seen how to:
- create a local certificate authority with `mkcert`
- issue a certificate for your computer/laptop
- use that certificate to secure the transferring of a file with `qrcp`

If you want to learn more about HTTPS and secure connections, have a look at the [How HTTPS works ...in a comic!](https://howhttps.works/) website.

## Useful Links

- [qrcp homepage](https://github.com/claudiodangelis/qrcp)
- [mkcert homepage](https://github.com/FiloSottile/mkcert)
- [qrcp Telegram channel](https://t.me/qrcp_dev)