/*
 * Copyright (c) 1999 Distributed Processing Technology Corporation
 * All rights reserved.
 *
 * Redistribution and use in source form, with or without modification, are
 * permitted provided that redistributions of source code must retain the
 * above copyright notice, this list of conditions and the following disclaimer.
 *
 * This software is provided `as is' by Distributed Processing Technology and
 * any express or implied warranties, including, but not limited to, the
 * implied warranties of merchantability and fitness for a particular purpose,
 * are disclaimed. In no event shall Distributed Processing Technology be
 * liable for any direct, indirect, incidental, special, exemplary or
 * consequential damages (including, but not limited to, procurement of
 * substitute goods or services; loss of use, data, or profits; or business
 * interruptions) however caused and on any theory of liability, whether in
 * contract, strict liability, or tort (including negligence or otherwise)
 * arising in any way out of the use of this driver software, even if advised
 * of the possibility of such damage.
 *
 * UNIX locking mechanism
 */
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include "lockunix.h"

static char Lock[] = "/tmp/%s.LOCK";
extern int errno;

static char *
getName(char * name)
{
	extern char ** Argv;

	if (name) {
		return (name);
	}
	if ((name = strrchr (Argv[0], '/')) != (char *)NULL) {
		return (name);
	}
	return (Argv[0]);
}

#if defined __GNUC__
int
MkLock __P((char * name))
#else
int
MkLock (char * name)
#endif
{   
#if defined _DPT_SPARC
    extern char * mktemp __P((char * origTemplate));
#endif //_DPT_SPARC
    static char Template[] = "/tmp/%s.XXXXXX";
	char buffer[10];
	int fd;
	char * Name = getName (name);
	char * templateBuffer = (char *)malloc (strlen(Name)+sizeof(Template)-1);
	char * lock;

	if (templateBuffer == (char *)NULL) {
		return (-1);
	}
	if ((lock = (char *)malloc (strlen(Name)+sizeof(Lock)-1)) == (char *)NULL) {
		free (templateBuffer);
		return (-1);
	}
	sprintf (templateBuffer, Template, Name);
	if ((fd = open(mktemp (templateBuffer), O_WRONLY|O_CREAT|O_EXCL)) < 0) {
		free (templateBuffer);
		free (lock);
		return (-1);
	}
	sprintf (buffer, "%d\n", (int)getpid());
	write (fd, buffer, strlen(buffer));
	close (fd);
	sprintf (lock, Lock, Name);
	if (link (templateBuffer, lock) == 0) {
		unlink (templateBuffer);
		free (templateBuffer);
		free (lock);
		return (0);
	}
	/* Check if the file is valid, and thus has a proper PID */
	if ((fd = open(lock, O_RDONLY)) >= 0) {
		read (fd, buffer, sizeof(buffer));
		close (fd);
		if ((fd = atoi (buffer)) == 0) {	/* Empty file is a lock */
			errno = EPERM;
		} else if (kill (fd, 0) == 0) {
			unlink (templateBuffer);
			free (templateBuffer);
			free (lock);
			return (fd != getpid());	/* Is it me? */
		}
		/* Unlink the lock */
		if (errno != EPERM) {
			unlink (lock);
		}
	}
	fd = link (templateBuffer, lock);
	unlink (templateBuffer);
	free (templateBuffer);
	free (lock);
	return (fd);
}

#if defined __GNUC__
void
RmLock __P((char * name))
#else 
void
RmLock (char * name)
#endif
{   char * Name = getName (name);
    char * lock = (char *)malloc (strlen(Name) + sizeof(Lock) - 1);

	if (lock != (char *)NULL) {
		sprintf (lock, Lock, Name);
		unlink (lock);
		free (lock);
	}
}

#if defined __GNUC__
void 
ChLock __P((char * name, int pid))
#else
void
ChLock (char * name, int pid)
#endif
{	char * Name = getName (name);
	int fd;
	char buffer[10];
	char * lock = (char *)malloc (strlen(Name) + sizeof(Lock) - 1);

	if (lock != (char *)NULL) {
		sprintf (lock, Lock, Name);
	    if ((fd = open(lock, O_WRONLY|O_CREAT|O_TRUNC)) >= 0) {
			sprintf (buffer, "%d\n", pid);
			write (fd, buffer, strlen(buffer));
			close (fd);
		}
		free (lock);
	}
}

#if defined __GNUC__
int 
IsLock __P((char * name))
#else 
int
IsLock (char * name)
#endif
{	char * Name = getName (name);
	int fd;
	char buffer[10];
	char * lock = (char *)malloc (strlen(Name) + sizeof(Lock) - 1);

	if (lock == (char *)NULL) {
		return (-1);
	}
	sprintf (lock, Lock, Name);
	fd = open(lock, O_RDONLY);
	free (lock);
	if (fd < 0) {
		return (0);
	}
	read (fd, buffer, sizeof(buffer));
	close (fd);
	if ((fd = atoi (buffer)) == 0) {	/* Empty file is a lock */
		return (-1);					/* In transition, try again */
	} else if (kill (fd, 0) == 0) {
		return (1);
	}
	return (errno == EPERM);	/* Not Locked, permission denied excepted */
}
