/*
   Raw photo loader plugin for The GIMP
   by Pawel T. Jochym jochym at ifj edu pl,
   
   Based on the gimp plug-in by Dave Coffin
   http://www.cybercom.net/~dcoffin/

   This code is licensed under the same terms as The GIMP.
   To simplify maintenance, it calls my command-line "dcraw"
   program to do the actual decoding. 
   

ChangeLog:

Pre-cvs:
2004/04/18 -- 0.1 First public release at http://ptj.rozeta.com.pl/ functional, beta-quality code.
2004/04/21 -- 0.2 lots of corrections - semi functional
2004/04/23 -- 0.3 first functional version, basic functions work

$Log: rawphoto.c,v $
Revision 1.19  2004/10/22 07:27:37  jochym
Endianess handling added. This should work on PPC machines.

Revision 1.18  2004/10/12 23:08:13  jochym
Version string changed to include the hour and minute.
Small tweaks in UI. Histogram debugging code added.

Revision 1.17  2004/10/12 22:23:56  jochym
Some cleanup (not finished yet).
Logarithmic histogram with divisions.
Corrected some errors with histogram ploting.

Revision 1.16  2004/10/12 21:25:33  jochym
RGB Histogram added by Keir Mierle,
Interface improvements (notebook style): Marek Olszewski
Some cleanup

Revision 1.15  2004/10/12 20:36:44  jochym
Just small cosmetics

Revision 1.14  2004/05/27 10:32:38  jochym
This version may compile under windows.
Saturation control added. Small tweaks to
the color processing the shadows curve is improved.
Still problems with some canon files. Very saturated
colors just look dim, the images from Nikon and Minolta
are fine with me (at least the samples I have).
Anyone can help with this? Any ideas? Maybe Canon
uses some non-standard filters in its matrix ?

Revision 1.13  2004/05/18 19:43:18  jochym
Additional supression of the shadows added. Slightly better makefiles.
The greens are still problematic at least for canon 300D files.
I have constant trouble getting the greens to have similar colours
to the ones produced by the canon firmware (embeded JPEG).
They are constantly too dark.
Some adjustement of the curve in the higlights seams necessary too.
I've not decided the shape yet.

Revision 1.12  2004/05/07 14:01:59  jochym
Spot WB! This is the main reason for release.
Plus some small bug fixes for the gui. Just click on the
image and you will get the WB for the 9x9 square around
cursor. So you need just something neutral in the image
to set custom WB. The versioning scheme changed to the
CVS revision number.

Revision 1.11  2004/05/07 10:47:08  jochym
Small bugs in the gui corrected. Un-clipping of the saturated
channels work (usually not recommended to use but you can
if you like to). The plugin is now able to produce satisfactory
results. The spot white balance is next on the list ...

Revision 1.10  2004/05/05 10:08:09  jochym
I think I've got the transfer function right this time (finaly!).
Hue and saturation is unchanged when exposition and
black level are changed (it was ambarasingly simple :).
The same for gamma and Auto-adjust function. The code
need to be cleaned up a bit but works.

Revision 1.9  2004/05/03 22:45:07  jochym
Rewrite of the color handling finished.
The algorithm is slower but correct (I belive). Thanks to
discussions with David and readin of the dcraw code.
Better feedback when loading files (not perfect yet).
The output is of much better quality. The "pale colors"
problem is solved. Some cleanup of the code remains.
This is released as version 0.4

Revision 1.8  2004/05/03 15:29:52  jochym
Complete rewrite of the color handling functions.
This is 0.4pre version.
The cleanup of the code is still pending but the output
is much better then the 0.3.
There are still some buggy corner cases in the code.
This will be cleaned up in the next release.

Revision 1.7  2004/04/27 00:30:43  jochym
Correction to final loading with custom WB. Exposition correction
partially made in dcraw (to get green component at 1.0).

Revision 1.6  2004/04/26 23:27:19  jochym
Avoid clipping in dcraw for preview by pre-mul of
R and B channels in preview and use of the dcraw color scaling
in final load. Code cleanup. Still some cleanup to do.

Revision 1.5  2004/04/26 14:09:14  jochym
Pre-cvs Changelog added.

*/

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <gtk/gtk.h>

#include <libgimp/gimp.h>
#include <libgimp/gimpui.h>

#include <libintl.h>
#include <locale.h>

#define _(String) gettext(String)

#include "blackbody.h"

#define PLUG_IN_VERSION  "$Id: rawphoto.c,v 1.19 2004/10/22 07:27:37 jochym Exp $"

#define DEBUG

#ifdef WIN32
#define PIPE_MODE "rb"
#else
#define PIPE_MODE "r"
#endif

/* Max input sample */
#define MAXSMPL 65535
/* Max output sample */
#define MAXOUT 255

static void query(void);
static void run(const gchar *name,
		gint nparams,
		const GimpParam *param,
		gint *nreturn_vals,
		GimpParam **return_vals);

static gint  load_dialog (gchar *name);
static gint32 load_image (gchar *filename);

static void update_scales(void);
#ifdef PLOTLUT
static void plotLUT(void);
#endif

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_procedure */
  NULL,  /* quit_procedure */
  query, /* query_procedure */
  run,   /* run_procedure */
};

static struct {
	gboolean check_val[5];
	gfloat Temperature;
	gfloat Green;
	gfloat Gamma;  
	gfloat Exposition;
	gfloat Black;
	gfloat Dark;
	gfloat Saturation;
	gboolean OverExp, ClipSat, WBind;
	gint Scale;
} cfg = {
  { FALSE, FALSE, FALSE, FALSE, FALSE},
   4.75, 1.2, 0.6, 0.0,  0.0, 0.5, 1.0, TRUE, TRUE, FALSE, 2
};

typedef struct {
  guint16 R, G, B;
} rgb16;

static GdkPixbuf *previewPixbuf=NULL;
static GdkPixmap *histoPixmap=NULL;
static GtkWidget *preview=NULL, *histo=NULL;
static GtkWidget *ProgressBar=NULL;
static rgb16 *rawBuf=NULL;

static gint16 prevH, prevW;
static guchar LUT[65536];
static gfloat curve[65536];
static guint16	maxR, maxG, maxB, minR, minG, minB, minV, maxV;
static gfloat mr=1.0, mg=1.0, mb=1.0;
static guint16 rgbMax=65535;
static gint32 BP=0, WP=MAXSMPL;

static GtkAdjustment *adjTemperature=NULL, 
			*adjExposition=NULL,
		    *adjGamma=NULL,
		    *adjGreen=NULL,
		    *adjBlack=NULL,
		    *adjDark=NULL,
		    *adjSaturation=NULL;

static gchar *fileName;

#define RHISTLEN 65536
#define RGBHISTLEN 256

static guint32 	histogramR[RHISTLEN];  /* RAW data histogram */

/* A histogram for each channel */
static guint32 	histogramRGB[3][RGBHISTLEN]; 

/* for histogram -- notice 512 == 2x256 */
#define HISTOGRAM_WIDTH 260 
#define HISTOGRAM_HEIGHT 100


MAIN ()

static void query (void)
{
  static GimpParamDef load_args[] =
  {
    { GIMP_PDB_INT32,      "run_mode",     "Interactive, non-interactive" },
    { GIMP_PDB_STRING,     "filename",     "The name of the file to load" },
    { GIMP_PDB_STRING,     "raw_filename", "The name of the file to load" },
  };
 static GimpParamDef load_return_vals[] =
{
	{ GIMP_PDB_IMAGE,      "image",        "Output image" },
};

static gint num_load_args =
	sizeof load_args / sizeof load_args[0];
static gint num_load_return_vals =
	sizeof load_return_vals / sizeof load_return_vals[0];

gimp_install_procedure ("file_rawphoto_load",
			  "Loads raw digital camera files",
			  "This plug-in loads raw digital camera files.",
			  "Pawel T. Jochym jochym at ifj edu pl",
			  "Copyright 2003 by Dave Coffin\n"
			  "Copyright 2004 by Pawel Jochym",
			  PLUG_IN_VERSION,
			  "<Load>/rawphoto",
			  NULL,
			  GIMP_PLUGIN,
			  num_load_args,
			  num_load_return_vals,
			  load_args,
			  load_return_vals);

 gimp_register_load_handler ("file_rawphoto_load",
    "bay,bmq,cr2,crw,cs1,dc2,dcr,fff,jpg,k25,kdc,mrw,mos,nef,orf,pef,raf,raw,rdc,srf,tif,x3f", "");
}

static void run (const gchar *name,
		gint nparams,
		const GimpParam *param,
		gint *nreturn_vals,
		GimpParam **return_vals)
{
	static GimpParam values[2];
	GimpRunMode run_mode;
	GimpPDBStatusType status;
	gint32 image_id = -1;
	gchar *command, *fname;
	int stat;
	
	*nreturn_vals = 1;
	*return_vals = values;
	
	status = GIMP_PDB_CALLING_ERROR;
	if (strcmp (name, "file_rawphoto_load")) goto done;
	
	status = GIMP_PDB_EXECUTION_ERROR;
	fname = param[1].data.d_string;
	command = (gchar *) g_malloc (strlen(fname)+20);
	if (!command) goto done;
/*
	Is the file really a raw photo?  If not, try loading it
  as a regular JPEG or TIFF.
 */
	sprintf (command, "dcraw -i '%s'\n",fname);
	fputs (command, stderr);
	stat = system (command);
	g_free (command);
	if (stat) {
		if (stat > 0x200)
			g_message (_("The \"rawphoto\" plugin won't work because "
					"there is no \"dcraw\" executable in your path."));
		if (!strcasecmp (fname + strlen(fname) - 4, ".jpg"))
			*return_vals = gimp_run_procedure2("file_jpeg_load", nreturn_vals, nparams, param);
		else
			*return_vals = gimp_run_procedure2("file_tiff_load", nreturn_vals, nparams, param);
		return;
	}
	gimp_get_data ("plug_in_rawphoto", &cfg);
	status = GIMP_PDB_CANCEL;
	run_mode = (GimpRunMode) param[0].data.d_int32;
	if (run_mode == GIMP_RUN_INTERACTIVE)
    if (!load_dialog (param[1].data.d_string)) goto done;

	status = GIMP_PDB_EXECUTION_ERROR;
	image_id = load_image (param[1].data.d_string);
	if (image_id == -1) goto done;

	*nreturn_vals = 2;
	values[1].type = GIMP_PDB_IMAGE;
	values[1].data.d_image = image_id;
	status = GIMP_PDB_SUCCESS;
	gimp_set_data ("plug_in_rawphoto", &cfg, sizeof cfg);

done:
	values[0].type = GIMP_PDB_STATUS;
	values[0].data.d_status = status;
	
}

static void setRGBmult( void )
{
	gint32 t;
	gfloat mi;
	
	if (cfg.Temperature>7) cfg.Temperature=7;
	t=cfg.Temperature*100-200;
#ifdef DEBUG
	printf("t: %d\n", t);
#endif	
	mr=1/bbWB[t][0];
	mg=1/bbWB[t][1];
	mb=1/bbWB[t][2];
	mg*=cfg.Green;
	/* Normalize to at least 1.0, so we are
		not dimming colors only bumping */
	mi=MIN(mr,mg);
	mi=MIN(mi,mb);
	mr/=mi; mg/=mi; mb/=mi;
}

static void setLUTv(gint mv)
{
	gint32 i;
	gdouble p, v, b, g;

	b=mg*pow(2,cfg.Exposition);
	g=cfg.Gamma;
	BP=rgbMax*cfg.Black;
	WP=rgbMax/b;
	if (WP-BP<1) WP=BP+1;

#ifdef DEBUG	
	printf("T: %fK => R:%f G:%f B:%f BP:%d WP:%d\n", cfg.Temperature, mr, mg, mb, BP, WP);
#endif	
	curve[0] = 0;
	for (i=1; i < 0x10000; i++) {
		gfloat x=(gfloat)(i-BP)/(WP-BP);
		curve[i] = (i<BP)?0:MAXOUT*pow (x, g);
		curve[i] *= (1-cfg.Dark*exp(-x*x/0.002));
		curve[i]/=(gfloat)i;
	}
	
	for (i=0; i<MAXSMPL+1; i++) {
		p=(cfg.ClipSat && i>rgbMax)?rgbMax:i;
		v=pow(p*b/MAXSMPL,g); v=(v>BP)?MAXOUT*(v-BP)/(1-BP):0;
		LUT[i]=(guchar)((v>256)?mv:v);
	}
#ifdef PLOTLUT
	plotLUT();
#endif
}

static void setLUT()
{
	setRGBmult();
	mg=1.0;
	setLUTv(cfg.OverExp?0:255);
	setRGBmult();
}

static void setLUTfinal()
{
	setRGBmult();
	mr=mb=1.0;
	if (!cfg.ClipSat) mg=1.0; 
	setLUTv(255);
	setRGBmult();
}

static gint32 load_image (gchar *filename)
{
	int		tile_height, width, height, row, nrows, c;
	FILE		*pfp;
	gint32	image, layer;
	GimpDrawable	*drawable;
	GimpPixelRgn	pixel_region;
	guchar	*pixel, *p;
	guint16 	*rawBuf, *rp;
	gint16		x, y;
	char		*command, nl;
	gdouble	mults[3];

	setlocale (LC_NUMERIC, "C");
	command = (char *) g_malloc (strlen(filename)+100);
	if (!command) return -1;
	gimp_progress_init("Loading raw image...");

	setRGBmult();
	gimp_progress_update(0.02);
	sprintf (command,
		"dcraw -v -c%s%s%s%s%s -4 -r %f -l %f -b %f '%s'\n",
		cfg.check_val[0] ? " -q":"",
		cfg.check_val[1] ? " -h":"",
		cfg.check_val[2] ? " -f":"",
		cfg.check_val[3] ? " -d":"",
		cfg.check_val[4] ? " -w":"",
		(cfg.ClipSat)?(mr/mg):0.5, (cfg.ClipSat)?(mb/mg):0.5, 1.0,
		filename );
	fputs (command, stderr);
	pfp = popen (command, PIPE_MODE);
	g_free (command);
	if (!pfp) {
		perror ("dcraw");
		return -1;
	}

	gimp_progress_update(0.05);

	if (fscanf (pfp, "P6 %d %d %d%c", &width, &height, &c, &nl) != 4) {
		pclose (pfp);
		g_message ("Not a raw digital camera image.\n");
		return -1;
	}
	rgbMax=c;

	gimp_progress_update(0.1);

	image = gimp_image_new (width, height, GIMP_RGB);
	if (image == -1) {
		pclose (pfp);
		g_message ("Can't allocate new image.\n");
		return -1;
	}

	gimp_image_set_filename (image, filename);
	
	/* Create the "background" layer to hold the image... */
	layer = gimp_layer_new (image, "Background", width, height,
				GIMP_RGB_IMAGE, 100, GIMP_NORMAL_MODE);
	gimp_image_add_layer (image, layer, 0);
	
	/* Get the drawable and set the pixel region for our load... */
	drawable = gimp_drawable_get (layer);
	gimp_pixel_rgn_init (&pixel_region, drawable, 0, 0, drawable->width,
			drawable->height, TRUE, FALSE);
	
	/* Temporary buffers... */
	tile_height = gimp_tile_height();
	rawBuf = g_new (guint16, tile_height * width * 3);
	pixel = g_new (guchar, tile_height * width * 3);

	
	/* Load the image... */
	setLUTfinal();
	mults[0]=2*mr; mults[1]=mg; mults[2]=2*mb;
	for (row = 0; row < height; row += tile_height) {
		nrows = height - row;
		if (nrows > tile_height)
		nrows = tile_height;
		fread (rawBuf, width * 3 *sizeof(guint16), nrows, pfp);
#if BYTE_ORDER == LITTLE_ENDIAN
		swab((const char *) rawBuf, (char *) rawBuf, width * 3 *sizeof(guint16) * nrows);
#endif		
		p = pixel;
		rp = rawBuf;
/*		if (ProgressBar!=NULL) {
			gtk_progress_bar_set_fraction((GtkProgressBar *)ProgressBar, (float)row/height);
			gtk_widget_draw(ProgressBar,(GdkRectangle *)(&(ProgressBar->allocation)) );
		}			
*/
		gimp_progress_update(0.1+0.9*(float)row/height);
		for (y=0 ; y<nrows ; y++) {
			for (x=0 ; x<width ; x++, p+=3, rp+=3) {
				guint32 i, v;
				
				if (cfg.ClipSat) {
					v=MAX(rp[0],rp[1]); v=MAX(v,rp[2]);
				} else {
					v=MAX(mults[0]*rp[0],mults[1]*rp[1]); v=MAX(v,mults[2]*rp[2]);
				}
				i=(guint16)v;
	
				for (c=0 ; c<3 ; c++) {
					gint32 r, o;
					
					r=rp[c];
					if (!cfg.ClipSat) r*=mults[c];
					r=(cfg.ClipSat && r>rgbMax) ? rgbMax : r;
					o=((v-cfg.Saturation*(v-r))*curve[v]);
					o=MAX(o,0);
					o=MIN(o,MAXOUT);
					p[c]=(guchar)o;
				}
			}
		}
		gimp_pixel_rgn_set_rect (&pixel_region, pixel, 0, row, width, nrows);
	}

	pclose (pfp);
	g_free (pixel);
	
	gimp_drawable_flush (drawable);
	gimp_drawable_detach (drawable);
	
	return image;
}

/* Create a new backing pixmap for the histogram */
static gboolean
configure_histogram_event( GtkWidget *widget, GdkEventConfigure *event )
{
	if (histoPixmap)
		gdk_pixmap_unref(histoPixmap);

	histoPixmap = gdk_pixmap_new(widget->window,
				     widget->allocation.width,
				     widget->allocation.height,
				     -1);
	return TRUE;
}

/* Redraw the histogram from the backing pixmap */
static gboolean
expose_histogram_event( GtkWidget *widget, GdkEventExpose *event )
{
	g_assert (histoPixmap != NULL);
	gdk_draw_pixmap(widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
			histoPixmap,
			event->area.x, event->area.y,
			event->area.x, event->area.y,
			event->area.width, event->area.height);
	return FALSE;
}

void plotHistogram( void )
{
	gint w,h;
	guint32 i,j;

	GdkGC *gc;
	GdkColor color;

	gc = gdk_gc_new (histoPixmap);
	color.red = color.green = color.blue = 32768;
	gdk_gc_set_rgb_fg_color (gc, &color);

	w = histo->allocation.width;
	h = histo->allocation.height;

	/* Find biggest bin for normalizing */
	gint max_val = 0;
	for ( i=0 ; i < 3; i++ )
		for ( j=0 ; j < RGBHISTLEN; j++ )
			max_val = MAX(histogramRGB[i][j], max_val);

	/* Avoid divide by zero */
	if (!max_val) return;

	/* Clear previous histogram */
	gdk_draw_rectangle(histoPixmap,
			   gc,
			   TRUE,
			   0,0,
			   w,h);

#define DRAW_BAR(y1,y2) gdk_draw_line(histoPixmap, \
			      gc, \
			      i, h-(h*((y1>0)?log(y1)/log(max_val):0)), \
			      i, h-(h*((y2>0)?log(y2)/log(max_val):0)));

	printf("Max value: %d\n", max_val);

	/* Draw histogram bars */
	gint bottom, top, middle;
	gint r,g,b;
	for ( i=0 ; i < RGBHISTLEN; i++ ) {
		r = histogramRGB[0][i];
		g = histogramRGB[1][i];
		b = histogramRGB[2][i];

		bottom = MIN(r,g);
		bottom = MIN(b,bottom);

		top = MAX(r,g);
		top = MAX(b,top);

		if (r < top && r > bottom) {
			middle = r;
		} else if (g < top && g > bottom) {
			middle = g;
		} else if (r == bottom && g == bottom) {
			middle = g;
		} else {
			middle = b;
		}

		/* First draw white */
		color.red = color.green = color.blue = 65535;
		gdk_gc_set_rgb_fg_color (gc, &color);
		DRAW_BAR(0, bottom);

		/* Drop out middle color */
		if (bottom == r) {
			color.red = 0;
		} else if (bottom == g) {
			color.green = 0;
		} else {
			color.blue = 0;
		}
		gdk_gc_set_rgb_fg_color (gc, &color);
		DRAW_BAR(bottom, middle);

		/* Now get last bit of bar */
		if (top == r) {
			color.green = color.blue = 0;
			color.red = 65535;
		} else if (top == g) {
			color.red = color.blue = 0;
			color.green = 65535;
		} else {
			color.red = color.green = 0;
			color.blue = 65535;
		}
		gdk_gc_set_rgb_fg_color (gc, &color);
		DRAW_BAR(middle, top);
	}
	/* Draw the divisions */
	color.red = color.green = color.blue = 0;
	gdk_gc_set_rgb_fg_color (gc, &color);
	for(i=w/5; i<(w-w/5); i+=(w/5)) DRAW_BAR(0,max_val);
	
	/* Notify GTK that the widget needs to be redrawn (via an expose) */
	GdkRectangle update_rect;
	update_rect.x = 0;
	update_rect.y = 0;
	update_rect.width = HISTOGRAM_WIDTH;
	update_rect.height = HISTOGRAM_HEIGHT;
	gtk_widget_draw (histo, &update_rect);

#ifdef DEBUG_HISTOGRAM
	/* Just cut and paste the output of this into a Python Numeric session
	 * and go wild. */
	printf("hist_data = array([");
	for ( i=0 ; i < 3; i++ ) {
		printf("[");
		for ( j=0 ; j < RGBHISTLEN; j++ )
			printf("%d,", histogramRGB[i][j]);
		printf("%d]", histogramRGB[i][j]);
		if (i != 2)
			printf(",");
	}
	printf("])\n");
#endif
}

void calcStats( void )
{
	gint32 i;
	guint16 *b;
	guint32 sum;
	
	/* Now calculate the statistics */
	minR=minG=minB=minV=65535;
	maxR=maxG=maxB=maxV=0;
	for (i=0; i<RHISTLEN; i++) histogramR[i]=0;
	for (i=0, b=(guint16 *)rawBuf; i<(gint32)(prevW/cfg.Scale)*(prevH/cfg.Scale)*3 ; i+=3, b+=3) {
		minR=(b[0]<minR)?b[0]:minR;
		minG=(b[1]<minG)?b[1]:minG;
		minB=(b[2]<minB)?b[2]:minB;
		minV=(minR*mr<minV)?minR*mr:minV;
		minV=(minG*mg<minV)?minG*mg:minV;
		minV=(minB*mb<minV)?minB*mb:minV;
		maxR=(b[0]>maxR)?b[0]:maxR;
		maxG=(b[1]>maxG)?b[1]:maxG;
		maxB=(b[2]>maxB)?b[2]:maxB;
		maxV=(maxR*mr>maxV)?maxR*mr:maxV;
		maxV=(maxG*mg>maxV)?maxG*mg:maxV;
		maxV=(maxB*mb>maxV)?maxB*mb:maxV;
		sum=(b[0]+b[1]+b[2])/3;
		//sum=(sum>b[1])?sum:b[1];
		//sum=(sum>b[2])?sum:b[2];
		/* Logarithmic histogram not used for now
		if (sum>0) {
			sum=RHISTLEN+*log(1.0*sum/MAXSMPL)/log(2);
			sum=(sum>=RHISTLEN)?RHISTLEN-1:sum;
			sum=(sum<0)?0:sum;
			histogramR[sum]++;
		} else {
			histogramR[0]++;
		}
		*/
		sum=MAX(b[0],b[1]);
		sum=MAX(sum,b[2]);
		histogramR[sum]++;
	}
#ifdef DEBUG	
	printf("Ready:\n R:%d - %d\n G:%d - %d\n B:%d - %d\n",
			minR,maxR,minG,maxG,minB,maxB);
#endif
	
}

void autoAdjust (void )
{
	guint32 sum, stop;
	guint32 i;
	
	/* Calculate optimal exposition and black level */
	
	/* Exposition - do not clip anything */
	/* cfg.Exposition=-log((maxV+1.0)/MAXSMPL)/log(2); */
	
	/* cutoff at 0.5% of the histogram */
	stop=((guint32)(prevW/cfg.Scale)*(guint32)(prevH/cfg.Scale))/200;
	for (i=rgbMax, sum=0; (i>=0) && (sum<stop); i--){
		sum+=histogramR[i];
	}
	cfg.Exposition=-log((gfloat)(i+1)/rgbMax)/log(2);
	printf("White level at %d\n", i);
	
	/* cutoff at 0.5% of the histogram */
	stop=((guint32)(prevW/cfg.Scale)*(guint32)(prevH/cfg.Scale))/200;
	for (i=1, sum=0; (i<RHISTLEN) && (sum<stop); i++){
		sum+=histogramR[i];
	}
	cfg.Black=(gfloat)i/rgbMax;
	cfg.Black/=2;
	printf("Black: %f  Exposition: %f\n", cfg.Black, cfg.Exposition);
	update_scales();
}

static gint32 load_preview (gchar *filename)
{
	FILE		*pfp;
	char		*command, nl;
	int		w, h, c, x, y;
	guint16 *b, *d, *buf;
	guint32	i, k, sum;

	fileName=filename;
	setlocale (LC_NUMERIC, "C");
	command = (char *) g_malloc (strlen(filename)+100);
	if (!command) return -1;
	/* set -l and -r to 0.25 to avoid any clipping in dcraw */
	sprintf (command, "dcraw -v -c -4 -h%s%s%s%s '%s'\n",
								cfg.check_val[0] ? " -q":"",
								cfg.check_val[2] ? " -f":"",
								cfg.check_val[3] ? " -d":" -l 0.25 -r 0.25 ",
								cfg.check_val[4] ? " -w":"",
								filename );
	fputs (command, stderr);
	pfp = popen (command, PIPE_MODE);
	g_free (command);
	if (!pfp) {
		perror ("dcraw");
		return -1;
	}
	
	if (fscanf (pfp, "P6 %d %d %d%c", &w, &h, &c, &nl) != 4) {
		pclose (pfp);
		g_message ("Not a raw digital camera image.\n");
		return -1;
	}
	rgbMax=c;
	
	printf("W:%d H:%d RGBmax:%d\n", prevW=w, prevH=h, rgbMax); 
	w/=400;
	h/=400;
	cfg.Scale=MAX(w,h);
	cfg.Scale=MAX(1,cfg.Scale);
	printf("Scale: %d\n", cfg.Scale);
	buf = (guint16 *) g_malloc ((prevW)*cfg.Scale*sizeof(rgb16));
	rawBuf = (rgb16 *) g_malloc ((prevW/cfg.Scale)*(prevH/cfg.Scale)*sizeof(rgb16));
	if (rawBuf == NULL) {
		pclose (pfp);
		g_message ("Can't allocate raw preview buffer.\n");
		return -1;
	}
	
	/* Load the image... */
	printf("Loading...\n");
  
	/* scalling down by scale, this is rather crude approach 
	** since if the scale-sized box does not fit into the picture
	** we just crop the picture from down and right to correct size
	*/
	for (y=0; y<prevH-cfg.Scale; y+=cfg.Scale) {
		fread (buf, prevW * sizeof(rgb16), cfg.Scale, pfp);
#if BYTE_ORDER == LITTLE_ENDIAN
		swab(buf, buf, prevW * sizeof(rgb16)*cfg.Scale);
#endif		
		if (ProgressBar!=NULL) {
			gtk_progress_bar_set_fraction((GtkProgressBar *)ProgressBar, (float)y/prevH);
			gtk_widget_draw(ProgressBar,(GdkRectangle *)(&(ProgressBar->allocation)) );
		}
		for (c=0; c<3; c++) {
			b=buf+c;  
			d=(guint16 *)(rawBuf+(prevW/cfg.Scale)*(y/cfg.Scale))+c;
			for (x=0; x<prevW-cfg.Scale; x+=cfg.Scale, b+=3*cfg.Scale, d+=3) {
				sum=0;
				for( i=0; i<3*cfg.Scale; i+=3) 
					for( k=0; k<3*cfg.Scale; k+=3) 
						sum+=b[k+prevW*i];
				d[0]=((c!=1 && !cfg.check_val[3] )?4L:1L)*sum/(cfg.Scale*cfg.Scale);
			}
		}
	}
  
/* FIXME */
/* No error check */
	
	pclose (pfp);
	g_free(buf);

/* Call setRGBmult to set mr/mg/mb */
	setRGBmult();
	calcStats();
	setLUT();
	update_scales();
	
	if (ProgressBar!=NULL) gtk_progress_bar_set_fraction((GtkProgressBar *)ProgressBar, 0);
	
	plotHistogram();

	return 1;
}

void render_preview( void )
{
	int width, height, x, y, rowstride, n_channels, rawRS, c;
	guchar *p;
	guint16 *rp;
	guint32 i,j;
	gdouble mults[3];
	
	if (preview==NULL || previewPixbuf==NULL) return;
		
	n_channels = gdk_pixbuf_get_n_channels (previewPixbuf);
	
	g_assert (gdk_pixbuf_get_colorspace (previewPixbuf) == GDK_COLORSPACE_RGB);
	g_assert (gdk_pixbuf_get_bits_per_sample (previewPixbuf) == 8);
	g_assert (n_channels == 3);
	
	width = gdk_pixbuf_get_width (previewPixbuf);
	height = gdk_pixbuf_get_height (previewPixbuf);
	
	rowstride = gdk_pixbuf_get_rowstride (previewPixbuf);
	p = gdk_pixbuf_get_pixels (previewPixbuf);
	rp = (guint16 *)rawBuf;
	rawRS=width*sizeof(rgb16);
	
	printf("Rendering (%d+%d x %d) ...\n",width, rowstride-n_channels*width, height);
	setLUT();

	/* Clear RGB histograms */
	for (i=0; i<3; i++)
		for (j=0; j<RGBHISTLEN; j++)
			histogramRGB[i][j]=0;

	printf("LUT ready, Rendering...\n");
	
	mults[0]=mr; mults[1]=mg; mults[2]=mb;
	
	for (y=0 ; y<height ; y++, p+=rowstride-(n_channels*width)) {
		for (x=0 ; x<width ; x++, p+=n_channels, rp+=3) {
			guint32 v, rv[3];
			
			rv[0]=rp[0]*mr;
			rv[1]=rp[1]*mg;
			rv[2]=rp[2]*mb;
			v=MAX(rv[0],rv[1]); v=MAX(v,rv[2]); 
			if (cfg.ClipSat) v=MIN(v,rgbMax);
			i=(guint16)v;

			for (c=0 ; c<3 ; c++) {
				gint32 r, o;
				
				r=(cfg.ClipSat && rv[c]>rgbMax) ? rgbMax : rv[c];

				if (v<=BP) 
					p[c]=o=0;
				else if (cfg.OverExp && v>WP) {
					if (cfg.WBind)
						r=(rv[c]>WP)?0:r;
					else
						r=0;
				}
				
				o=((i-cfg.Saturation*(i-r))*curve[i]);
				o=MAX(o,0);
				p[c]=o=(o>MAXOUT)?255:(guchar)o;

				/* Count them crunchy channels... */
				histogramRGB[c][(guchar)(o)]++;
					
			}
		}
	}
	printf("Ready, redrawing\n");
	gtk_widget_queue_draw(preview);
	plotHistogram();
}

void load_and_render_preview( void )
{
	printf("loading... %s\n",fileName);
	/* printLUT(); */
	load_preview(fileName);
	render_preview();
}

void update_scales( void )
{
	if (adjTemperature!=NULL) gtk_adjustment_set_value(adjTemperature, cfg.Temperature);
	if (adjGreen!=NULL) gtk_adjustment_set_value(adjGreen, cfg.Green);
	if (adjExposition!=NULL) gtk_adjustment_set_value(adjExposition, cfg.Exposition);
	if (adjGamma!=NULL) gtk_adjustment_set_value(adjGamma, cfg.Gamma);
	if (adjBlack!=NULL) gtk_adjustment_set_value(adjBlack, cfg.Black);
	if (adjDark!=NULL) gtk_adjustment_set_value(adjDark, cfg.Dark);
	if (adjSaturation!=NULL) gtk_adjustment_set_value(adjSaturation, cfg.Saturation);
};	

void reset_to_neutral( void )
{
	cfg.Temperature=4.750;
	cfg.Green=1.2;
	cfg.Exposition=0.0;
	cfg.Gamma=0.6;
	cfg.Black=0;
	cfg.Saturation=1.0;
	update_scales();
}

void update_scale_value( GtkRange *r, gfloat *t )
{
	*t=gtk_adjustment_get_value(gtk_range_get_adjustment(r));
}


static gboolean
  set_spot_WB (GtkWidget      *event_box, 
                         GdkEventButton *event,
                         gpointer        data)
{
	gint16 x, y, i, k, bs, w, h, l, r, m;
	gdouble sR, sG, sB, mRB, t;
	
    g_print ("Event box clicked at coordinates %f,%f\n", 
             event->x, event->y);

	if (preview==NULL || previewPixbuf==NULL || rawBuf==NULL) return TRUE;
		
	/* avarage box size */
	bs=4;
	sR=sG=sB=0;
	x=event->x;
	y=event->y;
	
	w = gdk_pixbuf_get_width (previewPixbuf);
	h = gdk_pixbuf_get_height (previewPixbuf);
	
	x=MAX(0,x-bs);
	y=MAX(0,y-bs);
	x=MIN(w,x+bs);
	y=MIN(h,y+bs);
	
	for( i=y-bs; i<=y+bs; i++) {
		guint16 *rp;
		rp=(guint16 *)(rawBuf+(i*w+x-bs));
		
		for ( k=x-bs; k<=x+bs; k++, rp+=3) {
			sR+=rp[0];
			sG+=rp[1];
			sB+=rp[2];
		}
	}
	sR/=(2*bs+1)*(2*bs+1);
	sG/=(2*bs+1)*(2*bs+1);
	sB/=(2*bs+1)*(2*bs+1);
	
	t=MAX(sR,sG);
	t=MAX(t,sB);
	sR/=t; sG/=t; sB/=t;

	mRB=sR/sB;

	printf("Sums: %f  %f  %f \n", sR, sG, sB);
	
	l=0; r=sizeof(bbWB)/(sizeof(float)*3);
	m=(r+l)/2;
	for (l=0, r=sizeof(bbWB)/(sizeof(float)*3), m=(l+r)/2 ; r-l>1 ; m=(l+r)/2) {
		if (bbWB[m][0]/bbWB[m][2] > mRB) 
			l=m;
		else
			r=m;
		printf("L,M,R: %d, %d, %d bbWB[m]=%f\n", l,m,r,bbWB[m][0]/bbWB[m][2]);
	}
	g_print("Temperature %f K\n", m*10.0+2000.0);
	t=(bbWB[m][1]/bbWB[m][0])/(sG/sR);
	g_print("Green component %f\n", t);
	
	cfg.Temperature=m/100.0+2.0;
	cfg.Green=t;
	update_scales();
    /* Returning TRUE means we handled the event, so the signal 
     * emission should be stopped (don't call any further 
     * callbacks that may be connected). Return FALSE
     * to continue invoking callbacks.
     */
    return TRUE;
}
  
#define NCHECK (sizeof cfg.check_val / sizeof (gboolean))

gint load_dialog (gchar * name)
{
	GtkWidget *dialog;
	GtkWidget *table;
	GtkObject *adj;
	GtkWidget *widget;
	GtkWidget *hbox1, *vbox1, *button1, *vbox3, *vbox4, *event_box, *hbox2, *frame, *notebook;
	GdkColor color;

	int i;
	static const char *label[10] =
	{ "Quick interpolation", "Half-size interpolation",
    "Four color interpolation", "Grayscale document", "Camera white balance",
    "Gamma", "Brightness", "Red Multiplier", "Blue Multiplier"};

	fileName=name;
	gimp_ui_init ("rawphoto", TRUE);

	dialog = gimp_dialog_new (_("Raw Photo Loader "PLUG_IN_VERSION), "rawphoto",
			NULL, (GtkDialogFlags) 0,
			gimp_standard_help_func, "rawphoto",

			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OK,     GTK_RESPONSE_OK,

			NULL);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 9);

	hbox1=gtk_hbox_new (FALSE, 12);
	gtk_widget_show (hbox1);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox1, TRUE, TRUE, 0);

/* Image portion of the dialog */

	vbox3 = gtk_vbox_new (FALSE, 4);
	gtk_widget_show (vbox3);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox3, TRUE, TRUE, 0);

    frame = gtk_frame_new(fileName);
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (vbox3), frame, TRUE, TRUE, 0 );
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_widget_set_size_request (frame, 500, 405);
	color.red = color.green = color.blue = 5535;
    gtk_widget_modify_fg(GTK_WIDGET(frame), GTK_STATE_NORMAL, &color);

	preview=gtk_image_new_from_pixbuf (previewPixbuf);
	gtk_widget_set_size_request (preview, 256, 0);
	event_box = gtk_event_box_new ();
    gtk_container_add (GTK_CONTAINER (event_box), preview);
	gtk_signal_connect (GTK_OBJECT (event_box), "button_press_event",
                      GTK_SIGNAL_FUNC (set_spot_WB),
                      preview);
    gtk_container_add(GTK_CONTAINER(frame), event_box);
	gtk_widget_show(preview);
	gtk_widget_show(event_box);
  
	ProgressBar = gtk_progress_bar_new ();
	gtk_widget_show (ProgressBar);
	gtk_box_pack_start (GTK_BOX (vbox3), ProgressBar, FALSE, FALSE, 0);
	/*gtk_progress_bar_set_text (GTK_PROGRESS_BAR (progressbar1), _("Loading..."));*/
	


	
	render_preview();

    /* Histogram */ 

	vbox1 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox1);
	gtk_box_pack_start (GTK_BOX (hbox1), vbox1, FALSE, FALSE, 0);

    frame = gtk_frame_new(NULL);
	gtk_widget_show (frame);
	gtk_widget_set_size_request (frame, HISTOGRAM_WIDTH, HISTOGRAM_HEIGHT);
	gtk_box_pack_start (GTK_BOX (vbox1), frame, FALSE, FALSE, 0);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    
	histo = gtk_drawing_area_new ();
	gtk_widget_show (histo);
	gtk_widget_set_size_request (histo, HISTOGRAM_WIDTH, HISTOGRAM_HEIGHT);
    gtk_container_add(GTK_CONTAINER(frame), histo);
	gtk_signal_connect (GTK_OBJECT (histo), "expose_event",
			    (GtkSignalFunc) expose_histogram_event, NULL);
	gtk_signal_connect (GTK_OBJECT(histo),"configure_event",
			    (GtkSignalFunc) configure_histogram_event, NULL);

    /* Controls */
	table = gtk_table_new (3, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(table), 6);
	gtk_box_pack_start(GTK_BOX(vbox1), table, FALSE, FALSE,0);
	gtk_widget_show (table);
	
	widget = gtk_check_button_new_with_label("Overexposure indication");
    gtk_toggle_button_set_active	(GTK_TOGGLE_BUTTON (widget), cfg.OverExp);
    gtk_table_attach	(GTK_TABLE(table), widget, 0, 2, 0, 1, GTK_FILL, GTK_FILL, 0, 0);
    gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (gimp_toggle_button_update),
			&cfg.OverExp);
	gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (render_preview),NULL);
	gtk_widget_show (widget);

	widget = gtk_check_button_new_with_label("WB indicator on saturated");
    gtk_toggle_button_set_active	(GTK_TOGGLE_BUTTON (widget), cfg.WBind);
    gtk_table_attach	(GTK_TABLE(table), widget, 0, 2, 1, 2, GTK_FILL, GTK_FILL, 0, 0);
    gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (gimp_toggle_button_update),
			&cfg.WBind);
	gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (render_preview),NULL);
	gtk_widget_show (widget);

	widget = gtk_check_button_new_with_label("Clip saturated pixels");
    gtk_toggle_button_set_active	(GTK_TOGGLE_BUTTON (widget), cfg.ClipSat);
    gtk_table_attach	(GTK_TABLE(table), widget, 0, 2, 2, 3, GTK_FILL, GTK_FILL, 0, 0);
    gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (gimp_toggle_button_update),
			&cfg.ClipSat);
	gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (render_preview),NULL);
	gtk_widget_show (widget);
    


    notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox1), notebook, TRUE, TRUE, 0);
    gtk_notebook_set_tab_pos (GTK_NOTEBOOK(notebook), GTK_POS_TOP);
	gtk_widget_show(notebook);
 
    frame = gtk_frame_new("");
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_widget_show (frame);

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
    gtk_container_add(GTK_CONTAINER(frame), vbox4);

	table = gtk_table_new (10, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(table), 6);
	gtk_box_pack_start(GTK_BOX(vbox4), table, FALSE, FALSE,0);
	gtk_widget_show (table);

for (i=0; i < NCHECK; i++) {
    widget = gtk_check_button_new_with_label
	(_(label[i]));
    gtk_toggle_button_set_active
	(GTK_TOGGLE_BUTTON (widget), cfg.check_val[i]);
    gtk_table_attach
	(GTK_TABLE(table), widget, 0, 2, i, i+1, GTK_FILL, GTK_FILL, 0, 0);
    gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (gimp_toggle_button_update),
			&cfg.check_val[i]);
/*
	gtk_signal_connect (GTK_OBJECT (widget), "toggled",
			GTK_SIGNAL_FUNC (load_and_render_preview),NULL);
*/	
	gtk_widget_show (widget);
}

	button1 = gtk_button_new_with_mnemonic (_("Update"));
	gtk_widget_show (button1);
	gtk_box_pack_start (GTK_BOX (vbox4), button1, FALSE, FALSE, 2);
	gtk_signal_connect_after (GTK_OBJECT (button1), "clicked",
			GTK_SIGNAL_FUNC (load_and_render_preview),NULL);

    gtk_notebook_prepend_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new("Import"));	

    frame = gtk_frame_new("");
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_widget_show (frame);

	vbox4 = gtk_vbox_new (FALSE, 0);
	gtk_widget_show (vbox4);
    gtk_container_add(GTK_CONTAINER(frame), vbox4);

	table = gtk_table_new (14, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER(table), 6);
	gtk_box_pack_start(GTK_BOX(vbox4), table, FALSE, FALSE,0);
	gtk_widget_show (table);
	gtk_widget_show (widget);

	adj=gimp_scale_entry_new(GTK_TABLE(table),0,0,"WB(x 1000K)",100,60,cfg.Temperature,2.2,7.0,0.01,0.1,2,TRUE,0,0,"White balance colour temperature (K)","");
	adjTemperature=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Temperature);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);
  
	adj=gimp_scale_entry_new(GTK_TABLE(table),0,1,"Green",100,60,cfg.Green,0.2,2.5,0.01,0.1,2,TRUE,0,0,"Magenta colour cast removal","");
	adjGreen=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Green);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);

	adj=gimp_scale_entry_new(GTK_TABLE(table),0,2,"Gamma",100,60,cfg.Gamma,0.01,1.5,0.01,0.1,2,TRUE,0,0,"Gamma corection","");
	adjGamma=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Gamma);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);

	adj=gimp_scale_entry_new(GTK_TABLE(table),0,3,"Brightness",100,60,cfg.Exposition,-6,8,0.01,0.1,2,TRUE,0,0,"Exposition (EV)","");
	adjExposition=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Exposition);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);

	adj=gimp_scale_entry_new(GTK_TABLE(table),0,4,"Black level",100,60,cfg.Black,0,0.05,0.01,0.1,3,TRUE,0,0,"Black level","");
	adjBlack=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Black);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);
			
	
	adj=gimp_scale_entry_new(GTK_TABLE(table),0,5,"Shadows",100,60,cfg.Dark,0.0,1.0,0.01,0.1,2,TRUE,0,0,"Shadows noise suppresion","");
	adjDark=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Dark);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);
			
	adj=gimp_scale_entry_new(GTK_TABLE(table),0,6,"Saturation",100,60,cfg.Saturation,0.0,2.0,0.01,0.1,2,TRUE,0,0,"Saturation","");
	adjSaturation=GTK_ADJUSTMENT(adj);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (gimp_float_adjustment_update),&cfg.Saturation);
	gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
			GTK_SIGNAL_FUNC (render_preview),NULL);

	hbox2 = gtk_hbox_new (FALSE, 0);
	gtk_widget_show (hbox2);
	gtk_box_pack_start (GTK_BOX(vbox4), hbox2, FALSE, FALSE, 2);
	
	
	button1 = gtk_button_new_with_mnemonic (_("Reset to neutral"));
	gtk_widget_show (button1);
	gtk_box_pack_start (GTK_BOX (hbox2), button1, TRUE, TRUE, 2);
	gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			GTK_SIGNAL_FUNC (reset_to_neutral),NULL);
	gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			GTK_SIGNAL_FUNC (render_preview),NULL);

	button1 = gtk_button_new_with_mnemonic (_("Auto adjust"));
	gtk_widget_show (button1);
	gtk_box_pack_start (GTK_BOX (hbox2), button1, TRUE, TRUE, 2);
	gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			GTK_SIGNAL_FUNC (autoAdjust),NULL);
	gtk_signal_connect (GTK_OBJECT (button1), "clicked",
			GTK_SIGNAL_FUNC (render_preview),NULL);

    gtk_notebook_prepend_page(GTK_NOTEBOOK(notebook), frame, gtk_label_new("Corrections"));	
    gtk_notebook_set_page (GTK_NOTEBOOK(notebook), 0);

	/*
	vpaned = gtk_vpaned_new ();
	gtk_widget_show (vpaned);
	gtk_paned_pack2 (GTK_PANED (hpaned), vpaned, TRUE, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (vpaned), 1);
	gtk_paned_set_position (GTK_PANED (vpaned), 0);
*/
	gtk_widget_show (dialog);

	load_preview(name);
  
	printf("Allocating pixbuf...\n");
	previewPixbuf=gdk_pixbuf_new
		(GDK_COLORSPACE_RGB, FALSE, 8, prevW/cfg.Scale, prevH/cfg.Scale);

	if ( previewPixbuf == NULL ) {
		g_message ("Can't allocate preview buffer.\n");
		return -1;
	}

	gtk_image_set_from_pixbuf ( GTK_IMAGE (preview), previewPixbuf);
	gtk_widget_set_size_request (preview, prevW/cfg.Scale, prevH/cfg.Scale);	
	
	render_preview();
	
	i = gimp_dialog_run (GIMP_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	ProgressBar=NULL;
	printf("Finito\n");
	return i == GTK_RESPONSE_OK;
}
