// Package rast provides tools for working with Rego's AST library (not RoAST)
package rast

import (
	"bytes"
	"fmt"
	"os"
	"reflect"
	"slices"
	"strconv"
	"strings"

	"github.com/open-policy-agent/opa/v1/ast"
	outil "github.com/open-policy-agent/opa/v1/util"

	"github.com/open-policy-agent/regal/internal/util"
)

// IsBodyGenerated checks if the body of a rule is generated by the parser.
func IsBodyGenerated(rule *ast.Rule) bool {
	if rule.Default || len(rule.Body) == 0 {
		return true
	}

	if rule.Head != nil && rule.Body[0] != nil {
		bodyStart := rule.Body[0].Location
		if bodyStart == rule.Location || rule.Head.Value != nil && bodyStart == rule.Head.Value.Location {
			return true
		}

		if rule.Head.Key != nil &&
			rule.Body[0].Location.Row == rule.Head.Key.Location.Row &&
			rule.Body[0].Location.Col < rule.Head.Key.Location.Col {
			// This is a quirk in the original AST — the generated body will have a location
			// set before the key, i.e. "message"
			return true
		}
	}

	return false
}

// RefStringToBody converts a simple dot-delimited string path to an ast.Body.
// This is a lightweight alternative to ast.ParseBody that avoids the overhead of parsing,
// and benefits from using interned terms when possible. It is also nowhere near as competent,
// and can only handle simple string paths without vars, numbers, etc. Suitable for use with
// e.g. rego.ParsedQuery and other places where a simple ref is needed. Do *NOT* use the returned
// ast.Body anywhere it might be mutated (like having location data added), as that modifies the
// globally interned terms.
//
// Implementations tested:
// -----------------------
// 333.6 ns/op	     472 B/op	      19 allocs/op - SplitSeq
// 330.7 ns/op	     496 B/op	      16 allocs/op - Split
// 269.1 ns/op	     400 B/op	      15 allocs/op - IndexOf for loop (current).
func RefStringToBody(path string) ast.Body {
	return ast.NewBody(ast.NewExpr(ast.NewTerm(RefStringToRef(path))))
}

// RefStringToRef converts a simple dot-delimited string path to an ast.Ref in the most
// efficient way possible, using interned terms where possible. See RefStringToBody for
// more details on the limitations of this function.
func RefStringToRef(path string) ast.Ref {
	before, after, found := strings.Cut(path, ".")
	terms := append(make([]*ast.Term, 0, strings.Count(path, ".")+1), refHeadTerm(before))

	for found {
		before, after, found = strings.Cut(after, ".")
		terms = append(terms, ast.InternedTerm(before))
	}

	return ast.Ref(terms)
}

// LinesArrayTerm converts a string with newlines into an ast.Term array holding each line.
func LinesArrayTerm(content string) *ast.Term {
	return ArrayTerm(strings.Split(strings.ReplaceAll(content, "\r\n", "\n"), "\n"))
}

func ArrayTerm(a []string) *ast.Term {
	if len(a) == 0 {
		return ast.InternedEmptyArray
	}

	return ast.ArrayTerm(util.Map(a, ast.InternedTerm)...)
}

func AppendLocation(buf []byte, location *ast.Location) []byte {
	endRow, endCol := location.Row, location.Col
	if textLen := len(location.Text); textLen > 0 {
		numNewLines := util.SafeIntToUint(bytes.Count(location.Text, []byte("\n")))

		endRow += util.SafeUintToInt(numNewLines)
		if numNewLines == 0 {
			endCol += textLen
		} else {
			endCol = textLen - bytes.LastIndexByte(location.Text, '\n')
		}
	}

	n := 3 + // 3 colons
		outil.NumDigitsInt(location.Row) + outil.NumDigitsInt(location.Col) +
		outil.NumDigitsInt(endRow) + outil.NumDigitsInt(endCol)

	if buf == nil {
		buf = make([]byte, 0, n)
	} else {
		buf = slices.Grow(buf, n)
	}

	buf = append(strconv.AppendInt(buf, int64(location.Row), 10), ':')
	buf = append(strconv.AppendInt(buf, int64(location.Col), 10), ':')
	buf = append(strconv.AppendInt(buf, int64(endRow), 10), ':')

	return strconv.AppendInt(buf, int64(endCol), 10)
}

func refHeadTerm(name string) *ast.Term {
	switch name {
	case "data":
		return ast.DefaultRootDocument
	case "input":
		return ast.InputRootDocument
	default:
		return ast.VarTerm(name)
	}
}

// StructToValue converts a struct to ast.Value using 'json' struct tags (e.g., `json:"field,omitempty"`)
// but without an expensive JSON roundtrip.
// Experimental: this is new and not yet battle-tested, so use with caution.
func StructToValue(input any) ast.Value {
	v, t := reflect.ValueOf(input), reflect.TypeOf(input)

	if v.Kind() == reflect.Ptr {
		v, t = v.Elem(), t.Elem()
	}

	kvs := make([][2]*ast.Term, 0, t.NumField())

	for i := range t.NumField() {
		tag := t.Field(i).Tag.Get("json")
		if tag == "" || tag == "-" {
			continue
		}

		value := v.Field(i)

		if strings.Contains(tag, ",") {
			parts := strings.Split(tag, ",")
			tag = parts[0]

			if slices.Contains(parts[1:], "omitempty") && value.IsZero() {
				continue
			}
		}

		kvs = append(kvs, ast.Item(ast.InternedTerm(tag), ast.NewTerm(toAstValue(value.Interface()))))
	}

	return ast.NewObject(kvs...)
}

func toAstValue(v any) ast.Value {
	if v == nil {
		return ast.NullValue
	}

	rv := reflect.ValueOf(v)
	for rv.Kind() == reflect.Ptr {
		if rv.IsNil() {
			return ast.NullValue
		}

		rv = rv.Elem()
	}

	//nolint:exhaustive
	switch rv.Kind() {
	case reflect.Struct:
		return StructToValue(rv.Interface())
	case reflect.Slice, reflect.Array:
		l := rv.Len()
		if l == 0 {
			return ast.InternedEmptyArrayValue
		}

		arr := make([]*ast.Term, 0, l)
		for i := range l {
			arr = append(arr, internedAny(rv.Index(i).Interface()))
		}

		return ast.NewArray(arr...)
	case reflect.Map:
		kvs := make([][2]*ast.Term, 0, rv.Len())

		for _, key := range rv.MapKeys() {
			var k *ast.Term

			ki := key.Interface()
			if s, ok := ki.(string); ok {
				k = ast.InternedTerm(s)
			} else {
				k = ast.InternedTerm(fmt.Sprintf("%v", ki))
			}

			kvs = append(kvs, [2]*ast.Term{k, internedAny(rv.MapIndex(key).Interface())})
		}

		return ast.NewObject(kvs...)
	case reflect.String:
		return ast.String(rv.String())
	case reflect.Int, reflect.Int8, reflect.Int16, reflect.Int32, reflect.Int64:
		return ast.Number(strconv.FormatInt(rv.Int(), 10))
	case reflect.Uint, reflect.Uint8, reflect.Uint16, reflect.Uint32, reflect.Uint64:
		return ast.Number(strconv.FormatUint(rv.Uint(), 10))
	case reflect.Float32, reflect.Float64:
		return ast.Number(fmt.Sprintf("%v", rv.Float()))
	case reflect.Bool:
		return ast.InternedTerm(rv.Bool()).Value
	}
	// Fallback: string representation
	fmt.Fprintln(os.Stderr, "WARNING: Unsupported type for conversion to ast.Value:", rv.Kind())

	return ast.String(fmt.Sprintf("%v", v))
}

func internedAny(v any) *ast.Term {
	switch value := v.(type) {
	case bool:
		return ast.InternedTerm(value)
	case string:
		return ast.InternedTerm(value)
	case int:
		return ast.InternedTerm(value)
	case uint:
		return ast.InternedTerm(int(value)) // //nolint:gosec
	case int64:
		return ast.InternedTerm(int(value))
	case float64:
		return ast.FloatNumberTerm(value)
	default:
		return ast.NewTerm(toAstValue(v))
	}
}
