/*
 * nasd_pdrive_client.h
 *
 * defs/structs used by clients of NASD drives
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1996,1997,1998,1999,2000.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#ifndef _NASD__NASD_PDRIVE_CLIENT_H_
#define _NASD__NASD_PDRIVE_CLIENT_H_

#include <nasd/nasd_options.h>
#include <nasd/nasd_itypes.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_types.h>
#include <nasd/nasd_drive_types.h>
#include <nasd/nasd_threadstuff.h>
#include <nasd/nasd_control.h>
#include <nasd/nasd_common.h>
#include <nasd/nasd_shutdown.h>
#include <nasd/nasd_pipe.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_nonce_mgmt.h>

typedef  struct nasd_drive_default_bindings_s   nasd_drive_default_bindings_t;
typedef  struct nasd_drive_handle_s            *nasd_drive_handle_t;
typedef  struct nasd_cl_p_rpcmod_tab_s          nasd_cl_p_rpcmod_tab_t;

#define NASD_CL_ERROR_STRING_LEN 160
typedef char nasd_error_string_t[NASD_CL_ERROR_STRING_LEN];
typedef nasd_uint32 nasd_rpc_status_t;

typedef struct nasd_cl_p_otw_buf_s       nasd_cl_p_otw_buf_t;
/* caller must allocate space for sizeof(nasd_cl_p_otw_buf_t) +
   sizeof(nasd_cl_p_blah_dr_args_otw_t) + sizeof(nasd_cl_p_blah_dr_res_otw_t)
   and set args_otw/res_otw pointers correctly. */
struct nasd_cl_p_otw_buf_s {
  nasd_security_param_otw_t   sec_param_otw;
  nasd_capability_otw_t       capability_otw;
  nasd_digest_nonce_otw_t     args_digest_otw;
  nasd_digest_nonce_otw_t     res_digest_otw;
  nasd_iv_t                   icv;
  nasd_status_t               sec_rc;
  nasd_status_t               skew_rc;
  int                         skew_try_count;
  nasd_otw_base_t            *args_otw;
  nasd_otw_base_t            *res_otw;
};

#define NASD_BIND_BOGUS                 0
#define NASD_BIND_DEFAULT               1
#define NASD_BIND_COLOCATE              2
#define NASD_BIND_MSGQ                  3
#define NASD_BIND_KPDEV                 4
#define NASD_BIND_KPDEV_DEFAULT         5
#define NASD_BIND_KPDEV_DEFAULT_FAKE_RW 6
#define NASD_BIND_DCE_DIRECT_UDP        7
#define NASD_BIND_DCE_DIRECT_TCP        8
#define NASD_BIND_SRPC                  9

#define nasd_cl_error_string(_handle_,_status_,_str_) \
  _nasd_cl_error_string(_handle_,_status_,_str_,__FILE__,__LINE__)

#define nasd_drive_handle_type_name(_handle_) \
  ((_handle_) ? \
    ((_handle_)->rpc_tab ? (_handle_)->rpc_tab->name : "invalid rpc_tab") : \
    "NULL handle")

/*
 * init/shutdown
 */
nasd_status_t nasd_cl_p_real_init(void);
void nasd_cl_p_real_shutdown(void);
void nasd_cl_p_sys_init(void);
nasd_status_t nasd_cl_p_init(void);
void nasd_cl_p_shutdown(void);

/*
 * Switch table types
 */

typedef nasd_status_t (*nasd_cl_pr_init_t)(
  nasd_shutdown_list_t  *sl);

typedef nasd_status_t (*nasd_cl_pr_bind_t)(
  nasd_drive_handle_t   handle,
  char                 *drive_name,
  char                 *portnum,
  int                   binding_type,
  void                 *binding_param,
  int                   binding_param_len);

typedef nasd_status_t (*nasd_cl_pr_unbind_t)(
  nasd_drive_handle_t  handle);

typedef void (*nasd_cl_pr_null_t)(
  nasd_drive_handle_t   handle,
  nasd_res_t           *res,
  nasd_rpc_status_t    *status);

typedef void (*nasd_cl_pr_sync_t)(
  nasd_drive_handle_t   handle,
  nasd_res_t           *res,
  nasd_rpc_status_t    *status);

typedef void (*nasd_cl_pr_part_creat_t)(
  nasd_drive_handle_t           handle,
  nasd_key_t                    in_key,
  nasd_security_param_t        *sec_param,
  nasd_capability_t            *capability,
  nasd_p_part_creat_dr_args_t  *args,
  nasd_p_part_creat_dr_res_t   *res,
  nasd_rpc_status_t            *status);

typedef void (*nasd_cl_pr_create_otw_provided_t)(
  nasd_drive_handle_t       handle,
  nasd_key_t                in_key,
  nasd_security_param_t    *sec_param,
  nasd_capability_t        *capability,
  nasd_p_create_dr_args_t  *args,
  nasd_p_create_dr_res_t   *res,
  nasd_cl_p_otw_buf_t      *otw,
  nasd_rpc_status_t        *status);

typedef void (*nasd_cl_pr_getattr_t)(
  nasd_drive_handle_t        handle,
  nasd_key_t                 in_key,
  nasd_security_param_t     *sec_param,
  nasd_capability_t         *capability,
  nasd_p_getattr_dr_args_t  *args,
  nasd_p_getattr_dr_res_t   *res,
  nasd_rpc_status_t         *status);

typedef void (*nasd_cl_pr_write_simple_t)(
  nasd_drive_handle_t         handle,
  nasd_key_t                  in_key,
  nasd_security_param_t      *sec_param,
  nasd_capability_t          *capability,
  nasd_p_smpl_op_dr_args_t   *args,
  void                       *buf,
  nasd_p_fastwrite_dr_res_t  *res,
  nasd_rpc_status_t          *status);

typedef void (*nasd_cl_pr_range_write_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_smpl_op_dr_args_t    *args,
  nasd_mem_list_t             *memlist,
  nasd_p_fastwrite_dr_res_t   *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_read_simple_t)(
  nasd_drive_handle_t        handle,
  nasd_key_t                 req_key,
  nasd_security_param_t     *sec_param,
  nasd_capability_t         *capability,
  int                        is_read2,
  nasd_p_smpl_op_dr_args_t  *args,
  void                      *buf,
  nasd_p_fastread_dr_res_t  *res,
  nasd_rpc_status_t         *status);

typedef void (*nasd_cl_pr_range_read_t)(
  nasd_drive_handle_t         handle,
  nasd_key_t                  in_key,
  nasd_security_param_t      *sec_param,
  nasd_capability_t          *capability,
  int                         is_read2,
  nasd_p_smpl_op_dr_args_t   *args,
  nasd_mem_list_t            *memlist,
  nasd_p_fastread_dr_res_t   *res,
  nasd_rpc_status_t          *status);

typedef void (*nasd_cl_pr_tread_simple_t)(
  nasd_drive_handle_t         handle,
  nasd_key_t                  in_key,
  nasd_security_param_t      *sec_param,
  nasd_capability_t          *capability,
  int                         is_read2,
  nasd_p_thrtl_op_dr_args_t  *args,
  void                       *buf,
  nasd_p_fastread_dr_res_t   *res,
  nasd_rpc_status_t          *status);

typedef void (*nasd_cl_pr_range_tread_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  int                          is_read2,
  nasd_p_thrtl_op_dr_args_t   *args,
  nasd_mem_list_t             *memlist,
  nasd_p_fastread_dr_res_t    *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_setattr_otw_provided_t)(
  nasd_drive_handle_t        handle,
  nasd_key_t                 in_key,
  nasd_security_param_t     *sec_param,
  nasd_capability_t         *capability,
  nasd_p_setattr_dr_args_t  *args,
  nasd_p_setattr_dr_res_t   *res,
  nasd_cl_p_otw_buf_t       *otw,
  nasd_rpc_status_t         *status);

typedef void (*nasd_cl_pr_flush_obj_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_flush_obj_dr_args_t  *args,
  nasd_p_flush_obj_dr_res_t   *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_eject_obj_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_eject_obj_dr_args_t  *args,
  nasd_p_eject_obj_dr_res_t   *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_remove_t)(
  nasd_drive_handle_t       handle,
  nasd_key_t                in_key,
  nasd_security_param_t    *sec_param,
  nasd_capability_t        *capability,
  nasd_p_remove_dr_args_t  *args,
  nasd_p_remove_dr_res_t   *res,
  nasd_rpc_status_t        *status);

typedef void (*nasd_cl_pr_initialize_t)(
  nasd_drive_handle_t           handle,
  nasd_p_initialize_dr_args_t  *args,
  nasd_p_initialize_dr_res_t   *res,
  nasd_rpc_status_t            *status);

typedef void (*nasd_cl_pr_strt_iread_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_strt_iread_dr_args_t *args,
  nasd_p_strt_iread_dr_res_t  *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_stop_iread_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_stop_iread_dr_args_t *args,
  nasd_p_stop_iread_dr_res_t  *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_rshutdown_t)(
  nasd_drive_handle_t          handle,
  nasd_key_t                   in_key,
  nasd_security_param_t       *sec_param,
  nasd_capability_t           *capability,
  nasd_p_rshutdown_dr_args_t  *args,
  nasd_p_rshutdown_dr_res_t   *res,
  nasd_rpc_status_t           *status);

typedef void (*nasd_cl_pr_getinfo_t)(
  nasd_drive_handle_t       handle,
  nasd_p_getinfo_dr_res_t  *res,
  nasd_rpc_status_t        *status);

typedef void (*nasd_cl_pr_remote_attach_t)(
  nasd_drive_handle_t              handle,
  nasd_key_t                      in_key,
  nasd_security_param_t          *sec_param,
  nasd_capability_t              *capability,
  nasd_p_remote_attach_dr_args_t   *args,
  void                             *buf,
  nasd_p_remote_attach_dr_res_t    *res,
  nasd_rpc_status_t                *status);
 
typedef void (*nasd_cl_pr_remote_detach_t)(
  nasd_drive_handle_t              handle,
  nasd_key_t                      in_key,
  nasd_security_param_t          *sec_param,
  nasd_capability_t              *capability,
  nasd_p_remote_detach_dr_args_t   *args,
  nasd_p_remote_detach_dr_res_t    *res,
  nasd_rpc_status_t                *rpc_status);

typedef void (*nasd_cl_pr_remote_invoke_t)(
  nasd_drive_handle_t              handle,
  nasd_key_t                       in_key,
  nasd_security_param_t            *sec_param,
  nasd_capability_t                *capability,
  nasd_p_smpl_op_dr_args_t         *args,
  void                             *buf,
  nasd_p_fastread_dr_res_t         *res,
  nasd_rpc_status_t                *rpc_status);

typedef void (*nasd_cl_pr_error_string_t)(
  nasd_drive_handle_t   handle,
  nasd_rpc_status_t     status,
  nasd_error_string_t   str,
  char                 *file,
  int                   line);

struct nasd_cl_p_rpcmod_tab_s {
  nasd_cl_pr_init_t          init;
  nasd_status_t              init_status;

  nasd_cl_pr_bind_t          bind;
  nasd_cl_pr_unbind_t        unbind;

  nasd_cl_pr_null_t          null;
  nasd_cl_pr_sync_t          sync;
  nasd_cl_pr_part_creat_t    part_creat;
  nasd_cl_pr_getattr_t       getattr;
  nasd_cl_pr_write_simple_t  write_simple;
  nasd_cl_pr_range_write_t   range_write;
  nasd_cl_pr_read_simple_t   read_simple;
  nasd_cl_pr_range_read_t    range_read;
  nasd_cl_pr_tread_simple_t  tread_simple;
  nasd_cl_pr_range_tread_t   range_tread;
  nasd_cl_pr_flush_obj_t     flush_obj;
  nasd_cl_pr_eject_obj_t     eject_obj;
  nasd_cl_pr_remove_t        remove;
  nasd_cl_pr_initialize_t    initialize;
  nasd_cl_pr_strt_iread_t    strt_iread;
  nasd_cl_pr_stop_iread_t    stop_iread;
  nasd_cl_pr_rshutdown_t     rshutdown;
  nasd_cl_pr_getinfo_t       getinfo;

  nasd_cl_pr_remote_attach_t remote_attach;
  nasd_cl_pr_remote_detach_t remote_detach;
  nasd_cl_pr_remote_invoke_t remote_invoke;

  nasd_cl_pr_create_otw_provided_t   create_otw_provided;
  nasd_cl_pr_setattr_otw_provided_t  setattr_otw_provided;

  nasd_cl_pr_error_string_t  error_string;

  int                        binding_type;
  char                      *name;
  int                        spec_size;

  nasd_cl_p_rpcmod_tab_t    *next;
};

struct nasd_drive_default_bindings_s {
  int  *bindings;
  int   nbindings;
};

extern nasd_drive_default_bindings_t *nasd_drive_default_bindings;

/*
 * !!! note that nasd_drive_handle_t is a pointer
 * to the structure, not the structure itself, which
 * is not the NASD standard. This is for dealing with
 * older client code, and code which isn't aware of the
 * structured nature of this type. This also allows
 * NULL/non-NULL checks, and lets us more easily pass
 * pointers to the structure on the stack without
 * changing legacy code.
 */
struct nasd_drive_handle_s {
  int                      type;      /* binding type */
  void                    *rpc_specific_handle;
  int                      rpc_specific_handle_size;
  nasd_cl_p_rpcmod_tab_t  *rpc_tab;
  NASD_DECLARE_MUTEX(drive_delta_lock)
  nasd_timespec_t          drive_delta;
  int                      drive_is_ahead;
  int                      skew_retries;
  nasd_drive_handle_t      next;
  nasd_drive_handle_t      prev;
  nasd_sec_nonce_mgr_t    *nonce_mgr;
};

/*
 * module management
 */
extern nasd_status_t nasd_cl_p_mods_init(nasd_shutdown_list_t *sl);

extern void nasd_cl_p_mods_load();

extern void nasd_cl_p_mod_register(nasd_cl_p_rpcmod_tab_t *tab,
  int counter_lock_held);
extern void nasd_cl_p_mod_unregister(nasd_cl_p_rpcmod_tab_t *tab,
  int counter_lock_held);

/*
 * handle management
 */
extern nasd_status_t nasd_bind_to_drive(char *servername, char *portnum,
  int binding_type, void *binding_param, int binding_param_len,
  nasd_drive_handle_t *handle);

extern nasd_status_t nasd_unbind_drive(nasd_drive_handle_t *handle);


/*
 * RPC error wrappers
 */
extern char *_nasd_cl_error_string(nasd_drive_handle_t handle,
  nasd_rpc_status_t status, nasd_error_string_t str,
  char *file, int line);

/*
 * RPC wrappers
 */

extern void nasd_cl_p_null_dr(nasd_drive_handle_t handle,
  nasd_status_t *nasd_status, nasd_rpc_status_t *status);

extern void nasd_cl_p_sync_dr(nasd_drive_handle_t handle,
  nasd_status_t *nasd_status, nasd_rpc_status_t *status);

extern void nasd_cl_p_part_creat_dr(nasd_drive_handle_t handle, 
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_part_creat_dr_args_t *args,
  nasd_p_part_creat_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_create_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_create_dr_args_t *args, 
  nasd_p_create_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_create_dr__otw_provided(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_p_create_dr_args_t *args, nasd_p_create_dr_res_t *res,
  nasd_cl_p_otw_buf_t *otw,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_getattr_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_getattr_dr_args_t *args,
  nasd_p_getattr_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_write_simple_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args, void *buf, 
  nasd_p_fastwrite_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_read_simple_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args, void *buf,
  nasd_p_fastread_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_read2_simple_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args, void *buf,
  nasd_p_fastread_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_tread_simple_dr(nasd_drive_handle_t handle, 
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_thrtl_op_dr_args_t *args, void *buf,
  nasd_p_fastread_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_range_read_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args,
  nasd_mem_list_t *memlist, nasd_p_fastread_dr_res_t *res,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_range_tread_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_thrtl_op_dr_args_t *args,
  nasd_mem_list_t *memlist, nasd_p_fastread_dr_res_t *res,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_range_write_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args,
  nasd_mem_list_t *memlist, nasd_p_fastwrite_dr_res_t *res,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_setattr_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability,
  nasd_p_setattr_dr_args_t *args, nasd_p_setattr_dr_res_t *res,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_setattr_dr__otw_provided(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param,  nasd_capability_t *capability,
  nasd_p_setattr_dr_args_t *args, nasd_p_setattr_dr_res_t *res,
  nasd_cl_p_otw_buf_t *otw,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_flush_obj_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_flush_obj_dr_args_t *args, 
  nasd_p_flush_obj_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_eject_obj_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_eject_obj_dr_args_t *args, 
  nasd_p_eject_obj_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_remove_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_remove_dr_args_t *args,
  nasd_p_remove_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_initialize_dr(nasd_drive_handle_t handle,
  nasd_p_initialize_dr_args_t *args, nasd_p_initialize_dr_res_t *res,
  nasd_rpc_status_t *status);

extern void nasd_cl_p_strt_iread_dr(nasd_drive_handle_t handle,
  nasd_key_t in_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_strt_iread_dr_args_t *args,
	nasd_p_strt_iread_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_stop_iread_dr(nasd_drive_handle_t handle,
  nasd_key_t in_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_stop_iread_dr_args_t *args,
  nasd_p_stop_iread_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_rshutdown_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_rshutdown_dr_args_t *args,
  nasd_p_rshutdown_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_getinfo_dr(nasd_drive_handle_t handle,
  nasd_p_getinfo_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_remote_attach_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_remote_attach_dr_args_t *args, void *buf,
  nasd_p_remote_attach_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_remote_detach_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_remote_detach_dr_args_t *args, 
  nasd_p_remote_detach_dr_res_t *res, nasd_rpc_status_t *status);

extern void nasd_cl_p_remote_invoke_dr(nasd_drive_handle_t handle,
  nasd_key_t req_key, nasd_security_param_t *sec_param,
  nasd_capability_t *capability, nasd_p_smpl_op_dr_args_t *args, void *buf,
  nasd_p_fastread_dr_res_t *res, nasd_rpc_status_t *status);

extern nasd_status_t nasd_cl_p_update_skew(nasd_drive_handle_t handle);

extern void nasd_drive_handle_get_time(nasd_drive_handle_t handle,
  nasd_timespec_t *tsp);

extern nasd_status_t nasd_cl_p_ctrl_rdwr_stats(
  nasd_drive_handle_t handle, int partnum, nasd_identifier_t nasdid,
  nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  void *info, nasd_offset_t offset, int read);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_info(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability, nasd_ctrl_drive_info_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_info(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability, nasd_ctrl_drive_info_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_cache_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_cache_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_op_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_drive_opstats_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_op_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_drive_opstats_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_cache_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_cache_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_io_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_io_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_io_read_size_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_size_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_io_read_size_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_size_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_io_write_size_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_size_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_io_write_size_stats(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_io_size_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_layout_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_layout_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_layout_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_layout_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_ioqueue_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_ioqueue_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_ioqueue_stats(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_ioqueue_stat_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_iread_cfg(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_index_read_cfg_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_iread_cfg(
  nasd_drive_handle_t handle, nasd_key_t in_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_index_read_cfg_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_part_info(nasd_drive_handle_t handle,
  nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  int partnum, nasd_ctrl_part_info_t *info);

extern nasd_status_t nasd_cl_p_ctrl_get_drive_trace_basic(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_trace_basic_t *info);

extern nasd_status_t nasd_cl_p_ctrl_put_drive_trace_basic(
  nasd_drive_handle_t handle, nasd_key_t req_key,
  nasd_security_param_t *sec_param, nasd_capability_t *capability,
  nasd_ctrl_trace_basic_t *info);

#endif /* !_NASD__NASD_PDRIVE_CLIENT_H_ */

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
