/*
 * pcrpart.c
 *
 * Create partition on a NASD drive.
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1997,1998,1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>

char *progname;

int binding_type = NASD_BIND_DEFAULT;

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername partnum nblocks min_protection_level master_password\n", progname);
  fprintf(stderr, "    partnum starts at 1\n");
  fprintf(stderr, "    nblocks in NASD starts at 1\n");

  fprintf(stderr, "  where minimum protection levels are \n" );
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -M     use message queues\n");
#if NASD_SECURE_RPCS_ENABLE > 0
  fprintf(stderr, "  -s     use security on RPC ops\n");
#endif /* NASD_SECURE_RPCS_ENABLE */
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T     use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */

  fflush(stderr);
  exit(1);
}

int
main(argc, argv)
  int     argc;
  char  **argv;
{
  nasd_sec_keyring_t keys;
  nasd_security_param_t sec_param;
  nasd_p_part_creat_dr_args_t args;
  nasd_p_part_creat_dr_res_t res;
  nasd_error_string_t error_text;
  nasd_rpc_status_t status;
  nasd_drive_handle_t h;
  int partnum, nblocks;
  char *server_name, c;
  nasd_status_t rc;
  int min_sec_level = 0;
  int sec_level = 0;
  nasd_uint16 protection;

  progname = argv[0];

  while(nasd_getopt(argc, argv, "MTs", &c)) {
    switch(c) {
      case 'M':
        binding_type = NASD_BIND_MSGQ;
        break;
      case 's':
        sec_level = NASD_MAX_SECURITY_LEVEL;
        break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
      case 'T':
        binding_type = NASD_BIND_DCE_DIRECT_TCP;
        break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &partnum) != 1)
    usage();
  nasd_optind++;
  if (partnum < 1)
    usage();

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &nblocks) != 1)
    usage();
  nasd_optind++;
  if (nblocks < 1)
    usage();

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind],"%d",&min_sec_level) != 1 ) 
    usage();
  nasd_optind++;

  bzero((char *)&args, sizeof(nasd_p_part_creat_dr_args_t)); 

  rc = nasd_sec_seclevel_to_protection(min_sec_level, &args.in_min_protection);
  if(rc) {
    fprintf(stderr, "Unknown security level %d\n", min_sec_level);
    usage();
  }
  printf("Creating partition with security level %d (protection 0x%x)\n",
         min_sec_level, args.in_min_protection);

  rc = nasd_cl_p_init();
  if (rc) {
    printf("ERROR (%s:%d): cannot init client library, rc=%ld\n",
      __FILE__, __LINE__, (long)rc);
    return(rc);
  }

  if (nasd_optind >= argc)
    usage();
  nasd_sec_password_to_keys(argv[nasd_optind], partnum, &keys);
  nasd_optind++;

  if (nasd_optind < argc)
    usage();

  rc = nasd_bind_to_drive(server_name, NASD_PDRIVE_PORT,
    binding_type, NULL, 0, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s\n", server_name);
    fflush(stderr);
    exit(1);
  }

  /* Fill in capability and security param */
  nasd_sec_seclevel_to_protection(NASD_MAX_SECURITY_LEVEL, &protection);
  sec_param.actual_protection = protection;
  sec_param.type = NASD_DRIVE_KEY;
  sec_param.partnum = partnum;
  args.in_partnum = partnum;
  args.in_blkcnt = nblocks;

  /* initial keys for this partition */
  bcopy(keys.part_key, args.in_partition_key, sizeof(nasd_key_t));
  bcopy(keys.red_key, args.in_red_key, sizeof(nasd_key_t));
  bcopy(keys.black_key, args.in_black_key, sizeof(nasd_key_t));

  nasd_cl_p_part_creat_dr(h, keys.drive_key, &sec_param, NULL,
                          &args, &res, &status);

  if (res.nasd_status || status) {
    fprintf(stderr, "ERROR: nasd_status 0x%x (%s) status 0x%x (%s)\n",
      res.nasd_status, nasd_error_string(res.nasd_status),
      status, nasd_cl_error_string(h, status, error_text));
    exit(2);
  }

  nasd_unbind_drive(&h);
  nasd_cl_p_shutdown();

  printf("Succeeded\n");
  fflush(stdout);

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
