/*
 * nasd_out.c
 *
 * Read data from a NASD drive
 *
 * Author: Jim Zelenka
 */
/*
 * Copyright (c) of Carnegie Mellon University, 1999.
 *
 * Permission to reproduce, use, and prepare derivative works of
 * this software for internal use is granted provided the copyright
 * and "No Warranty" statements are included with all reproductions
 * and derivative works. This software may also be redistributed
 * without charge provided that the copyright and "No Warranty"
 * statements are included in all redistributions.
 *
 * NO WARRANTY. THIS SOFTWARE IS FURNISHED ON AN "AS IS" BASIS.
 * CARNEGIE MELLON UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER
 * EXPRESSED OR IMPLIED AS TO THE MATTER INCLUDING, BUT NOT LIMITED
 * TO: WARRANTY OF FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY
 * OF RESULTS OR RESULTS OBTAINED FROM USE OF THIS SOFTWARE. CARNEGIE
 * MELLON UNIVERSITY DOES NOT MAKE ANY WARRANTY OF ANY KIND WITH RESPECT
 * TO FREEDOM FROM PATENT, TRADEMARK, OR COPYRIGHT INFRINGEMENT.
 */


#include <nasd/nasd_options.h>
#include <nasd/nasd_getopt.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <nasd/nasd_pdrive.h>
#include <nasd/nasd_pdrive_client.h>
#include <nasd/nasd_pdrive_client_kpdev.h>
#include <nasd/nasd_security.h>
#include <nasd/nasd_mem.h>

char *progname;

int                        binding_type;
int                        binding_args_len;
void                      *binding_args;
char                      *binding_port = NASD_PDRIVE_PORT;
nasd_drive_param_kpdev_t   kpdev_args;
int                        nondefault_binding = 0;
nasd_error_string_t        error_text;
int                        partnum;
nasd_sec_keyring_t         keys;
nasd_uint16                protection;

int                         verbose;
nasd_drive_handle_t         h;
char                       *master_password;
nasd_security_param_t       sec_param;
nasd_p_smpl_op_dr_args_t    rd_args;
nasd_p_fastread_dr_res_t    rd_res;
nasd_cookie_t               rd_cookie;
nasd_error_string_t         err_str;

#define DEFAULT_BUFLEN 131072
char *buf;
int buflen = DEFAULT_BUFLEN;

void
usage()
{
  int i;
  fprintf(stderr, "USAGE: %s [options] servername partnum master_password "
    "objectid filename [objectid filename [objectid filename...]]\n",
    progname);
  fprintf(stderr, "Options:\n");
  fprintf(stderr, "  -b buflen [default %d]\n", DEFAULT_BUFLEN);
  fprintf(stderr, "  -k use kernel device\n");
  fprintf(stderr, "  -M use message queues\n");
  fprintf(stderr, "  -s sec_level\n");
  fprintf(stderr, "  where sec_level is one of\n");
  for(i = 0; i <= NASD_MAX_SECURITY_LEVEL; i++) {
    fprintf(stderr, "     %d %s\n", i, nasd_sec_level_string(i));
  }
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
  fprintf(stderr, "  -T use DCE-TCP\n");
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
  fprintf(stderr, "  -v verbose\n");
  fflush(stderr);
  exit(1);
}

void
do_object(
  nasd_identifier_t   nid,
  char               *filename)
{
  int no_close, fd, got, want, rc;
  nasd_rpc_status_t status;
  nasd_uint64 total_len;
  nasd_offset_t offset;
  nasd_timespec_t tm;
  char *say_filename;

  total_len = 0;

  bzero((char *)&rd_args, sizeof(rd_args));

  if (verbose) {
    printf("Begin processing 0x%" NASD_ID_FMT "\n", nid);
  }

  /*
   * Open file
   */

  no_close = 0;
  if ((filename[0] == '-') && (filename[1] == '\0')) {
    no_close = 1;
    fd = fileno(stdout);
    say_filename = "stdout";
  }
  else {
    fd = open(filename, O_CREAT|O_TRUNC|O_WRONLY, S_IRUSR|S_IWUSR);
    if (fd < 0) {
      perror("open");
      fprintf(stderr, "ERROR: could not open %s for writing\n", filename);
      fflush(stderr);
      exit(1);
    }
    say_filename = filename;
  }

  if (verbose) {
    printf("%s opened as %d\n", filename, fd);
  }

  nasd_drive_handle_get_time(h, &tm);
  tm.ts_sec += (60*60);
  rd_args.in_identifier = nid;

  nasd_sec_build_capability(partnum,
    rd_args.in_identifier, NASD_ACCESS_RIGHTS_READ,
    0, tm.ts_sec, protection, NASD_BLACK_CAPABILITY,
    0, 0, 7, keys.black_key, &rd_cookie);

  sec_param.type = NASD_BLACK_CAPABILITY;
  sec_param.partnum = partnum;
  sec_param.actual_protection = protection;
  rd_args.in_partnum = partnum;
  rd_args.in_len = buflen;

  /*
   * Loop, reading drive and writing file
   */
  offset = 0;
  while(1) {
    /*
     * Read a chunk from the drive
     */

    rd_args.in_offset = offset;

    nasd_cl_p_read_simple_dr(h, rd_cookie.key, &sec_param,
                             &rd_cookie.capability, &rd_args, buf,
                             &rd_res, &status);

    if (status || rd_res.nasd_status) {
      fprintf(stderr, "ERROR: reading 0x%" NASD_ID_FMT " for %s"
        ", status=0x%x (%s) nasd_status=0x%x (%s)\n",
        nid, say_filename,
        status, nasd_cl_error_string(h, status, err_str),
        rd_res.nasd_status, nasd_error_string(rd_res.nasd_status));
      fflush(stderr);
      exit(1);
    }

    if (verbose) {
      printf("read %d bytes from 0x%" NASD_ID_FMT "\n",
        rd_res.out_datalen, nid);
    }

    want = rd_res.out_datalen;
    offset += rd_res.out_datalen;

    if (want == 0)
      break;


    /*
     * Write to file
     */
    for(got=0;got<want;) {
      rc = write(fd, &buf[got], want-got);

      if (rc < 0) {
        perror("write");
        fprintf(stderr, "ERROR: could not write %s\n",
          say_filename);
        fflush(stderr);
        exit(1);
      }

      if (rc == 0) {
        fprintf(stderr, "ERROR: could not write %s (rc is 0)\n",
          say_filename);
        fflush(stderr);
        exit(1);
        break;
      }

      total_len += rc;
      got += rc;
    }

    if (verbose) {
      printf("wrote %d bytes to %s\n", want, say_filename);
    }
  }

  /*
   * Close file
   */

  if (no_close == 0) {
    close(fd);
  }

  printf("Wrote 0x%" NASD_ID_FMT " to %s (%" NASD_64u_FMT " total bytes)\n",
    nid, say_filename, total_len);
}

int
main(
  int     argc,
  char  **argv)
{
  nasd_identifier_t nid;
  char *server_name;
  nasd_status_t rc;
  char c;
  int i;
  int sec_level = 0;

  progname = argv[0];

  binding_type = NASD_BIND_DEFAULT;
  binding_args = NULL;
  binding_args_len = 0;

  while (nasd_getopt(argc, argv, "b:kMs:Tv", &c)) {
    switch(c) {
      case 'b':
        if (sscanf(nasd_optarg, "%d", &buflen) != 1)
          usage();
        if (buflen < 1)
          usage();
        break;
      case 'k':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_KPDEV_DEFAULT;
        binding_args = &kpdev_args;
        binding_args_len = sizeof(kpdev_args);
        strcpy(kpdev_args.devname, "/dev/nasdkp0");
        break;
      case 'M':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_MSGQ;
        break;
      case 's':
        sec_level = atoi(nasd_optarg);
        break;
#if NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE
      case 'T':
        if (nondefault_binding)
          usage();
        nondefault_binding = 1;
        binding_type = NASD_BIND_DCE_DIRECT_TCP;
        break;
#endif /* NASD_RPC_PACKAGE == NASD_RPC_PACKAGE_DCE */
      case 'v':
        verbose = 1;
        break;
      default:
        fprintf(stderr, "Unknown option '%c'\n", nasd_optopt);
        usage();
    }
  }

  if (nasd_optind >= argc)
    usage();
  server_name = argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  if (sscanf(argv[nasd_optind], "%d", &partnum) != 1)
    usage();
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();
  master_password=argv[nasd_optind];
  nasd_optind++;

  if (nasd_optind >= argc)
    usage();

  rc = nasd_sec_seclevel_to_protection(sec_level, &protection);
  if(rc) {
    fprintf(stderr, "ERROR: invalid security level %d\n", sec_level);
    usage();
  }

  rc = nasd_cl_p_init();
  if (rc) {
    fprintf(stderr, "ERROR: cannot init client library, rc=0x%x (%s)\n",
      rc, nasd_error_string(rc));
    exit(1);
  }

  nasd_sec_password_to_keys(master_password, partnum, &keys);

  NASD_Malloc(buf, buflen, (char *));
  if (buf == NULL) {
    fprintf(stderr, "ERROR: could not allocate %d bytes for buffer\n",
      buflen);
    fflush(stderr);
    exit(1);
  }

  rc = nasd_bind_to_drive(server_name, binding_port,
    binding_type, binding_args, binding_args_len, &h);
  if (rc) {
    fprintf(stderr, "ERROR: cannot bind to server %s\n", server_name);
    fflush(stderr);
    exit(1);
  }

  if ((argc-nasd_optind)%2)
    usage();

  for(i=nasd_optind;i<argc;i+=2) {
    rc = nasd_str_to_nasd_id(argv[i], &nid);
    if (rc) {
      fprintf(stderr, "ERROR: Could not parse %s into"
        " a nasd object identifier\n",
        argv[i]);
      fflush(stderr);
      exit(1);
    }
    NASD_ASSERT((i+1) < argc);
    do_object(nid, argv[i+1]);
  }

  NASD_Free(buf, buflen);

  rc = nasd_unbind_drive(&h);
  if (rc) {
    fprintf(stderr, "ERROR: got 0x%x (%s) from nasd_unbind_drive()\n",
      rc, nasd_error_string(rc));
    fflush(stderr);
    exit(2);
  }

  nasd_cl_p_shutdown();

  exit(0);
}

/* Local Variables:  */
/* indent-tabs-mode: nil */
/* tab-width: 2 */
/* End: */
