/*
 * scamper_sniff_warts.c
 *
 * Copyright (C) 2011      The University of Waikato
 * Copyright (C) 2014      The Regents of the University of California
 * Copyright (C) 2016-2023 Matthew Luckie
 * Copyright (C) 2025      The Regents of the University of California
 * Author: Matthew Luckie
 *
 * $Id: scamper_sniff_warts.c,v 1.18 2025/10/19 21:53:46 mjl Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "internal.h"

#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_sniff.h"
#include "scamper_sniff_int.h"
#include "scamper_file.h"
#include "scamper_file_warts.h"
#include "scamper_sniff_warts.h"

#include "mjl_list.h"
#include "utils.h"

#define WARTS_SNIFF_LIST        1
#define WARTS_SNIFF_CYCLE       2
#define WARTS_SNIFF_USERID      3
#define WARTS_SNIFF_SRC         4
#define WARTS_SNIFF_START       5
#define WARTS_SNIFF_FINISH      6
#define WARTS_SNIFF_STOP_REASON 7
#define WARTS_SNIFF_LIMIT_PKTC  8
#define WARTS_SNIFF_LIMIT_TIME  9
#define WARTS_SNIFF_PKTC        10
#define WARTS_SNIFF_ICMPID      11
#define WARTS_SNIFF_ERRMSG      12

static const warts_var_t sniff_vars[] =
{
  {WARTS_SNIFF_LIST,         4},
  {WARTS_SNIFF_CYCLE,        4},
  {WARTS_SNIFF_USERID,       4},
  {WARTS_SNIFF_SRC,         -1},
  {WARTS_SNIFF_START,        8},
  {WARTS_SNIFF_FINISH,       8},
  {WARTS_SNIFF_STOP_REASON,  1},
  {WARTS_SNIFF_LIMIT_PKTC,   4},
  {WARTS_SNIFF_LIMIT_TIME,   2},
  {WARTS_SNIFF_PKTC,         4},
  {WARTS_SNIFF_ICMPID,       2},
  {WARTS_SNIFF_ERRMSG,      -1},
};
#define sniff_vars_mfb WARTS_VAR_MFB(sniff_vars)

#define WARTS_SNIFF_PKT_TIME     1
#define WARTS_SNIFF_PKT_DATALEN  2
#define WARTS_SNIFF_PKT_DATA     3

static const warts_var_t sniff_pkt_vars[] =
{
  {WARTS_SNIFF_PKT_TIME,            8},
  {WARTS_SNIFF_PKT_DATALEN,         2},
  {WARTS_SNIFF_PKT_DATA,           -1},
};
#define sniff_pkt_vars_mfb WARTS_VAR_MFB(sniff_pkt_vars)

typedef struct warts_sniff_pkt
{
  uint8_t               flags[sniff_pkt_vars_mfb];
  uint16_t              flags_len;
  uint16_t              params_len;
} warts_sniff_pkt_t;

static void warts_sniff_pkt_params(const scamper_sniff_pkt_t *pkt,
				   warts_sniff_pkt_t *state, uint32_t *len)
{
  const warts_var_t *var;
  int max_id = 0;
  size_t i;

  memset(state->flags, 0, sniff_pkt_vars_mfb);
  state->params_len = 0;

  for(i=0; i<sizeof(sniff_pkt_vars) / sizeof(warts_var_t); i++)
    {
      var = &sniff_pkt_vars[i];

      if(var->id == WARTS_SNIFF_PKT_DATA)
        {
	  if(pkt->len == 0)
	    continue;

	  state->params_len += pkt->len;
	  flag_set(state->flags, var->id, &max_id);
	  continue;
        }

      assert(var->size >= 0);
      state->params_len += var->size;
      flag_set(state->flags, var->id, &max_id);
    }

  state->flags_len = fold_flags(state->flags, max_id);

  *len += state->flags_len + state->params_len;

  if(state->params_len != 0)
    *len += 2;

  return;
}

static scamper_sniff_pkt_t *warts_sniff_pkt_read(warts_state_t *state,
						 uint8_t *buf, uint32_t *off,
						 uint32_t len)
{
  scamper_sniff_pkt_t *pkt = NULL;
  uint8_t *data = NULL;
  struct timeval tv;
  uint16_t plen;
  warts_param_reader_t handlers[] = {
    {&tv,    (wpr_t)extract_timeval,      NULL},
    {&plen,  (wpr_t)extract_uint16,       NULL},
    {&data,  (wpr_t)extract_bytes_ptr,   &plen},
  };
  const int handler_cnt = sizeof(handlers)/sizeof(warts_param_reader_t);

  if(warts_params_read(buf, off, len, handlers, handler_cnt) != 0 ||
     plen == 0 || data == NULL ||
     (pkt = scamper_sniff_pkt_alloc(data, plen, &tv)) == NULL)
    goto err;

  return pkt;

 err:
  if(pkt != NULL) scamper_sniff_pkt_free(pkt);
  return NULL;
}

static int warts_sniff_pkt_write(const scamper_sniff_pkt_t *pkt,
				 const scamper_file_t *sf, uint8_t *buf,
				 uint32_t *off, const uint32_t len,
				 warts_sniff_pkt_t *state)
{
  uint16_t dl = pkt->len;
  warts_param_writer_t handlers[] = {
    {&pkt->tv,    (wpw_t)insert_timeval,       NULL},
    {&pkt->len,   (wpw_t)insert_uint16,        NULL},
    {pkt->data,   (wpw_t)insert_bytes_uint16, &dl},
  };
  const int handler_cnt = sizeof(handlers)/sizeof(warts_param_writer_t);
  warts_params_write(buf, off, len, state->flags, state->flags_len,
		     state->params_len, handlers, handler_cnt);
  return 0;
}

static int warts_sniff_params(const scamper_sniff_t *sniff,
			      warts_addrtable_t *table, uint8_t *flags,
			      uint16_t *flags_len, uint16_t *params_len)
{
  const warts_var_t *var;
  int max_id = 0;
  size_t i;

  /* Unset all flags */
  memset(flags, 0, sniff_vars_mfb);
  *params_len = 0;

  for(i=0; i<sizeof(sniff_vars)/sizeof(warts_var_t); i++)
    {
      var = &sniff_vars[i];

      /* Skip the variables for which we have no data */
      if((var->id == WARTS_SNIFF_LIST && sniff->list == NULL) ||
	 (var->id == WARTS_SNIFF_CYCLE && sniff->cycle == NULL) ||
	 (var->id == WARTS_SNIFF_USERID && sniff->userid == 0) ||
	 (var->id == WARTS_SNIFF_SRC && sniff->src == NULL) ||
	 (var->id == WARTS_SNIFF_ERRMSG && sniff->errmsg == NULL))
	continue;

      /* Set the flag for the rest of the variables */
      flag_set(flags, var->id, &max_id);

      /* Variables that don't have a fixed size */
      if(var->id == WARTS_SNIFF_SRC)
        {
	  if(warts_addr_size(table, sniff->src, params_len) != 0)
	    return -1;
        }
      else if(var->id == WARTS_SNIFF_ERRMSG)
	{
	  if(warts_str_size(sniff->errmsg, params_len) != 0)
	    return -1;
	}
      else
	{
	  /* The rest of the variables have a fixed size */
	  assert(var->size >= 0);
	  *params_len += var->size;
	}
    }

  *flags_len = fold_flags(flags, max_id);
  return 0;
}

static int warts_sniff_params_read(scamper_sniff_t *sniff,
				   warts_addrtable_t *table,
				   warts_state_t *state,
				   uint8_t *buf, uint32_t *off, uint32_t len)
{
  uint16_t limit_time = 0;
  warts_param_reader_t handlers[] = {
    {&sniff->list,         (wpr_t)extract_list,         state},
    {&sniff->cycle,        (wpr_t)extract_cycle,        state},
    {&sniff->userid,       (wpr_t)extract_uint32,       NULL},
    {&sniff->src,          (wpr_t)extract_addr,         table},
    {&sniff->start,        (wpr_t)extract_timeval,      NULL},
    {&sniff->finish,       (wpr_t)extract_timeval,      NULL},
    {&sniff->stop_reason,  (wpr_t)extract_byte,         NULL},
    {&sniff->limit_pktc,   (wpr_t)extract_uint32,       NULL},
    {&limit_time,          (wpr_t)extract_uint16,       NULL},
    {&sniff->pktc,         (wpr_t)extract_uint32,       NULL},
    {&sniff->icmpid,       (wpr_t)extract_uint16,       NULL},
    {&sniff->errmsg,       (wpr_t)extract_string,       NULL},
  };
  const int handler_cnt = sizeof(handlers)/sizeof(warts_param_reader_t);
  int rc;
  if((rc = warts_params_read(buf, off, len, handlers, handler_cnt)) != 0)
    return rc;
  if(sniff->src == NULL)
    return -1;
  sniff->limit_time.tv_sec = limit_time;
  return 0;
}

static int warts_sniff_params_write(const scamper_sniff_t *sniff,
				    const scamper_file_t *sf,
				    warts_addrtable_t *table,
				    uint8_t *buf, uint32_t *off,
				    const uint32_t len, const uint8_t *flags,
				    const uint16_t flags_len,
				    const uint16_t params_len)
{
  uint32_t list_id, cycle_id;
  uint16_t limit_time;

  /* Specifies how to write each variable to the warts file. */
  warts_param_writer_t handlers[] = {
    {&list_id,             (wpw_t)insert_uint32,       NULL},
    {&cycle_id,            (wpw_t)insert_uint32,       NULL},
    {&sniff->userid,       (wpw_t)insert_uint32,       NULL},
    {sniff->src,           (wpw_t)insert_addr,         table},
    {&sniff->start,        (wpw_t)insert_timeval,      NULL},
    {&sniff->finish,       (wpw_t)insert_timeval,      NULL},
    {&sniff->stop_reason,  (wpw_t)insert_byte,         NULL},
    {&sniff->limit_pktc,   (wpw_t)insert_uint32,       NULL},
    {&limit_time,          (wpw_t)insert_uint16,       NULL},
    {&sniff->pktc,         (wpw_t)insert_uint32,       NULL},
    {&sniff->icmpid,       (wpw_t)insert_uint16,       NULL},
    {sniff->errmsg,        (wpw_t)insert_string,       NULL},
  };
  const int handler_cnt = sizeof(handlers)/sizeof(warts_param_writer_t);

  if(warts_list_getid(sf,  sniff->list,  &list_id)  == -1) return -1;
  if(warts_cycle_getid(sf, sniff->cycle, &cycle_id) == -1) return -1;
  limit_time = sniff->limit_time.tv_sec;

  warts_params_write(buf, off, len, flags, flags_len, params_len,
		     handlers, handler_cnt);

  return 0;
}

int scamper_file_warts_sniff_read(scamper_file_t *sf, const warts_hdr_t *hdr,
				 scamper_sniff_t **sniff_out)
{
  scamper_sniff_t *sniff = NULL;
  scamper_sniff_pkt_t *pkt = NULL;
  warts_addrtable_t *table = NULL;
  warts_state_t *state = scamper_file_getstate(sf);
  uint8_t *buf = NULL;
  uint32_t off = 0;
  uint32_t i, pktc;
  slist_t *list = NULL;

  /* Read in the header */
  if(warts_read(sf, &buf, hdr->len) != 0)
    {
      goto err;
    }

  if(buf == NULL)
    {
      *sniff_out = NULL;
      return 0;
    }

  /* Allocate space for a sniff object */
  if((sniff = scamper_sniff_alloc()) == NULL)
    {
      goto err;
    }

  if((table = warts_addrtable_alloc_byid()) == NULL)
    goto err;

  /* Read in the sniff data from the warts file */
  if(warts_sniff_params_read(sniff, table, state, buf, &off, hdr->len) != 0)
    {
      goto err;
    }

  /* Determine how many sniff pkts to read */
  if(sniff->pktc > 0)
    {
      /* for each sniff packet, read it and put it in a temporary list */
      pktc = sniff->pktc; sniff->pktc = 0;
      if((list = slist_alloc()) == NULL)
	goto err;
      for(i=0; i<pktc; i++)
	{
	  if((pkt = warts_sniff_pkt_read(state, buf, &off, hdr->len)) == NULL ||
	     slist_tail_push(list, pkt) == NULL)
	    goto err;
	  pkt = NULL;
	}

      if(scamper_sniff_pkts_alloc(sniff, pktc) != 0)
	goto err;
      while((pkt = slist_head_pop(list)) != NULL)
	sniff->pkts[sniff->pktc++] = pkt;
      slist_free(list); list = NULL;
    }

  warts_addrtable_free(table);
  *sniff_out = sniff;
  free(buf);
  return 0;

 err:
  if(list != NULL) slist_free_cb(list, (slist_free_t)scamper_sniff_pkt_free);
  if(pkt != NULL) scamper_sniff_pkt_free(pkt);
  if(table != NULL) warts_addrtable_free(table);
  if(buf != NULL) free(buf);
  if(sniff != NULL) scamper_sniff_free(sniff);
  return -1;
}

/* Write data from a scamper sniff object to a warts file */
int scamper_file_warts_sniff_write(const scamper_file_t *sf,
				   const scamper_sniff_t *sniff, void *p)
{
  warts_addrtable_t *table = NULL;
  warts_sniff_pkt_t *pkts = NULL;
  uint8_t *buf = NULL;
  uint8_t  flags[sniff_vars_mfb];
  uint16_t flags_len, params_len;
  uint32_t len, i, off = 0;
  size_t size;

  if((table = warts_addrtable_alloc_byaddr()) == NULL)
    goto err;

  /* Set the sniff data (not including the packets) */
  if(warts_sniff_params(sniff, table, flags, &flags_len, &params_len) != 0)
    goto err;
  len = 8 + flags_len + params_len + 2;

  if(sniff->pktc > 0)
    {
      /* Allocate memory for the state */
      size = sniff->pktc * sizeof(warts_sniff_pkt_t);
      if((pkts = (warts_sniff_pkt_t *)malloc_zero(size)) == NULL)
	goto err;

      for(i=0; i<sniff->pktc; i++)
	warts_sniff_pkt_params(sniff->pkts[i], &pkts[i], &len);
    }

  /* Allocate memory to store all of the data (including packets) */
  if((buf = malloc_zero(len)) == NULL)
    goto err;
  insert_wartshdr(buf, &off, len, SCAMPER_FILE_OBJ_SNIFF);

  /* Write the sniff data (excluding packets) to the buffer */
  if(warts_sniff_params_write(sniff, sf, table, buf, &off, len,
			      flags, flags_len, params_len) != 0)
    {
      goto err;
    }

  if(sniff->pktc > 0)
    {
      for(i=0; i<sniff->pktc; i++)
	warts_sniff_pkt_write(sniff->pkts[i], sf, buf, &off, len, &pkts[i]);
      free(pkts); pkts = NULL;
    }

  assert(off == len);

  /* Write the whole buffer to a warts file */
  if(warts_write(sf, buf, len, p) == -1)
    goto err;

  warts_addrtable_free(table);
  free(buf);
  return 0;

err:
  if(table != NULL) warts_addrtable_free(table);
  if(pkts != NULL) free(pkts);
  if(buf != NULL) free(buf);
  return -1;
}
