/*
 * Copyright (C) 1983-1989  Masatoshi Kurihara <kurihara@sra.co.jp>
 * Copyright (C) 1999, 2000 and 2001
 * Jun-ichiro itojun Hagino <itojun@iijlab.net>
 * All rights reserved.
 *
 * Note well: this is not a normal 3-clause BSD copyright;
 * commercial use of the software is restricted.
 *
 * Redistribution and non-commercial use in source and binary
 * forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the authors nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * Print Personal Schedule
 *
 *	% prsch [-[month/]day] [user | -f schedule]
 *		month, day -- print start date
 *		schedule   -- schedule file, default is ~/.schrc
 *		user       -- user name, file is ~user/.schrc
 *
 *	% cc -s -O -o prsch prsch.c
 *		-DBSD4_2  -- Berkeley 4.2 version
 *              -DMSKANJI -- mskanji version
 *		-DSRA     -- use local termcap name
 *
 *	Author	Masatoshi Kurihara
 *	Date	Sat Jun  4 13:12:39 JST 1983
 *		Thu Dec  5 10:32:59 JST 1985
 *	Office	Software Tools and Technology Group
 *		Software Research Associates, Inc.
 *		1-1-1 Hirakawa-cho Chiyoda-ku Tokyo, Japan
 *		03 (234) 2611 ext 153
 *		03 (234) 2615 (night only)
 *	Home	Waseda House 2A 
 *		1-9-22 Nishiwaseda Shinjuku-ku Tokyo, Japan
 *		03 (208) 3875
 *
 *	Copyright (C) 1985 - 1989 by Software Research Associates, Inc.
 */

static	char	sccsid[] = "@(#)prsch.c\t1.3 (SRA) 5/30/86";

#include <common.h>
#ifdef MULTIBYTE
# include <wchar.h>
#endif
#include <curses.h>
#include <err.h>

#include <entry.h>

#define	MAXCHR	160
#define	COLMON	0
#define	COLDAY	3
#define	COLMSG	35
#define POSBEG	7
#define	POSEND	13

#define	FNAME	"/.schrc"

#define	VT100GR	"\033(0"
#define	VT100TX	"\033(B"
#define	VT100SO	"\033[7m"
#define	VT100SE	"\033[0m"
#define	PC98SO	"\033[46m"
#define	PC98SE	"\033[0m"

#define	V1 "lqqqqqqqqqqqqqwqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqqqwqk"
#define	V2 "x  DATE/TIME  x 9  10  11  12   1   2   3   4   5   6   7   8   9 x"
#define	V3 "tqqqqqqqqqqqqqnqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqqqvqu"
#define	V4 "mqqqqqqqqqqqqqvqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqj"
#define	P1 "Ь"
#define	P2 " DATE/TIME  9  10  11  12   1   2   3   4   5   6   7   8   9 "
#define	P3 "ତ"
#define	P4 "ج"
#define	H1 "+-----------------------------------------------------------------+"
#define	H2 "|  date/time  | 9  10  11  12   1   2   3   4   5   6   7   8   9 |"
#define	H3 "|-------------+-*---*---*---*---*---*---*---*---*---*---*---*---*-|"
#define	H4 "+-----------------------------------------------------------------+"
#define	FM "| %2d/%2d (%3s) |                                                   |"

#ifdef	MSKANJI
static	int	mskanji = NULL;
#endif

#ifdef MULTIBYTE
static void mbdsp __P((wchar_t *));
#else
static char *jisdsp __P((char *));
#endif

/*
 * Print Personal Schedule
 *
 *	% prsch [-[month/]day] [user | -f schedule]
 *		month, day -- print start date
 *		schedule   -- schedule file, default is ~/.schrc
 *		user       -- user name, file is ~user/.schrc
 */

void
usage()
{
	fprintf(stderr, "usage: prsch [-d [month/}day] [user | -f schedule]\n");
}

int
main(argc, argv)
	int argc;
	char *argv[];
{
	struct tm *tm;
	struct passwd *pwd;
	FILE *schfile;
	char *pch, *term;
	int cnt, indch, indCH, mon, day, sanday, today;
	int stmon, stday, edmon, edday, stpos, edpos;
	char getbuf[MAXCHR], username[MAXCHR], schfname[MAXCHR];
	char timetbl[8][80];
#ifdef MULTIBYTE
	wchar_t msgtbl[30][40];
#else
	char msgtbl[30][40];
#endif
	time_t ltime;
	static int vt100 = NULL, pc98 = NULL, maxmsg = NULL;
	static int montbl[] =
		{ 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31, 30, 31, 0 };
	static char *week[] =
		{ "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
	static char *callid = "You have";
	struct sch *sch;
	int filever;
	int fileargs = 0;
	extern char *optarg;
	extern int optind;
	int ch;

	/*
	 * Date Set
	 */
	time(&ltime);
	tm = localtime(&ltime);
	if ((tm->tm_year % 4) == 0 && (tm->tm_year % 400) != 0)
		montbl[2]++;

	stmon = tm->tm_mon + 1;
	stday = tm->tm_mday + 1;
	while ((ch = getopt(argc, argv, "d:f")) != EOF) {
		switch (ch) {
		case 'd':
			if (sscanf(optarg, "%d/%d", &stmon, &stday) == 2)
				break;
			if (sscanf(optarg, "%d", &stday) == 1) {
				stmon = tm->tm_mon + 1;
				break;
			}
			errx(1, "bad date %s", optarg);
			/*NOTREACHED*/
		case 'f':
			fileargs = 1;
			break;
		default:
			usage();
			exit(1);
		}
	}
	argc -= optind;
	argv += optind;

#ifdef MULTILINGUAL
    {
	char *opt;
	if (opt = getenv("LANG"))
		setrunelocale(opt);
	else
		setrunelocale("C");
    }
#endif /*MULTILINGUAL*/


	for (today = 0, cnt = 1; cnt < stmon; cnt++)
		today += montbl[cnt];
	if ((today += stday) > (sanday = tm->tm_yday - tm->tm_wday + 1))
		for (; today > sanday + 6; sanday += 7) ;
	else
		for (; today < sanday; sanday -= 7) ;
	if (sanday < 1) {
		fprintf(stderr, "prsch: can't print first week on Janualy.\n");
		exit(1);
	}
	for (stmon = 1, cnt = montbl[1]; cnt < sanday; cnt += montbl[++stmon]) ;
	stday = sanday - (cnt - montbl[stmon]);
	if (stday + 7 > montbl[stmon]) {
		edmon = stmon + 1;
		edday = 6 - (montbl[stmon] - stday);
	}
	else {
		edmon = stmon;
		edday = stday + 6;
	}

	/*
	 * File Set
	 */
	if (argc == 0) {
		strcpy(username, getpwuid(geteuid())->pw_name);
		strcpy(schfname, getenv("HOME"));
		strcat(schfname, FNAME);
	} else if (argc == 1) {
		if (fileargs) {
			strcpy(username, getpwuid(geteuid())->pw_name);
			strcpy(schfname, *argv);
		} else {
			if ((pwd = getpwnam(*argv)) != NULL) {
				strcpy(username, pwd->pw_name);
				strcpy(schfname, pwd->pw_dir );
				strcat(schfname, FNAME);
				callid = "He has";
			} else {
				errx(1, "unknown user %s", *argv);
				/*NOTREACHED*/
			}
		}
	} else {
		usage();
		exit(1);
	}
	endpwent();
#if 0
	username[0] = toupper(username[0]);
#endif
	if ((schfile = fopen (schfname, "r")) == NULL) {
		fprintf(stderr, "prsch: can't open %s.\n", schfname);
		exit(1);
	}

	/*
	 * Terminal Set
	 */
#ifdef	SRA
#define	VT(x) (! strcmp (term, x))
	if (term = getenv ("TERM")) {
		if (VT ("d0" ) || VT ("ds" ) || VT ("d0b") || VT ("d0s") ||
		    VT ("dsb") || VT ("d2" ) || VT ("d2b") || VT ("d2s") ||
		    VT ("d2sb")|| VT("vt100")|| VT("vt220"))
			vt100++;
	}
	if (VT ("py" ) || VT ("p9y") || VT ("pv" ) || VT ("p9v") ||
	    VT ("pc9800y") || VT ("pc9800v" ) || VT ("pc9800")) {
		pc98++;
	}
#endif
#ifdef	MSKANJI
	if (term = getenv ("TTYPE")) {
		if (! strcmp (term, "mskanji"))
			mskanji++;
	}
#endif

	/*
	 * Table Initialize
	 */
	for (mon = stmon, day = stday, cnt = 0; cnt < 7; cnt++) {
		sprintf(timetbl[cnt], FM, mon, day, week[cnt]);
		if (++day > montbl[mon]) {
			if (++mon > 12)
				mon = 1;
			day = 1;
		}
	}

	/*
	 * Get Data
	 */
	filever = 0;
	while (fgets (getbuf, MAXCHR, schfile)) {
		/*
		 * Check Date
		 */
		if ((pch = index(getbuf, '\n')) != NULL)
			*pch = NULL;

		if (filever == 0) {
			sch = sch_oparse(getbuf);
			if (sch == NULL)
				filever = 1;
		}
		if (filever == 1)
			sch = sch_parse(getbuf);

		if (sch == NULL)
			continue;

		mon = sch->month;
		day = sch->day;
		if (mon > edmon || (mon == edmon && day > edday))
			break;
		if (mon < stmon || (mon == stmon && day < stday))
			continue;

		/*
		 * Time Set
		 */
		if (mon > stmon)
			today = day + (montbl[stmon] - stday);
		else
			today = day - stday;
		stpos = 16 + (sch->sthour - 9) * 4 + (sch->stmin) / 15;
		edpos = 16 + (sch->enhour - 9) * 4 + (sch->enmin) / 15;

		/*
		 * Time and Message Save
		 */
		if (stpos >= 16 && *(getbuf + COLMSG) && maxmsg < 25) {
			indch = 'a' + maxmsg;
			indCH = 'A' + maxmsg;
			for (cnt = stpos; cnt < edpos; cnt++)
				if (timetbl[today][cnt] == ' ')
					timetbl[today][cnt] = indch;
				else
					timetbl[today][cnt] = indCH;
			strcpy(msgtbl[maxmsg], sch->msg);
			maxmsg++;
		}

		free(sch);
	}
	fclose(schfile);

	/*
	 * Print Schedule
	 */
	if (vt100) {
		printf("%s %s %s %s",
			VT100SO, username, VT100SE, ctime(&ltime));
		fputs(VT100GR, stdout);
		puts(V1);
		puts(V2);
		puts(V3);
		fputs(VT100TX, stdout);
		for (cnt = 0; cnt < 7; cnt++) {
			for (pch = timetbl[cnt]; *pch; pch++) {
				if (*pch == '|') {
					fputs(VT100GR, stdout);
					putchar('x');
					fputs(VT100TX, stdout);
				}
				else
					putchar(*pch);
			}
			putchar('\n');
		}
		fputs(VT100GR, stdout);
		puts(V4);
		fputs(VT100TX, stdout);
	}
#if 0
	else if (pc98) {
		printf("%s %s %s %s",
			PC98SO, username, PC98SE, ctime(&ltime));
		puts(P1);
		puts(P2);
		puts(P3);
		for (cnt = 0; cnt < 7; cnt++) {
			for (pch = timetbl[cnt]; *pch; pch++) {
				if (*pch == '|') {
					fputs("", stdout);
					pch++;
				}
				else
					putchar(*pch);
			}
			putchar('\n');
		}
		puts(P4);
	}
#endif
	else {
		printf("%s: %s", username, ctime(&ltime));
		puts(H1);
		puts(H2);
		puts(H3);
		for (cnt = 0; cnt < 7; cnt++)
			puts(timetbl[cnt]);
		puts(H4);
	}
	putchar('\n');

	/*
	 * Print Message
	 */
	if (maxmsg) {
		for (cnt = 0; cnt < maxmsg; cnt++) {
#ifdef MULTIBYTE
			printf("%c: ", 'a' + cnt);
			mbdsp(msgtbl[cnt]);
			putchar('\n');
#else
			printf("%c: %-31s", 'a' + cnt, jisdsp(msgtbl[cnt]));
			if (++cnt < maxmsg)
				printf("   %c: %-31s\n",
					'a' + cnt, jisdsp(msgtbl[cnt]));
			else
				putchar('\n');
#endif
		}
	}
	else
		printf("%s no plan on this week.  Enjoy free time!\n", callid);

	exit(0);
}

#ifdef MULTIBYTE
static void
mbdsp(msg)
wchar_t *msg;
{
	const wchar_t *p;
	wchar_t wc[3];
	char mb[32];	/*XXX*/
	mbstate_t state;
	int i;

	for (p = (const wchar_t *)msg; *p; p++) {
		wcrtomb(NULL, '\0', &state);	/*initialize state*/
		mb[0] = '\0';
#if 1
		wc[0] = *p;
		wc[1] = '\0';
		i = wcsrtombs(mb, &wc[0], sizeof(mb), &state);
		if (0 < i)
			mb[i] = '\0';
#else
		i = wcrtomb(mb, *p, &state);
		if (0 < i)
			mb[i] = '\0';
#endif
		if (mb[0] != '\033' && !(mb[0] & 0x80))
			printf("\033(B");
		printf("%s", mb);
	}
}
#else
static char *
jisdsp(msg)
register char	*msg;
{
	static   char	msgbuf[80];
#ifdef	MSKANJI
	register int	ch;
	register char	*from, *to;

	from = msg;
	to = msgbuf;
	while ((ch = *from) && (from - msg < 30)) {
		if (iskanji(ch)) {
			if (mskanji) {
				*to++ = ch;
				*to++ = *++from;
			}
			else {
				*to++ = '[';
				*to++ = ']';
				++from;
			}
		}
		else
			*to++ = ch;
		from++;
	}
	*to = NULL;
#else
	strncpy(msgbuf, msg, 30);
	msgbuf[30] = NULL;
#endif
	return (msgbuf);
}
#endif /*MULTIBYTE*/
