/*	$Id: listener.c,v 1.11 2009/11/16 14:30:13 steve Exp $	*/

/*-
 * Copyright (c) 2001 Steve C. Woodford.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Steve C. Woodford.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <sys/types.h>
#include <sys/socket.h>

#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#ifdef LIBWRAP
#include <ctype.h>
#include <tcpd.h>
#ifndef LIBWRAP_ALLOW_FACILITY
# define LIBWRAP_ALLOW_FACILITY LOG_AUTH
#endif
#ifndef LIBWRAP_ALLOW_SEVERITY
# define LIBWRAP_ALLOW_SEVERITY LOG_INFO
#endif
#ifndef LIBWRAP_DENY_FACILITY
# define LIBWRAP_DENY_FACILITY LOG_AUTH
#endif
#ifndef LIBWRAP_DENY_SEVERITY
# define LIBWRAP_DENY_SEVERITY LOG_WARNING
#endif
int allow_severity = LIBWRAP_ALLOW_FACILITY|LIBWRAP_ALLOW_SEVERITY;
int deny_severity = LIBWRAP_DENY_FACILITY|LIBWRAP_DENY_SEVERITY;

extern const char *pname;
#endif

#include "context.h"
#include "buffer.h"
#include "config.h"
#include "listener.h"
#include "telnet.h"

static const char *cf_listener_init(void *, char **, int, int, void *);
static int	listener_init(struct client_ctx *, const void *);
static void	listener_destroy(struct client_ctx *);
static int	listener_event(struct client_ctx *, int);
static int	listener_ioctl(struct client_ctx *, int, void *,
		    struct client_ctx *);

struct client_ops listener_ops = {
	"listener",
	cf_listener_init, listener_init, listener_destroy, listener_event,
	NULL, NULL, NULL, listener_ioctl
};

static int	full_init(struct client_ctx *, const void *);
static void	full_destroy(struct client_ctx *);
static int	full_event(struct client_ctx *, int);
static int	full_write_pending(struct client_ctx *);

struct client_ops full_ops = {
	"full",
	NULL, full_init, full_destroy, full_event,
	NULL, NULL, full_write_pending, NULL
};

static struct client_ops *listen_protocols[] = {
	&telnet_ops,
	&raw_ops,
	NULL
};

static const char *cf_listener_address(void *, char **, int, int, void *);
static const char *cf_listener_port(void *, char **, int, int, void *);
static const char *cf_listener_maxclients(void *, char **, int, int, void *);
static const char *cf_listener_readonly(void *, char **, int, int, void *);
static const char *cf_listener_allowbreak(void *, char **, int, int, void *);
static const char *cf_listener_wrappers(void *, char **, int, int, void *);
static const char *cf_listener_protocol(void *, char **, int, int, void *);

static struct config_tokens cf_listener_tokens[] = {
 {"address",     1,                         cf_listener_address},
 {"port",        1,                         cf_listener_port},
 {"maxclients",  1,                         cf_listener_maxclients},
 {"readonly",    1,                         cf_listener_readonly},
 {"allowbreak",  1,                         cf_listener_allowbreak},
 {"tcpwrappers", 1,                         cf_listener_wrappers},
 {"protocol",    1 | CFG_ARGC_COMPOUNDABLE, cf_listener_protocol},
 {NULL, 0, NULL}
};


struct listener_args {
	char			*la_address;
	char			*la_port;
	int			la_maxclients;
	int			la_use_wrappers;
	struct addrinfo		*la_addrinfo;
	struct client_ops	*la_client;
	struct client_options	la_copts;
};

struct listener_ctx {
	struct listener_args	lc_args;
	struct client_qhead	lc_children;
	int			lc_client_cnt;
};

static char full_message[] =
"Sorry, the maximum number of connections to this service has been reached.\r\n"
"Please try again later or contact the system administrator.\r\n\r\n";


static int
listener_init(struct client_ctx *cc, const void *arg)
{
	const struct listener_args *la = arg;
	struct listener_ctx *lc;
	struct addrinfo *ai = la->la_addrinfo;
	int on = 1;

	if ((lc = calloc(1, sizeof(*lc))) == NULL)
		return (-1);

	lc->lc_args = *la;
	lc->lc_client_cnt = 0;
	TAILQ_INIT(&lc->lc_children);

	cc->cc_name = NULL;
	cc->cc_data = (void *) lc;
	cc->cc_fd = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
	if (cc->cc_fd < 0) {
		(void) free(lc);
		return (-1);
	}

	(void) setsockopt(cc->cc_fd, SOL_SOCKET, SO_REUSEADDR,
	    (char *)((void *)&on), sizeof(on));
	(void) setsockopt(cc->cc_fd, SOL_SOCKET, SO_KEEPALIVE,
	    (char *)((void *)&on), sizeof(on));

	if (bind(cc->cc_fd, ai->ai_addr, ai->ai_addrlen) < 0 ||
	    listen(cc->cc_fd, 4) < 0) {
		(void) free(lc);
		(void) close(cc->cc_fd);
		return (-1);
	}

	return (0);
}

static void
listener_destroy(struct client_ctx *cc)
{
	struct listener_ctx *lc = cc->cc_data;
	struct client_ctx *ccc;

	(void) close(cc->cc_fd);

	while ((ccc =
	    TAILQ_FIRST((volatile struct client_qhead *)&lc->lc_children)) !=
	    NULL)
		context_del_client(cc->cc_ctx, ccc);

	if (lc->lc_args.la_address)
		(void) free(lc->lc_args.la_address);
	if (lc->lc_args.la_port)
		(void) free(lc->lc_args.la_port);

	(void) free(lc);
}

static int
listener_event(struct client_ctx *cc, int events)
{
#ifdef LIBWRAP
	struct request_info req;
#endif
	struct listener_ctx *lc = cc->cc_data;
	struct client_ops *client;
	struct sockaddr_storage sas;
	struct client_ctx *new_cc;
	socklen_t slen;
	int fd;

	if ((events & POLLRDNORM) == 0)
		return (0);

	slen = sizeof(sas);
	if ((fd = accept(cc->cc_fd, (struct sockaddr *)((void *)&sas),
	    &slen)) < 0) {
		syslog(LOG_ALERT, "listener_event(): accept() failed: %m");
		return (0);
	}

#ifdef LIBWRAP
	if (lc->lc_args.la_use_wrappers) {
		request_init(&req, RQ_DAEMON,
		    isdigit((unsigned char)lc->lc_args.la_port[0]) ? pname :
		    lc->lc_args.la_port, RQ_FILE, fd, NULL);
		fromhost(&req);
		if (hosts_access(&req) == 0) {
			(void) close(fd);
			syslog(LOG_ALERT, "refused connection from %.500s\n",
			    eval_client(&req));
			return (0);
		}
	}
#endif

	client = lc->lc_args.la_client;

	if (lc->lc_args.la_maxclients &&
	    lc->lc_client_cnt >= lc->lc_args.la_maxclients) {
		syslog(LOG_ALERT,
		    "Maximum connection count reached for %s - %s:%s\n",
		    cc->cc_ctx->c_name,
		    lc->lc_args.la_address ? lc->lc_args.la_address : "*.*",
		    lc->lc_args.la_port);
		client = &full_ops;
	}

	if (client_init(&new_cc, cc, client,
	    &lc->lc_args.la_copts, (void *)&fd) < 0) {
		(void) close(fd);
		return (0);
	}

	if (context_add_client(cc->cc_ctx, new_cc) < 0)
		client_destroy(new_cc);

	return (0);
}

/* ARGSUSED */
static int
listener_ioctl(struct client_ctx *cc, int cmd, void *arg,
    struct client_ctx *sender)
{
	struct listener_ctx *lc = cc->cc_data;

	switch (cmd) {
	case CLIENT_IOCTL_NEW_CHILD:
		TAILQ_INSERT_TAIL(&lc->lc_children, sender, cc_qent_parent);
		lc->lc_client_cnt++;
		break;

	case CLIENT_IOCTL_CHILD_DELETED:
		TAILQ_REMOVE(&lc->lc_children, sender, cc_qent_parent);
		lc->lc_client_cnt--;
		break;
	}

	return (0);
}

/* ARGSUSED */
static const char *
cf_listener_init(void *cs, char **argv, int argc, int is_compound, void *arg)
{
	struct context *ctx = arg;
	struct client_ctx *cc;
	struct listener_args la;
	struct addrinfo hints, *res, *res0;
	const char *errstr;
	int rv;

	memset(&la, 0, sizeof(la));
	la.la_copts.co_allowbreak = 1;

	errstr = config_parse(cs, cf_listener_tokens, (void *)&la);

	if (errstr == NULL && la.la_port == NULL)
		errstr = config_err(cs, "Missing `port'");

	if (errstr == NULL && la.la_client == NULL)
		la.la_client = &telnet_ops;

	if (errstr == NULL && la.la_address &&
	    (strcasecmp(la.la_address, "all") == 0 ||
	     strcasecmp(la.la_address, "ANY") == 0 ||
	     strcmp(la.la_address, "*.*") == 0)) {
		(void) free(la.la_address);
		la.la_address = NULL;
	}

	if (errstr == NULL) {
		memset(&hints, 0, sizeof(hints));
		hints.ai_flags = AI_PASSIVE;
		hints.ai_family = PF_UNSPEC;
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_protocol = 0;

		rv = getaddrinfo(la.la_address, la.la_port, &hints, &res0);
		if (rv != 0)
			errstr = config_err(cs, "%s:%s - %s",
			    la.la_address ? la.la_address : "*.*",
			    la.la_port, gai_strerror(rv));

		for (res = res0; res && errstr == NULL; res = res->ai_next) {
			la.la_addrinfo = res;

			if (client_init(&cc, NULL, &listener_ops, &la.la_copts,
			    (const void *)&la) < 0) {
				if (errno == EPROTONOSUPPORT)
					continue;

				errstr = config_err(cs,
				    "listener_init(client_init(): %s:%s): %s",
			    	    la.la_address ? la.la_address : "*.*",
				    la.la_port, strerror(errno));
			} else
			if (context_add_client(ctx, cc) < 0) {
				errstr = config_err(cs,
				    "listener_init(context(): %s:%s): %s",
			    	    la.la_address ? la.la_address : "*.*",
				    la.la_port, strerror(errno));
				client_destroy(cc);
			}
		}
	}

	if (errstr != NULL) {
		if (la.la_address)
			(void) free(la.la_address);
		if (la.la_port)
			(void) free(la.la_port);
		la.la_address = NULL;
		la.la_port = NULL;
	}

	return (errstr);
}

/* ARGSUSED */
static const char *
cf_listener_address(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	if (la->la_address != NULL)
		return (config_err(cs, "listen address: already specified"));

	if ((la->la_address = strdup(argv[1])) == NULL)
		return (config_err(cs, "listen address: %s", strerror(errno)));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_listener_port(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	if (la->la_port != NULL)
		return (config_err(cs, "listen port: already specified"));

	if ((la->la_port = strdup(argv[1])) == NULL)
		return (config_err(cs, "listen port: %s", strerror(errno)));

	return (NULL);
}

/* ARGSUSED */
static const char *
cf_listener_maxclients(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	return (config_integer(cs, argv[1], &la->la_maxclients));
}

/* ARGSUSED */
static const char *
cf_listener_readonly(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	return (config_boolean(cs, argv[1], &la->la_copts.co_readonly));
}

/* ARGSUSED */
static const char *
cf_listener_allowbreak(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	return (config_boolean(cs, argv[1], &la->la_copts.co_allowbreak));
}

/* ARGSUSED */
static const char *
cf_listener_wrappers(void *cs, char **argv, int argc, int cmpnd, void *arg)
{
	struct listener_args *la = arg;

	return (config_boolean(cs, argv[1], &la->la_use_wrappers));
}

/* ARGSUSED */
static const char *
cf_listener_protocol(void *cs, char **argv, int argc, int is_compound,
    void *arg)
{
	struct listener_args *la = arg;
	struct client_ops **co;

	if (la->la_client != NULL)
		return (config_err(cs, "Only one client allowed"));

	for (co = listen_protocols; *co; co++)
		if (strcasecmp((*co)->co_name, argv[1]) == 0)
			break;

	if (*co == NULL)
		return (config_err(cs, "Unknown protocol `%s'", argv[1]));

	la->la_client = *co;

	if (is_compound)
		return (cf_client_options(cs, &la->la_copts));

	return (NULL);
}

static int
full_init(struct client_ctx *cc, const void *arg)
{
	struct buffer_ctx *bc;

	cc->cc_fd = *((const int *)arg);

	if (buffer_init(&bc, cc->cc_fd) < 0)
		return (-1);

	cc->cc_name = NULL;
	cc->cc_data = (void *) bc;

	if (buffer_fill(bc, full_message, sizeof(full_message) - 1) < 0) {
		(void) close(cc->cc_fd);
		buffer_destroy(bc);
		return (-1);
	}

	return (0);
}

static void
full_destroy(struct client_ctx *cc)
{

	buffer_destroy((struct buffer_ctx *) cc->cc_data);
	(void) close(cc->cc_fd);
}

static int
full_event(struct client_ctx *cc, int events)
{
	struct buffer_ctx *bc = cc->cc_data;

	if (buffer_length(bc) == 0)
		return (-1);

	if ((events & POLLWRNORM) != 0)
		return ((int) buffer_drain(bc));

	return (0);
}

/* ARGSUSED */
static int
full_write_pending(struct client_ctx *cc)
{

	return (1);
}
