#! /bin/sh

# $Id: sparc-fpu-auto.sh,v 1.6 2009/11/08 16:27:12 fredette Exp $

# ic/sparc-fpu-auto.sh - automatically generates C code for many SPARC FPU
# emulation instructions:

#
# Copyright (c) 2005 Matt Fredette
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
#      This product includes software developed by Matt Fredette.
# 4. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
#

header=false

for option
do
    case $option in
    --header) header=true ;;
    esac
done

PROG=`basename $0`
cat <<EOF
/* automatically generated by $PROG, do not edit! */
_TME_RCSID("\$Id: sparc-fpu-auto.sh,v 1.6 2009/11/08 16:27:12 fredette Exp $");
EOF

# the precision information helper script:
#
ieee754_precision_sh=`echo $0 | sed -e "s%$PROG%../ieee754/ieee754-precision.sh%"`

# permute for the different precisions:
#
for precision in single double quad; do
    
    # get information about this precision:
    #
    eval `sh ${ieee754_precision_sh} ${precision}`

    dst_formats="(TME_FLOAT_FORMAT_IEEE754_${capprecision} | TME_FLOAT_FORMAT_IEEE754_${capprecision}_BUILTIN)"

    cat <<EOF

/* this sets the floating-point condition codes after a
   ${precision}-precision operation: */
static void inline
_tme_sparc_fpu_fcc_${precision}(struct tme_sparc *ic, const struct tme_float *dst, int trap_on_nan)
{
  tme_uint32_t fcc;
  unsigned int cc;

  /* set fcc: */
  fcc = (tme_float_is_nan(dst, ${dst_formats})
         ? TME_SPARC_FSR_FCC_UN
	 : tme_float_is_zero(dst, ${dst_formats})
	 ? TME_SPARC_FSR_FCC_EQ
	 : tme_float_is_negative(dst, ${dst_formats})
	 ? TME_SPARC_FSR_FCC_LT
	 : TME_SPARC_FSR_FCC_GT);

  /* if this is an FCMPE and this is a NaN, we always cause an invalid exception: */
  if (trap_on_nan && fcc == TME_SPARC_FSR_FCC_UN) {
    _tme_sparc_fpu_exception_ieee754(&ic->tme_sparc_fpu_ieee754_ctl, TME_FLOAT_EXCEPTION_INVALID);
  }

  /* set the floating-point condition codes: */
  if (TME_SPARC_VERSION(ic) >= 9) {
    cc = TME_FIELD_MASK_EXTRACTU(TME_SPARC_INSN, (0x3 << 25));
    if (cc != 0) {
      fcc = TME_FIELD_MASK_EXTRACTU(fcc, TME_SPARC_FSR_FCC);
      ic->tme_sparc_fpu_xfsr
	= ((ic->tme_sparc_fpu_xfsr
	    & ~ (tme_uint32_t) (0x3 << (2 * (cc - 1))))
	   | (fcc << (2 * (cc - 1))));
      return;
    }
  }
  ic->tme_sparc_fpu_fsr = (ic->tme_sparc_fpu_fsr & ~TME_SPARC_FSR_FCC) | fcc;
}

/* if the most significant bit of the NaN fraction is zero,
   this is a signaling NaN: */
#define _TME_SPARC_FPU_IS_SNAN_${capprecision}(a) (((*(a))${chunk_member_0} & ((${chunk_mask_0} | (${chunk_mask_0} >> 1)) ^ (${chunk_mask_0} >> 1))) != 0)
static tme_int8_t
_tme_sparc_fpu_is_snan_${precision}(${integral} *value)
{
  return (_TME_SPARC_FPU_IS_SNAN_${capprecision}(value));
}

/* ${precision}-precision NaN propagation: */
static void
_tme_sparc_fpu_nan_from_nans_${precision}(struct tme_ieee754_ctl *ctl, 
				     const ${integral} *a,
				     const ${integral} *b,
				     ${integral} *z)
{
  struct tme_sparc *ic;
  int a_is_snan;
  int b_is_snan;

  /* recover our data structure : */
  ic = ctl->tme_ieee754_ctl_private;    

  /* see if any of the NaNs are signaling NaNs: */
  a_is_snan = _TME_SPARC_FPU_IS_SNAN_${capprecision}(a);
  b_is_snan = _TME_SPARC_FPU_IS_SNAN_${capprecision}(b);

  /* if either operand is a signaling NaN: */
  if (a_is_snan || b_is_snan) {

    /* signal the signaling NaN: */
    _tme_sparc_fpu_exception_ieee754(ctl, TME_FLOAT_EXCEPTION_INVALID);
  }

  /* if and only if a (corresponding to f[rs1]) is a signaling NaN, do 
     we return a. at all other times we return b (corresponding to f[rs2]): */
  if (a_is_snan) {
    b = a;
  }

  /* return the NaN, but make sure it's nonsignaling: */
  *z = *b;
  (*(z))${chunk_member_0} |= ((${chunk_mask_0} | (${chunk_mask_0} >> 1)) ^ (${chunk_mask_0} >> 1));
}

EOF

done

if $header; then :; else
    echo "#define _TME_SPARC_FPU_UNIMPL tme_sparc_fpu_exception(ic, TME_SPARC_FSR_FTT_unimplemented_FPop)"
    echo "#define _TME_SPARC_FPU_UNIMPL_IF(flags) do { if ((ic->tme_sparc_fpu_flags & (flags)) != 0) { _TME_SPARC_FPU_UNIMPL; } } while (/* CONSTCOND */ 0)"
fi
quad="_TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_QUAD);"

# permute over fpop1/fpop2:
#
for fpop in fpop1 fpop2; do

    # placeholder for another permutation:
    #
    :
	
	# open the function:
	#
	echo ""
	echo "void"
	echo "tme_sparc_fpu_${fpop}(struct tme_sparc *ic)"
	echo "{"
	echo "  tme_uint8_t rounding_mode;"
	echo "  unsigned int opf;"
	echo "  unsigned int fpreg_rd_number_encoded;"
	echo "  unsigned int fpreg_rd_number;"
	echo "  const struct tme_float *fpreg_rs1;"
	echo "  const struct tme_float *fpreg_rs2;"
	if test ${fpop} = fpop1; then
	    echo "  struct tme_float fpreg_rs1_buffer;"
	    echo "  struct tme_float fpreg_rs2_buffer;"
	else
	    echo "  unsigned int cc;"
	    echo "  tme_uint32_t conds_mask;"
	    echo "  unsigned int cc_i;"
	    echo "  tme_uint32_t cond;"
	fi
	echo "  struct tme_float fpreg_rd;"
	echo "  unsigned int fpreg_rd_format;"

	echo ""
	echo "  /* set the rounding mode: */"
	echo "  switch (ic->tme_sparc_fpu_fsr & TME_SPARC_FSR_RND) {"
	echo "  default: assert(FALSE);"
	echo "  case TME_SPARC_FSR_RND_RN: rounding_mode = TME_FLOAT_ROUND_NEAREST_EVEN; break;"
	echo "  case TME_SPARC_FSR_RND_RZ: rounding_mode = TME_FLOAT_ROUND_TO_ZERO; break;"
	echo "  case TME_SPARC_FSR_RND_RM: rounding_mode = TME_FLOAT_ROUND_DOWN; break;"
	echo "  case TME_SPARC_FSR_RND_RP: rounding_mode = TME_FLOAT_ROUND_UP; break;"
	echo "  }"
	echo "  ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = rounding_mode;"

	echo ""
	echo "  /* decode the rd and opf fields: */"
	echo "  fpreg_rd_number_encoded = TME_FIELD_MASK_EXTRACTU(TME_SPARC_INSN, TME_SPARC_FORMAT3_MASK_RD);"
	echo "  opf = TME_FIELD_MASK_EXTRACTU(TME_SPARC_INSN, (0x1ff << 5));"

	echo ""
	echo "  /* silence uninitialized variable warnings: */"
	echo "  fpreg_rd_number = 0;"

	echo ""
	echo "#ifdef _TME_SPARC_RECODE_VERIFY"
	echo "  /* clear the rd buffer: */"
	echo "  memset(&fpreg_rd, 0, sizeof(fpreg_rd));"
	echo "#endif /* _TME_SPARC_RECODE_VERIFY */"

	fmovcc=
	if test ${fpop} = fpop2; then
	    fmovcc=cc
	    echo ""
	    echo "  /* if this is an FMOVcc: */"
	    echo "  if (((opf - 1) & 0x3f) < 3) {"
	    echo ""
	    echo "    /* if opf bit eight is set, this uses integer condition codes: */"
	    echo "    if (opf & TME_BIT(8)) {"
	    echo ""
	    echo "      /* if opf bit six is set, this is unimplemented: */"
	    echo "      if (__tme_predict_false(opf & TME_BIT(6))) {"
	    echo "        _TME_SPARC_FPU_UNIMPL;"
	    echo "      }"
	    echo ""
	    echo "      /* get %icc or %xcc, depending on opf bit seven: */"
	    echo "      cc = ic->tme_sparc64_ireg_ccr;"
	    echo "      if (opf & TME_BIT(7)) {"
	    echo "        cc /= (TME_SPARC64_CCR_XCC / TME_SPARC64_CCR_ICC);"
	    echo "      }"
	    echo "      cc = TME_FIELD_MASK_EXTRACTU(cc, TME_SPARC64_CCR_ICC);"
	    echo ""
	    echo "      /* get the conditions mask: */"
	    echo "      conds_mask = _tme_sparc_conds_icc[cc];"
	    echo "    }"
	    echo ""
	    echo "    /* otherwise, this uses floating-point condition codes: */"
	    echo "    else {"
	    echo ""
	    echo "      /* get the right %fcc: */"
	    echo "      cc_i = TME_FIELD_MASK_EXTRACTU(opf, (0x3 << 6));"
	    echo "      if (cc_i == 0) {"
	    echo "        cc = TME_FIELD_MASK_EXTRACTU(ic->tme_sparc_fpu_fsr, TME_SPARC_FSR_FCC);"
	    echo "      }"
	    echo "      else {"
	    echo "        cc = (ic->tme_sparc_fpu_xfsr >> (2 * (cc_i - 1))) & 0x3;"
	    echo "      }"
	    echo ""
	    echo "      /* get the conditions mask: */"
	    echo "      conds_mask = _tme_sparc_conds_fcc[cc];"
	    echo "    }"
	    echo ""
	    echo "    /* add the not-conditions to the conditions mask: */"
	    echo "    conds_mask += ((~conds_mask) << 8);"
	    echo ""
	    echo "    /* get the cond field: */"
	    echo "    cond = TME_BIT(TME_FIELD_MASK_EXTRACTU(TME_SPARC_INSN, (0xf << 14)));"
  	    echo ""
	    echo "    /* if the condition is not true: */"
	    echo "    if (!(conds_mask & cond)) {"
	    echo ""
	    echo "      /* return now: */"
	    echo "      /* NB that this may expose us to guests, since we do not check"
	    echo "         that the floating-point register numbers are valid: */"
	    echo "      return;"
	    echo "    }"
	    echo ""
	    echo "    /* clear bits six, seven, and eight in opf: */"
	    echo "    opf &= 0x3f;"
	    echo "  }"
	fi

	echo ""
	echo "  /* dispatch on the opf field: */"
	echo "  switch (opf) {"
	echo "#define _TME_SPARC_FPU_FORMAT_RS1(format) fpreg_rs1 = tme_sparc_fpu_fpreg_read(ic, TME_SPARC_FORMAT3_MASK_RS1, (format))"
	echo "#define _TME_SPARC_FPU_FORMAT_RS2(format) fpreg_rs2 = tme_sparc_fpu_fpreg_read(ic, TME_SPARC_FORMAT3_MASK_RS2, (format))"
	echo "#define _TME_SPARC_FPU_FORMAT_RD(format) do { fpreg_rd_format = (format) | TME_IEEE754_FPREG_FORMAT_BUILTIN; fpreg_rd_number = tme_sparc_fpu_fpreg_decode(ic, fpreg_rd_number_encoded, fpreg_rd_format); } while (/* CONSTCOND */ 0)"
	echo ""

	# permute over the opf field:
	#
	opf_decimal=-1
	while test ${opf_decimal} != 511; do
	    opf_decimal=`expr ${opf_decimal} + 1`

	    # make the binary version of the opf field:
	    #
	    bits=9
	    opf=
	    opf_shifted=${opf_decimal}
	    while test ${bits} != 0; do
		bits=`expr ${bits} - 1`
		opf_shifted_next=`expr ${opf_shifted} / 2`
		opf_test=`expr ${opf_shifted_next} \* 2`
		if test ${opf_test} = ${opf_shifted}; then
		    opf="0${opf}"
		else
		    opf="1${opf}"
		fi
		opf_shifted=${opf_shifted_next}
	    done

	    # dispatch on the fpop/opf combination:
	    #
	    default=false
	    case "${fpop}:${opf}" in
	    fpop1:011000100)
		echo "  case ${opf_decimal}:  /* ${opf} FiTOs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_from_int32,"
		echo "                              fpreg_rs2->tme_float_value_ieee754_single, &fpreg_rd);"
		;;
	    fpop1:011001000)
		echo "  case ${opf_decimal}:  /* ${opf} FiTOd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_int32,"
		echo "                              fpreg_rs2->tme_float_value_ieee754_single, &fpreg_rd);"
		;;
	    fpop1:011001100)
		echo "  case ${opf_decimal}:  /* ${opf} FiTOq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_int32,"
		echo "                              fpreg_rs2->tme_float_value_ieee754_single, &fpreg_rd);"
		;;
	    fpop1:010000100)
		echo "  case ${opf_decimal}:  /* ${opf} FxTOs: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_from_int64,"
		echo "                                fpreg_rs2->tme_float_value_ieee754_double.tme_value64_int, &fpreg_rd);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:010001000)
		echo "  case ${opf_decimal}:  /* ${opf} FxTOd: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_int64,"
		echo "                                fpreg_rs2->tme_float_value_ieee754_double.tme_value64_int, &fpreg_rd);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:010001100)
		echo "  case ${opf_decimal}:  /* ${opf} FxTOq: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      ${quad}"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_int64,"
		echo "                                fpreg_rs2->tme_float_value_ieee754_double.tme_value64_int, &fpreg_rd);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:011010001)
		echo "  case ${opf_decimal}:  /* ${opf} FsTOi: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_SINGLE;"
		echo "    ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_to_int32,"
		echo "                              fpreg_rs2, (tme_int32_t *) &fpreg_rd.tme_float_value_ieee754_single);"
		;;
	    fpop1:011010010)
		echo "  case ${opf_decimal}:  /* ${opf} FdTOi: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_SINGLE;"
		echo "    ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_to_int32,"
		echo "                              fpreg_rs2, (tme_int32_t *) &fpreg_rd.tme_float_value_ieee754_single);"
		;;
	    fpop1:011010011)
		echo "  case ${opf_decimal}:  /* ${opf} FqTOi: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_SINGLE;"
		echo "    ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_to_int32,"
		echo "                              fpreg_rs2, (tme_int32_t *) &fpreg_rd.tme_float_value_ieee754_single);"
		;;
	    fpop1:010000001)
		echo "  case ${opf_decimal}:  /* ${opf} FsTOx: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_DOUBLE;"
		echo "      ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_to_int64,"
		echo "                                fpreg_rs2, &fpreg_rd.tme_float_value_ieee754_double.tme_value64_int);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:010000010)
		echo "  case ${opf_decimal}:  /* ${opf} FdTOx: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_DOUBLE;"
		echo "      ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_to_int64,"
		echo "                                fpreg_rs2, &fpreg_rd.tme_float_value_ieee754_double.tme_value64_int);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:010000011)
		echo "  case ${opf_decimal}:  /* ${opf} FqTOx: */"
		echo "#ifdef TME_HAVE_INT64_T"
		echo "    if (__tme_predict_true(TME_SPARC_VERSION(ic) >= 9)) {"
		echo "      ${quad}"
		echo "      _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "      _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "      _TME_SPARC_FPU_BEGIN;"
		echo "      fpreg_rd.tme_float_format = TME_FLOAT_FORMAT_IEEE754_DOUBLE;"
		echo "      ic->tme_sparc_fpu_ieee754_ctl.tme_ieee754_ctl_rounding_mode = TME_FLOAT_ROUND_TO_ZERO;"
		echo "      _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_to_int64,"
		echo "                                fpreg_rs2, &fpreg_rd.tme_float_value_ieee754_double.tme_value64_int);"
		echo "      break;"
		echo "    }"
		echo "#endif /* TME_HAVE_INT64_T */"
		echo "    _TME_SPARC_FPU_UNIMPL;"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop1:011001001)
		echo "  case ${opf_decimal}:  /* ${opf} FsTOd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_single,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:011001101)
		echo "  case ${opf_decimal}:  /* ${opf} FsTOq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_single,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:011000110)
		echo "  case ${opf_decimal}:  /* ${opf} FdTOs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_from_double,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:011001110)
		echo "  case ${opf_decimal}:  /* ${opf} FdTOq: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_double,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:011000111)
		echo "  case ${opf_decimal}:  /* ${opf} FqTOs: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_from_quad,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:011001011)
		echo "  case ${opf_decimal}:  /* ${opf} FqTOd: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_quad,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:000000001 | fpop2:000000001)
		echo "  case ${opf_decimal}:  /* ${opf} FMOVs${fmovcc}: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		;;
	    fpop1:000000010 | fpop2:000000010)
		echo "  case ${opf_decimal}:  /* ${opf} FMOVd${fmovcc}: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		;;
	    fpop1:000000101)
		echo "  case ${opf_decimal}:  /* ${opf} FNEGs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		echo "    fpreg_rd.tme_float_value_ieee754_single ^= 0x80000000;"
		;;
	    fpop1:000000110)
		echo "  case ${opf_decimal}:  /* ${opf} FNEGd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		echo "    fpreg_rd.tme_float_value_ieee754_double.tme_value64_uint32_hi ^= 0x80000000;"
		;;
	    fpop1:000001001)
		echo "  case ${opf_decimal}:  /* ${opf} FABSs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		echo "    fpreg_rd.tme_float_value_ieee754_single &= ~0x80000000;"
		;;
	    fpop1:000001010)
		echo "  case ${opf_decimal}:  /* ${opf} FABSd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_BEGIN;"
		echo "    fpreg_rd = *fpreg_rs2;"
		echo "    fpreg_rd.tme_float_value_ieee754_double.tme_value64_uint32_hi &= ~0x80000000;"
		;;
	    fpop1:000101001)
		echo "  case ${opf_decimal}:  /* ${opf} FSQRTs: */"
		echo "    _TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_FSQRT);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_single_sqrt,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:000101010)
		echo "  case ${opf_decimal}:  /* ${opf} FSQRTd: */"
		echo "    _TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_FSQRT);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_sqrt,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:000101011)
		echo "  case ${opf_decimal}:  /* ${opf} FSQRTq: */"
		echo "    _TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_FSQRT | TME_SPARC_FPU_FLAG_NO_QUAD);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_sqrt,"
		echo "                              fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000001)
		echo "  case ${opf_decimal}:  /* ${opf} FADDs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_add,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000010)
		echo "  case ${opf_decimal}:  /* ${opf} FADDd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_add,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000011)
		echo "  case ${opf_decimal}:  /* ${opf} FADDq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_add,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000101)
		echo "  case ${opf_decimal}:  /* ${opf} FSUBs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000110)
		echo "  case ${opf_decimal}:  /* ${opf} FSUBd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001000111)
		echo "  case ${opf_decimal}:  /* ${opf} FSUBq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001001001)
		echo "  case ${opf_decimal}:  /* ${opf} FMULs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_mul,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001001010)
		echo "  case ${opf_decimal}:  /* ${opf} FMULd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_mul,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001001011)
		echo "  case ${opf_decimal}:  /* ${opf} FMULq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_mul,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001101001)
		echo "  case ${opf_decimal}:  /* ${opf} FsMULd: */"
		echo "    _TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_FMUL_WIDER);"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_single,"
		echo "                              fpreg_rs1, &fpreg_rs1_buffer);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_double_from_single,"
		echo "                              fpreg_rs2, &fpreg_rs2_buffer);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_mul,"
		echo "                             &fpreg_rs1_buffer, &fpreg_rs2_buffer, &fpreg_rd);"
		;;
	    fpop1:001101110)
		echo "  case ${opf_decimal}:  /* ${opf} FdMULq: */"
		echo "    _TME_SPARC_FPU_UNIMPL_IF(TME_SPARC_FPU_FLAG_NO_FMUL_WIDER | TME_SPARC_FPU_FLAG_NO_QUAD);"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_double,"
		echo "                              fpreg_rs1, &fpreg_rs1_buffer);"
		echo "    _TME_SPARC_FPU_OP_MONADIC(tme_ieee754_ops_quad_from_double,"
		echo "                              fpreg_rs2, &fpreg_rs2_buffer);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_mul,"
		echo "                             &fpreg_rs1_buffer, &fpreg_rs2_buffer, &fpreg_rd);"
		;;
	    fpop1:001001101)
		echo "  case ${opf_decimal}:  /* ${opf} FDIVs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_SINGLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_div,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001001110)
		echo "  case ${opf_decimal}:  /* ${opf} FDIVd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_DOUBLE);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_div,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop1:001001111)
		echo "  case ${opf_decimal}:  /* ${opf} FDIVq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RD(TME_IEEE754_FPREG_FORMAT_QUAD);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_div,"
		echo "                              fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		;;
	    fpop2:001010001)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_single(ic, &fpreg_rd, FALSE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop2:001010010)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_double(ic, &fpreg_rd, FALSE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop2:001010011)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_quad(ic, &fpreg_rd, FALSE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop2:001010101)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPEs: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_SINGLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_single_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_single(ic, &fpreg_rd, TRUE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop2:001010110)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPEd: */"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_DOUBLE | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_double_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_double(ic, &fpreg_rd, TRUE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    fpop2:001010111)
		echo "  case ${opf_decimal}:  /* ${opf} FCMPEq: */"
		echo "    ${quad}"
		echo "    _TME_SPARC_FPU_FORMAT_RS1(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_FORMAT_RS2(TME_IEEE754_FPREG_FORMAT_QUAD | TME_IEEE754_FPREG_FORMAT_BUILTIN);"
		echo "    _TME_SPARC_FPU_OP_DYADIC(tme_ieee754_ops_quad_sub,"
		echo "                             fpreg_rs1, fpreg_rs2, &fpreg_rd);"
		echo "    _tme_sparc_fpu_fcc_quad(ic, &fpreg_rd, TRUE);"
		echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
		;;
	    *) default=true ;;
	    esac
	    if $default; then :; else echo "    break;"; echo ""; fi
	done
	echo "  default:"
	echo "    _TME_SPARC_FPU_UNIMPL;"
	echo "    fpreg_rd_format = TME_IEEE754_FPREG_FORMAT_NULL;"
	echo "    break;"
	echo ""
	echo "#undef _TME_SPARC_FPU_FORMAT_RS1"
	echo "#undef _TME_SPARC_FPU_FORMAT_RS2"
	echo "#undef _TME_SPARC_FPU_FORMAT_RD"
	echo "  }"

	echo ""
	echo "  /* store any destination: */"
	echo "  if (fpreg_rd_format != TME_IEEE754_FPREG_FORMAT_NULL) {"
	echo "    tme_sparc_fpu_fpreg_format(ic, fpreg_rd_number, fpreg_rd_format);"
	echo "    ic->tme_sparc_fpu_fpregs[fpreg_rd_number] = fpreg_rd;"
	echo "    TME_SPARC_FPU_DIRTY(ic, fpreg_rd_number);"
	echo "  }"

	echo ""
	echo "}"
   : 
done

# done:
#
exit 0
