#!/usr/bin/ruby
#
# Author:: Oliver M. Bolzer <oliver@fakeroot.net>
# Copyright:: Copyright (c) 2002 Oliver M. Bolzer. All rights reserved.
# Licence:: Ruby licence.

require 'test/unit'
require 'mock'

require 'vapor/transaction'
require 'vapor/utils'

## test class for Transaction 
class Test_Transaction < Test::Unit::TestCase
  include Vapor
  include Vapor::Exceptions
  # test set-up
  def setup
     @mock_pmgr = Mock.new("PersistenceManager")
     @mock_datastore = Mock.new("Datastore")
     Persistable.persistence_manager = @mock_pmgr

     @trans = Transaction.new( @mock_pmgr, @mock_datastore )
  end # setup()

  # test constructor, if publicly readable attributes are set
  def test_initialize()
    assert_raises( ArgumentError ){ Transaction.new() }
    assert_raises( ArgumentError ){ Transaction.new( nil ) }
    assert_raises( ArgumentError ){ Transaction.new( nil, nil, nil ) }

    assert_same( @mock_pmgr, @trans.persistence_manager )
  end # test constructor

  # test transaction start
  def test_begin
    assert_respond_to( @trans, :begin )
    assert_respond_to( @trans, :active? )
    
    assert_equal( false, @trans.active? )
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){ |flag| assert_equal( false, flag ) }
    @mock_datastore.__next( :begin_transaction ){}
    assert_nothing_raised{ @trans.begin }
    assert_equal( true, @trans.active? )

    @mock_pmgr.__verify
    @mock_datastore.__verify

    # starting an already started transaction
    assert_raises( NestedTransactionError ){ @trans.begin }
    assert_equal( true, @trans.active? )

  end # test_begin()

  # test commit
  def test_commit
    assert_respond_to( @trans, :commit )
    assert_respond_to( @trans, :active? )

    # committing non-active transaction gives error
    assert_equal( false, @trans.active? )
    assert_raises( StaleTransactionError ){ @trans.commit }

    ## start transaction, so we can commit it :-)
    assert_respond_to( @trans, :begin )
    @mock_datastore.__next( :begin_transaction ){}
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){}
    assert_nothing_raised{ @trans.begin }
    assert_equal( true, @trans.active? )
    @mock_pmgr.__verify
    @mock_datastore.__verify
    
    ## let's try to commit
    @mock_pmgr.__next( :try_change_state ){|block,obj| block.call }
    @mock_pmgr.__next( :next_oid ){ 123 }
    @mock_pmgr.__next( :new_object ){}
    @mock_pmgr.__next( :flush_all ){}
    @mock_datastore.__next( :transaction_log= ){}
    @mock_datastore.__next( :commit_transaction ){}
    @mock_datastore.__next( :transaction_log= ){}
    assert_nothing_raised{ @trans.commit }
    assert_equal( false, @trans.active? )
    @mock_pmgr.__verify
    @mock_datastore.__verify

  end # test_commit

  # test rollback
  def test_rollback
    assert_respond_to( @trans, :rollback )
   
    @mock_datastore.__next( :rollback_transaction ){}
    @mock_pmgr.__next( :rollback_all ){}
    assert_nothing_raised{ @trans.rollback }
    assert_equal( false, @trans.active? )
    @mock_datastore.__verify
    @mock_pmgr.__verify
  end # test_rollback()

  # test creation and saving, logging of TransactionLog
  def test_log

    # existence of log entry retrieval method
    assert_respond_to( @trans, :log_entry )

    # no transaction log present, after Transaction freshly created
    assert_kind_of( Vapor::TransactionLog, @trans.log_entry )

    # new log entry created on begin
    @mock_datastore.__next( :begin_transaction ){}
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){}
    @mock_datastore.__next( :rollback_transaction ){}
    @mock_pmgr.__next( :rollback_all ){} 
    @mock_datastore.__next( :begin_transaction ){}
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){}
    assert_nothing_raised{ @trans.begin } 
    assert_kind_of( Vapor::TransactionLog, @trans.log_entry )
    oldid = @trans.log_entry.__id__
    assert_nothing_raised{
      @trans.abort
      @trans.begin
    }
    newid = @trans.log_entry.__id__
    assert_not_equal( newid, oldid )
    @mock_pmgr.__verify
    
    # log written on commit
    log = @trans.log_entry
    @mock_pmgr.__next( :try_change_state ){|block, obj|
      assert_same( log, obj )
      block.call
    }
    @mock_pmgr.__next( :next_oid ){ 123 }
    @mock_pmgr.__next( :new_object ){|obj|
      assert_same( log, obj )
    }
    @mock_pmgr.__next( :flush_all ){}
    @mock_datastore.__next( :transaction_log= ){}
    @mock_datastore.__next( :commit_transaction ){}
    @mock_datastore.__next( :transaction_log= ){}
    @trans.commit
    @mock_pmgr.__verify

  end # test_log()

  # test wheter message and committer are properly set
  def test_message
    assert_respond_to( @trans, :message= )
    assert_respond_to( @trans, :committer= )
    assert_respond_to( @trans, :log_entry )
    
    @mock_datastore.__next( :begin_transaction ){}
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){}
    assert_nothing_raised{ @trans.begin }

    assert_equal( "", @trans.log_entry.message )
    assert_equal( "", @trans.log_entry.committer )

    message = "commit message"
    committer = "somebody"
    assert_nothing_raised{
      @trans.message = message 
      @trans.committer = committer 
    }
    assert_equal( message, @trans.log_entry.message )
    assert_equal( committer, @trans.log_entry.committer )

    # only committer should be kept, message should be discarded
    @mock_pmgr.__next( :try_change_state ){|block, obj|}
    @mock_pmgr.__next( :flush_all ){}
    @mock_datastore.__next( :transaction_log= ){}
    @mock_datastore.__next( :commit_transaction ){}
    @mock_datastore.__next( :transaction_log= ){}
    @mock_datastore.__next( :begin_transaction ){}
    @mock_pmgr.__next( :autocommit_in_progress? ){ false }
    @mock_pmgr.__next( :autocommit= ){}
    assert_nothing_raised{ @trans.commit; @trans.begin }

    assert_equal( committer, @trans.log_entry.committer )
    assert_equal( "", @trans.log_entry.message )

  end # test_message()

end # class Test_Transaction
