/* Copyright 2014-present Facebook, Inc.
 * Licensed under the Apache License, Version 2.0 */

#include <windows.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>

// Super simple utility to suspend or resume all threads in a target process.
// We use this in place of `kill -STOP` and `kill -CONT`

typedef LONG (NTAPI *sus_res_func)(HANDLE proc);

const char *win32_strerror(DWORD err) {
  static char msgbuf[1024];
  FormatMessageA(
      FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
      NULL, err, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
      msgbuf, sizeof(msgbuf)-1, NULL);
  return msgbuf;
}

int apply(DWORD pid, BOOL suspend) {
  sus_res_func func;
  const char *name = suspend ? "NtSuspendProcess" : "NtResumeProcess";
  HANDLE proc;
  DWORD res;

  func = (sus_res_func)GetProcAddress(
      GetModuleHandle("ntdll"),
      name);

  if (!func) {
    printf("Failed to GetProcAddress(%s): %s\n", name,
        win32_strerror(GetLastError()));
    return 1;
  }

  proc = OpenProcess(PROCESS_ALL_ACCESS, FALSE, pid);
  if (proc == INVALID_HANDLE_VALUE) {
    printf("Failed to OpenProcess(%d): %s\n",
        pid, win32_strerror(GetLastError()));
    return 1;
  }

  res = func(proc);

  printf("%s(%d) returns %x: %s\n",
      name, pid, res, win32_strerror(res));

  CloseHandle(proc);

  return res == 0 ? 0 : 1;
}

void usage() {
  printf(
      "Usage: susres suspend [pid]\n"
      "       susres resume  [pid\n");
  exit(1);
}

int main(int argc, char **argv) {
  DWORD pid;
  BOOL suspend;

  if (argc != 3) {
    usage();
  }

  if (!strcmp(argv[1], "suspend")) {
    suspend = TRUE;
  } else if (!strcmp(argv[1], "resume")) {
    suspend = FALSE;
  } else {
    usage();
  }

  pid = _atoi64(argv[2]);

  return apply(pid, suspend);
}

/* vim:ts=2:sw=2:et:
 */
