/*
 * Copyright (c) 1997, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/pixmap.c,v 1.12 2004/04/24 07:58:40 danny Exp $
 */
static char id[] = "$Header: /pack/anoncvs/xinvest/src/pixmap.c,v 1.12 2004/04/24 07:58:40 danny Exp $";

#include <stdio.h>
#include <stdlib.h>

#include <Xm/Xm.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif
#include <xpm.h>

#include "color.h"
#include "pixmap.h"

#ifdef XQUOTE
#include "pixmapPXq.h"
#else
#include "pixmapPXi.h"
#endif

#include "xutil.h"

#define gray_width 16
#define gray_height 16
static unsigned char gray_bits[] = {
   0x99, 0x99, 0xAA, 0xAA, 0x66, 0x66, 0x55, 0x55, 
   0x99, 0x99, 0xAA, 0xAA, 0x66, 0x66, 0x55, 0x55, 
   0x99, 0x99, 0xAA, 0xAA, 0x66, 0x66, 0x55, 0x55,
   0x99, 0x99, 0xAA, 0xAA, 0x66, 0x66, 0x55, 0x55};

/* forward declarations */
static Pixmap   MakeWindowsPixmap ( Widget, Pixmap );
static Pixmap   MakeGrayPixmap ( Widget, Pixmap );

#ifdef	SIMPLE_PIXMAP

static Pixmap pixmap[XtNumber(pixname)];
static Pixmap inspixmap[XtNumber(pixname)];
static Pixmap shapemask[XtNumber(pixname)];

Colormap InitPixmaps( Widget top, Colormap cmap )
{
  Display *dpy = XtDisplay( top );
  Window  win = RootWindowOfScreen (XtScreen( top ));
  Pixel  bg;
  Pixmap junk;  /* Unused, except to code around xpm-3.4h bug */

  XpmColorSymbol backg;
  XpmColorSymbol mask[2];
  XpmAttributes attrib;

  int code;
  int i = 0;
  int colorVisual = isColorVisual(dpy);

  XtVaGetValues( top, XmNbackground, &bg, NULL );

  /* Override background color with background color of parent */
  backg.name = "mask";
  backg.value = NULL;
  backg.pixel = bg;

  /* 
  ** Used to generate shapemask for insensitive pixmap. Set mask and light
  ** to transparent.
  */
  mask[0].name = "mask";
  mask[0].value = "none";
  mask[0].pixel = (Pixel)NULL;

  mask[1].name = "light";
  mask[1].value = "none";
  mask[1].pixel = (Pixel)NULL;

  attrib.numsymbols = 1;
  attrib.colorsymbols = &backg;
  attrib.closeness = 40000;
  attrib.colormap = cmap;
  attrib.valuemask = XpmColorSymbols | XpmCloseness | XpmColormap;

#ifdef MONO_TEST
  attrib.color_key = XPM_MONO;
  attrib.valuemask |= XpmColorKey;
#endif

  do {
     /* Make the normal (sensitive) pixmap. */
     code = XpmCreatePixmapFromData( dpy, win,
                                     pixname[i], &(pixmap[i]), &(shapemask[i]),
                                     &attrib );
     switch (code) {

         case XpmSuccess:

                if (i <= LASTINSENS) {  /* No insensitive for others */
                  /* 
                  ** Now make the insensitive pixmap.  If we're color-challenged
                  ** make the fuzzed out pixmap, otherwise get the windows
                  ** type of pixmap.
                  */

                  if ( colorVisual ) {
                    attrib.colorsymbols = &mask[0];
                    attrib.numsymbols = 2;
              
                    /* 
                    ** We assume this succeeds because the above did. If 
                    ** it didn't we'll get the XCreateInsensitive 
                    ** pixmap instead of the MakeWindowsPixmap variety.
                    */
                    code = XpmCreatePixmapFromData( dpy, win,
                                               pixname[i], 
                                               &junk, &(shapemask[i]),
                                               &attrib );
                    /* Junk is not needed, but we can't pass NULL to
                    ** XPM as the pixmap.  This is a bug in XPM. */
                    XFreePixmap ( dpy, junk);

                    attrib.colorsymbols = &backg;
                    attrib.numsymbols = 1;

                    if (shapemask[i])
                      inspixmap[i] = MakeWindowsPixmap ( top, shapemask[i]);
                    else
                      inspixmap[i] = MakeGrayPixmap( top, pixmap[i] );

                  } else  /* MONO visual */
                    inspixmap[i] = MakeGrayPixmap( top, pixmap[i] );

                  if ( inspixmap[i] == (Pixmap)NULL)
                    return (Colormap)NULL;

                } /* if pixmap needs insensitive */

                i++;
                break;

         case XpmColorFailed:
              if ( (cmap = NewColormap( dpy, cmap)) == (Colormap)NULL)
                 return (Colormap)NULL;
              attrib.colormap = cmap;
              attrib.valuemask |= XpmColormap;
              break;

         default:
              return (Colormap)NULL;
              /* break; */
    }
  } while (i <= LASTPIXMAP);

  return cmap;
}

Pixmap GetPixmap(int which, int type, Display *dpy)
{
  switch (type) {
    case NORMAL:  return pixmap[which];

    case INSENS:  return inspixmap[which];

    case MASK:    return shapemask[which];

    default:      return (Pixmap) XmUNSPECIFIED_PIXMAP;
  }
}

#else	/* ~ SIMPLE_PIXMAP */

struct pixmaplist {
	int	which;	/* its name */
	int	tp;	/* type */
	Display	*dpy;	/* for which display */
	Pixmap	pm;
};

static	struct pixmaplist	*PL = NULL;
static int	nPL = 0,
		mPL = 0;

#define	MDL	4	/* Can cope with this many displays */
static int	nd = 0;

static struct {
	Display		*dpy;
	Colormap	cmap;
	Widget		top;
} displaylist[MDL];
	
Colormap InitPixmaps(Widget top, Colormap cmap)
{
	displaylist[nd].top = top;
	displaylist[nd].cmap = cmap;
	displaylist[nd].dpy = XtDisplay(top);

	nd++;

	return cmap;
}

static void StorePixmap(Pixmap pm, Display *dpy, int which, int t)
{
#ifdef	DEBUG
	fprintf(stderr, "StorePixmap(%p, dpy %p, which %d, tp %d)\n",
		pm, dpy, which, t);
#endif

	if (nPL == mPL) {
		mPL += 64;
		PL = (struct pixmaplist *)XtRealloc((char *)PL, mPL * sizeof(struct pixmaplist));
	}

	/* There is empty space */
	PL[nPL].which = which;
	PL[nPL].pm = pm;
	PL[nPL].dpy = dpy;
	PL[nPL].tp = t;

	nPL++;
}

/*
 * Get a Pixmap in a lazy way.
 */
static Pixmap GetNewPixmap(Widget top, Display *dpy, Colormap cmap, int which, int t)
{
	Window	win = RootWindowOfScreen(XtScreen(top));
	Pixel	bg;
	Pixmap	junk;  /* Unused, except to code around xpm-3.4h bug */
	Pixmap	pm, sm;

	XpmColorSymbol backg;
	XpmColorSymbol mask[2];
	XpmAttributes attrib;

	int code;
	int colorVisual = isColorVisual(dpy);

#ifdef DEBUG
	fprintf(stderr, "GetNewPixmap(which %d, t %s, pn %s\n",
		which,
		(t == NORMAL) ? "NORMAL" :
		(t == MASK) ? "MASK" :
		(t == INSENS) ? "INSENS" :
		"??",
		pixname[which][0]);
#endif
	XtVaGetValues( top, XmNbackground, &bg, NULL );

	/* Override background color with background color of parent */
	backg.name = "mask";
	backg.value = NULL;
	backg.pixel = bg;

	/* 
	 * Used to generate shapemask for insensitive pixmap. Set mask and light
	 * to transparent.
	 */
	mask[0].name = "mask";
	mask[0].value = "none";
	mask[0].pixel = (Pixel)NULL;

	mask[1].name = "light";
	mask[1].value = "none";
	mask[1].pixel = (Pixel)NULL;

	attrib.numsymbols = 1;
	attrib.colorsymbols = &backg;
	attrib.closeness = 40000;
	attrib.colormap = cmap;
	attrib.valuemask = XpmColorSymbols | XpmCloseness | XpmColormap;

#ifdef MONO_TEST
	attrib.color_key = XPM_MONO;
	attrib.valuemask |= XpmColorKey;
#endif

	switch (t) {
	case NORMAL:
	case MASK:
		/* Make the normal (sensitive) pixmap. */
		code = XpmCreatePixmapFromData(dpy, win, pixname[which], &pm, &sm, &attrib);

		if (code == XpmSuccess) {
			StorePixmap(pm, dpy, which, NORMAL);
			StorePixmap(sm, dpy, which, MASK);
			if (t == NORMAL)
				return pm;
			return sm;
		} else if (code == XpmColorFailed) {
#if 0
			if ( (cmap = NewColormap( dpy, cmap)) == (Colormap)NULL)
				return (Colormap)NULL;
			attrib.colormap = cmap;
			attrib.valuemask |= XpmColormap;
#endif
			return XmUNSPECIFIED_PIXMAP;	/* FIX ME */
		}

	case INSENS:
		/* 
		** Now make the insensitive pixmap.  If we're color-challenged
		** make the fuzzed out pixmap, otherwise get the windows
		** type of pixmap.
		*/

		if (colorVisual) {
			attrib.colorsymbols = &mask[0];
			attrib.numsymbols = 2;

			/* 
			** We assume this succeeds because the above did. If 
			** it didn't we'll get the XCreateInsensitive 
			** pixmap instead of the MakeWindowsPixmap variety.
			*/
			code = XpmCreatePixmapFromData(dpy, win, pixname[which], &junk, &sm, &attrib);
			/* Junk is not needed, but we can't pass NULL to
			** XPM as the pixmap.  This is a bug in XPM. */
			XFreePixmap ( dpy, junk);

			attrib.colorsymbols = &backg;
			attrib.numsymbols = 1;

			if (sm)
				pm = MakeWindowsPixmap(top, sm);
			else
				pm = MakeGrayPixmap(top, GetPixmap(which, NORMAL, dpy));
			StorePixmap(pm, dpy, which, t);
			return pm;

		} else { /* MONO visual */
			pm = MakeGrayPixmap(top, GetPixmap(which, NORMAL, dpy));
			StorePixmap(pm, dpy, which, t);
			return pm;
		}
		break;
	default:
		return XmUNSPECIFIED_PIXMAP;
	}

#if 0
	switch (code) {
	case XpmSuccess:

		if ( inspixmap[i] == (Pixmap)NULL)
			return (Colormap)NULL;

		break;

	default:
		return (Colormap)NULL;
	}
#endif
}

/*
 * This function now does lazy lookup.
 * The pixmap is not created until the application wants it.
 *
 * The only reason for this is that we can have more than one Display, for
 * printing.
 */
Pixmap GetPixmap(int which, int type, Display *dpy)
{
	int	i;
	Pixmap	pm;

	for (i=0; i<nPL; i++) {
		if (PL[i].which == which && PL[i].tp == type && PL[i].dpy == dpy) {
#ifdef	DEBUG
			fprintf(stderr, "GetPixmap(%d,%d,%p)->%p\n", which, type, dpy, PL[i].pm);
#endif
			return PL[i].pm;
		}
	}
	for (i=0; i<MDL; i++) {
		if (displaylist[i].dpy == dpy) {
			pm = GetNewPixmap(displaylist[i].top, dpy,
				displaylist[i].cmap, which, type);

#ifdef	DEBUG
			fprintf(stderr, "GetPixmap(%d,%d,%p)->%p\n", which, type, dpy, pm);
#endif
			return pm;
		}
	}
	/* Houston, we have a problem. */
	fprintf(stderr, "GetPixmap(%d,%d,%p)->NULL\n", which, type, dpy);
	return None;
	abort();
}
#endif	/* SIMPLE_PIXMAP */

static	Pixmap	MakeGrayPixmap( Widget parent, Pixmap image )
{
    static Pixmap       graypm = (Pixmap) NULL;
    static GC           graygc = (GC) NULL;

    Display		*display = XtDisplay(parent);
    Pixmap		new;
    int			x, y;
    Window		root;
    unsigned int	w, h, d, b;
    Pixel		bg;

    /* how big is the original? */
    XGetGeometry( display, image, &root, &x, &y, &w, &h, &b, &d );

    /* find/create the tile */
    if ( graypm == 0 ) {
	graypm = XCreateBitmapFromData( display,
					root,
					(char *) gray_bits,
					gray_width,
					gray_height
					);
	graygc = XCreateGC( display, root, 0, 0 );
	XSetStipple( display, graygc, graypm );
	XSetFillStyle( display, graygc, FillStippled );

	XtVaGetValues( parent, XmNbackground, &bg, 0 );
	XSetForeground( display, graygc, bg );
    }

    /* create a copy and gray it out */
    new = XCreatePixmap( display, root, w, h, d );
    XCopyArea( display, image, new, graygc, 0, 0, w, h, 0, 0 );
    XFillRectangle( display, new, graygc, 0, 0, w+1, h+1 );

    return new;
}


void MakeIconWindow( Widget top )
{
   Window window, root;
   unsigned int width, height, border_width, depth;
   int x, y;
   Display *dpy = XtDisplay (top);
   Pixmap iconpix, maskpix;
   int shape_event, shape_error;

   XpmColorSymbol mask;
   XpmAttributes attrib;

   /* Set color "mask", the bg color, to NULL making it transparent */
   mask.name = "mask";
   mask.value = "none";
   mask.pixel = (Pixel)NULL;

   attrib.numsymbols = 1;
   attrib.colorsymbols = &mask;
   attrib.closeness = 40000;
   attrib.valuemask = XpmColorSymbols | XpmCloseness;

   /* Create ICON pixmap and window mask */
   if ( XpmCreatePixmapFromData( dpy, RootWindowOfScreen ( XtScreen(top) ), 
                                 pixname[PICON], &iconpix, &maskpix, &attrib ) 
        != XpmSuccess )
     return;

   /* Get current icon window */
   XtVaGetValues ( top, XmNiconWindow, &window, NULL);

   /* If we don't have one, make a simple window as large as the pixmap */
   if (!window) {
     if (!XGetGeometry ( dpy, iconpix, &root, 
                         &x, &y, &width, &height, &border_width, &depth ) ||
         !( window = XCreateSimpleWindow ( dpy, root, 0, 0, width, height,
                         (unsigned) 0, CopyFromParent, CopyFromParent ))) {

       /* If either fails, just set the icon pixmap as a fallback */
       XtVaSetValues ( top, XmNiconPixmap, iconpix, NULL);
       return;
     }

#ifdef SHAPE
     /* Set window mask to get transparent icon background */
     if ( XShapeQueryExtension ( dpy, &shape_event, &shape_error) )
        XShapeCombineMask ( dpy, window, ShapeBounding, 0, 0, 
                            maskpix, ShapeSet );
 
#endif
     XFreePixmap (dpy, iconpix);
     XFreePixmap (dpy, maskpix);

     /* Created ok, set the window */
     XtVaSetValues ( top, XmNiconWindow, window, NULL);

   }
                          
   /* Set window (icon) background to our icon pixmap */
   XSetWindowBackgroundPixmap(dpy, window, GetPixmap(PICON, NORMAL, dpy));

   /* Force redisplay */
   XClearWindow ( dpy, window);
}

static Pixmap MakeWindowsPixmap(Widget parent, Pixmap image)
{
    static GC           graygc = (GC) NULL;

    Display             *dpy = XtDisplay (parent);
    int                 depth = DefaultDepthOfScreen( XtScreen(parent));

    Pixel               bg, shadow;
    Pixmap		new;

    int			x, y;
    Window		root;
    unsigned int	w, h, d, b;

    /* how big is the original? */
    XGetGeometry( dpy, image, &root, &x, &y, &w, &h, &b, &d );

    if (graygc == (GC) NULL) {
       graygc = XCreateGC(dpy, root, 0, 0);
       XSetFillStyle(dpy, graygc, FillStippled);
    }

    new = XCreatePixmap(dpy, root, w, h, depth);

    XSetClipMask(dpy, graygc, None);

    XtVaGetValues(parent, XmNbackground, &bg, 0 );
    XmGetColors(XtScreen(parent), GetColormap(), bg, 
                  NULL, NULL, &shadow, NULL);

    /* fill pixmap in background color */
    XSetForeground(dpy, graygc, bg); 
    XFillRectangle(dpy, new, graygc, 0, 0, w, h);

    /* Copy mask onto new pixmap, but offset a bit and in white */
    XSetClipMask(dpy, graygc, image);
    XSetClipOrigin(dpy, graygc, 1, 1);
    XSetForeground(dpy, graygc, GetColor(WHITE)); 

    XFillRectangle(dpy, new, graygc, 0, 0, w, h);

    /* Copy mask onto new pixmap in an appropriate dark color */
    XSetForeground(dpy, graygc, shadow); 
    XSetClipOrigin(dpy, graygc, 0, 0);
    XFillRectangle(dpy, new, graygc, 0, 0, w, h);

    return new;
}

