/*
 * Copyright (c) 1997, 1998, 1999, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2001, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/report.c,v 1.42 2004/04/29 19:41:14 danny Exp $
 */
static char id[] = "$Header: /pack/anoncvs/xinvest/src/report.c,v 1.42 2004/04/29 19:41:14 danny Exp $";

#include <float.h>
#include <math.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <Xm/XmAll.h>

#include "account.h"
#include "calendar.h"
#include "color.h"
#include "nav.h"
#include "pixmap.h"
#include "pref.h"
#include "rate.h"
#include "report.h"
#include "reportP.h"
#include "resource.h"
#include "status.h"
#include "tooltips.h"
#include "trans.h"
#include "util.h"
#include "xutil.h"

#include "xinvest.h"
#ifdef	HAVE_LIBDT
#include <Dt/Print.h>
#endif

/*
** Globals (all static)
*/
static Widget	reportDialog = (Widget)NULL;
static Widget	reportDrawing = (Widget)NULL;

static int reportAccount = 0;
static int reportTime = 0;
static int reportValue = 0;

static char datestr[40];

static struct daterange {
  long start;
  long end;
} reportDateRange[REPORT_TIMES];

static Calendar reportCal;        /* Current calendar */
static long Customstart;          /* Custom start and end dates. Copied into */
static long Customend;            /* reportDateRange in reportGenerate() */
static int reportResource;        /* Number used to query report resources */

/* 
** Forwards
*/
static int reportDraw (Widget, XtPointer, XtPointer);


/* Initialize reports */
void reportInit (Widget Toplevel, char *appname)
{
  /* propnames must match names in Xinvest.ad */
  char propnames[][MAXRESCHAR] = { "value", "price", "share", "dValue", 
                                   "dPrice", "dShare", "cost", "debit", "fee",
				   "tr", "irr", "yield", "gain", "dist",
                                   "na", "port"};

  /* Read any non-application/non-widget resources */
  reportResource = readResourceDatabase (Toplevel, appname, "Report", "report", 
                                         propnames, XtNumber(propnames));
}


/* Initialize calendar */
static void reportCustCalInit()
{
  Customstart = convertDate(activeAccount(), FIRST_DATE);
  Customend = convertDate(activeAccount(), CURRENT_DATE);
}

/* Sets/clears time frames to report on */
/* ARGSUSED */
void reportSetTime (Widget w, int which, XmToggleButtonCallbackStruct *calldata)
{
  static Boolean first = True;
  Widget ok;
  int row_num;

  /* Distinguish between short and long term rows of time frames */
  XtVaGetValues (XtParent(w), XmNuserData, &row_num, NULL);
  if (row_num == 1) which += (REPORT_TIMES/2);

  if (calldata->set == XmSET) {
    reportTime |= (1 << which);
    if (which == TIME_CUSTOM) {
      XtVaGetValues (w, XmNuserData, &ok, NULL);
      XtSetSensitive (ok, True);

      if (first) {
        reportCustCalInit();
        first = False;
      }
    }
  } else {
    reportTime &= ~(1 << which);
    if (which == TIME_CUSTOM) {
      XtVaGetValues (w, XmNuserData, &ok, NULL);
      XtSetSensitive (ok, False);
    }
  }

  XtVaGetValues (XtParent(XtParent(w)), XmNuserData, &ok, NULL);
  if (ok && reportAccount && reportTime)
    XtSetSensitive (ok, True);
  else
    XtSetSensitive (ok, False);

}

/* Sets/clears report options */
/* ARGSUSED */
void reportSetValue (Widget w, int which, 
                     XmToggleButtonCallbackStruct *calldata)
{
  if (calldata->set == XmSET)
    reportValue |= (1 << which);
  else
    reportValue &= ~(1 << which);
}

/* Sets/clears accounts to report on */
/* ARGSUSED */
void reportSetAccount (Widget w, int which, 
                       XmToggleButtonCallbackStruct *calldata)
{
  Widget button;
  char name[10];

  if (calldata->set == XmSET) {
    reportAccount |= (1 << which);

    /* Can't have 'this account' AND 'all accounts' set simultaneously */
    if (which == 0 || which == 1) {
      reportAccount &= ~(1 << (1-which));
      sprintf (name, "button_%d", 1-which);
      if ((button = XtNameToWidget (XtParent(w), name)) != (Widget)NULL)
        XmToggleButtonSetState (button, False, False);
    }
  } else {
    reportAccount &= ~(1 << which);
  }

  XtVaGetValues (XtParent(w), XmNuserData, &button, NULL);
  if (button && reportAccount && reportTime)
    XtSetSensitive (button, True);
  else
    XtSetSensitive (button, False);
}


/* Redisplay the custom calendar and/or report.  Useful on account switch */
void displayReport (int account)
{
  void *trans;

  /* If user hasn't touched calendar, change default dates to new account */
  if (!calUserTouched(&reportCal)) {
    getAccount (account, ACCOUNT_TRANS, &trans);
    if (trans) {
      if (Customstart == -1)   /* Displaying start */
        calLtoCal (&reportCal, GetTransLDate (trans, 1));
      else                     /* Displaying end */
        calLtoCal (&reportCal, convertDate( account, CURRENT_DATE));
    }
    reportCustCalInit();
  }
  calSetDate (&reportCal);

  /* Force new drawing if in single window mode */
  if (reportDrawing && reportValue & REPORT_SINGLEWIN_ON)
    reportDraw (reportDrawing, (XtPointer)ALLOW, (XtPointer)NULL);
}

void reportCustomStartEnd (Widget w, XtPointer client_data, 
                           XmAnyCallbackStruct *call_data)
{
  long date = calCaltoL (&reportCal);

  if ((int)client_data == 0) {
    Customend = date;                     /* Save end */
    calLtoCal (&reportCal, Customstart);  /* Display start */
    Customstart = -1L;                    /* Start is in reportCal */
  } else {
    Customstart = date;
    calLtoCal (&reportCal, Customend);
    Customend = -1L;
  }
   
  calSetDate (&reportCal);
}

static void reportCustomWarp (Widget w, XtPointer client_data,
                       XmPushButtonCallbackStruct *call_data)
{
  int account = activeAccount();
  void *trans;
  int numTrans, i;

  long currentDate = calCaltoL (&reportCal);
  long nextDate;

  getAccount (account, ACCOUNT_TRANS, &trans);
  getAccount (account, ACCOUNT_NUM_TRANS, &numTrans);

  if (trans == NULL)
    return;

  switch ((int) client_data) {
  
  case 0: /* previous transaction */
          for (i=numTrans; i >= 1; i--) {
            nextDate = GetTransLDate (trans, i);
            if (nextDate < currentDate)
              break;
          }
          break;
  case 1: /* next transaction */
          for (i=1; i <= numTrans; i++) {
            nextDate = GetTransLDate (trans, i);
            if (nextDate > currentDate)
              break;
          }
          break;
  case 2: /* first transaction */
          nextDate = GetTransLDate (trans, 1);
          break;
  case 3: /* last transaction */
          nextDate = GetTransLDate (trans, numTrans);
          break;
  default: break;
  }

  if (nextDate && nextDate != currentDate) {
    calLtoCal (&reportCal, nextDate);
    calSetDate (&reportCal);
  }
}

void reportCustomTime (Widget w, XtPointer client_data, 
                       XmPushButtonCallbackStruct *call_data)
{
  static Widget form = (Widget)NULL;
  Widget calendar, optmenu, button1, button2;
  Widget reportform = (Widget)client_data;
  Boolean mapped;

  if (form == (Widget)NULL) {

    /* Starting date */
    calLtoCal (&reportCal, convertDate (activeAccount(), 
                                        reportDateRange[TIME_CUSTOM].start));

    /* Same size as parent */
    form = XtVaCreateWidget ("reportform2", 
                       xmFormWidgetClass, XtParent(reportform),
                       XmNmappedWhenManaged, False,
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_FORM,
                       XmNrightAttachment, XmATTACH_FORM,
                       XmNbottomAttachment, XmATTACH_FORM,
                       NULL);
    calendar = calCreate (form, &reportCal);
    XtVaSetValues (calendar, 
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_FORM,
                   NULL);
    XtManageChild (calendar);
    optmenu = XmVaCreateSimpleOptionMenu (form, "reportoptmenu",
                       NULL, (KeySym) NULL, 0,
                       (XtCallbackProc) reportCustomStartEnd,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmVaPUSHBUTTON, NULL, NULL, NULL, NULL,
                       XmNtopAttachment, XmATTACH_FORM,
                       XmNleftAttachment, XmATTACH_WIDGET,
                       XmNleftWidget, calendar,
                       NULL);
    XtDestroyWidget(XmOptionLabelGadget(optmenu));
    XtManageChild (optmenu);
    /* previous transaction button */
    button1 =  XtVaCreateManagedWidget ("prevTrans",
         xmPushButtonWidgetClass, form,
         XmNlabelType,       XmPIXMAP,
         XmNlabelPixmap,     GetPixmap(PPREVTRANS, NORMAL, XtDisplay(form)),
         XmNlabelInsensitivePixmap,   GetPixmap(PPREVTRANS, NORMAL, XtDisplay(form)),
         XmNsensitive,       True,
         XmNtopAttachment,   XmATTACH_WIDGET,
         XmNtopWidget,       optmenu,
         XmNleftAttachment,  XmATTACH_WIDGET,
         XmNleftWidget,      calendar,
         NULL);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)0 );
    installToolTips (button1);
    /* next transaction button */
    button2 = XtVaCreateManagedWidget ("nextTrans",
         xmPushButtonWidgetClass,  form,
         XmNlabelType,             XmPIXMAP,
         XmNlabelPixmap,           GetPixmap(PNEXTTRANS, NORMAL, XtDisplay(form)),
         XmNlabelInsensitivePixmap,   GetPixmap(PNEXTTRANS, NORMAL, XtDisplay(form)),
         XmNsensitive,             True,
         XmNtopAttachment,         XmATTACH_WIDGET,
         XmNtopWidget,             optmenu,
         XmNleftAttachment,        XmATTACH_WIDGET,
         XmNleftWidget,            button1,
         NULL);
    XtAddCallback (button2, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)1 );
    installToolTips (button2);
    /* 1st transaction button */ 
    button1 =  XtVaCreateManagedWidget ("firstTrans",
         xmPushButtonWidgetClass, form,
         XmNlabelType,            XmPIXMAP,
         XmNlabelPixmap,          GetPixmap(PFIRST, NORMAL, XtDisplay(form)),
         XmNlabelInsensitivePixmap,   GetPixmap(PFIRST, NORMAL, XtDisplay(form)),
         XmNsensitive,            True,
         XmNtopAttachment,        XmATTACH_WIDGET,
         XmNtopWidget,            button1,
         XmNleftAttachment,       XmATTACH_WIDGET,
         XmNleftWidget,           calendar,
         NULL);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)2 );
    installToolTips (button1);
    /* last transaction button */
    button2 =  XtVaCreateManagedWidget ("lastTrans",
         xmPushButtonWidgetClass,  form,
         XmNlabelType,             XmPIXMAP,
         XmNlabelPixmap,           GetPixmap(PLAST, NORMAL, XtDisplay(form)),
         XmNlabelInsensitivePixmap,   GetPixmap(PLAST, NORMAL, XtDisplay(form)),
         XmNsensitive,             True,
         XmNtopAttachment,         XmATTACH_OPPOSITE_WIDGET,
         XmNtopWidget,             button1,
         XmNleftAttachment,        XmATTACH_WIDGET,
         XmNleftWidget,            button1,
         NULL);
    XtAddCallback (button2, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomWarp, (XtPointer)3 );
    installToolTips (button2);

    /* Page return button */
    button1 = XtVaCreateManagedWidget (
                "Screenbut", xmPushButtonWidgetClass, form,
                XmNlabelType,    XmPIXMAP,
                XmNlabelPixmap,  GetPixmap(PPREV,NORMAL, XtDisplay(form)),
                XmNlabelInsensitivePixmap, GetPixmap(PPREV,INSENS, XtDisplay(form)),
                XmNrightAttachment, XmATTACH_FORM,
                XmNbottomAttachment, XmATTACH_FORM,
                NULL);
    installToolTips (button1);
    XtAddCallback (button1, XmNactivateCallback, 
                   (XtCallbackProc)reportCustomTime, (XtPointer)reportform);

    XtManageChild (form);

    XtVaSetValues (form, XmNnoResize, True, NULL);
    CenterWidget (XtParent(form), form);

  }

  XtVaGetValues (form, XmNmappedWhenManaged, &mapped, NULL);
  if (mapped == False) {
    /* Turn on second page */
    XtSetMappedWhenManaged (reportform, False);
    XtSetMappedWhenManaged (form, True);
  } else {
    /* Return to 1st page */
    XtSetMappedWhenManaged (reportform, True);
    XtSetMappedWhenManaged (form, False);
  }
}

static int reportDrawArrow (double val, char *caption,
		            int pos_x, int pos_y, int size,
		            struct report_attrib *attrib)
{
	int one_third, one_half, two_third;
	int symbol_border = (int)(0.1*size);
	int symbol_size =   (int)(0.8*size);
	XPoint point[7]; 
	int neg = (val < 0);

	one_third = symbol_size/3;
	two_third = one_third *2;
	one_half = symbol_size/2;

	/******************
	**       ^6
	**     /   \
	**  0--1   4--5
	**     |___|
	**     2   3
	*******************/
	point[0].x = pos_x;
	point[0].y = (neg) ? (pos_y + symbol_border + one_third +1):
		(pos_y + symbol_border + two_third +1);
	point[1].x = one_third;
	point[1].y = 0;
	point[2].x = 0;
	point[2].y = (neg)?-one_third:one_third;
	point[3].x = one_third;
	point[3].y = 0;
	point[4].x = 0;
	point[4].y = -point[2].y;
	point[5].x = point[1].x;
	point[5].y = 0;
	point[6].x = -one_half;
	point[6].y = (neg)?two_third:-two_third;

#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
		XSetForeground(attrib->dpy, attrib->gc, attrib->shad );
		XFillPolygon(attrib->dpy, attrib->pix, attrib->gc,
			point, 7, Convex, CoordModePrevious);
	}
	point[0].x--;
	point[0].y--;

	if (neg)
		XSetForeground (attrib->dpy, attrib->gc, GetColor(RED) );
	else
		XSetForeground (attrib->dpy, attrib->gc, GetColor(GREEN) );
	if (val == 0.0)
		XSetForeground (attrib->dpy, attrib->gc, GetColor(GREY) );

	XFillPolygon(attrib->dpy, attrib->pix, attrib->gc,
		point, 7, Convex, CoordModePrevious );

	XSetForeground(attrib->dpy, attrib->gc, attrib->fg);

	/* Draw caption */
	if (caption) {
		int adjust_x = XTextWidth (attrib->font, caption, strlen(caption))/2;
		int adjust_y = (neg)?(attrib->font->ascent):(-attrib->font->descent);

		XSetFont (attrib->dpy, attrib->gc, attrib->font->fid);

		/*
		* When displaying in a window, draw text twice in different
		* colors for visual appearance; don't do this when printing.
		*/
#ifdef	XPRINT
		if (! per->pctxt)
#endif
		{
			XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
			XDrawString (attrib->dpy, attrib->pix, attrib->gc,
				pos_x + symbol_size/2 + attrib->shadThick - adjust_x, 
				point[0].y + adjust_y + attrib->shadThick,
				caption, strlen (caption) );
		}

		XSetForeground(attrib->dpy, attrib->gc, attrib->fg);
		XDrawString(attrib->dpy, attrib->pix, attrib->gc,
			pos_x + symbol_size/2 - adjust_x, 
			point[0].y + adjust_y,
			caption, strlen (caption));
	}
	return (symbol_size);
}


static void reportRenderText(char *valhdr, char *valval,
		int inc_x, int adjust,
		struct report_attrib *attrib)
{ 
	int adjust_x = 0, adjust_y = attrib->sfont->ascent;
	int inc_y = attrib->font->ascent + attrib->font->descent
		+ attrib->sfont->ascent + attrib->sfont->descent;

	/* Draw heading */
	if (strlen(valhdr)) {
		if (adjust) { /* Center justify */
			adjust_x = inc_x*attrib->avgWidth - 
				XTextWidth(attrib->sfont, valhdr, strlen(valhdr));
			if (adjust_x >= 0) 
				adjust_x /= 2;
			else
				adjust_x = 5;
		}
		XSetFont(attrib->dpy, attrib->gc, attrib->sfont->fid);

		/*
		 * When displaying in a window, draw text twice in different
		 * colors for visual appearance; don't do this when printing.
		 */
#ifdef	XPRINT
		if (! per->pctxt)
#endif
		{
			XSetForeground(attrib->dpy, attrib->gc, attrib->shad);
			XDrawString(attrib->dpy, attrib->pix, attrib->gc,
				attrib->x + adjust_x + attrib->shadThick, 
				attrib->y + adjust_y + attrib->shadThick,
				valhdr, strlen (valhdr));
		}
		XSetForeground(attrib->dpy, attrib->gc, attrib->hilite);
		XDrawString(attrib->dpy, attrib->pix, attrib->gc,
			attrib->x + adjust_x, attrib->y + adjust_y,
			valhdr, strlen (valhdr) );
	}

	/* Draw value */
	if (strlen(valval)) {
		if (adjust) { /* Right justify */
			adjust_x = XTextWidth (attrib->font, valval, strlen(valval));
			adjust_x -=  inc_x*attrib->avgWidth;
			if (adjust_x >= 0) 
				adjust_x = 0;
		}
		XSetFont(attrib->dpy, attrib->gc, attrib->font->fid);

		/*
		 * When displaying in a window, draw text twice in different
		 * colors for visual appearance; don't do this when printing.
		 */
#ifdef	XPRINT
		if (! per->pctxt)
#endif
		{
			XSetForeground (attrib->dpy, attrib->gc, attrib->shad );
			XDrawString (attrib->dpy, attrib->pix, attrib->gc,
				attrib->x + attrib->shadThick - adjust_x, 
				attrib->y + adjust_y + attrib->font->ascent +attrib->shadThick,
				valval, strlen (valval) );
		}

		XSetForeground(attrib->dpy, attrib->gc, attrib->fg );
		XDrawString(attrib->dpy, attrib->pix, attrib->gc,
			attrib->x - adjust_x, 
			attrib->y + adjust_y + attrib->font->ascent,
			valval, strlen (valval) );
	}
	/* Draw vertical border to right of text */
	if (adjust) {
		XSetForeground(attrib->dpy, attrib->gc, attrib->tshad);
		XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
			attrib->x + inc_x*attrib->avgWidth, attrib->y, 
			attrib->x + inc_x*attrib->avgWidth, attrib->y + inc_y);
	}
}

static void reportSummary(int account, struct report_attrib *attrib)
{
	int k; 
	int inc_x, inc_y;
	char valval[80], valhdr[80];

	double price = 0.0;
	double value = 0.0;
	double shares = 0.0;
	long daymin = LONG_MAX; long daymax = 0L;
	char title[40];

	/* Calculate values  */
	if (account == REPORT_PORTFOLIO_ACCOUNT) {
		for (k=1; k <= numAccounts(); k++) {
			if (convertDate (k, LAST_DATE) > daymax)
				daymax = convertDate (k, CURRENT_DATE);
			if (convertDate(k, FIRST_DATE) < daymin)
				daymin = convertDate(k, FIRST_DATE);
			value += rate_value (k, FIRST_DATE, convertDate(k, CURRENT_DATE) );
		}
	} else {
		daymax   = convertDate (account, CURRENT_DATE);
		daymin   = convertDate (account, FIRST_DATE);
		value    = rate_value (account, FIRST_DATE, daymax);
		shares   = rate_shares (account, FIRST_DATE, daymax, TRUE);
		price    = rate_price (account, daymax, TRUE);
	}

	attrib->y += (attrib->font->ascent + attrib->font->descent); 

  /* TITLE */
	inc_x = 0;
	inc_y = attrib->font->ascent + attrib->font->descent;
	attrib->x = BORDER/2;

	/* Copy logo to front of title if there is enough vertical room */
	if (inc_y >= 16) {
		Pixmap	pm;

		/* Not the most robust code, 16's will get us into trouble someday.*/
		XSetClipMask(attrib->dpy, attrib->gc,
      			GetPixmap(PTICON, MASK, attrib->dpy));
		XSetClipOrigin (attrib->dpy, attrib->gc, attrib->x, attrib->y);

		pm = GetPixmap(PTICON, NORMAL, attrib->dpy);
		XCopyArea(attrib->dpy, pm, attrib->pix, attrib->gc,
			0, 0, 16, 16, attrib->x, attrib->y);
		XSetClipMask(attrib->dpy, attrib->gc, None);
		inc_x = 20;
	}

	if (account == REPORT_PORTFOLIO_ACCOUNT) {
		strcpy (title, getResource(reportResource, REPPORT));
	} else {
		char *name = createAccountName (account);
		sprintf(title, "%-.40s", name);
		XtFree(name);
	}

	XSetForeground(attrib->dpy, attrib->gc, attrib->shad);
	XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
		attrib->x, attrib->y + inc_y, 
		BORDER/2+REPORT_CHAR_WIDTH*attrib->avgWidth,
		attrib->y + inc_y);
#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
		XSetForeground(attrib->dpy, attrib->gc, attrib->fg);
		XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
			attrib->x, attrib->y + inc_y - attrib->shadThick, 
			BORDER/2+REPORT_CHAR_WIDTH*attrib->avgWidth, 
			attrib->y + inc_y - attrib->shadThick);
	}
	XSetFont(attrib->dpy, attrib->gc, attrib->font->fid);

	/*
	 * When displaying in a window, draw text twice in different
	 * colors for visual appearance; don't do this when printing.
	 */
#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
		XSetForeground(attrib->dpy, attrib->gc, attrib->shad );
		XDrawString(attrib->dpy, attrib->pix, attrib->gc,
			attrib->x + inc_x + attrib->shadThick, 
			attrib->y + attrib->font->ascent + attrib->shadThick,
			title, strlen (title) );
	}
	XSetForeground(attrib->dpy, attrib->gc, attrib->hilite);
	XDrawString(attrib->dpy, attrib->pix, attrib->gc,
		attrib->x + inc_x, attrib->y + attrib->font->ascent,
		title, strlen(title));

	attrib->y += inc_y;

	inc_y = attrib->font->ascent + attrib->font->descent
		+ attrib->sfont->ascent + attrib->sfont->descent;

	/* VALUE */
	strcpy(valhdr, getResource(reportResource, REPVALUE));
	sprintf(valval, "%.20s", utilCurrOut(value, UTILCURR));
	inc_x = 20; 
	attrib->x = BORDER/2 + 20*attrib->avgWidth;
	reportRenderText (valhdr, valval, inc_x, True, attrib);
	attrib->x += inc_x * attrib->avgWidth;

	/* PRICE */
  if (account == REPORT_PORTFOLIO_ACCOUNT) {
    strcpy (valhdr, getResource(reportResource, REPPRICE));
    sprintf (valval, "%.20s", getResource(reportResource, REPNA));
  } else {
    NAV *navp;
    getAccount (account, ACCOUNT_NAV, &navp);
    if (isNavChanged (account, navp))
      sprintf (valhdr, "%.10s (Live Quote)", 
               getResource(reportResource, REPPRICE));
    else
      strcpy (valhdr, getResource(reportResource, REPPRICE));
    sprintf (valval, "%.20s", utilCurrOut(price, UTILCURR));
  }
  inc_x = 20; 
  reportRenderText (valhdr, valval, inc_x, True, attrib);
  attrib->x += inc_x * attrib->avgWidth;

	/* SHARES */
  strcpy (valhdr, getResource(reportResource, REPSHARE));
  if (account == REPORT_PORTFOLIO_ACCOUNT)
    sprintf (valval, "%.16s", getResource(reportResource, REPNA));
  else
    sprintf (valval, "%.16s", utilCurrOut(shares,UTILNUM));
  inc_x = 16; 
  reportRenderText (valhdr, valval, inc_x, True, attrib);
  attrib->x += inc_x * attrib->avgWidth;

	/* Reduce the clutter by drawing a line */
  XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
  XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
             BORDER/2 + 20*attrib->avgWidth, attrib->y + inc_y, 
	     BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, attrib->y + inc_y);
#ifdef XPRINT
  if (! per->pctxt)
#endif
  {
	XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
	XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
		BORDER/2 + 20*attrib->avgWidth, 
		attrib->y + inc_y - attrib->shadThick, 
		BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, 
		attrib->y + inc_y - attrib->shadThick);
  }

  attrib->y += inc_y;
}

static void reportTimeVal (int account, struct report_attrib *attrib)
{
  extern Widget Timesform, Timelform;
  Widget w;

  int i, k; 

  double   price, value, shares, cost, debit, fee, distrib;
  double   prevPrice, prevValue, prevShares, prevCost, prevDebit, prevFee, 
           prevDistrib;

  long     daymin, daymax;

  /* Draw selected times */
  for (i=0; i < REPORT_TIMES; i++) {
    if (reportTime & (1 << i)) {
      char timename[20], *timestr;
      XmString timeframe;

      if (i < REPORT_TIMES/2) {  /* Short time frame form */
        sprintf (timename, "button_%d", i);
        w = XtNameToWidget (Timesform, timename);
      } else {                   /* Long time frame form */
        sprintf (timename, "button_%d", i-REPORT_TIMES/2);
        w = XtNameToWidget (Timelform, timename);
      }
      if (w != (Widget)NULL) {

        XtVaGetValues (w, XmNlabelString, &timeframe, NULL);
        XmStringGetLtoR (timeframe, XmFONTLIST_DEFAULT_TAG, &timestr);

        /* Calculate values  */
        price = 0.0;   prevPrice = 0.0;
        value = 0.0;   prevValue = 0.0;
        cost = 0.0;    prevCost = 0.0;
        debit = 0.0;   prevDebit = 0.0;
        fee = 0.0;     prevFee = 0.0;
        distrib = 0.0; prevDistrib = 0.0;
        shares = 0.0;  prevShares = 0.0;
        daymin = LONG_MAX; daymax = 0L;

        if (account == REPORT_PORTFOLIO_ACCOUNT) {
          for (k=1; k <= numAccounts(); k++) {
            if (convertDate (k, reportDateRange[i].end) > daymax)
              daymax = convertDate (k, reportDateRange[i].end);
            if (convertDate(k, reportDateRange[i].start) < daymin)
              daymin = convertDate(k, reportDateRange[i].start);
            value    += rate_value (k, FIRST_DATE, reportDateRange[i].end );
            prevValue += rate_value (k, FIRST_DATE, daymin-1);
            cost     += rate_cost (k, FIRST_DATE, reportDateRange[i].end );
            prevCost += rate_cost (k, FIRST_DATE, daymin-1);
            debit    += rate_withdrawal (k, FIRST_DATE, reportDateRange[i].end);
            prevDebit += rate_withdrawal (k, FIRST_DATE, daymin-1);
            fee      += rate_fee (k, FIRST_DATE, reportDateRange[i].end); 
            prevFee  += rate_fee (k, FIRST_DATE, daymin-1);
            distrib  += rate_distrib (k, FIRST_DATE, reportDateRange[i].end);
            prevDistrib += rate_distrib( k, FIRST_DATE, daymin-1);
          }
        } else {
          daymax   = convertDate (account, reportDateRange[i].end);
          daymin   = convertDate (account, reportDateRange[i].start);
          value    = rate_value (account, FIRST_DATE, reportDateRange[i].end);
          prevValue = rate_value (account, FIRST_DATE, daymin-1);
          cost     = rate_cost (account, FIRST_DATE, reportDateRange[i].end);
          prevCost = rate_cost (account, FIRST_DATE, daymin-1);
          debit = rate_withdrawal (account, FIRST_DATE, reportDateRange[i].end);
          prevDebit = rate_withdrawal (account, FIRST_DATE, daymin-1);
          fee      = rate_fee (account, FIRST_DATE, reportDateRange[i].end); 
          prevFee  = rate_fee (account, FIRST_DATE, daymin-1);
          distrib  = rate_distrib (account, FIRST_DATE, reportDateRange[i].end);
          prevDistrib = rate_distrib (account, FIRST_DATE, daymin-1);
          shares   = rate_shares (account, FIRST_DATE, reportDateRange[i].end, TRUE);
          prevShares = rate_shares (account, FIRST_DATE, daymin-1, TRUE);
          price     = rate_price (account, reportDateRange[i].end, TRUE);
          /* If we precede first trans, use first trans info to calc price */
          if (convertDate (account, FIRST_DATE) > daymin-1) 
            prevPrice = rate_price (account, FIRST_DATE, TRUE);
	  else
            prevPrice = rate_price (account, daymin-1, TRUE);
        }

#ifdef DEBUG
        fprintf( stderr, 
                 "daymax -> %ld\n" "daymin -> %ld\n" "value     -> %f\n"
                 "prevalue  -> %f\n" "cost      -> %f\n" "precost   -> %f\n"
                 "debit     -> %f\n" "predebit  -> %f\n" "fee       -> %f\n"
                 "prefee    -> %f\n" "distrib   -> %f\n" "predistrib-> %f\n"
                 "shares    -> %f\n",
                 daymax, daymin, value, prevValue, cost, prevCost, debit,
                 prevDebit, fee, prevFee, distrib, prevDistrib, shares );
#endif
        /* 
        ** Generate strings 
        */
	     
        /* Draw selected values */
	{
          char valval[80], valhdr[80];
          int inc_x, inc_y;
          YIELD result;

	  inc_y = attrib->font->ascent + attrib->font->descent
	          + attrib->sfont->ascent + attrib->sfont->descent;

          /* 
	  ** TIME FRAME 
	  */
          attrib->x = BORDER/2;
          inc_x = 20; 

	  valhdr[0] = '\0';
          if (i == TIME_CUSTOM) {       /* Custom */
	    char *start;  
	    sprintf (valhdr, "%-.20s", timestr);
            /* utilOutDate uses static return, can't call twice below.*/
            start = XtNewString (utilDateOut(reportDateRange[i].start));
            sprintf (valval, "%.10s - %.10s", start,
                     utilDateOut(reportDateRange[i].end) );
	    valval[20] = '\0';
            XtFree (start);
          } else if (i == TIME_INCEP) { /* Since inception */
	    sprintf (valhdr, "%-.20s", timestr);
            sprintf (valval, "%.20s", utilDateOut(daymin));
	    valval[20] = '\0';
          } else if (i == TIME_YTD) {   /* YTD */
 	    sprintf (valhdr, "%-.20s", timestr);
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
              sprintf (valval, "%.20s", 
		     utilDateOut(convertDate(activeAccount(), CURRENT_DATE)));
	    } else {
              sprintf (valval, "%.20s", 
		       utilDateOut(convertDate(account, CURRENT_DATE)));
	    }
	    valval[20] = '\0';
	  } else {                            /* Others */
            sprintf (valval, "%-.20s", timestr);
	  }
          XtFree (timestr);
          reportRenderText (valhdr, valval, inc_x, False, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* Transaction details line 1 */

          if (reportValue & REPORT_DETAIL_ON) {
            /* 
	    ** delta VALUE 
	    */
            inc_x = 20; 
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
            strcpy (valhdr, getResource(reportResource, REPDVALUE));
	    sprintf (valval, "%.20s", utilCurrOut(value-prevValue, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta PRICE 
	    */
            strcpy (valhdr, getResource(reportResource, REPDPRICE));
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
	      sprintf (valval, "%.19s", getResource(reportResource, REPNA));
            } else {
	      char *change;  
              /* utilOutDate uses static return, can't call twice below.*/
              change = XtNewString (utilCurrOut(price-prevPrice, UTILCURR));
	      sprintf (valval, "%.9s %8.8s%%", change,
		       utilCurrOut((price-prevPrice)/prevPrice*100.0, UTILNUM));
              XtFree (change);
            }
            inc_x = 20; 
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta SHARES 
	    */
            inc_x = 16; 
            strcpy (valhdr, getResource(reportResource, REPDSHARE));
            if (account == REPORT_PORTFOLIO_ACCOUNT) {
	      sprintf (valval, "%.16s", getResource(reportResource, REPNA));
            } else {
	      sprintf (valval, "%.16s", 
		       utilCurrOut(shares-prevShares,UTILNUM));
            }
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
          }

          /* Transaction details line 2 */

          if (reportValue & REPORT_DETAIL_ON) {
            /* 
	    ** delta COST 
	    */
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
	    attrib->y += inc_y;
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPCOST));
	    sprintf (valval, "%.20s", utilCurrOut(cost-prevCost, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta DEBIT 
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPDEBIT));
	    sprintf (valval, "%.20s", utilCurrOut(debit-prevDebit, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta FEE 
	    */
            inc_x = 16; 
            strcpy (valhdr, getResource(reportResource, REPFEE));

            if (fee > 0.0)
              fee = -fee;
            if (prevFee > 0.0)
              prevFee = -prevFee;
	    sprintf (valval, "%.16s", utilCurrOut(fee-prevFee, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
	  }

          /* Return line */

          /* 
	  ** TR 
	  */
          attrib->x = BORDER/2 + 20*attrib->avgWidth;
          if (reportValue & (REPORT_DETAIL_ON))
	    attrib->y += inc_y;
          inc_x = 20;   
          strcpy (valhdr, getResource(reportResource, REPTR));
          {
            double rate, annRate;
            double years = (double)(daymax - daymin) / 365.0;
            double dCost = cost;
            double dValue = (value-prevValue) + (debit-prevDebit) -
                            (cost-prevCost);

            /* If we're under one less digit than max user preference
               precision, remove any weird significance behavior.     */
            if (fabs(dValue) < DBL_EPSILON*1.0E+07) 
              dValue = 0.0;

            if (dCost == (double)0.0) {
              if (dValue == (double)0.0)
                rate = annRate = 0.0;
              else
                rate = annRate = LONG_MAX; /* trigger inf. format */
            } else {
              rate = dValue / dCost;
              annRate = (pow (1.0+rate, 1.0/years) - 1.0) * 100.0;
              rate *= 100;
            }
	    sprintf (valval, "%.19s%%", utilCurrOut(rate, UTILNUM));

	    /* If we had good results, let them know */
            if (reportValue & (REPORT_DETAIL_ON))
              reportDrawArrow (rate, valval, BORDER/2 + 5*attrib->avgWidth, 
		               attrib->y-inc_y, 6*attrib->avgWidth, attrib);
          }
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* 
	  ** IRR 
	  */
          inc_x = 20; 
          strcpy (valhdr, getResource(reportResource, REPIRR));
          {
            double rate;

            rateClrAccount();

            if (account != REPORT_PORTFOLIO_ACCOUNT) {
              rate_add_account (account, reportDateRange[i].start,
                                reportDateRange[i].end);
            } else {
              for (k=1; k<=numAccounts(); k++) {
                rate_add_account (k, reportDateRange[i].start,
                                  reportDateRange[i].end);
              }
            }
            rate = rate_IRR (value + (distrib-prevDistrib));
	    sprintf (valval, "%.19s%%", utilCurrOut(rate, UTILNUM));
          }
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* 
	  ** YIELD 
	  */
          inc_x = 16; 
          strcpy (valhdr, getResource(reportResource, REPYIELD));
          if (account != REPORT_PORTFOLIO_ACCOUNT) {
            rate_yield (account, reportDateRange[i].start,
                        reportDateRange[i].end, &result);
          } else {
            YIELD acct;
            double years = (double)(daymax - daymin) / 365.0;
	    result.div = result.value = 0.0;
            for (k = 1; k <= numAccounts(); k++) {
              rate_yield (k, reportDateRange[i].start,
                          reportDateRange[i].end, &acct);
              result.div += acct.div;
              result.value += acct.value;
            }
            /* Guard against divide by zero */
            if (result.value == 0.0) result.value = 1.0;
            result.yield = result.div/result.value;
            result.annYield = (pow (1.0+result.yield, 1.0/years) - 1.0) * 100.0;
            result.yield *= 100.0;
          }
          sprintf (valval, "%.15s%%", utilCurrOut(result.yield, UTILNUM));
          reportRenderText (valhdr, valval, inc_x, True, attrib);
          attrib->x += (inc_x*attrib->avgWidth); 

          /* Return details line */

          if (reportValue & REPORT_RTN_DETAIL_ON) {
            inc_x = 20; 
            attrib->x = BORDER/2 + 20*attrib->avgWidth;
	    attrib->y += inc_y;

            /* 
	    ** delta VALUE due to price change
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPGAIN));
	    { 
	      double dValue =  (value-prevValue)+(debit-prevDebit)
                              -(cost-prevCost)-result.div;
              if (fabs(dValue) < DBL_EPSILON*1.0E+07) 
                dValue = 0.0;
              sprintf (valval, "%.20s", utilCurrOut(dValue, UTILCURR));
	    }
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

            /* 
	    ** delta VALUE due to distributions
	    */
            inc_x = 20; 
            strcpy (valhdr, getResource(reportResource, REPDIST));
            sprintf (valval, "%.20s", utilCurrOut(result.div, UTILCURR));
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 

	    /* Place holder for possible dividend use */
            inc_x = 16; 
            strcpy (valhdr, "");
            strcpy (valval, "");
            reportRenderText (valhdr, valval, inc_x, True, attrib);
            attrib->x += (inc_x*attrib->avgWidth); 
	  }

	  /* Reduce the clutter by drawing a line after each time frame */
          XSetForeground (attrib->dpy, attrib->gc, attrib->shad);
          XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
                     BORDER/2 + 20*attrib->avgWidth, attrib->y + inc_y, 
	             REPORT_CHAR_WIDTH*attrib->avgWidth, attrib->y + inc_y);
#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
          XSetForeground (attrib->dpy, attrib->gc, attrib->fg);
          XDrawLine(attrib->dpy, attrib->pix, attrib->gc, 
	             BORDER/2 + 20*attrib->avgWidth, 
		     attrib->y + inc_y - attrib->shadThick, 
	             BORDER/2 + REPORT_CHAR_WIDTH*attrib->avgWidth, 
		     attrib->y + inc_y - attrib->shadThick);
	}

        } /* for values */

        attrib->y += (attrib->font->ascent + attrib->font->descent +
		      attrib->sfont->ascent + attrib->sfont->descent); 

      } /* if widget found */
    } /* if time on */
  } /* for time */
}


/* ARGSUSED */
static void reportRedraw (Widget w, XtPointer client_data, XtPointer call_data)
{
	Dimension width, height;
	struct report_attrib *attrib;

	XtVaGetValues(w, XmNwidth, &width,
			XmNheight, &height,
			XmNuserData, &attrib,
		NULL);
	if (attrib && attrib->pix && attrib->gc)
		XCopyArea (XtDisplay(w),
			attrib->pix, XtWindow(w), attrib->gc,
			0, 0, width, height, 0, 0 );
}

/*
 * This function has become a bit more complicated than it was.
 *
 * When creating a report for display in a window, performance is
 * important. Therefore, a pixmap is created in which the whole
 * report is created in one go.
 *
 * Redisplay is handled in another function which simply copies the
 * pixmap onto the window again. Fast and easy.
 *
 * When printing the report, we know we'll be doing this only once,
 * so there's not much point in storing all this in a pixmap. So we
 * draw on the "window" directly. Also we need to take care not to
 * print outside the boundaries of the paper, so we'll have to paginate.
 *
 * The way this works is that we return, and get called again to print
 * the next page. So this function is restartable through the per->busy
 * and per->count flags. Also the attrib pointer is remembered there.
 *
 * Determining whether we're printing is simple: per->ctxt is non-NULL
 * when printing.
 */

/* ARGSUSED */
static int reportDraw(Widget w, XtPointer client_data, XtPointer call_data)
{
	int accounts = 0, times = 0;
	unsigned int inner, outer;
	int i;
	Display *dpy = per->dpy;
	GC gc;
	Pixmap draw;
	Dimension height, width;
	extern char version[];
	char line[80];

	static char atoZ[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
	static char dig[] = "01234567890$% .,";
	struct report_attrib *attrib;

	if (per->attrib) {
		attrib = (struct report_attrib *)per->attrib;
	} else {
		attrib = (struct report_attrib *)XtCalloc(1, sizeof(struct report_attrib));
		attrib->font = per->large;
		attrib->sfont = per->small;
		attrib->avgWidth = XTextWidth(attrib->font, atoZ, strlen(atoZ));
		attrib->avgWidth += XTextWidth(attrib->font, dig, strlen(dig));
		attrib->avgWidth = attrib->avgWidth / (strlen(atoZ) + strlen(dig));

		per->attrib = (void *)attrib;
	}

	/* 
	 * How many lines 
	 */
	if (reportAccount & REPORT_ALL_ACCOUNTS)
		accounts += numAccounts();
	if (reportAccount & REPORT_ACTIVE_ACCOUNT)
		accounts++;
	if (reportAccount & REPORT_PORTFOLIO)
		accounts++;

	for (i=0; i < REPORT_TIMES; i++) {
		if (reportTime & (1 << i))
			times++;
	}

	if (accounts == 0 || times == 0) {
		write_status("No account(s) and/or time frame(s)\n"
			"selected, report not generated.", ERR);
		return 0;
	}

	inner = 1*(attrib->font->ascent + attrib->font->descent)
		+ 1*(attrib->sfont->ascent + attrib->sfont->descent);
	if (reportValue & REPORT_DETAIL_ON) 
		inner += (2*(attrib->font->ascent + attrib->font->descent) +
			2*(attrib->sfont->ascent + attrib->sfont->descent));
	if (reportValue & REPORT_RTN_DETAIL_ON) 
		inner += (1*(attrib->font->ascent + attrib->font->descent) +
			1*(attrib->sfont->ascent + attrib->sfont->descent));

	outer = 3*(attrib->font->ascent + attrib->font->descent) +
		1*(attrib->sfont->ascent + attrib->sfont->descent);

#ifdef XPRINT
	if (per->pctxt) {
		/* Print */
		width = per->wid - BORDER;
		height = per->ht - BORDER;
		draw = XtWindow(w);
	} else
#endif
	{
		/* Display in a window */
		setBusyCursor(w, True);

		height = accounts*outer + accounts*times*inner + 
			(attrib->sfont->ascent + attrib->sfont->descent);
		width = REPORT_CHAR_WIDTH*attrib->avgWidth;

		/* Create pixmap */
		draw = XCreatePixmap (dpy, RootWindowOfScreen (XtScreen(w)),
			width+BORDER, height+BORDER,
			DefaultDepthOfScreen (XtScreen(w)));
	}

#ifdef DEBUG
	fprintf(stderr, 
		"accounts -> %d\n" "times    -> %d\n"
		"inner   -> %d  outer   -> %d\n"
		"attrib sfont ascent -> %d\n" "attrib sfont descent-> %d\n"
		"attrib font ascent  -> %d\n" "attrib font descent -> %d\n"
		"height   -> %d\n" "width    -> %d\n",
		accounts, times, inner, outer, 
		attrib->sfont->ascent, attrib->sfont->descent,
		attrib->font->ascent, attrib->font->descent, height, width);
#endif

	if (attrib->shadThick == 0)
		attrib->shadThick = 1;

#ifdef	XPRINT
	if (per->pctxt) {
		/* Have a different color allocation for printing */
		attrib->fg = per->color[13];	/* foreground */
		attrib->bg = per->color[14];	/* background */
		attrib->shad =
		attrib->tshad =
		attrib->hilite = per->color[15];	/* illustrations */
	} else
#endif
	{
		/* Clear background, copy header */
		XtVaGetValues(w, 
				XmNbackground,		&attrib->bg,
				XmNforeground,		&attrib->fg,
				XmNhighlightColor,	&attrib->hilite,
			NULL);

		if (isColorVisual(dpy)) {
			Pixel top, bot;

#ifdef DEBUG
			fprintf(stderr, "Cmap %p for XmGetColors()\n", GetColormap());
#endif

			XmGetColors(XtScreen(w), GetColormap(), attrib->bg,
				NULL, &top, &bot, NULL);
			if (isDarkPixel (dpy, attrib->fg)) {
				attrib->tshad = bot;
				attrib->shad = top;
			} else {
				attrib->tshad = top;
				attrib->shad = bot;
			}
		} else {
			attrib->shad = attrib->bg;
		}
	}

	if (attrib->gc) {
		gc = attrib->gc;
	} else {
		gc = XCreateGC(dpy, RootWindowOfScreen(XtScreen(w)), 0, NULL);
		XSetLineAttributes(dpy, gc, attrib->shadThick, 
			LineSolid, CapNotLast, JoinBevel);
	}

#ifdef	XPRINT
	if (per->pctxt) {
		/*
		 * Print : make it simple, just fill out the whole page with
		 * the background color.
		 */
		XSetForeground(dpy, gc, attrib->bg);
		XFillRectangle(dpy, draw, gc, 0, 0, per->wid, per->ht);
	} else
#endif
	{
		/* Window */
		/*
		 * Draw fancy bitmap on background, then other stuff inside,
	 	 * to give impression of having a report on a page.
		 */
		XSetTile(dpy, gc, GetPixmap(PSTIPPLE, NORMAL, dpy));
		XSetFillStyle(dpy, gc, FillTiled );
		XFillRectangle(dpy, draw, gc, 0, 0, width+BORDER, height+BORDER);
		XSetFillStyle(dpy, gc, FillSolid);
		XSetForeground(dpy, gc, attrib->bg);
		XFillRectangle(dpy, draw, gc, BORDER/2-4, BORDER/2-4, width+7, height+7); 
		XSetForeground(dpy, gc, attrib->shad);
		XDrawRectangle(dpy, draw, gc, BORDER/2-3, BORDER/2-3, width+4, height+4); 
		XSetForeground(dpy, gc, attrib->fg);
		XDrawRectangle(dpy, draw, gc, BORDER/2-3, BORDER/2-3, width+4, height+4); 
	}


	XSetLineAttributes(dpy, gc, attrib->shadThick, LineSolid, CapNotLast, JoinBevel);

	attrib->dpy = dpy;
	attrib->gc = gc;
	attrib->pix = draw;
	attrib->x = BORDER/2;
	attrib->y = BORDER/2;

	DEBUGXY;
	/* Write report */
	if (reportAccount & REPORT_ACTIVE_ACCOUNT) {
		reportSummary(activeAccount(), attrib);
		DEBUGXY;
		reportTimeVal(activeAccount(), attrib);
		DEBUGXY;
	}

	if (reportAccount & REPORT_ALL_ACCOUNTS) {
		if (per->count)
			i = per->count;
		else
			i = 1;
		for (; i <= numAccounts(); i++) {
#ifdef	XPRINT
			if (per->pctxt) {
				/*
				 * Figure out whether we have enough
				 * space left on the page.
				 */
				if (attrib->y + outer + times * inner > per->ht) {
					/* Start a new page */
					per->count = i;
					return 1;
				}
      			}
#endif

			reportSummary(i, attrib);
			DEBUGXY;
			reportTimeVal(i, attrib);
			DEBUGXY;
		}
	}

	/*
	 * Figure out whether we have enough space left on the page.
	 */
#ifdef	XPRINT
	if (per->pctxt && (attrib->y + outer + times * inner > per->ht)) {
		/* Start a new page */
		per->count = i;
		return 1;
	}
#endif

	if (reportAccount & REPORT_PORTFOLIO) {
		reportSummary(REPORT_PORTFOLIO_ACCOUNT, attrib);
		reportTimeVal(REPORT_PORTFOLIO_ACCOUNT, attrib);
	}

	/* Time stamp the report */
	sprintf(line, "%s: Generated %s", 
		version, utilDateOut(reportDateRange[0].end));
	XSetFont(attrib->dpy, attrib->gc, attrib->sfont->fid);

	/*
	 * When displaying in a window, draw text twice in different colors for
	 * visual style. Don't do this when printing.
	 */
#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
		XSetForeground(attrib->dpy, attrib->gc, attrib->shad);
		XDrawString (attrib->dpy, attrib->pix, attrib->gc,
			BORDER/2 + attrib->shadThick, 
			height + BORDER/2 - attrib->shadThick - attrib->sfont->descent,
			line, strlen (line));
	}

	XSetForeground(attrib->dpy, attrib->gc, attrib->fg);
	XDrawString(attrib->dpy, attrib->pix, attrib->gc,
		BORDER/2, 
		height + BORDER/2 - attrib->sfont->descent,
		line, strlen(line));

	width += BORDER;  height += BORDER;

  /* Resize drawing area (not when printing !). */
#ifdef	XPRINT
	if (per->pctxt == 0)
#endif
	{
		Dimension rheight, rwidth;
		Dimension dheight, dwidth;

		XtVaGetValues(XtParent(reportDrawing), 
				XmNwidth, &dwidth,
				XmNheight, &dheight, 
			NULL);
		XtVaGetValues(reportDialog,
				XmNwidth, &rwidth,
				XmNheight, &rheight,
			NULL);

		XtVaSetValues(reportDrawing,
				XmNwidth, width,
				XmNheight, height,
			NULL);

		/* Resize dialog to make it fit if allowed to */
		if ((int)client_data == ALLOW) {
			/* Figure out which size to display */
			Dimension	dh, mh;

			/* Get the screen height */
			if (!per->scr)
				per->scr = XtScreen(reportDialog);
			mh = XHeightOfScreen(per->scr) - 80;
			dh = height + BORDER + rheight - dheight;
			if (dh > mh) {
				dh = mh;
			} else {
				/* We already know DH fits inside the screen. */
				/* We know rheight and dheight differ by an interesting amount */
				/* Add that to the bitmap height and resize to that. */
				dh = height + rheight - dheight;
			}
			XtVaSetValues(reportDialog, 
				XmNwidth, width + (rwidth-dwidth), 
				XmNheight, dh,
				NULL);
		}
	}

#ifdef	XPRINT
	if (! per->pctxt)
#endif
	{
		/* Remove initializer, replace with copier */
		XtRemoveCallback(w, XmNexposeCallback,
			(XtCallbackProc)reportDraw, (XtPointer)ALLOW);

		/* Save for redraw */
		XtVaSetValues(w, XmNuserData, attrib, NULL);

		/* Write to screen */
		XCopyArea(dpy, draw, XtWindow(w), gc, 0, 0, width, height, 0, 0);
		XtAddCallback(w, XmNexposeCallback, 
	                 (XtCallbackProc)reportRedraw, (XtPointer)NULL);
		setBusyCursor(w, False);
	}

	/*
	 * No need to keep attrib if we're printing and we're done.
	 * In the window case, this is done in reportDestroy().
	 */
#ifdef	XPRINT
	if (per->pctxt) {
		XtFree((char *)attrib);
		per->attrib = 0;

		per->count = 0;
	}
#endif
	return 0;	/* We're done */
}


/* ARGSUSED */
void reportDestroy(Widget w, Widget client_data, XtPointer call_data)
{
  struct report_attrib *attrib;

  XtVaGetValues(client_data, XmNuserData, &attrib, NULL);
  per->attrib = 0;

  if (attrib) {
    if (attrib->pix)
      XFreePixmap(XtDisplay(w), attrib->pix);
    if (attrib && attrib->gc)
      XFreeGC(XtDisplay(w), attrib->gc);
    XtFree((char *)attrib);
  }
  XtDestroyWidget(GetTopShell(w));
  reportDialog = (Widget)NULL;
  reportDrawing = (Widget)NULL;
}

void reportPageCB(Widget w, Widget client, XtPointer call)
{
	Widget				drawing = (Widget)client;
	XmPrintShellCallbackStruct	*cbp = (XmPrintShellCallbackStruct *)call;
	int				i = 0;

	per = &Global[1];
	
	if (per->busy) {
		/* This is not the first page */
#ifdef DEBUG
		fprintf(stderr, "Print page (count %d)\n", per->count);
#endif
		i=reportDraw(w, client, call);
	} else {
		/* This is the first page */
#ifdef DEBUG
		fprintf(stderr, "Print first page\n");
#endif
		i = reportDraw(drawing, (XtPointer)ALLOW, NULL);
	}

	if (i) {
		per->busy = 1;
		cbp->last_page = False;
	} else {
		per->busy = 0;
		cbp->last_page = True;
	}

	per = &Global[0];
}

void reportPrintOk(Widget w, Widget client, XtPointer call)
{
	/* Print */
#ifdef	XPRINT
#ifdef	HAVE_LIBDT
	DtPrintSetupCallbackStruct	*pbs = call;
	Screen				*scr;
	Display				*pdpy;
	unsigned short			wid, ht;
	XRectangle			rect;
	Widget				ps, drawing;
	Arg				al[5];
	int				ac;

	/* Get the info from the widget */
	pdpy = pbs->print_data->print_display;
	if (!pdpy) {
		fprintf(stderr, "Xprint error: no display\n");
		return;
	}
	scr = DefaultScreenOfDisplay(pdpy);

	per = &Global[1];

	per->dpy = pdpy;
	per->win = RootWindowOfScreen(scr);
	per->_cmap = per->cmap = DefaultColormapOfScreen(scr);
	per->pctxt = pbs->print_data->print_context;
	per->gc = XCreateGC(per->dpy, per->win, 0, NULL);

	if (! XpGetPageDimensions(pdpy, per->pctxt, &wid, &ht, &rect)) {
		/*
		 * FIX ME how do I indicate failure in a decent way ?
		 */
		fprintf(stderr, "Couldn't figure out paper size\n");
		return;
	}

	/* Page size */
	per->wid = wid;
	per->ht = ht;
#if 0
	/*
	 * Before this can work, we need to create appdata.print_font_list
	 * by means of the print Display, not the video display.
	 */

  /* Get fonts for printing */
	if (XmFontListInitFontContext(&fontContext, appdata.print_font_list) == True) {
		XmFontListEntry entry;
		XmFontType      fonttype =  XmFONT_IS_FONT;
		char *tag;

		while ((entry = XmFontListNextEntry(fontContext)) != NULL) {
			tag = XmFontListEntryGetTag ( entry );
			if ( strcmp( tag, "small") == 0 )
			    per->small = (XFontStruct *)XmFontListEntryGetFont(entry, &fonttype);
			if ( strcmp( tag, "large") == 0 )
			    per->large = (XFontStruct *)XmFontListEntryGetFont(entry, &fonttype);

			if ( tag != NULL )
			    XtFree (tag);
		}

		XmFontListFreeFontContext ( fontContext );
	}
#else
	/* Hack for now */
	per->small = XLoadQueryFont(pdpy,
		"-adobe-courier-medium-r-normal--*-60-300-300-p-*-iso8859-1"
	);
	per->large = XLoadQueryFont(pdpy,
		"-adobe-courier-medium-r-normal--*-120-300-300-p-*-iso8859-1");
#endif

	ac = 0;
	XtSetArg(al[ac], XmNheight, ht); ac++;
	XtSetArg(al[ac], XmNwidth, wid); ac++;
	per->Toplevel = ps = XmPrintSetup(w, scr, "printShell", al, ac);

	AllocateAllColors();

	per->cmap = per->_cmap = InitColor(per->Toplevel, per->cmap, per->color);
#ifdef DEBUG
	fprintf(stderr, "Print dft cmap %p\n", per->cmap);      /* Danny */
#endif

	InitPixmaps(ps, DefaultColormapOfScreen(XtScreen(ps)));

	XpStartJob(XtDisplay(ps), XPSpool);	/* FIX ME for print to file */

	/* Create a widget, draw in it */
	drawing = XtVaCreateWidget("Graph", xmDrawingAreaWidgetClass, ps, NULL);
	XtManageChild(drawing);
	XtAddCallback(ps, XmNpageSetupCallback,
		(XtCallbackProc)reportPageCB, drawing);

	XtRealizeWidget(ps);
	per = &Global[1];
#endif
#endif
}

void reportPrint(Widget w, Widget client, XtPointer call)
{
#ifdef	HAVE_LIBDT
	Widget	d;
	Arg	al[10];
	int	ac;
	Widget	drawing = (Widget)client;

	ac = 0;
	d = DtCreatePrintSetupDialog(w, "ReportPrintDialog", al, ac);
	XtManageChild(d);
	XtAddCallback(d, DtNprintCallback,
		(XtCallbackProc)reportPrintOk, drawing);
#endif
}

/* ARGSUSED */
void reportGenerate (Widget w, XtPointer client_data, XtPointer calldata)
{
  Widget scroll;
  Widget pane, form, button;

  int i, resize = DISALLOW;

  time_t t = time ((time_t *)NULL);
  struct tm *today = localtime (&t);
  char datebuf[40];
  long ldate;

  /*
  ** Set time ranges 
  */
  strftime (datebuf, 20, "%m/%d/%Y", today);
  ldate = calStrtoL (datebuf);
  sprintf (datestr, " - %.30s", utilDateOut (ldate));

  for (i=0; i < REPORT_TIMES; i++) {
    reportDateRange[i].end = ldate;
 
    switch (i) {
      case 0:  reportDateRange[i].start = ldate - 1L; break;
      case 1:  reportDateRange[i].start = ldate - 7L; break;
      case 2:  reportDateRange[i].start = ldate - 30L; break;
      case 3:  reportDateRange[i].start = ldate - (365L/4L); break;
      case 4:  reportDateRange[i].start = ldate - (long) (today->tm_yday); 
               break;
      case 5:  reportDateRange[i].start = ldate - 365L; break;
      case 6:  reportDateRange[i].start = ldate - 2*365L; break;
      case 7:  reportDateRange[i].start = ldate - 3*365L; break;
      case 8:  reportDateRange[i].start = ldate - 5*365L; break;
      case 9:  reportDateRange[i].start = ldate - 10*365L; break;
      case 10: reportDateRange[i].start = FIRST_DATE; break;
      case 11: if (Customstart == -1L) {
                 reportDateRange[i].start = calCaltoL(&reportCal);
                 reportDateRange[i].end = Customend;
               } else if (Customend == -1L) {
                 reportDateRange[i].start = Customstart;
                 reportDateRange[i].end = calCaltoL(&reportCal);
               } else {
                 reportDateRange[i].start = Customstart;
                 reportDateRange[i].end = Customend;
               }
               break;
    }
  }

  /* Check custom time for errors if enabled */
  i = TIME_CUSTOM;
  if (reportTime & (1 << i)) {
    if (reportDateRange[i].start >= reportDateRange[i].end) {
      write_status ("Custom report period end date\n"
                    "must be later than start date.",
                    ERR);
      return;
    }
  }

  /* 
  ** Create scrolled window dialog if in single window mode and none exists,
  ** or multi-window mode. 
  */
  if (((reportValue & REPORT_SINGLEWIN_ON) && reportDialog == (Widget)NULL)
      || !(reportValue & REPORT_SINGLEWIN_ON)) {
    reportDialog = XtVaCreatePopupShell ("Report", xmDialogShellWidgetClass,
                            GetTopShell (w), NULL );
    resize = ALLOW;
    pane = XtVaCreateWidget ("ReportPane", 
                           xmPanedWindowWidgetClass, reportDialog,
                           XmNsashWidth, 1,
                           XmNsashHeight, 1,
                           NULL ); 
    scroll = XtVaCreateManagedWidget ("ReportScroll", 
                           xmScrolledWindowWidgetClass, pane,
                           XmNscrollingPolicy, XmAUTOMATIC,
                           XmNvisualPolicy,    XmCONSTANT,
                           NULL);
    reportDrawing = XtVaCreateManagedWidget ("ReportDrawing", 
                           xmDrawingAreaWidgetClass, scroll,
                           NULL );
    XtAddCallback( reportDrawing, XmNexposeCallback,
                 (XtCallbackProc) reportDraw, (XtPointer) ALLOW);

    /* Dialog control */
    form = XtVaCreateWidget ("form", xmFormWidgetClass, pane,
                           XmNfractionBase, 7,
                           XmNskipAdjust, True,
                           NULL );
    button = XtVaCreateManagedWidget ("Ok", xmPushButtonWidgetClass, form,
			XmNtopAttachment,    XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNleftAttachment,   XmATTACH_POSITION,
			XmNleftPosition,     2,
			XmNrightAttachment,  XmATTACH_POSITION,
			XmNrightPosition,    3,
			XmNshowAsDefault,    True,
			XmNdefaultButtonShadowThickness, 1,
		NULL );
    XtAddCallback (button, XmNactivateCallback,
                 (XtCallbackProc) reportDestroy, reportDrawing);

    button = XtVaCreateManagedWidget("Print", xmPushButtonWidgetClass, form,
			XmNtopAttachment,    XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNleftAttachment,   XmATTACH_POSITION,
			XmNleftPosition,     4,
			XmNrightAttachment,  XmATTACH_POSITION,
			XmNrightPosition,    5,
			XmNshowAsDefault,    True,
			XmNdefaultButtonShadowThickness, 1,
		NULL );
#ifdef	HAVE_LIBDT
    XtAddCallback(button, XmNactivateCallback,
		(XtCallbackProc)reportPrint, reportDrawing);
#else
    XtSetSensitive(button, False);
#endif

    XtManageChild (form);
    XtManageChild (pane);
  }

  XtManageChild (reportDialog);
  XtPopup (reportDialog, XtGrabNone);

  /* Force new drawing if in single window mode */
  if (reportDrawing && reportValue & REPORT_SINGLEWIN_ON)
    reportDraw (reportDrawing, (XtPointer)resize, (XtPointer)NULL);

}
