/*
 * audioIO_alsa.c  alsa lowlevel acess
 *
 * Copyright (C) 1997-1998 Masaki Chikama (Wren) <chikama@kasumi.ipl.mech.nagoya-u.ac.jp>
 *               1998-                           <masaki-c@is.aist-nara.ac.jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
*/

#include <stdio.h>
#include <stdlib.h>
#include <sys/asoundlib.h>
#include "audio.h"


#if SND_PCM_VERSION > SND_PROTOCOL_VERSION(1,0,1) || SND_LIB_VERSION >= SND_PROTOCOL_VERSION(0,5,0)
static snd_pcm_t *pcm_handle;
#else
static void *pcm_handle;
#endif

#if SND_PROTOCOL_MAJOR(SND_PCM_VERSION)>1 || SND_LIB_VERSION >= SND_PROTOCOL_VERSION(0,5,0)
#define MODE SND_PCM_MODE_BLOCK
#define CHANNEL SND_PCM_CHANNEL_PLAYBACK
int flush_f=0;
unsigned char slice=0;

int audioOpen(DSPFILE *dfile, WAVFILE *wfile, char *device) {
	unsigned int bps,size;
	snd_pcm_format_t fmt;
	snd_pcm_channel_info_t info;
	snd_pcm_channel_params_t param;
	snd_pcm_channel_setup_t setup;

	if (snd_pcm_open(&pcm_handle, atoi(device), dfile->fd, SND_PCM_OPEN_PLAYBACK) < 0) {
		fprintf(stderr, "audioOpen(): Opening audio device %s failed\n", device);
		return -1;
	}
	//fprintf(stderr,"audoIO_alsa: open pid %d\n",getpid());

	memset(&info,0,sizeof(info));
	info.channel=CHANNEL;
	if (snd_pcm_plugin_info(pcm_handle, &info) < 0) {
		fprintf(stderr, "audioOpen(): param get failed\n");
		return -1;
	}

	memset(&fmt,0,sizeof(fmt));
	fmt.rate     = wfile->wavinfo.SamplingRate;
	fmt.format   = wfile->wavinfo.DataBits == 8 ?  SND_PCM_SFMT_U8 : SND_PCM_SFMT_S16_LE;
	slice = wfile->wavinfo.DataBits == 8 ? 0x80 : 0;
	fmt.voices = wfile->wavinfo.Channels == Stereo ? 2 : 1;
	fmt.interleave = 1;

	memset(&param,0,sizeof(param));
	param.mode=MODE;
	param.channel=CHANNEL;
	param.start_mode = SND_PCM_START_FULL;
	param.stop_mode = SND_PCM_STOP_STOP;
	memcpy(&param.format,&fmt,sizeof(snd_pcm_format_t));

	param.buf.block.frag_size = info.max_fragment_size/4;
	param.buf.block.frags_max = -1;
	param.buf.block.frags_min = 1;
	if( snd_pcm_plugin_params(pcm_handle,&param)<0 ) {
		fprintf(stderr, "audioOpen(): unable to set channel params\n");
		return -1;
	}
	snd_pcm_plugin_flush(pcm_handle,CHANNEL);
	if (snd_pcm_plugin_prepare(pcm_handle,CHANNEL ) < 0) {
		fprintf(stderr, "audioOpen(): unable to prepare channel\n");
		return -1;
	}
	memset(&setup,0,sizeof(setup));
	setup.mode=MODE;
	setup.channel=CHANNEL;
	if( snd_pcm_plugin_setup(pcm_handle,&setup)<0 ) {
		fprintf(stderr, "audioOpen(): unable to obtain setup\n");
		return -1;
	}
	
	dfile->dspblksiz = setup.buf.block.frag_size;
	if ( (dfile->dspbuf = (char *) malloc(dfile->dspblksiz)) == NULL ) {
		fprintf(stderr, "audioOpen(): For DSP I/O buffer\n");
		return -1;
	}
	flush_f=0;
	return 0;
}
int audioClose(void) {
	int err,bc;
	snd_pcm_channel_status_t ps;
	
	if( pcm_handle ) {
		memset(&ps,0,sizeof(ps));
		if( (err=snd_pcm_plugin_status(pcm_handle,&ps))< 0 )
		{
			fprintf(stderr,"playback status err 1: %s\n",snd_strerror(err));
			return snd_pcm_close(pcm_handle);
		}

		while( ps.count )
		{
			memset(&ps,0,sizeof(ps));
			bc=ps.count;
			snd_pcm_plugin_status(pcm_handle,&ps);
			if( bc == ps.count ) snd_pcm_plugin_flush(pcm_handle,CHANNEL);
			usleep(1000);
		}
		//fprintf(stderr,"audoIO_alsa: close pid %d\n",getpid());
		return snd_pcm_close(pcm_handle);
	} else {
		return 0;
	}
}
int audioWrite(DSPFILE *dfile, int cnt) {
	int ret;
	snd_pcm_channel_status_t ps;
    int i;
    int err;
	snd_pcm_channel_status_t status;
    memset(&ps,0,sizeof(ps));
    if( (err=snd_pcm_plugin_status(pcm_handle,&ps))< 0 )
    {
        fprintf(stderr,"playback status err 1: %s\n",snd_strerror(err));
    }
//    printf("playback status %d %d\n",ps.free,ps.underrun);
	if( cnt < dfile->dspblksiz )
		memset(dfile->dspbuf+cnt,slice,dfile->dspblksiz-cnt);
	ret=snd_pcm_plugin_write(pcm_handle, dfile->dspbuf, dfile->dspblksiz);
    
	if( ret == dfile->dspblksiz ) return cnt;
	if( ret < 0 ) fprintf(stderr,"audioWrite: err %s\n",snd_strerror(ret));
	memset(&status,0,sizeof(status));
	if (snd_pcm_plugin_status(pcm_handle, &status) < 0) {
		fprintf(stderr, "ALSA: could not get channel status %d\n",ret);
	} else {
		if( status.status != SND_PCM_STATUS_RUNNING &&
			status.status != SND_PCM_STATUS_PREPARED )
			snd_pcm_plugin_prepare(pcm_handle,CHANNEL);
	}
	return ret;
}
void audioFlush(DSPFILE *dfile) {
	snd_pcm_channel_status_t ps;
    int i;
    int err;
	snd_pcm_channel_status_t status;
    memset(&ps,0,sizeof(ps));

    if( (err=snd_pcm_plugin_status(pcm_handle,&ps))< 0 )
    {
        fprintf(stderr,"playback status err 1: %s\n",snd_strerror(err));
    }
//    printf("playback status %d %d\n",ps.free,ps.underrun);
	if( ps.status ==  SND_PCM_STATUS_READY&& flush_f ) snd_pcm_plugin_flush(pcm_handle,CHANNEL); else flush_f=1;
}

/* ALSA  0.6.0  0.5.0 κۼ */
#if SND_LIB_VERSION >= SND_PROTOCOL_VERSION(0,6,0)
#  define SND_STATUS snd_pcm_stream_status_t
#  define SND_GET_STATUS snd_pcm_stream_status
#  define SND_BYTES bytes_used
#else /* 0.5.0 */
#  define SND_STATUS snd_pcm_channel_status_t
#  define SND_GET_STATUS snd_pcm_channel_status
#  define SND_BYTES count
#endif /* 0.6.0 */

int audioRest(DSPFILE *dfile) {

	SND_STATUS status;
	int err;

	memset(&status,0,sizeof(status));
	if ((err=SND_GET_STATUS(pcm_handle, &status)) < 0) {
		fprintf(stderr, "ALSA:(pid %d) could not get channel status %s\n",getpid(),snd_strerror(err));
		return 0;
	};
	return status.SND_BYTES;
}

/* äޥ̵ˤ */
#undef SND_STATUS
#undef SND_GET_STATUS
#undef SND_BYTES


#else
int audioOpen(DSPFILE *dfile, WAVFILE *wfile, char *device) {
	snd_pcm_format_t fmt;
	snd_pcm_playback_info_t info;

	if (snd_pcm_open(&pcm_handle, atoi(device), dfile->fd, SND_PCM_OPEN_PLAYBACK) < 0) {
		fprintf(stderr, "audioOpen(): Opening audio device %s failed\n",				device);
		return -1;
	}

	fmt.rate     = wfile->wavinfo.SamplingRate;
	fmt.channels = wfile->wavinfo.Channels == Stereo ? 2 : 1;
	fmt.format   = wfile->wavinfo.DataBits == 8 ?  SND_PCM_SFMT_U8 : SND_PCM_SFMT_S16_LE;
	if (snd_pcm_playback_format(pcm_handle, &fmt) < 0) {
		fprintf(stderr, "audioOpen(): format set failed\n");
		return -1;
	}

	if (snd_pcm_playback_info(pcm_handle, &info) < 0) {
		fprintf(stderr, "audioOpen(): param get failed\n");
		return -1;
	}

	dfile->dspblksiz = info.max_fragment_size; 
	if ( (dfile->dspbuf = (char *) malloc(dfile->dspblksiz)) == NULL ) {
		fprintf(stderr, "audioOpen(): For DSP I/O buffer\n");
		return -1;
	}

	return 0;
}
int audioClose(void) {
	int err;
	snd_pcm_playback_status_t ps;

	if( pcm_handle ) {
		if( (err=snd_pcm_playback_status(pcm_handle,&ps))< 0 )
		{
			fprintf(stderr,"playback status err 1: %s\n",snd_strerror(err));
			return snd_pcm_close(pcm_handle);
		}

		while( ps.queue )
		{
			snd_pcm_playback_status(pcm_handle,&ps);
			if( ps.queue ) snd_pcm_flush_playback(pcm_handle);
			usleep(1000);
		}
		return snd_pcm_close(pcm_handle);
	} else {
		return 0;
	}
}
int audioWrite(DSPFILE *dfile, int cnt) {
	int ret;

	ret=snd_pcm_write(pcm_handle, dfile->dspbuf, cnt);
	if( ret < 0 ) fprintf(stderr,"audioWrite: err %s\n",snd_strerror(ret));
	return ret;
}
void audioFlush(DSPFILE *dfile) {
	snd_pcm_flush_playback(pcm_handle);
}

int audioRest(DSPFILE *dfile) {
	return 0;
}
#endif

int audioCheck(DSPINFO *info, char *device) {
	/* ???? */
	info->Available = TRUE;
	return OK;
}
void audioStop(void) {
	int err;

	if( pcm_handle)
    {
#if SND_LIB_VERSION >= SND_PROTOCOL_VERSION(0,5,0)
        if( (err=snd_pcm_plugin_playback_drain(pcm_handle)) < 0 ) {
#elif SND_PROTOCOL_MAJOR(SND_PCM_VERSION)>1
		if( (err=snd_pcm_plugin_drain_playback(pcm_handle)) < 0 ) {
#else
		if( (err=snd_pcm_drain_playback(pcm_handle)) < 0 ) {
#endif
			fprintf(stderr,"drain err: %s\n",snd_strerror(err));
		}
	}
}

#include "audioMix_alsa.c"

