/*-
 * @(#)gdkmm_image.cc -- Gdk--'s Gdk_Image with TrueType font
 */

/*
 * This class defines Gdk_Image_FreeType and some extensions.
 * -- Kazunori Ueno <jagarl@creater.club.ne.jp>
 */

#include <gdk--/types.h>
#include <gdk--/colormap.h>
#include <gdk--/gc.h>
#include <gdk/gdk.h>

/* we need memcpy() */
#include <stdlib.h>
#include <stdio.h>

#include "gdkmm_image.h"

extern "C" {
    #include "chconv.h"
}

Gdk_Image_FreeType::Gdk_Image_FreeType(GdkImageType type, Gdk_Visual &visual, 
				       gint width, gint height)
    : Gdk_Image(type, visual, width, height)
{
}

Gdk_Image_FreeType::Gdk_Image_FreeType(const Gdk_Image& image)
    : Gdk_Image(image)
{
}

Gdk_Image_FreeType::~Gdk_Image_FreeType()
{
}

static int CalcBpp(Gdk_Image* orig) {
	GdkImage* im = orig->gdkobj();
	GdkVisual* vis = im->visual;
	if (vis == NULL) vis = gdk_visual_get_system();
	int bpp = 16;
	if (vis->depth == 16) {
		if (vis->red_mask == 0x7c00 &&
			vis->green_mask == 0x3e0 &&
			vis->blue_mask == 0x1f) {
			bpp= 15;
		}
	} else {
		bpp = (im->bpl / im->width) * 8;
	}
	return bpp;
}

static unsigned short bpp15_table1[0x20*3];
static unsigned short bpp15_table2[0x20*3];
static unsigned short bpp15_table3[0x20*3];
static unsigned short bpp16_table1[0x20*3];
static unsigned short bpp16_table2[0x40*3];
static int is_init = 0;
#define bpp16_table3 bpp15_table3
static void InitTable(void) {
	if (is_init) return;
	int i; for (i=0; i<3; i++) {
		int n = (2-i)*0x20; int x = 0; // x == j*(i+1)
		int j; for (j=0; j<0x20; j++) {
			int y = x>>2;
			bpp15_table1[j+n] = y << 10;
			bpp15_table2[j+n] = y << 5;
			bpp15_table3[j+n] = y;
			bpp16_table1[j+n] = y << 11;
			x += i+1;
		}
		n = (2-i)*0x40; x = 0; // x == j*(i+1)
		for (j=0; j<0x40; j++) {
			bpp16_table2[j+n] = (x>>2) << 5;
			x += i+1;
		}
	}
	is_init = 1;
}

void
Gdk_Image_FreeType::draw_marged_text(Gdk_Font_FreeType &font,
				     gint x, gint y, gint code, guint32 pixel)
{
    if (!font.is_usable_freetype())
	return;
    InitTable();

    char *bitmap; char *bitmap_orig;
    int	width, height; int fbpl;
    bitmap_orig = font.get_bitmap(code, width, height);
    bitmap = bitmap_orig; fbpl = width;

    /* ɸ */
    if (x<0) {
	width += x; bitmap -= x; x = 0;
    }
    if (y<0) {
        height += y; bitmap -= y*fbpl; y=0;
    }
    if (x+width > gdkobj()->width) {
	width = gdkobj()->width - x;
    }
    if (y+height > gdkobj()->height) {
	height = gdkobj()->height - y;
    }
    if (width <= 0 || height <= 0) return;

    unsigned int bpl = gdkobj()->bpl;
    char *mem = (char *)gdkobj()->mem;
    mem += x * (bpl/gdkobj()->width) + y * bpl;

    int bpp = CalcBpp(this);
	unsigned int color;
	unsigned int color_table[5]; // alpha ͤȤ˷׻

	unsigned int cr = (pixel>>16)&0xff;
	unsigned int cg = (pixel>>8)&0xff;
	unsigned int cb = (pixel)&0xff;
	if (bpp == 15) {
		color = ((cr&0xf8)<<7) | ((cg&0xf8)<<2) | ((cb&0xf8)>>3);
		color_table[0] = 0;
		color_table[1]=
			((cr&0xe0)<<5) | ((cg&0xe0)) | ((cb&0xe0)>>5);
		color_table[2]=
			((cr&0xf0)<<6) | ((cg&0xf0)<<1) | ((cb&0xf0)>>4);
		color_table[3] = color_table[1] + color_table[2];
		color_table[4] = color;
	} else if (bpp == 16) {
		color = ((cr&0xf8)<<8) | ((cg&0xfc)<<3) | ((cb&0xf8)>>3);
		color_table[0] = 0;
		color_table[1]=
			((cr&0xe0)<<6) | ((cg&0xf0)<<1) | ((cb&0xe0)>>5);
		color_table[2]=
			((cr&0xf0)<<7) | ((cg&0xf8)<<2) | ((cb&0xf0)>>4);
		color_table[3] = color_table[1] + color_table[2];
		color_table[4] = color;

	} else { /* bpp == 24 / 32 */
		color = pixel;
		color_table[0] = 0;
		color_table[1]=
			((cr&0xfc)<<14) | ((cg&0xfc)<<6) | ((cb&0xfc)>>2);
		color_table[2]=
			((cr&0xfe)<<15) | ((cg&0xfe)<<7) | ((cb&0xfe)>>1);
		color_table[3] = color_table[1] + color_table[2];
		color_table[4] = color;
	}
	
	if (bpp == 15) {
		for (int j=0; j<height; j++) {
			char* bitmap_ptr = bitmap + j * fbpl;
			unsigned short* image_ptr = (unsigned short*)(mem + j * bpl);
			for (int i=0; i<width; i++) {
				char m = *bitmap_ptr;
				if (m == 0) ;
				else if (m == 4) *image_ptr = color;
				else {
					unsigned short d = color_table[m];
					m = (m-1)*0x20;
					unsigned short s = *image_ptr;
					s = bpp15_table1[m + (s>>10)&0x1f] |
						bpp15_table2[m + (s>>5)&0x1f] |
						bpp15_table3[m + (s)&0x1f];
					// s += d;
					unsigned int m = ( ((s&d)<<1) + ( (s^d)&0x7bde ) ) & 0x8420;
					m = ( ((m&0x8420)>>5) + 0x3def) ^ 0x3def;
					*image_ptr = (s + d - m) | m;
				}
				bitmap_ptr++; image_ptr++;
			}
		}
	} else if (bpp == 16) {
		for (int j=0; j<height; j++) {
			char* bitmap_ptr = bitmap + j * fbpl;
			unsigned short* image_ptr = (unsigned short*)(mem + j * bpl);
			for (int i=0; i<width; i++) {
				char m = *bitmap_ptr;
				if (m == 0) ;
				else if (m == 4) *image_ptr = color;
				else {
					unsigned short d = color_table[m];
					m = (m-1)*0x20;
					unsigned short s = *image_ptr;
					s = bpp16_table1[m + (s>>11)&0x1f] |
						bpp16_table2[m*2 + (s>>5)&0x3f] |
						bpp16_table3[m + (s)&0x1f];
					// s += d;
					unsigned int m = ( ((s&d)<<1) + ( (s^d)&0xf7de ) ) & 0x10820;
					m = ( (((m*3)&0x20840)>>6) + 0x7bef) ^ 0x7bef;
					*image_ptr = (s + d - m) | m;
				}
				bitmap_ptr++; image_ptr++;
			}
		}
	} else if (bpp == 24) {
		for (int j=0; j<height; j++) {
			char* bitmap_ptr = bitmap + j * fbpl;
			char* image_ptr = (mem + j * bpl);
			for (int i=0; i<width; i++) {
				char m = *bitmap_ptr;
				if (m == 0) ;
				else if (m == 4) {
					image_ptr[0]=cb; image_ptr[1]=cg; image_ptr[2]=cr;
				}else {
					image_ptr[0] = (int(*(unsigned char*)(image_ptr+0) )*(4-m) + cb*m)>>2; 
					image_ptr[1] = (int(*(unsigned char*)(image_ptr+1) )*(4-m) + cb*m)>>2; 
					image_ptr[2] = (int(*(unsigned char*)(image_ptr+2) )*(4-m) + cb*m)>>2; 
				}
				bitmap_ptr++; image_ptr+=3;
			}
		}
	} else if (bpp == 32) {
		for (int j=0; j<height; j++) {
			char* bitmap_ptr = bitmap + j * fbpl;
			char* image_ptr = (mem + j * bpl);
			for (int i=0; i<width; i++) {
				char m = *bitmap_ptr;
				if (m == 0) ;
				else if (m == 4) {
					image_ptr[0]=cb; image_ptr[1]=cg; image_ptr[2]=cr;
				}else {
					image_ptr[0] = (int(*(unsigned char*)(image_ptr+0) )*(4-m) + cb*m)>>2; 
					image_ptr[1] = (int(*(unsigned char*)(image_ptr+1) )*(4-m) + cb*m)>>2; 
					image_ptr[2] = (int(*(unsigned char*)(image_ptr+2) )*(4-m) + cb*m)>>2; 
				}
				bitmap_ptr++; image_ptr+=4;
			}
		}
	}

    delete[] bitmap_orig;
}

void
Gdk_Image_FreeType::draw_image(Gdk_Image &image, gint xsrc, gint ysrc,
			       gint xdest, gint ydest, gint width, gint height)
{
    /* width or height not specified */
    if (width == -1)
	width = image.gdkobj()->width - xsrc;
    if (height == -1)
	height = image.gdkobj()->height - ysrc;

    /* minus value assigend */
    if (xsrc < 0)
	{ xdest += -xsrc; width -= -xsrc; xsrc = 0; }
    if (ysrc < 0)
	{ ydest += -ysrc; height -= ysrc; ysrc = 0; }
    if (xdest < 0)
	{ xsrc += -xdest; xdest = 0; }
    if (ydest < 0)
	{ ysrc += -ydest; ydest = 0; }

    /* destination is larger than source */
    if (width + xsrc > image.gdkobj()->width)
	width = gdkobj()->width - xsrc;
    if (height + ysrc > gdkobj()->height)
	height = image.gdkobj()->height - ysrc;
    if (width + xdest > gdkobj()->width)
	width = gdkobj()->width - xdest;
    if (height + ydest > gdkobj()->height)
	height = gdkobj()->height - ydest;

    /* no width or height */
    if (width < 0 || height < 0)
	return;

    int src_bpl  = image.gdkobj()->bpl;
    int dest_bpl = gdkobj()->bpl;
    int src_bpp  = src_bpl / image.gdkobj()->width;
    int dest_bpp = dest_bpl / gdkobj()->width;

    char *src_data  = (char *)image.gdkobj()->mem;
    char *dest_data = (char *)gdkobj()->mem;

    src_data += src_bpp * xsrc + src_bpl * ysrc;
    dest_data += dest_bpp * xdest + dest_bpl * ydest;

    for (int i = 0; i < height; i++) {
	 memcpy(dest_data, src_data, dest_bpp * width);
	 dest_data += dest_bpl;
	 src_data += src_bpl;
    }
}

/* gdkmm_image.cc ends here */
