/*
 * Copyright (c) 1997, 1999, 2000, 2001, Mark Buser.
 * Copyright (c) 2001, 2002, 2003, 2004, Danny Backx.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the names the authors (see above), nor the names of other
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /pack/anoncvs/xinvest/src/optnet.c,v 1.41 2004/12/23 19:55:42 danny Exp $
 */
#include <sys/types.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>

#include <X11/Xos.h>
#include <Xm/XmAll.h>

#include "about.h"
#include "optdetail.h"
#include "optnet.h"
#include "opttick.h"
#include "optupdate.h"
#include "parse.h"
#include "remoteXq.h"
#include "server.h"
#include "status.h"
#include "tape.h"
#include "xutil.h"
#include "xquote.h"

#ifdef XQUOTE
#include "xquote.h"
#else
#include "xinvest.h"
#endif

#define	REREAD_LMT	4

/* Globals */
int netFinished = True;
XtWorkProcId work = (XtWorkProcId)NULL;
int	reread_lmt = REREAD_LMT;

/*
** Net option support
*/
static void netTellXinvest ()
{
  Display *dpy = XtDisplay (per->Toplevel);
  int i;
  char *price;
  char date[11];

  double nav;
  time_t t = time ( (time_t *)NULL ); 
  struct tm *now = localtime(&t);

  char **commands = NULL;
  int num_commands = 0;

#if !defined (NO_XMU)
  write_status_line ("Contacting Xinvest...");
  XmUpdateDisplay(per->Toplevel);

  /* 
  ** Send "NAV ticker mm/dd/yyyy fffff.fff" for each ticker.
  */ 
  for (i=0; i < tickGetNum(); i++) {
    QUERY_STRUCT *data = tickGetDetail(i);
    if (data && data->values[DETAIL_PRICE]) {

      price = data->values[DETAIL_PRICE];
      if (strcmp (price, "N/A") == 0) {     /* no price */
        continue;
      } else if (strchr (price, '/')) {     /* fraction or decimal */
	int whole, num, den;
	sscanf( price, "%d %d / %d", &whole, &num, &den);
	nav = (double)whole + ((double)num/(double)den);
      } else {
	char *p;
	/* Try to figure out European/American number notation */
	if (strchr(price, '.')) {		/* American notation */
		nav = strtod (price, NULL);
	} else if (p = strchr(price, ',')) {	/* Europe */
		*p = '.';
		nav = strtod (price, NULL);
		*p = ',';
	} else {				/* Dunno, don't bother */
		nav = strtod (price, NULL);
	}
	
      }

      /* Dates from the net could be anything, punt and use today */
      strftime ( date, 11, "%m/%d/%Y", now);

      /* Add new command plus NULL command */
      commands = (char **)XtRealloc ( (char *)commands, 
		                      (num_commands+2) * sizeof(char *));
      commands[num_commands] = XtCalloc ( 40, sizeof(char) );
      sprintf ( commands[num_commands++], "NAV %s %s %.3f", 
		tickGetName(i), date, nav);
      commands[num_commands] = NULL;
    }
  }

  if (num_commands)
    xinvest_remote_commands ( dpy, commands);

  for (i=0; i < num_commands; i++) 
    XtFree (commands[i]);
  XtFree ((char *)commands);
#endif
}

char *makeQuery (int proxy, int tick)
{
  char *url, *name, *x = NULL;
  char *insert, *nohost, *host;
  static char buf[BUFSIZ];

  if (tick != WORK_NEWS) {
    setCurServer (tickGetServer(tick));
    setCurServerType (tickGetType(tick));
    name = tickGetName (tick);
  } else {
    setCurServer (NEWS);
    setCurServerType (0);
    name = "";
  }
  url = getServer(DETAIL_URL);
  insert = strstr (url, "SYMBOL");
  nohost = strchr (url, '/');

  host = malloc(strlen(url)+2);
  strcpy(host, url);
  x = strchr(host, '/');
  if (x) *x = '\0';
  x = NULL;

#ifdef DEBUG
fprintf(stderr, "makeQuery(%s,%s)\n", url, name);
#endif

  if (strchr(name, ' ')) {
	int	i, cnt, j;
	for (i=0,cnt=0; name[i]; i++)
		if (name[i] == ' ')
			cnt++;
	x = malloc(i+2*cnt+8);
	for (i=0,j=0; name[i]; i++) {
		if (name[i] == ' ') {
			x[j++] = '%';
			x[j++] = '2';
			x[j++] = '0';
		} else {
			x[j++] = name[i];
		}
	}
	x[j] = 0;
  }
  if (insert && nohost) {
    if (proxy)
      sprintf ( buf, "GET http://%.*s%s%s HTTP/1.0\r\n"
			"Host: %s\r\n"
                     "User-Agent: Xquote/2.0\r\n"
                     "Accept: */*\r\n"
                     "\r\n",
  	        insert-url, url,
		x ? x : name,
		insert + strlen("SYMBOL"),
		host);
    else
      sprintf ( buf, "GET %.*s%s%s HTTP/1.0\r\n"
			"Host: %s\r\n"
                     "User-Agent: Xquote/2.0\r\n"
                     "Accept: */*\r\n"
                     "\r\n",
  	        insert-nohost, nohost,
		x ? x : name,
		insert + strlen("SYMBOL"),
		host);

  } else 
    strcpy (buf, "");

#ifdef DEBUG
  fprintf (stderr, buf);
#endif

  if (x) free(x);
  free(host);

  return (buf);
}


static int readQuery(int s, char **html) 
{
  int num;
  int size = 0;
  char buf[BUFSIZ];
	  
  if (*html)
    size = strlen(*html);
  else
    size = 0;

  /* Read the page */
  if ( (num = read (s, buf, BUFSIZ-1)) > 0) {
    buf[num] = '\0';
    *html = XtRealloc ( *html, (size + num +1)*sizeof(char) );
    if (*html) {
      if (size == 0)
        strcpy ( *html, buf);
      else
        strcat ( *html, buf);
    } else {
      write_status ("Could not allocate memory for HTML.", ERR);
      return (0);
    }
  }

#ifdef DEBUG
  if (*html) fprintf (stderr, "ReadQuery(%s) -> [%s]\n", *html, buf);
#endif

  return (num);
}


#define IDLE    0
#define SOCKET  1
#define CONNECT 2
#define READ    3
#define WRITE   4
#define PARSE   5
#define TEST    6
#define UPDATE  7
#define END     8

int workNet (int command)
{
  static int state = IDLE;
  static int i = 0;

  static int s;
  int status;
  static struct sockaddr_in serv_addr;
  struct servent *sp;
  struct hostent *hp;
  char **name;
  int size;

  static int proxy;
  static char *serv_machine;
  static int port;

  static char *html;
  static int reread_cnt;

  if (command == WORK_CLEAN)
    state = END;

  if (command == UPDATE)
	  state = UPDATE;

  AllowShellResize (per->Toplevel, DISALLOW);
#ifdef DEBUG
  fprintf(stderr, "workNet(%s)\n",
		  (command == WORK_CLEAN) ? "WORK_CLEAN" :
		  (state == IDLE) ? "IDLE" :
		  (state == SOCKET) ? "SOCKET" :
		  (state == CONNECT) ? "CONNECT" :
		  (state == WRITE) ? "WRITE" :
		  (state == READ) ? "READ" :
		  (state == UPDATE) ? "UPDATE" :
		  (state == END) ? "END" :
		  (state == PARSE) ? "PARSE" :
		  (state == TEST) ? "TEST" : "??");
#endif

  switch (state) {

    case IDLE:
      cylonStart (True);

      setCurServer ( (command==WORK_NEWS)?NEWS:tickGetServer(i) );
      setCurServerType ( (command==WORK_NEWS)?0:tickGetType(i) );

      /* 
      ** Determine server and port
      */
      proxy = True;
      getProxy (&serv_machine, &port);
      if (serv_machine == NULL) {                    /* No proxy */
        size = strcspn ( getServer(DETAIL_URL), "/");       /* Get machine from URL */
        serv_machine = XtCalloc ( size+1, sizeof(char));
        sprintf ( serv_machine, "%.*s", size, getServer(DETAIL_URL));

        if ( (sp = getservbyname ("http", "tcp")) )  /* Get port from lookup */
          port = ntohs(sp->s_port);
        else
          port = 80;                                 /* or default */
        proxy = False;
      }
      sprintf (errmsg, "Connecting to %s...", serv_machine);
      write_status_line (errmsg);

      state = SOCKET;
      netFinished = False;
      return (False);
      /* break; */

    case SOCKET:
      /* 
      ** Get server address struct.
      */
      memset ( (char *)&serv_addr, 0, sizeof(serv_addr) );
      serv_addr.sin_port = htons(port);

      if ((serv_addr.sin_addr.s_addr = inet_addr(serv_machine)) != -1) {
        /* Dotted number address */
        serv_addr.sin_family = AF_INET;   

        s = socket (serv_addr.sin_family, SOCK_STREAM, 0);
        if ( s < 0 ) {
	  sprintf (errmsg, "Socket create failed:\n%.*s", 
		   (int)XtNumber(errmsg)-23, strerror(errno));
          write_status (errmsg, ERR);
          state = END;
          return (False);
        }
      } else {
        /* Dotted name address */
        if ((hp = gethostbyname(serv_machine)) == NULL) {
          sprintf(errmsg, "Unknown host: '%s'.\n", serv_machine);
          write_status ( errmsg, ERR);
          state = END;
          return (False);
        }
        serv_addr.sin_family = hp->h_addrtype;
 
#ifdef h_addr
        for (s = 0, name = hp->h_addr_list; *name; name++) {
          memcpy((char *) &serv_addr.sin_addr, *name, hp->h_length);

          if ((s = socket (serv_addr.sin_family, SOCK_STREAM, 0)) >= 0) 
            break;     /* Got one */
        }
#else
        memcpy((char *) &serv_addr.sin_addr, hp->h_addr, hp->h_length);
        s = socket (serv_addr.sin_family, SOCK_STREAM, 0);
        if ( s < 0 ) {
	  sprintf (errmsg, "Socket create failed:\n%.*s", 
		   XtNumber(errmsg)-23, strerror(errno));
          write_status (errmsg, ERR);
          state = END;
          return (False);
        }
#endif
      }
 
      /* Set non blocking, not fatal if we can't */
      if ( (status = fcntl (s, F_GETFL)) != -1 ) {
	status |= O_NONBLOCK;
        fcntl (s, F_SETFL, status);
      }

      state = CONNECT;
      return (False);
      /* break; */

    case CONNECT:
      /*
      ** Connect to http server 
      */
      if (connect(s, (struct sockaddr *)&serv_addr, sizeof (serv_addr)) < 0) {
	if (errno == EISCONN)
          state = WRITE;
        else if (errno == EALREADY || errno == EINPROGRESS || errno == EAGAIN)
          state = CONNECT;
	else {
          state = END;
	  sprintf (errmsg, "Socket connect failed:\n%.*s", 
		   (int)XtNumber(errmsg)-24, strerror(errno));
          write_status (errmsg, ERR);
          close(s);
        }
      } else
        state = WRITE;

      if (state == WRITE) {
        if (command == WORK_NEWS) {
	  html = makeQuery ( proxy, WORK_NEWS);
	} else {
	  html = makeQuery ( proxy, i);
	}
      }

      return (False);
      /* break; */

    case WRITE:

      status = write (s, html, strlen(html));
      if (status < 0) {                     /* ERROR */
	if (errno == EAGAIN) {    /* no, just can't write yet */
	  state = WRITE;
        } else {
	  sprintf (errmsg, "Write failed:\n%.*s", 
		 (int)XtNumber(errmsg)-15, strerror(errno));
	  write_status (errmsg, ERR);
	  html = NULL;
	  state = END;
	}
      } else if (status == strlen(html)) {  /* DONE */
        sprintf ( errmsg, "Reading %s...", (command!=WORK_NEWS) ? tickGetName(i):"Xquote News");
        write_status_line (errmsg);
        html = NULL;
        state = READ;
      } else {                              /* NOT DONE */
	html += status;
	state = WRITE;
      }

      /* Set last update to current time */
      set_current_time();

      return (False);
      /* break; */

    case READ:
      /* Read the query response */
      status = readQuery (s, &html);
      if (status < 0) {                  /* ERROR */
	if (errno == EAGAIN) {    /* no, just no data yet */
	  state = READ;
        } else {
	  sprintf (errmsg, "Read failed:\n%.*s", 
		 (int)XtNumber(errmsg)-14, strerror(errno));
	  write_status (errmsg, ERR);
	  XtFree (html);
	  html = NULL;
	  state = END;
	}
      } else if (status == 0) {  /* DONE */
	char *start, *end;
	int code, num;

        /* Check for error codes 'HTTP/1.0 200 OK\r\n' */
	if (html) {
	  sscanf ( html, "HTTP/%*f %d %n", &code, &num);
	  if (code != 200) {
		  if (code == 400) {
			  /* attempt to deal with code 400 bad request by re-reading */
			  if ( reread_cnt >= reread_lmt ) {    /* enough already give it up */
				  start = html + num;
				  if ( (end = strchr ( start, '\r')) == NULL)
					  end = start + 20;
				  sprintf ( errmsg, "%s: re-read limit reached on %d %.*s", 
					  (command!=WORK_NEWS) ? tickGetName(i):"Xquote News",
					code, end-start, start);
				  write_status (errmsg, ERR);
				  close(s);
				  state = TEST;
				  XtFree(html);
				  html = NULL;
				  /* Should there be a new state set here ? */
				  return (False);
			  } else {
				/* limit not reached try again */
				++reread_cnt;
				/*
				** go back to SOCKET and re do complete html query
				*/
				state = SOCKET;
				close(s);
				XtFree(html);
				html = NULL;
				/*
				 * FIX ME should probably implement a simple backoff
				 * delay here, to ensure an already overloaded server
				 * isn't getting even more overloaded.
				 */
				return (False);
				/*break;*/
			  }
		  } else {    /* code not 400 and not 200 */
		    start = html + num;
		    if ( (end = strchr ( start, '\r')) == NULL)
		      end = start + 20;
		    sprintf ( errmsg, "%s: %d %.*s", 
			(command!=WORK_NEWS) ? tickGetName(i):"Xquote News",
			code, end-start, start);
		    write_status (errmsg, ERR);
		    reread_cnt = 0;
		    /* Should there be a new state set here ? */
	          }
	  }
	}
	    reread_cnt = 0;
        state = PARSE;
      } else {  /* status != 0 */                /* MAYBE MORE */
        state = READ;
      }
      return (False);
      /* break; */

    case PARSE:
      close (s);
      if (command != WORK_NEWS) {
        if (html)
          tickSetDetail ( i, findPattern(html) );
        state = TEST;
      } else {
        if (html)
          aboutText( findNews(html) );
        state = END;
      }
      XtFree(html);
      html = NULL;

      return (False);
      /* break; */

    case TEST:
      if ( ++i < tickGetNum() ) {
	if (tickGetServer(i) != getCurServer())
	  state = IDLE;
	else
          state = SOCKET;
	/*
	 * Get Xquote to update its display whenever it gets a quote,
	 * not after it got them all.
	 */
#if 0
        detailUpdateMainWindow(DETAIL_UPDATE);
#endif
      } else {
        detailUpdateMainWindow(DETAIL_UPDATE);
        makeTextTape();
        state = UPDATE;
      }
      return (False);
      /* break; */

    case UPDATE:
      netTellXinvest();
      state = END;
      return (False);
      /* break; */

    case END:
    default:
      if (!proxy)
        XtFree (serv_machine);

      strcpy(errmsg, time_msg);
      write_status_line (errmsg);

      cylonStart (False);

      state = IDLE;
      i = 0;
      netFinished = True;
      work = (XtWorkProcId) NULL;
      return (True);
      /* break; */
  }
}
#undef IDLE
#undef SOCKET
#undef CONNECT
#undef READ
#undef PARSE
#undef TEST
#undef UPDATE
#undef END


/* ARGSUSED */
void procNet (Widget w, XtPointer which, XtPointer call_data)
{
  static Widget proxy = (Widget) NULL, port, retries;
  static char *server, *portno;
  char *s;

  if (proxy == (Widget)NULL) {
    proxy = XtNameToWidget (GetTopShell(w),
                            "NetPane.NetRow.NetProxy.form_0.text_0");
    port = XtNameToWidget  (GetTopShell(w),
                            "NetPane.NetRow.NetProxy.form_0.text_1");
    retries = XtNameToWidget(GetTopShell(w), "NetPane*NetRetries*text_0");
  }

  switch ((int) which) {

    case 0: /* Ok */
	    { 
              XtPopdown (GetTopShell(w));
              /* 
              ** PROXY
              */
	      if (server) XtFree (server);
	      if (portno) XtFree (portno);
	      server = XmTextFieldGetString (proxy);
	      portno = XmTextFieldGetString (port);
	      if (strlen(server)) {
                if (strlen(portno))
                  setProxy (server, portno);
	      } else {
                setProxy (NULL, "80");
              }
	      /*
	      ** RETRIES
	      */
	      s = XmTextFieldGetString(retries);
	      if (strlen(s)) {
		reread_lmt = atoi(s);
	      }
	      XtFree(s);
	    }
            break;

    case 1: /* Cancel */
            XtPopdown (GetTopShell(w));

	    /* Restore previous values */
	    if (server)
	      XmTextFieldSetString (proxy, server);
	    if (portno)
	      XmTextFieldSetString (port, portno);
	    s = XtMalloc(10);
	    sprintf(s, "%d", reread_lmt);
	    XmTextFieldSetString(retries, s);
	    XtFree(s);
	    
            break;

    default: break;
  }
}

Widget createNetDialog ()
{
  char *frames[] =  {"NetProxy", "NetRetries" };
  char *forms[] =   {"form_0"};
  char *buttons[] = {"button_0", "button_1"};

  Widget dialog, form, pane, row, frame, button, label, text;
  int num;

  Dimension width, height, border;

  dialog = XtVaCreatePopupShell ("OptionNet", 
                           xmDialogShellWidgetClass,
                           GetTopShell(per->Toplevel),
                           NULL);

  pane = XtVaCreateWidget ("NetPane", xmPanedWindowWidgetClass, 
                           dialog,
                           XmNsashWidth, 1,
                           XmNsashHeight, 1,
                           NULL);

  row = XtVaCreateWidget ("NetRow", xmRowColumnWidgetClass, pane,
                          XmNnavigationType, XmNONE,
                          XmNtopAttachment, XmATTACH_FORM,
                          XmNleftAttachment, XmATTACH_FORM,
                          NULL);

  for (num = 0; num < XtNumber(frames); num ++) {
    
     frame =   XtVaCreateManagedWidget( frames[num],
                          xmFrameWidgetClass, row,
                          NULL);
     XtVaCreateManagedWidget ( frames[num],
                          xmLabelGadgetClass, frame,
                          XmNchildType, XmFRAME_TITLE_CHILD,
                          XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
                          NULL);
     form = XtVaCreateWidget (forms[num], xmFormWidgetClass, frame, NULL );

     switch (num) {
     case 0:	/* Proxy */
     {
         char *host, port[12];
	 int portno;

	 getProxy (&host, &portno);
	 if (host)
	   sprintf ( port, "%d", portno );

           label = XtVaCreateManagedWidget ( "label_0",
                            xmLabelGadgetClass, form,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
           text = XtVaCreateManagedWidget ( "text_0",
                            xmTextFieldWidgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, label,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
	   if (host)
	     XmTextFieldSetString (text, host);

           label = XtVaCreateManagedWidget ( "label_1",
                            xmLabelGadgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, text,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
           text = XtVaCreateManagedWidget ( "text_1",
                            xmTextFieldWidgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, label,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
	   if (host)
	     XmTextFieldSetString (text, port);
     }
     break;
     case 1:	/* Retry count */
     {
	char	scount[10];
	text = XtVaCreateManagedWidget ( "text_0",
		xmTextFieldWidgetClass, form,
		XmNrightAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_FORM,
		XmNbottomAttachment, XmATTACH_FORM,
		NULL );
	label = XtVaCreateManagedWidget ( "label_0",
		xmLabelGadgetClass, form,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_WIDGET,
		XmNrightWidget,		text,
		XmNrightOffset,		5,
		XmNalignment,		XmALIGNMENT_BEGINNING,
		NULL );
	sprintf(scount, "%d", reread_lmt);
	XmTextFieldSetString (text, scount);
     }

	break;
     }

     XtManageChild (form);

  }

  XtManageChild (row);

  /* Buttons to add delete cancel exit */
  form = XtVaCreateWidget ( "ButForm", xmFormWidgetClass, pane,
                            XmNfractionBase, 5,
                            NULL );

  for (num=0; num < XtNumber(buttons); num++) {
    button = XtVaCreateManagedWidget ( buttons[num], 
                                       xmPushButtonWidgetClass, form,
                                       XmNtopAttachment, XmATTACH_FORM,
                                       XmNbottomAttachment, XmATTACH_FORM,
                                       XmNleftAttachment, XmATTACH_POSITION,
                                       XmNleftPosition, 2*num+1,
                                       XmNrightAttachment, XmATTACH_POSITION,
                                       XmNrightPosition, 2*num+2,
                                       XmNshowAsDefault, (num==0)?True:False,
                                       XmNdefaultButtonShadowThickness, 1,
                                       NULL );
    XtAddCallback ( button, XmNactivateCallback,
                            (XtCallbackProc) procNet, (XtPointer)num );
  }
  XtManageChild (form);
  XtManageChild (pane);

  /* Prevent pane from changing size */
  XtVaGetValues ( dialog, 
                  XmNwidth, &width,
                  XmNheight, &height,
                  XmNborderWidth, &border,
                  NULL );

  XtVaSetValues ( dialog,
                  XmNminWidth,  width +  border,
                  XmNmaxWidth,  width +  border,
                  XmNminHeight, height + border,
                  XmNmaxHeight, height + border,
                  NULL );
  return (dialog);
}
