/*
 *
 * Copyright (c) 1997 Charles D. Cranor.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following condition
 * is met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
/*
 * xuvmstat.c
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/time.h>
#if (__NetBSD_Version__ < 105000000)
#include <vm/vm.h>
#endif
#include <uvm/uvm_extern.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "libcdcx.h"

void get_uvmexp __P((struct uvmexp *));
void redraw __P((struct xdpy *, Window, GC, int));


u_long black, white, red, green, blue, magenta, orange, purple;
XFontStruct *fnt_fixed, *fnt_8x13bold;
struct uvmexp cexp, oexp;	/* current, old */
struct timeval tv, otv;
struct graph *gp_faults, *gp_traps, *gp_intrs, *gp_ctx;

#define WIDTH 300

main(argc, argv)

int argc;
char **argv;

{
  struct xdpy xdpy;
  Window win;
  int x,y,w,h, depth;
  Visual *visual = CopyFromParent;
  Colormap colormap;
  GC gc;
  fd_set fds;

  if (open_display(NULL, &xdpy) == NULL) 
    err(1, "open_display failed");

  if (setup_visual(&xdpy, xdpy.screen, &visual, &depth) != True) 
    errx(1, "setup_visual: unable to find pseudocolor/truecolor visual");

  x = y = 10;
  w = WIDTH;
  h = 685;
  win = open_window(&xdpy, xdpy.rootwin, x, y, w, h,
		    xdpy.black, xdpy.white, ExposureMask, visual, depth);

  set_stdhints(&xdpy, win, argv[0], argv[0], x, y, w, h);

  if (setup_colormap(&xdpy, xdpy.screen, win, visual, &colormap) != True) 
    errx(1, "setup_colormap failed");

  black = create_color(&xdpy, colormap, "black", xdpy.black);
  white = create_color(&xdpy, colormap, "white", xdpy.white);
  red = create_color(&xdpy, colormap, "red", black);
  green = create_color(&xdpy, colormap, "forestgreen", black);
  blue = create_color(&xdpy, colormap, "blue", black);
  magenta = create_color(&xdpy, colormap, "magenta", black);
  orange = create_color(&xdpy, colormap, "darkorange2", black);
  purple = create_color(&xdpy, colormap, "purple", black);

  gc = create_gc(&xdpy, win, xdpy.black, xdpy.white);

  fnt_fixed = create_font(&xdpy, "fixed", "fixed");
  fnt_8x13bold = create_font(&xdpy, "8x13bold", "fixed");

  XMapRaised(xdpy.dpy, win);
  XFlush(xdpy.dpy);

  get_uvmexp(&cexp);
  gettimeofday(&tv, NULL);
  get_uvmexp(&oexp);
  gettimeofday(&otv, NULL);

  {
    int count = 0, xfd = ConnectionNumber(xdpy.dpy), retval;
    struct timeval now, incr, ping, timer;
    XEvent event;
   
    incr.tv_sec = 0;
    incr.tv_usec = 500000;
    gettimeofday(&now, NULL);
    timeradd(&now, &incr, &ping);
    

    while (1) {
      FD_ZERO(&fds);
      FD_SET(xfd, &fds);
      gettimeofday(&now, NULL);
      if (timercmp(&now, &ping, <)) {
        timersub(&ping, &now, &timer);
        retval = select(xfd+1, &fds, NULL, NULL, &timer);
      } else {
        retval = 0;
      }

      if (retval < 0) {
        if (errno == EINTR) continue;
        err(1, "select");
      }

      if (retval == 0) {
        timeradd(&now, &incr, &ping);
        otv = tv;
        oexp = cexp;
        get_uvmexp(&cexp);
        gettimeofday(&tv, NULL);
        /* NEW GRAPH DATA HERE */

      } else {

        XNextEvent(xdpy.dpy, &event);

      }

      if (retval == 0 || event.type == Expose) {
        redraw(&xdpy, win, gc, retval == 0);
      }
    }
  }

  XFreeFont(xdpy.dpy, fnt_fixed);
  XCloseDisplay(xdpy.dpy);
}

/*
 * draw the screen
 */

void redraw(xdpy, win, gc, was_timeout)

struct xdpy *xdpy;
Window win;
GC gc;
int was_timeout;

{
  char buf[512], *tp;
  time_t tim;
  int y = 0;


  XSetFont(xdpy->dpy, gc, fnt_8x13bold->fid);

  tim = time(0);
  tp = ctime(&tim);
  snprintf(buf, sizeof(buf), "UVM Status at %.20s", tp);
  XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_8x13bold) + 5, 
	      buf, strlen(buf));
  y = get_fontheight(fnt_8x13bold) + 5 + 5;
  XDrawLine(xdpy->dpy, win, gc, 0, y, WIDTH, y);
  y += 5;

  XSetFont(xdpy->dpy, gc, fnt_fixed->fid); 
  snprintf(buf, sizeof(buf),"The UVM system manages %d %d-byte pages",
	   cexp.npages, cexp.pagesize);
  XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
  y += get_fontheight(fnt_fixed) + 5;

  {
    static char *names[] = { "active", "inactive", "free", "wired", "allocd" };
    u_long vals[5];
    static u_long colors[5], init = 0;

    if (init == 0) {
      init = 1;
      colors[0] = red;
      colors[1] = blue;
      colors[2] = green;
      colors[3] = orange;
      colors[4] = purple;
    }
 
    vals[0] = cexp.active;
    vals[1] = cexp.inactive;
    vals[2] = cexp.free;
    vals[3] = cexp.wired;
    vals[4] = cexp.npages - (vals[0]+vals[1]+vals[2]+vals[3]);
    y = draw_barbox(xdpy, win, gc, fnt_fixed, 5, 295, y, 5, cexp.npages, names,
		  vals, colors, white);
  }

  y += 8;
  {
    static char *names[] = { "file", "anon", "exec", "free", "kernel" };
    u_long vals[5];
    static u_long colors[5], init = 0;
    
    if (init == 0) { 
      init = 1;
      colors[0] = red;
      colors[1] = blue;
      colors[2] = green;
      colors[3] = orange; 
      colors[4] = purple;
    } 
  
    vals[0] = cexp.filepages;
    vals[1] = cexp.anonpages;
    vals[2] = cexp.execpages;
    vals[3] = cexp.free;
    vals[4] = cexp.npages - (vals[0]+vals[1]+vals[2]+vals[3]);
    y = draw_barbox(xdpy, win, gc, fnt_fixed, 5, 295, y, 5, cexp.npages, names,
                  vals, colors, white);
  }

  y += 8;

  {
    static char *names[] = { "freemin", "freetarget", "free" };
    u_long vals[3];
    u_long colors[3];

    vals[0] = cexp.freemin;  colors[0] = black;
    vals[1] = cexp.freetarg; colors[1] = black;
    vals[2] = cexp.free;     colors[2] = green;

    y = draw_barlvl(xdpy, win, gc, fnt_fixed, 5, 295, y, 3, cexp.npages, names,
		vals, colors, white);
  }

  y += 8;

  {
    static char *names[] = { "inactive target", "inactive" };
    u_long vals[2];
    u_long colors[2];

    vals[0] = cexp.inactarg;  colors[0] = black;
    vals[1] = cexp.inactive;  colors[1] = blue;

    y = draw_barlvl(xdpy, win, gc, fnt_fixed, 5, 295, y, 2, cexp.npages, names,
		vals, colors, white);
  }

  y += 8;

  {
    static char *names[] = { "wired max", "wired" };
    u_long vals[2];
    u_long colors[2];

    vals[0] = cexp.wiredmax;  colors[0] = black;
    vals[1] = cexp.wired;     colors[1] = orange;

    y = draw_barlvl(xdpy, win, gc, fnt_fixed, 5, 295, y, 2, cexp.npages, names,
		vals, colors, white);
  }

  y += 8;

  {
    static int init = 0;

    if (init == 0) {
      static char *names[] = { "faults", "anon", "ancow", "obj", 
	"copy", "zero" };
      static int *before[6], *after[6];
      static u_long color[6];
      init = 1;
      before[0] = &oexp.faults; after[0] = &cexp.faults; color[0] = black;
      before[1] = &oexp.flt_anon; after[1] = &cexp.flt_anon; color[1] = red;
      before[2] = &oexp.flt_acow; after[2] = &cexp.flt_acow; color[2] = blue;
      before[3] = &oexp.flt_obj; after[3] = &cexp.flt_obj; color[3] = green;
      before[4] = &oexp.flt_prcopy; after[4] = &cexp.flt_prcopy; color[4] = orange;
      before[5] = &oexp.flt_przero; after[5] = &cexp.flt_przero; color[5] = purple;
      gp_faults = create_graph(6, "FAULTS", names, before, after, color, white);
    }  

    draw_graph(xdpy, win, gc, fnt_fixed, 3, y, 145, 140, 
		gp_faults, was_timeout);

  }

  {
    static int init = 0;

    if (init == 0) {
      static char *names[] = { "traps" };
      static int *before[1], *after[1];
      static u_long color[1];
      init = 1;
      before[0] = &oexp.traps; after[0] = &cexp.traps; color[0] = black;
      gp_traps = create_graph(1, "TRAPS", names, before, after, color, white);
    }  

    draw_graph(xdpy, win, gc, fnt_fixed, 152, y, 145, 140, 
		gp_traps, was_timeout);

  }

  y += 148;

  {
    static int init = 0;

    if (init == 0) {
      static char *names[] = { "intrs", "soft", "syscall" };
      static int *before[3], *after[3];
      static u_long color[3];
      init = 1;
      before[0] = &oexp.intrs; after[0] = &cexp.intrs; color[0] = black;
      before[1] = &oexp.softs; after[1] = &cexp.softs; color[1] = blue;
      before[2] = &oexp.syscalls; after[2] = &cexp.syscalls; color[2] = red;
      gp_intrs = create_graph(3, "INTRS", names, before, after, color, white);
    }  

    draw_graph(xdpy, win, gc, fnt_fixed, 3, y, 145, 140, 
		gp_intrs, was_timeout);

  }

  {
    static int init = 0;

    if (init == 0) {
      static char *names[] = { "cswtch", "swin", "swout" };
      static int *before[3], *after[3];
      static u_long color[3];
      init = 1;
      before[0] = &oexp.swtch; after[0] = &cexp.swtch; color[0] = black;
      before[1] = &oexp.swapins; after[1] = &cexp.swapins; color[1] = green;
      before[2] = &oexp.swapouts; after[2] = &cexp.swapouts; color[2] = red;
      gp_ctx = create_graph(3, "SWITCH/SWAP", names, before, after, color, white);
    }  

    draw_graph(xdpy, win, gc, fnt_fixed, 152, y, 145, 140, 
		gp_ctx, was_timeout);

  }

  y += 148;

  {
    XSetForeground(xdpy->dpy, gc, black);
    snprintf(buf, sizeof(buf), "Page Daemon counts:");
    XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
    y += get_fontheight(fnt_fixed) + 5;

    snprintf(buf, sizeof(buf), " woke=%d, revs=%d, swout=%d",
	cexp.pdwoke, cexp.pdrevs, cexp.pdswout);
    XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
    y += get_fontheight(fnt_fixed) + 1;

    snprintf(buf, sizeof(buf), " scans=%d, anon_scans=%d, obj_scans=%d",
	cexp.pdscans, cexp.pdanscan, cexp.pdobscan);
    XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
    y += get_fontheight(fnt_fixed) + 1;

    snprintf(buf, sizeof(buf), " busy=%d, freed=%d, pending=%d",
	cexp.pdbusy, cexp.pdfreed, cexp.pdpending);
    XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
    y += get_fontheight(fnt_fixed) + 1;

    snprintf(buf, sizeof(buf), " reactivate=%d, deactivate=%d, pageouts=%d",
	cexp.pdreact, cexp.pddeact, cexp.pdpageouts);
    XDrawImageString(xdpy->dpy, win, gc, 5, get_fontheight(fnt_fixed) + y,
		   buf, strlen(buf)); 
    y += get_fontheight(fnt_fixed) + 1;

  }
  
  /* DONE */
  XSetForeground(xdpy->dpy, gc, black);
  XFlush(xdpy->dpy); 
}
