/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>

#include <X11/Xlib.h>		/* GC and other */
#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>
#include <X11/Xatom.h>		
#include <X11/cursorfont.h>

#include <X11/Xaw/Box.h>	/*Athena Widget Set */
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Simple.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>

#include "types.h"
#include "xwave.h"
#include "menu.h"
#include "button.h"
#include "status.h"
#include "audio.h"
#include "audio_file.h"
#include "misc.h"
#include "xwave_widget.h"
#include "sample_settings.h"
#include "graphics.h"
#include "record_dialog.h"
#include "filebrowse.h"
#include "fileop.h"
#include "bitmaps/xwave.xbm"
#include "bitmaps/canvas.xbm"

static void setwave_action (Widget w,Next_Wave *nw, XEvent *event, 
			    Boolean *flg);
static void delete_toplevel (Widget w,Main_Data *md, XEvent *event, 
			     Boolean *flg);
static void reparent_canvas (Widget w,Next_Wave *nw, XReparentEvent *event, 
			     Boolean *flg);
static void reparent_toplevel (Widget w,Main_Data *md, XReparentEvent *event, 
			       Boolean *flg);
static void pop_canvas_menu (Widget w,Main_Data *md, XButtonEvent *event, 
			     Boolean *flg);
static void usage();
static void XtWarningHandler(String msg);

/* in this struct we have all data about open files */

Main_Data *MD;

AppResources app_resources;

static Cursor previewC[3];
static Atom delwin=None,protocols=None,focus=None;
static char **load_files;
static int no_load_files;

static XtResource resources[] = {
      {"version","Version",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,version),XtRImmediate,"0.0"},
      {"device","Device",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,device),XtRImmediate,"/dev/dsp"},
      {"wdir","WDir",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,wdir),XtRImmediate,"."},
      {"tdir","TDir",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,tdir),XtRImmediate,"/tmp"},
      {"wavemainfgl","WaveMainFgl",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,wavemainfgl),XtRImmediate,"blue"},
      {"wavemainfgr","WaveMainFgr",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,wavemainfgr),XtRImmediate,"red"},
      {"markcolor","MarkColor",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,markcolor),XtRImmediate,"green"},
      {"default_recname","Default_recname",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,default_recname),XtRImmediate,"record"},
      {"default_newname","Default_newname",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,default_newname),XtRImmediate,"unnamed"},
      {"playlinefg","PlayLineFg",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,playlinefg),XtRImmediate,"black"},
      {"err_overwrite","Err_overwrite",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_overwrite),XtRImmediate,"not defined"},
      {"err_openfile","Err_openfile",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_openfile),XtRImmediate,"not defined"},
      {"err_fileinuse","Err_fileinuse",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_fileinuse),XtRImmediate,"not defined"},
      {"err_modified","Err_modified",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_modified),XtRImmediate,"not defined"},
      {"err_16bit","Err_16bit",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_16bit),XtRImmediate,"not defined"},
      {"err_stereo","Err_stereo",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_stereo),XtRImmediate,"not defined"},
      {"err_44khz","Err_44khz",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_44khz),XtRImmediate,"not defined"},
      {"err_44khzstereo","Err_44khzstereo",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_44khzstereo),
	    XtRImmediate,"not defined"},
      {"err_permissions","Err_permissions",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_permissions),
	    XtRImmediate,"not defined"},
      {"err_notsupported","Err_notsupported",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_notsupported),XtRImmediate,
	    "not defined"},
      {"err_wrongformat","Err_wrongformat",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_wrongformat),XtRImmediate,
	    "not defined"},
      {"err_mem","Err_mem",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_mem),XtRImmediate,"not defined"},
      {"err_input","Err_input",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_input),XtRImmediate,"not defined"},
      {"err_bg","Err_bg",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_bg),XtRImmediate,"not defined"},
      {"err_nohome","Err_nohome",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_nohome),XtRImmediate,"not defined"},
      {"err_notmp","Err_notmp",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_notmp),XtRImmediate,"not defined"},
      {"err_notimplement","Err_notimplement",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,err_notimplement),XtRImmediate,
	    "not defined"},
      {"default_ext","Default_ext",XtRString,sizeof(String),
	    XtOffsetOf(AppResources,default_ext),XtRImmediate,"*wav"},
      {"maxmem","MaxMem",XtRInt,sizeof(int),
	    XtOffsetOf(AppResources,maxmem),XtRImmediate,(XtPointer)1000},
      {"cv_maxpoints","CV_Maxpoints",XtRInt,sizeof(int),
	    XtOffsetOf(AppResources,cv_maxpoints),XtRImmediate,(XtPointer)10},
      {"default_ftype","Default_ftype",XtRInt,sizeof(int),
	    XtOffsetOf(AppResources,default_comp),XtRImmediate,(XtPointer)0},
      {"default_comp","Default_comp",XtRInt,sizeof(int),
	    XtOffsetOf(AppResources,default_comp),XtRImmediate,(XtPointer)0},
      {"doubleplay","DoublePlay",XtRBoolean,sizeof(Boolean),
	    XtOffsetOf(AppResources,doubleplay),XtRImmediate,(XtPointer)True},
      {"raisecanvas","RaiseCanvas",XtRBoolean,sizeof(Boolean),
	    XtOffsetOf(AppResources,raisecanvas),XtRImmediate,(XtPointer)True}
};

static XrmOptionDescRec options[] = {
      { "-device",".device",XrmoptionSepArg,(XtPointer)"/dev/dsp"},
      { "-wdir",".wdir",XrmoptionSepArg,(XtPointer)"."},
      { "-maxmem",".maxmem",XrmoptionSepArg,(XtPointer)1000}
};


static String fallback_resources[]={
    NULL,
};

static void usage()
{
    printf("XWave Version %s:\n\n",VERSION);
    printf("-?, -h, -help, display this help.\n");
    printf("-device, the play/rec device (usually /dev/dsp).\n");
    printf("-wdir, set working directory.\n");
    printf("-maxmem, max. amount of memory to load whole sample in memory.\n");
    printf("If sample is greater than maxmem, we do everything on HDD.\n");
    exit(1);
}

void XtWarningHandler(String msg)
{
    if (strncmp(msg,"XtRemoveGrab",strlen("XtRemoveGrab"))==0) return;
    if (MD!=NULL) 
      warn_popup(MD->mw->form,msg);
    else printf("%s\n",msg);
}


void main (int argc, char *argv[])
{
    XtAppContext context;
    Widget toplevel,form,widget,graph;
    Pixmap xwIcon;
    XGCValues xgcv;
    Dimension width,height;
    Display *dpy;
    XSetWindowAttributes wattr;

   /*
#if defined (_LITTLE_ENDIAN) || BYTE_ORDER == LITTLE_ENDIAN
    printf("little_endian\n");
#endif
#if defined (_BIG_ENDIAN) || BYTE_ORDER == BIG_ENDIAN
    printf("big_endian\n");
#endif
    */
   
   MD=NULL;
    toplevel = XtVaAppInitialize (&context,
				  "XWave",                  	 
				  options, XtNumber(options), 	 
				  &argc, argv,            
				  fallback_resources,
				  NULL);
    
    XtGetApplicationResources(toplevel,(XtPointer)&app_resources,
			      resources,XtNumber(resources),NULL,0);
    
    if (argc>1) {
	int i;
	if (strcmp(argv[1],"-?")==0) usage();
	if (strcmp(argv[1],"-h")==0) usage();
	if (strcmp(argv[1],"-help")==0) usage();
	if (strcmp(argv[1],"-wdir")==0) usage();
	if (strcmp(argv[1],"-maxmem")==0) usage();
	if (strcmp(argv[1],"-device")==0) usage();
	no_load_files=argc-1;
	load_files=(char **) XtCalloc(sizeof(char**),argc);
	for (i=1;i<argc;i++) {
	    load_files[i-1]=argv[i];
	}
    }
    
    if (strcmp(app_resources.version,"0.0")==0) {
	initwarn_popup(toplevel,NORESOURCE);
    }
    
    if (strcmp(app_resources.version,VERSION)!=0) {
	initwarn_popup(toplevel,WRONGRESOURCE);
    }
    
    XtAppSetWarningHandler(context,(XtErrorHandler) XtWarningHandler);
    dpy=XtDisplay(toplevel);
    
    /*
     XSynchronize(dpy,TRUE); 
    */
    /* alloc mem for Main_Data (this is global) */
    
    if ((MD=calloc(sizeof(Main_Data),1))==NULL) {
	initwarn_popup(toplevel,app_resources.err_mem);
	exit(-1);
    }
    MD->mb=calloc(sizeof(Main_Bool),1);
    MD->mg=calloc(sizeof(Main_Graphic),1);
    MD->mw=calloc(sizeof(Main_Widget),1);
    MD->cd=calloc(sizeof(Clip_Data),1);
    if ((MD->mb==NULL)||(MD->mg==NULL)||(MD->mw==NULL)||(MD->cd==NULL)) {
	initwarn_popup(toplevel,app_resources.err_mem);
	exit(-1);
    }
    MD->nw=NULL;
    MD->wd=NULL;
    if ((MD->cd->wd=calloc(sizeof(Wave_Data),1))==NULL) {
	initwarn_popup(toplevel,app_resources.err_mem);
	exit(-1);
    }
    if ((MD->cd->wd->name=
	 calloc(strlen(app_resources.tdir)+strlen(CLIPFILE)+1,1))==NULL) {
	initwarn_popup(toplevel,app_resources.err_mem);
	exit(-1);
    }
    sprintf(MD->cd->wd->name,"%s/%s",app_resources.tdir,CLIPFILE);
    
    MD->mg->pixmap=None;
    MD->maxmem=app_resources.maxmem * 1024;
    if ((MD->mg->fbuf=(ushort*) malloc(MAXUSHORT))==NULL)  {
	initwarn_popup(toplevel,app_resources.err_mem);
	exit(-1);
    }
    
    /* get environment variables */
    
    if ((MD->home=getenv(HOME_ENV))==NULL) {
	initwarn_popup(toplevel,app_resources.err_nohome);
	exit(-1);
    }
#ifdef sun
    if (getenv(AUDIODEV_ENV)!=NULL) app_resources.device=getenv(AUDIODEV_ENV);
#endif
    /* set the tmp directory, should really have permissions to write here */
    if (!can_write(app_resources.tdir)) {
	if (can_write(TMP)) {
	    if (strlen(app_resources.tdir)>=strlen(TMP)) 
	      strcpy(app_resources.tdir,TMP);
	    else app_resources.tdir=XtNewString(TMP);
	} else {
	    if (can_write(MD->home)) app_resources.tdir=MD->home;
	    else {
		initwarn_popup(toplevel,app_resources.err_notmp);
		exit(-1);
	    }
	}
    }
    check_audio(MD->mb);
    if (!MD->mb->canplay) app_resources.device=XtNewString("no device");
    
    MD->no=0;
    MD->mb->playing=False;
    MD->mb->exposing=False;
    MD->mb->isclip=False;
    
    form = MW ("main_form",formWidgetClass,toplevel,NULL);        
    MD->mw->form=form;
    
    widget= MW ("menu_form",formWidgetClass,form,
		XtNtop,XawChainTop,XtNbottom,XawChainTop,
		XtNleft,XawChainLeft,XtNright,XawChainRight,
		NULL);
    
    create_topmenu(widget);
    
    widget=MW ("main_line",formWidgetClass,form,
	       XtNfromVert,widget,XtNborderWidth,2,
	       XtNtop,XawChainTop,XtNbottom,XawChainTop,
	       XtNleft,XawChainLeft,XtNright,XawChainRight,
	       NULL);
    
    widget=MW ("border_form",formWidgetClass,form,
	       XtNfromVert,widget,XtNborderWidth,0,
	       XtNtop,XawChainTop,XtNbottom,XawChainTop,
	       XtNleft,XawChainLeft,XtNright,XawChainLeft,
	       NULL);
    
    create_button(widget);
    
    graph=MW ("main_graph",simpleWidgetClass,form,
	      XtNfromVert,widget,XtNheight,PREVIEWHEIGHT,
	      XtNtop,XawChainBottom,XtNbottom,XawChainBottom,NULL);
    
    MD->mw->graph=graph;    
    xgcv.foreground=ConvertColor(graph,app_resources.wavemainfgl);
    MD->mg->gcl=XtGetGC(MD->mw->graph,GCForeground, &xgcv);
    xgcv.foreground=ConvertColor(graph,app_resources.wavemainfgr);
    MD->mg->gcr=XtGetGC(MD->mw->graph,GCForeground, &xgcv);
    xgcv.foreground=ConvertColor(graph,app_resources.markcolor);
    xgcv.function=GXxor;
    MD->mg->markgc=XtGetGC(MD->mw->graph,GCForeground|GCFunction,&xgcv);
    xgcv.foreground=ConvertColor(graph,app_resources.playlinefg);
    MD->mg->playgc=XtGetGC(MD->mw->graph,GCForeground, &xgcv);
    
    XtVaGetValues(MD->mw->form,XtNcursor,&previewC[CURSOR_DEFAULT],NULL);
    previewC[CURSOR_MARK]=XCreateFontCursor(dpy,XC_sb_h_double_arrow);
    previewC[CURSOR_CLOCK]=XCreateFontCursor(dpy,XC_watch);
    
    widget=MW ("status_form",formWidgetClass,form,
	       XtNfromVert,graph,XtNborderWidth,0,
	       XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
	       XtNleft,XawChainLeft,XtNright,XawChainRight,
	       NULL);
    
    create_status(widget);
    
    /* create ATOM WM_DELETE_WINDOW and WM_TAKEFOCUS, this is for the DELETE
     * MenuItem of WINDOWMANAGERS
     * the Properties are only set, if an REPARENT EVENT occurs
     * most of the WM use this to set an own window (i hope)		
     */
    
    delwin=XInternAtom(dpy, "WM_DELETE_WINDOW", TRUE);
    focus=XInternAtom(dpy, "WM_TAKE_FOCUS", TRUE);
    protocols = XInternAtom(dpy, "WM_PROTOCOLS", TRUE);

    XtAddEventHandler(toplevel, StructureNotifyMask, FALSE,
		      (XtEventHandler)reparent_toplevel, (XtPointer) MD);
    
    create_canvas_menu(toplevel,MD->mw);
    
    XtRealizeWidget(toplevel);
    
    /* create the file_dialog datas, now we can popup file_dialog very fast */
    create_file_dialog(toplevel);
    
    /* create the sample_dialog data */
    create_sample_dialog(toplevel);
    
    /* create the record_dialog data */
    create_record_dialog(toplevel);
    
    /* set the icon for main window */
    xwIcon = XCBFD(dpy,XtWindow(form),(char *)xwave_bits,
		   xwave_width, xwave_height);
    
    /* set min,max width and height and icon */
    XtVaSetValues(toplevel,XtNiconPixmap,xwIcon,NULL);
    XtVaGetValues(toplevel,XtNheight,&height,XtNwidth,&width,NULL);
    XtVaSetValues(toplevel,XtNminWidth,width,XtNmaxHeight,height,XtNminHeight,
		  height,NULL);
    
    XtAddEventHandler(graph, ExposureMask, FALSE, 
		      (XtEventHandler)expose_action, (XtPointer) MD);
    XtAddEventHandler(graph, StructureNotifyMask, FALSE, 
		      (XtEventHandler)config_action, (XtPointer) MD);
    XtAddEventHandler(graph, Button1MotionMask, FALSE, 
		      (XtEventHandler)mark_action, (XtPointer) MD);
    XtAddEventHandler(graph, ButtonPressMask, FALSE, 
		      (XtEventHandler)buttonp_action, (XtPointer) MD);
    XtAddEventHandler(graph, ButtonReleaseMask, FALSE, 
		      (XtEventHandler)buttonr_action, (XtPointer) MD);
    
    /* if server can backing store, we want backing store */
    wattr.backing_store=DoesBackingStore(DefaultScreenOfDisplay(dpy));
    XChangeWindowAttributes(dpy,XtWindow(graph),CWBackingStore,&wattr);
    
    XtAppMainLoop(context);
}    

void new_canvas(Widget w,Next_Wave *nw)
{
    Widget popup,form; 
    Pixmap cwIcon;
    static XtTranslations trans;
    XGCValues xgcv;
    Dimension width,height;
    Display *dpy;
    XSetWindowAttributes wattr;
    
    trans= XtParseTranslationTable("<Btn3Down>: XtMenuPopup(cv_menu)");
    popup=XtVaCreatePopupShell(nw->wd->name,topLevelShellWidgetClass,w,NULL);
    
    nw->cw->popup=popup;
    nw->cg->pixmap=None;
    
    form  = MW ("cv_main_form",formWidgetClass,popup,
		XtNresizable,TRUE,
		XtNborderWidth,0,
		NULL);
    
    
    nw->cw->graph= MW ("cv_graph",simpleWidgetClass,form,
		       XtNtranslations, trans,
		       XtNtop,XawChainTop,XtNbottom,XawChainBottom,
		       XtNleft,XawChainLeft,XtNright,XawChainRight,
		       NULL);
    
    nw->cw->scroll = MW ("cv_pos_scroll", scrollbarWidgetClass,form,
			 XtNfromVert,nw->cw->graph,
			 XtNorientation,XtorientHorizontal,
			 XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
			 XtNleft,XawChainLeft,XtNright,XawChainRight,
			 NULL);
    
    XtVaGetValues(nw->cw->graph,XtNwidth,&width,NULL);
    XtVaSetValues(nw->cw->scroll,XtNwidth,width,NULL);
    
    nw->cw->zscroll = MW ("cv_zoom_scroll", scrollbarWidgetClass,form,
			  XtNfromHoriz,nw->cw->graph,
			  XtNorientation,XtorientVertical,
			  XtNtop,XawChainTop,XtNbottom,XawChainBottom,
			  XtNleft,XawChainRight,XtNright,XawChainRight,NULL);
    
    XtVaGetValues(nw->cw->graph,XtNheight,&height,NULL);
    XtVaSetValues(nw->cw->zscroll,XtNheight,height,NULL);
    
    xgcv.foreground=ConvertColor(nw->cw->graph,app_resources.wavemainfgl);
    nw->cg->gcl=XtGetGC(nw->cw->graph,GCForeground, &xgcv);
    xgcv.foreground=ConvertColor(nw->cw->graph,app_resources.wavemainfgr);
    nw->cg->gcr=XtGetGC(nw->cw->graph,GCForeground, &xgcv);
    xgcv.foreground=ConvertColor(nw->cw->graph,app_resources.markcolor);
    xgcv.function=GXxor;
    nw->cg->markgc=XtGetGC(nw->cw->graph,GCForeground|GCFunction,&xgcv);
    
    XtAddEventHandler(popup, StructureNotifyMask, False,
		      (XtEventHandler)reparent_canvas, (XtPointer) nw);
    
    XtAddEventHandler(nw->cw->graph, ButtonPressMask|Button1MotionMask, False,
		      (XtEventHandler)mark_canvas, (XtPointer) MD);
    
    XtPopup(popup,XtGrabNone);
    
    XtAddEventHandler(nw->cw->graph, ExposureMask, False,
		      (XtEventHandler)expose_canvas, (XtPointer) MD);
    XtAddEventHandler(nw->cw->graph, StructureNotifyMask, False,
		      (XtEventHandler)config_canvas, (XtPointer) MD);
    
    XtVaGetValues(popup,XtNheight,&height,XtNwidth,&width,NULL);
    XtVaSetValues(popup,XtNminWidth,width/2,XtNminHeight,height/2,NULL);
    
    XtAddCallback(nw->cw->scroll, XtNjumpProc, jump_canvas, (XtPointer) MD);
    XtAddCallback(nw->cw->scroll, XtNscrollProc, scroll_canvas, (XtPointer) MD);
    
    XtAddCallback(nw->cw->zscroll, XtNjumpProc, zjump_canvas, (XtPointer) MD);
    XtAddCallback(nw->cw->zscroll, XtNscrollProc, zscroll_canvas, 
		  (XtPointer) MD);
    
    XtAddEventHandler(nw->cw->graph, ButtonPressMask, False,
		      (XtEventHandler)pop_canvas_menu, (XtPointer) MD);
    
    /* if server can backing store, we want backing store */
    dpy=XtDisplay(nw->cw->graph);
    wattr.backing_store=DoesBackingStore(DefaultScreenOfDisplay(dpy));
    XChangeWindowAttributes(dpy,XtWindow(nw->cw->graph),CWBackingStore,&wattr);
    
    cwIcon = XCBFD(dpy,XtWindow(form),(char *)canvas_bits,
		   canvas_width, canvas_height);
    XtVaSetValues(popup,XtNiconPixmap,cwIcon,NULL);
    
    add_menu_entry(nw);
}

static void setwave_action (Widget w,Next_Wave *nw, XEvent *event, 
			    Boolean *flg)
{
    XClientMessageEvent *cevent=(XClientMessageEvent*)event;
    if (MD->mb->exposing) return;
    if (cevent->type!=ClientMessage) return;
    if (cevent->message_type==protocols && cevent->data.l[0]==focus)
      set_actual_canvas(nw);
    if (cevent->message_type==protocols && cevent->data.l[0]==delwin) { 
	if (MD->mb->playing) {
	    warn_popup(w,"Stop playing to close file !");  
	    return;
	}
	close_call(w,MD,NULL);
    }
}

void set_actual_canvas(Next_Wave *nw)
{
    
    if (MD->wd==nw->wd) return;
    if (MD->mb->playing) return;
    
    MD->wd=nw->wd;
    MD->cw=nw->cw;
    MD->cb=nw->cb;
    MD->cg=nw->cg;
    MD->mg->step=(float)MD->wd->tlength/(float)MD->mg->width;
    
    if (app_resources.raisecanvas) window_order(nw->cw->popup,Above);
    
    update_display(MD);
    return;
}

static void reparent_canvas (Widget w,Next_Wave *nw, XReparentEvent *event, 
			     Boolean *flg)
{
    if (event->type!=ReparentNotify) return;
    
    XtAddRawEventHandler(w, 0, True,
			 (XtEventHandler)setwave_action, (XtPointer) nw);
    if (delwin!=None || protocols!=None) {
	XChangeProperty(XtDisplay(w), XtWindow(w),
			protocols,XA_ATOM, 32, PropModeReplace,
			(unsigned char *) &delwin, 1);
    }
    if (focus!=None || protocols!=None) {
	XChangeProperty(XtDisplay(w), XtWindow(w),
			protocols,XA_ATOM, 32, PropModeAppend,
			(unsigned char *) &focus, 1);
    }
}

static void reparent_toplevel (Widget w,Main_Data *md, XReparentEvent *event, 
			       Boolean *flg)
{
    Display *dpy=XtDisplay(w);
    if (event->type!=ReparentNotify) return;
    
    XtAddRawEventHandler(w, 0, TRUE,
			 (XtEventHandler)delete_toplevel, (XtPointer) md);
    
    if (delwin!=None || protocols!=None) {
	XChangeProperty(dpy, XtWindow(w),
			protocols,XA_ATOM, 32, PropModeReplace,
			(unsigned char *) &delwin, 1);
    }
    md->mg->lpixmap=XCreatePixmap(dpy, XtWindow(md->mw->graph),1/*width*/,
				  PREVIEWHEIGHT,
				  DefaultDepth(dpy,DefaultScreen(dpy)));
    
    if (no_load_files>0) {
	int i,type;
	for (i=0;i<no_load_files;i++) {
	    fprintf(stderr,"Loading %s... ",load_files[i]);
	    fflush(stderr);
	    type=load_file(load_files[i],AF_NORAW,AF_COMP_UNDEFINED);
	    switch (type) {
	     case AF_ERROR:
		fprintf(stderr,"error.\n");
		break;
	     case AF_RAW: 
		fprintf(stderr,"can't load raw files from cmdline !\n");
		break;
	     case AF_NOTSUPPORTED:
		fprintf(stderr,"filetype not supported !\n");
		break;
	     default:
		fprintf(stderr,"done.\n");
		break;
	    }
	}
    }
    XtFree((char*)load_files);

    update_status(md);

    /* set working directory to wdir-Resource */
    chdir(app_resources.wdir);
    /*
     if (XInternAtom(dpy, "_SUN_WM_PROTOCOLS", TRUE)!=None)
     	warn_popup(w,"SUN WM present (olwm,olvwm), expect problems !");
     */
    if (DoesBackingStore(DefaultScreenOfDisplay(dpy))==NotUseful) {
	warn_popup(w,"Your X-Server doesn't support BackingStore !\nThis will slow down the screen updates !");
    }
    if (!MD->mb->canplay) {
	warn_popup(w,"Couldn't access audio device !\nNo play and record !");
    }
}

static void delete_toplevel (Widget w,Main_Data *md, XEvent *event, 
			     Boolean *flg)
{
    XClientMessageEvent *cevent=(XClientMessageEvent*)event;
    if (md->mb->exposing) return;
    if (cevent->type!=ClientMessage) return;
    if (cevent->message_type==protocols && cevent->data.l[0]==delwin) 
      if (md->mb->playing) {
	  warn_popup(w,"Stop playing to quit !");
	  return;
      }
    quit_call(w,md,NULL);
}

void set_cursor(Widget w,byte which)
{
    XtVaSetValues(w,XtNcursor,previewC[which],NULL);
}

void watch_cursor(bool state)
{
    Window win=XtWindow(FindWmShell(MD->mw->form));
    Display *dpy=XtDisplay(MD->mw->form);
    Next_Wave *nw=MD->nw;
    
    if (state) {
	XDefineCursor(dpy,win,previewC[CURSOR_CLOCK]);
	while (nw!=NULL) {
	    win=XtWindow(FindWmShell(nw->cw->popup));
	    dpy=XtDisplay(nw->cw->popup);
	    XDefineCursor(dpy,win,previewC[CURSOR_CLOCK]);
	    nw=nw->nw;
	}
    } else {
	XUndefineCursor(dpy,win);
	while (nw!=NULL) {
	    win=XtWindow(FindWmShell(nw->cw->popup));
	    dpy=XtDisplay(nw->cw->popup);
	    XUndefineCursor(dpy,win);
	    nw=nw->nw;
	}
    }
    XFlush(dpy);
}

static void pop_canvas_menu (Widget w,Main_Data *md, XButtonEvent *event, 
			     Boolean *flg)
{
    if (event->button==3) { /* right mbutton (3-chord-mouse) */
	Screen *scr=XtScreen(w);
	Position x,y;
	Dimension width,height;
	
	x=event->x_root;
	y=event->y_root;
	XtVaGetValues(md->mw->cmenu,XtNwidth,&width,XtNheight,&height,NULL);
	if (y>scr->height-height) y-=y+height-scr->height;
	else y-=10;
	if (x<width/2) x=0;
	else if (x<scr->width-width/2) x-=width/2;
	else {
	    width-=scr->width-x;
	    x-=width;
	}
	XtVaSetValues(md->mw->cmenu,XtNx,x,XtNy,y,NULL);
    }
    if (!md->mb->playing) {
	set_cmenu_state(CM_ZOOM,CM_COPY,md->wd->ismark);
	set_cmenu_state(CM_PASTE_I,CM_PASTE_M,md->mb->isclip);
    }
}

