/* -*- mode: c ; c-file-style: "canonware-c-style" -*-
 ****************************************************************************
 *
 * Copyright (C) 1996-1999 Jason Evans <jasone@canonware.com>.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ****************************************************************************
 *
 * Version: s19990912a
 *
 * <<< Description >>>
 *
 * Public interface for the log (logging) class.  Output can be sent to a file
 * (stderr by default).  Functions similar to printf() are provided for
 * formatted output.
 *
 ****************************************************************************/

typedef struct cw_log_s cw_log_t;

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * None.
 *
 * <<< Output(s) >>>
 *
 * retval : Pointer to a log, or NULL.
 *          NULL : Memory allocation error.
 *
 * <<< Description >>>
 *
 * Constructor.
 *
 ****************************************************************************/
cw_log_t *
log_new(void);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * <<< Output(s) >>>
 *
 * None.
 *
 * <<< Description >>>
 *
 * Destructor.
 *
 ****************************************************************************/
void
log_delete(cw_log_t * a_log);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_logfile : Pointer to a string that represents a file to log to.
 *
 * a_overwrite : TRUE == truncate a_logfile, FALSE == append to a_logfile.
 *
 * <<< Output(s) >>>
 *
 * retval : FALSE == success, TRUE == error.
 *          TRUE : Error opening a_logfile.
 *               : fflush() error.
 *               : fclose() error.
 *               : Memory allocation error.
 *
 * <<< Description >>>
 *
 * Open file for logging.  If another file is already being used, close it
 * first.
 *
 ****************************************************************************/
cw_bool_t
log_set_logfile(cw_log_t * a_log, const char * a_logfile,
		cw_bool_t a_overwrite);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_format : Formatting string.
 *
 * ... : Arguments.
 *
 * <<< Output(s) >>>
 *
 * retval : Number of characters printed.
 *
 * <<< Description >>>
 *
 * printf() wrapper.
 *
 ****************************************************************************/
int
log_printf(cw_log_t * a_log, const char * a_format, ...);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_filename : __FILE__, or NULL.
 *
 * a_line_num : __LINE__.  Ignored if a_filename == NULL.
 *
 * a_func_name : __FUNCTION__, or NULL.
 *
 * a_format : Formatting string.
 *
 * ... : Arguments.
 *
 * <<< Output(s) >>>
 *
 * retval : Number of characters printed.
 *
 * <<< Description >>>
 *
 * printf() wrapper.  Optional arguments prepend filename, line number, and
 * function name.
 *
 ****************************************************************************/
int
log_eprintf(cw_log_t * a_log, 
	    const char * a_filename,
	    int a_line_num,
	    const char * a_func_name,
	    const char * a_format,
	    ...);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_size : Maximum number of bytes to print.
 *
 * a_format : Formatting string.
 *
 * ... : Arguments.
 *
 * <<< Output(s) >>>
 *
 * retval : Number of bytes printed, or -1.
 *          -1 : Memory allocation error.
 *
 * <<< Description >>>
 *
 * printf() wrapper, but print no more than a_size characters.
 *
 ****************************************************************************/
int
log_nprintf(cw_log_t * a_log,
	    cw_uint32_t a_size,
	    const char * a_format,
	    ...);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_format : Formatting string.
 *
 * ... : Arguments.
 *
 * <<< Output(s) >>>
 *
 * retval : Number of bytes printed.
 *
 * <<< Description >>>
 *
 * printf() wrapper, with a time stamp prepended.
 *
 ****************************************************************************/
int
log_lprintf(cw_log_t * a_log, const char * a_format, ...);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_log : Pointer to a log.
 *
 * a_filename : __FILE__, or NULL.
 *
 * a_line_num : __LINE__.  Ignored if a_filename == NULL.
 *
 * a_func_name : __FUNCTION__, or NULL.
 *
 * a_format : Formatting string.
 *
 * ... : Arguments.
 *
 * <<< Output(s) >>>
 *
 * retval : Number of characters printed.
 *
 * <<< Description >>>
 *
 * printf() wrapper, with a time stamp prepended.  Optional arguments prepend
 * filename, line number, and function name.
 *
 ****************************************************************************/
int
log_leprintf(cw_log_t * a_log,
	     const char * a_filename,
	     int a_line_num,
	     const char * a_func_name,
	     const char * a_format,
	     ...);

/****************************************************************************
 *
 * <<< Input(s) >>>
 *
 * a_val : Value to convert to a string.
 *
 * a_base : Number base to convert to.
 *
 * a_buf : A buffer of at least 65 bytes for base 2 conversion, 21 bytes
 *         for base 10 conversion, and 17 bytes for base 16 conversion.
 *
 * <<< Output(s) >>>
 *
 * retval : Pointer to a string that represents a_val, in base a_base (a_buf).
 *
 * <<< Description >>>
 *
 * Convert a_val to a number string in base a_base and put the result into
 * *a_buf.  Supported bases are 2, 10, and 16.
 *
 ****************************************************************************/
char *
log_print_uint64(cw_uint64_t a_val, cw_uint32_t a_base, char * a_buf);
