//
// $Id: BufferingOutStream.h,v 1.15 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(BUFFERINGOUTSTREAM_OL_GUARD)
#define BUFFERINGOUTSTREAM_OL_GUARD

#include <ObjectiveLib/LayeredOutStream.h>

/**
 * A layered out stream that filters its data through a buffer. This concrete class channels
 * all written data through a buffer, which in some cases can dramatically improve
 * performance, for example when the layer below the buffering stream performs a very
 * time-consuming task. The buffer will be cleared on any call to #flush or #close.
 *
 * @sa OLBufferingInStream
 *
 * @ingroup Streams
 */
@interface OLBufferingOutStream : OLLayeredOutStream
{
@protected
    /**
     * The buffer that temporarily holds data written to the stream
     */
    uint8_t*        buffer;

    /**
     * The number of bytes currently in the buffer
     */
    unsigned        count;

    /**
     * The capacity of the buffer
     */
    unsigned        capacity;
}

/**
 * Create and return a new stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses the default buffer size of 32K.
 *
 * @param underStream the stream to which this stream ultimately forwards its data
 * @return a reference to the newly created stream
 */
+ (id) streamWithOutStream: (OLOutStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses the default buffer size of 32K.
 *
 * @param underStream the stream to which this stream ultimately forwards its data
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream;

/**
 * Initialize the stream. The new stream uses @a underStream as the next lower
 * layer in the stream stack and uses @a size as the buffer size.
 *
 * @param underStream the stream to which this stream ultimately forwards its data
 * @param size the size of the buffer
 * @return a reference to this stream
 */
- (id) initWithOutStream: (OLOutStream*)underStream bufferSize: (unsigned)size;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */
- (void) flush;
#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Write a byte. This is an optimized version of writing a byte that is very fast if the
 * buffer is not full.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param byte the byte to write
 */
- (void) writeByte: (uint8_t)byte;

/**
 * Write a sequence of bytes to the stream. Bytes are copied into the buffer, which
 * is flushed as needed. For cases where @a num is larger than the size of the buffer,
 * the buffer is flushed and then ignored, thus eliminating the needless overhead
 * of using it.
 *
 * @exception OLInputOutputException if there is an error writing to the stream
 * @param bytes the bytes to write
 * @param num the number of bytes to write
 * @return the number of bytes written
 */
- (unsigned) writeBytes: (const uint8_t*)bytes count: (unsigned)num;

@end

#endif
