//
// $Id: GzipInStream.h,v 1.11 2007/03/06 20:42:19 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(GZIPINSTREAM_OL_GUARD)
#define GZIPINSTREAM_OL_GUARD

#include <ObjectiveLib/ZlibInStream.h>

@class OLMap;

/**
 * @class OLGzipInStream GzipInStream.h ObjectiveLib/GzipInStream.h
 *
 * A stream for zlib decompression that reads data in gzip format.
 * The data are read as if they were written with the format specified in
 * <a href=http://www.faqs.org/rfcs/rfc1952.html>RFC 1952</a>. The command-line tool
 * @c gzip writes data in this format, so this class can be used to read files
 * written by @c gzip.
 *
 * @sa OLGzipOutStream, OLZlibInStream
 *
 * @ingroup Streams
 */
@interface OLGzipInStream : OLZlibInStream
{
@protected
    /**
     * The original file name
     */
    char*       fileName;

    /**
     * The gzip comment
     */
    char*       comment;

    /**
     * The extra fields
     */
    OLMap*      extraFields;

    /**
     * The modification time of the original file
     */
    uint32_t    modTime;

    /**
     * The CRC-32 of the data
     */
    uint32_t    crc;

    /**
     * Whether we have reached the end of the stream
     */
    BOOL        endOfStream;
}

+ (id) streamWithInStream: (OLInStream*)underStream;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the stream. The stream is created with the given buffer size.
 * The given stream @a underStream will become
 * the next layer down in the stack of streams. If @a zlibHeader is YES, then
 * the stream will attempt to read the zlib header.
 *
 * @attention You probably do not want to use this initializer. This initializer is
 * overriden to ignore the given value of @a zlibHeader and always set it to NO.
 * The zlib header cannot be used when dealing with the gzip data format.
 *
 * @exception OLInputOutputException if the stream cannot be created for inflation
 * @param underStream the stream from which to read data
 * @param size the buffer size
 * @param zlibHeader it doesn't matter what you use for this, as it will be overriden
 * @return a reference to this stream
 */
- (id) initWithInStream: (OLInStream*)underStream bufferSize: (unsigned)size readZlibHeader: (BOOL)zlibHeader;
#if !defined(OL_NO_OPENSTEP)
- (void) dealloc;
#endif
/* @} */

/**
 * Return the comment in the gzip data. If the data contains no comment, then @c nil
 * is returned.
 *
 * @return the gzip comment or @c nil if there is no comment
 */
- (const char*) comment;

/**
 * Return an extra field. Extra fields are identified by a two-letter identifier. If the
 * requested extra field does not exist in the gzip data, then @c NULL is returned.
 *
 * @exception OLInputOutputException if the length of @a identifier is not equal to 2
 * @param identifier the identifier of the extra field
 * @param countOut a pointer to a unsigned int that will contain the length of the
 * extra field's data on exit, or zero if no extra field with the given identifier
 * can be found
 * @return the data of the extra field or @c nil if the extra field does not exist in
 * the data
 */
- (const uint8_t*) extraField: (const char*)identifier count: (unsigned*)countOut;

#if defined(OL_NO_OPENSTEP)
- (void) freeStreamResources;
#endif

/**
 * Return the modification time embedded in the data. This is expressed in Unix format,
 * which is the number of seconds since 00:00:00 GMT, Jan. 1, 1970. If no modification
 * time was available when the data were created, then 0 is returned.
 *
 * @return the modification time or zero if there is no modification time available
 */
- (uint32_t) modificationTime;

/**
 * Return the original file name associated with the data. If no file name was recorded
 * in the data, then @c nil is returned.
 *
 * @return the original file name or @c nil if no file name is present in the data
 */
- (const char*) originalFileName;
- (unsigned) readBytes: (uint8_t*)dest count: (unsigned)max;

@end

#endif
