package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````,"$```````!``````````%"U````
M`````````$``.``(`$``&@`9``8````$````0`````````!``````````$``
M````````P`$```````#``0````````@``````````P````0``````@``````
M```"``````````(````````7`````````!<``````````0`````````!````
M!0```````````````````````````````````)R>````````G)X`````````
M``$```````$````&````0*T```````!`K0$``````$"M`0``````2`4`````
M```(!@```````````0```````@````8```!8K0```````%BM`0``````6*T!
M``````#@`0```````.`!````````"``````````$````!````!@"````````
M&`(````````8`@```````"P`````````+``````````$`````````%#E=&0$
M````V)8```````#8E@```````-B6````````!`$````````$`0````````0`
M````````4N5T9`0```!`K0```````$"M`0``````0*T!``````#``@``````
M`,`"`````````0`````````O=7-R+VQI8F5X96,O;&0N96QF7W-O```'````
M!`````$```!.971"4T0``$`,JCL$````!`````,```!085@```````````!A
M````:```````````````,0````T``````````````#@````.````0```````
M```P``````````,```!,````80`````````?````2P````8````=````%```
M``````!"````5@```%T`````````%P```#P```!8``````````````!D````
M9@```"\````2````5P```$0````5````7````$4`````````2````"(````U
M````6P```"8````J``````````<```!.``````````L````L````9P```%H`
M``!*````(````$\```!E`````````%\`````````&````"X```!-````````
M````````````$0```#,``````````````&`````K``````````D`````````
M)````#D`````````4`````4```!&````&P```#L`````````````````````
M````20`````````Z````4P```&(`````````8P``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````!,`````````````````````````````````````
M```````````````C````(0`````````I``````````H````G````*````"T`
M``````````````P`````````'````!H````9```````````````0````````
M`#0````R````-P```!X`````````````````````````!```````````````
M00```"4````````````````````6```````````````_````0P````@`````
M```````````````^````4@``````````````50``````````````/0``````
M````````40````````!'````#P````````!4````60```#8```!>````````
M`````````````````````````````````````````````P`*`#`A````````
M`````````````````P`0`%"M`0````````````````"2````$@``````````
M```````````````,````$@`````````````````````````4````$@``````
M``````````````````!P`P``$@`*`#`K````````"``````````<````(@``
M```````````````````````K````$@`````````````````````````7`P``
M$@````````````````````````!)`P``$``5`(BR`0`````````````````Q
M````$@````````````````````````"!`P``$@`*`$`U````````(```````
M```X````$@`````````````````````````D`P``$@``````````````````
M``````!`````$@````````````````````````"<`P``$@`*`'`^````````
M9`````````"#````$@````````````````````````!8`P``$``5`$BS`0``
M``````````````!(````$@````````````````````````"H`P``$``4`(BR
M`0````````````````!8````$@````````````````````````!G````$@``
M``````````````````````!M````$@````````````````````````!T````
M$@````````````````````````!^````$@````````````````````````"(
M````$@````````````````````````!7`P``$``5`$BS`0``````````````
M``"1````$@`````````````````````````3`@``$@``````````````````
M``````"7````$@````````````````````````">````$@``````````````
M``````````"L````$@````````````````````````"]````$0``````````
M``````````````"H`@``(@````````````````````````#*````$@``````
M``````````````````"O`P``$@`*`&0M````````+`````````"\`P``$@`*
M`)`M````````;`````````#(`P``$@`*`&`V````````P`4```````#8````
M%@````````````````````````#K````$@`````````````````````````V
M`P``$@````````````````````````#V````$@``````````````````````
M``#_````$0`5`$"S`0``````"``````````'`0``$@``````````````````
M```````0`0``$@`````````````````````````-`P``$@``````````````
M```````````<`0``$@`````````````````````````C`0``$@``````````
M``````````````#5`P``$@`*`,!/````````*`$````````V`0``$0``````
M``````````````````!(`0``$@````````````````````````!8`0``$@``
M``````````````````````!F`0``$@````````````````````````!K`0``
M$@````````````````````````#@`P``$0`4`("R`0``````"`````````!Z
M`0``$@````````````````````````"&`0``$@``````````````````````
M``";`0``$@````````````````````````"F`0``$@``````````````````
M``````!K`P``$``5`$BS`0````````````````#.`@``$@``````````````
M``````````"Y`0``$@`````````````````````````;!```$@`*`#`A````
M````"`````````#'`0``$@````````````````````````#/`0``$@``````
M``````````````````#;`0``$@````````````````````````!C`P``$``5
M`$BS`0````````````````#J`P``$@`*```N````````Q`0```````#A`0``
M$@````````````````````````#L`0``$@````````````````````````#W
M`P``$@`*`/!0````````E`$```````#]`0``$@``````````````````````
M```!!```$``5`(BR`0`````````````````-!```$@`*`)0J````````0```
M```````5!```$@`*`%!3````````&`(````````$`@``$@``````````````
M``````````")`@``(@`````````````````````````:!```$@`*`#`A````
M````"``````````+`@``$@`````````````````````````B!```$@`*`$`K
M````````)`(````````O!```$@`*`&`U````````^``````````2`@``$@``
M``````````````````````!!!```$@`*`,0R````````<``````````;`@``
M$@````````````````````````"A`@``$@`````````````````````````L
M`@``$@````````````````````````!/!```$@`*`#0S````````G```````
M```[`@``$@````````````````````````!$`@``$@``````````````````
M``````!*`@``$@```````````````````````````P``$@``````````````
M``````````!/`@``$@````````````````````````#S`@``$0`5`,BR`0``
M````"`````````!7`@``$@`````````````````````````K`P``$@``````
M``````````````````!>`@``$@````````````````````````!7!```$@`*
M`&1/````````5`````````!E`@``$0````````````````````````#H`@``
M$0`4``"R`0``````"`````````!@!```$@`*`"`\````````2`(````````O
M`0``$@````````````````````````!J`@``$@``````````````````````
M```]`P``$@``````````````````````````;&EB<&5R;"YS;P!R96%L;&]C
M`&UE;6UO=F4`7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T<FYC;7``
M<W1R<F-H<@!B;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`86)O<G0`
M=6YL:6YK`%]?;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X:70`;65M
M8W!Y`'!E<FQ?9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D;U]U;F1U
M;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R;%]A;&QO
M8P!C;&]S961I<@!E;G9I<F]N`')E861L:6YK`%]?;W!E;F1I<C,P`&UE;7-E
M=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`4&5R;%]G
M=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A9`!097)L7W-Y<U]I;FET,P!0
M97)L7V=E=%]S=@!097)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VYE=UA3`%!E
M<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%]?97)R;F\`4&5R;%]M
M9U]S970`;6MD:7(`<&5R;%]P87)S90!097)L7VYE=UA37V9L86=S`&%C8V5S
M<P!G971P:60`;65M8VAR`'9F<')I;G1F`%!E<FQ?871F;W)K7VQO8VL`4&5R
M;%]S=E]S971P=FX`<&5R;%]R=6X`;'-E96L`;W!E;@!S<')I;G1F`&=E='5I
M9`!S=')L96X`7U]S1@!S=')C;7``;&EB;2YS;RXP`&QI8F-R>7!T+G-O+C$`
M7U]D97)E9VES=&5R7V9R86UE7VEN9F\`<W1R9'5P`%]?<F5G:7-T97)?9G)A
M;65?:6YF;P!L:6)P=&AR96%D+G-O+C$`<'1H<F5A9%]A=&9O<FL`;&EB8RYS
M;RXQ,@!?7W!R;V=N86UE`%]?<'-?<W1R:6YG<P!?7V=E='!W=6ED-3``7U]S
M>7-C86QL`%]?<W1P8W!Y7V-H:P!A=&5X:70`7VQI8F-?:6YI=`!S=')T;VL`
M7U]R96%D9&ER,S``7U]B<W-?<W1A<G1?7P!?7V)S<U]E;F1?7P!?7V5N9%]?
M`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!G971?=7-E<FYA;65?9G)O;5]G971P
M=W5I9`!P87)?8VQE86YU<`!?961A=&$`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`&UY7W!A<E]P;`!P87)?:6YI
M=%]E;G8`<VAA7V9I;F%L`%]?8G-S7W-T87)T`'AS7VEN:70`;6%I;@!?7W-T
M87)T`'!A<E]F:6YD<')O9P!P87)?<V5T=7!?;&EB<&%T:`!P87)?96YV7V-L
M96%N`'!A<E]D:64`<VAA7VEN:70`6%-?26YT97)N86QS7U!!4E]"3T]4`"]U
M<W(O<&MG+VQI8B]P97)L-2\U+C0P+C`O86%R8V@V-"UN971B<V0M=&AR96%D
M+6UU;'1I+T-/4D4Z+W5S<B]P:V<O;&EB````0*T!```````#!````````!!3
M````````2*T!```````#!````````)!2````````0*\!```````#!```````
M`%"M`0``````6*\!```````#!````````$"M`0``````>*\!```````#!```
M`````$"S`0``````B*\!```````#!````````$BM`0``````D*\!```````#
M!````````("R`0``````L*\!```````#!````````%BM`0``````R*\!````
M```#!````````,BR`0``````V*\!```````#!`````````"R`0``````X*\!
M```````#!````````"`\`````````+(!```````#!````````,"R`0``````
M"+(!```````#!`````````BR`0``````(+(!```````#!````````(!5````
M````*+(!```````#!````````(A5````````.+(!```````#!````````)!5
M````````0+(!```````#!````````*!5````````2+(!```````#!```````
M`*A5````````4+(!```````#!````````+!5````````6+(!```````#!```
M`````+A5````````:+(!```````#!````````,!5````````<+(!```````#
M!````````,A5````````@+(!```````#!````````%!9````````2*\!````
M```!!```!P``````````````4*\!```````!!```$P``````````````8*\!
M```````!!```(0``````````````:*\!```````!!```(@``````````````
M<*\!```````&!```)P``````````````@*\!```````!!```,@``````````
M````F*\!```````!!```.P``````````````J*\!```````!!```30``````
M````````P*\!```````!!```5```````````````T*\!```````!!```8@``
M`````````````+`!```````"!````P``````````````"+`!```````"!```
M!```````````````$+`!```````"!```!0``````````````&+`!```````"
M!```!P``````````````(+`!```````"!```"```````````````*+`!````
M```"!```"0``````````````,+`!```````"!```"P``````````````.+`!
M```````"!```#0``````````````0+`!```````"!```#@``````````````
M2+`!```````"!```#P``````````````4+`!```````"!```$0``````````
M````6+`!```````"!```%0``````````````8+`!```````"!```%@``````
M````````:+`!```````"!```%P``````````````<+`!```````"!```&```
M````````````>+`!```````"!```&0``````````````@+`!```````"!```
M&@``````````````B+`!```````"!```'```````````````D+`!```````"
M!```'0``````````````F+`!```````"!```'@``````````````H+`!````
M```"!```'P``````````````J+`!```````"!```(```````````````L+`!
M```````"!```(@``````````````N+`!```````"!```(P``````````````
MP+`!```````"!```*```````````````R+`!```````"!```*0``````````
M````T+`!```````"!```*@``````````````V+`!```````"!```+```````
M````````X+`!```````"!```+0``````````````Z+`!```````"!```+@``
M````````````\+`!```````"!```+P``````````````^+`!```````"!```
M,````````````````+$!```````"!```,P``````````````"+$!```````"
M!```-```````````````$+$!```````"!```-0``````````````&+$!````
M```"!```-@``````````````(+$!```````"!```.```````````````*+$!
M```````"!```.0``````````````,+$!```````"!```.@``````````````
M.+$!```````"!```/0``````````````0+$!```````"!```/@``````````
M````2+$!```````"!```0```````````````4+$!```````"!```00``````
M````````6+$!```````"!```0@``````````````8+$!```````"!```10``
M````````````:+$!```````"!```1@``````````````<+$!```````"!```
M2```````````````>+$!```````"!```3```````````````@+$!```````"
M!```30``````````````B+$!```````"!```3P``````````````D+$!````
M```"!```4@``````````````F+$!```````"!```50``````````````H+$!
M```````"!```5@``````````````J+$!```````"!```6```````````````
ML+$!```````"!```60``````````````N+$!```````"!```6@``````````
M````P+$!```````"!```6P``````````````R+$!```````"!```7```````
M````````T+$!```````"!```7@``````````````V+$!```````"!```7P``
M````````````X+$!```````"!```8```````````````Z+$!```````"!```
M90``````````````\+$!```````"!```9@``````````````^+$!```````"
M!```9P``````````````\'N_J=```+`1_D?Y$.(_D2`"']8?(`/5'R`#U1\@
M`]70``#0$0)`^1`"`)$@`A_6T```T!$&0/D0(@"1(`(?UM```-`1"D#Y$$(`
MD2`"'];0``#0$0Y`^1!B`)$@`A_6T```T!$20/D0@@"1(`(?UM```-`1%D#Y
M$*(`D2`"'];0``#0$1I`^1#"`)$@`A_6T```T!$>0/D0X@"1(`(?UM```-`1
M(D#Y$`(!D2`"'];0``#0$29`^1`B`9$@`A_6T```T!$J0/D00@&1(`(?UM``
M`-`1+D#Y$&(!D2`"'];0``#0$3)`^1""`9$@`A_6T```T!$V0/D0H@&1(`(?
MUM```-`1.D#Y$,(!D2`"'];0``#0$3Y`^1#B`9$@`A_6T```T!%"0/D0`@*1
M(`(?UM```-`11D#Y$"("D2`"'];0``#0$4I`^1!"`I$@`A_6T```T!%.0/D0
M8@*1(`(?UM```-`14D#Y$(("D2`"'];0``#0$59`^1"B`I$@`A_6T```T!%:
M0/D0P@*1(`(?UM```-`17D#Y$.("D2`"'];0``#0$6)`^1`"`Y$@`A_6T```
MT!%F0/D0(@.1(`(?UM```-`1:D#Y$$(#D2`"'];0``#0$6Y`^1!B`Y$@`A_6
MT```T!%R0/D0@@.1(`(?UM```-`1=D#Y$*(#D2`"'];0``#0$7I`^1#"`Y$@
M`A_6T```T!%^0/D0X@.1(`(?UM```-`1@D#Y$`($D2`"'];0``#0$89`^1`B
M!)$@`A_6T```T!&*0/D00@21(`(?UM```-`1CD#Y$&($D2`"'];0``#0$9)`
M^1""!)$@`A_6T```T!&60/D0H@21(`(?UM```-`1FD#Y$,($D2`"'];0``#0
M$9Y`^1#B!)$@`A_6T```T!&B0/D0`@61(`(?UM```-`1ID#Y$"(%D2`"'];0
M``#0$:I`^1!"!9$@`A_6T```T!&N0/D08@61(`(?UM```-`1LD#Y$((%D2`"
M'];0``"P$;9`^1"B!9$@`A_6T```L!&Z0/D0P@61(`(?UM```+`1OD#Y$.(%
MD2`"'];0``"P$<)`^1`"!I$@`A_6T```L!'&0/D0(@:1(`(?UM```+`1RD#Y
M$$(&D2`"'];0``"P$<Y`^1!B!I$@`A_6T```L!'20/D0@@:1(`(?UM```+`1
MUD#Y$*(&D2`"'];0``"P$=I`^1#"!I$@`A_6T```L!'>0/D0X@:1(`(?UM``
M`+`1XD#Y$`('D2`"'];0``"P$>9`^1`B!Y$@`A_6T```L!'J0/D00@>1(`(?
MUM```+`1[D#Y$&('D2`"'];0``"P$?)`^1""!Y$@`A_6T```L!'V0/D0H@>1
M(`(?UM```+`1^D#Y$,('D2`"'];0``"P$?Y`^1#B!Y$@`A_6X0,"JC0``!0`
M`````````/U[O:G```"0_0,`D0"@1_GU$P#YU0``D+4B-9&_`@#K@@$`5/-3
M`:D4!`#1E`(5RQ,`@-*4_D/3H'IS^```/]9_`A3K<P8`D8'__U3S4T&I]1-`
M^?U[PZC``U_6'R`#U?U[O:G]`P"1\U,!J=,``)#U$P#YU0``D'/21_FUWD?Y
M?P(5ZV(!`%1@"D"Y'R`0\8$!`%1T`D#Y<V(`D6""7_@``#_6@`(`^7\"%>OC
M_O]4\U-!J?430/G]>\.HP`-?UOS^_Y?]>[RI_0,`D?-3`:GU6P*I]QL`^6$)
M`+3T`P&JP0``D-4``)`AY$?Y@@)`^:2^1_E#`$#Y-```^8$*0/F!``#YP0``
MD",'`+0A[$?Y(P``^4(`0/E#`$`YXP``-$($`)%_O`!Q@?__5"(``/E#`$`Y
M8___-4```+3-_O^7F/__E\```)``V$?YP`8`M,```)#7``"0]P(UD0"L1_G_
M`@#K0@$`5!8$`-$3`(#2U@(7R];^0]/@>G/X```_UG\"%NMS!@"1@?__5```
M`)````61M_[_E]<``)#```"0]P(UD0#$1_G_`@#K0@$`5!8$`-$3`(#2U@(7
MR];^0]/@>G/X```_UM\"$^MS!@"1@?__5+6^1_F`"D"Y@0)`^:("0/D+#`"4
M@O[_ER'L1_G"``"P0@`+D2(``/D`^O^UT/__%P(``/!"H!618P*`TD$`@%*`
M`(#2ZO[_ER``@%*X_O^7C___E\K__Q<`````_7N\J?T#`)'S4P&IH`8`M/=C
M`ZGW`P"JP```D`"\1_D8`$#Y.`8`M/5;`JGV`P&J%0"`4N`"0#D?]`!Q!!A`
M>L$"`%03`T#YTP$`M+5^0)/T`QBJ'R`#U>(#%:KA`Q>JX`,3JG/^_Y>````U
M8&IU.!_T`'$``@!4DXY`^//^_[7U6T*I]V-#J>`#$ZKS4T&I_7O$J,`#7];U
M`Q>JH!Y`.!_T`'$$&$!ZH?__5+4"%TOE__\7E`(8R[4&`)%S`A6+E/Y#D]0"
M`+G@`Q.J\U-!J?5;0JGW8T.I_7O$J,`#7]83`(#2Z?__%Q,`@-+W8T.IYO__
M%_U[LJG!``"0_0,`D2'`1_DB`$#YXF\`^0(`@-+```"T\U,!J?,#`*H!`$`Y
M80$`-?-30:G```"0`,!'^>%O0/D"`$#Y(0`"ZP(`@-+A"`!4_7O.J,`#7]:-
M_O^7]`,`JH#^_[3W8P.I%P``\/@#`9'W8A>1]5L"J>`#%*H0__^7(`4`M`$T
M0#D5-`"1/[@`<6$``%2A!D`Y`?__-``T0#D?N`!Q``4`5.`#$ZKX_O^7]@,`
MJN`#%:KU_O^7P`(`"P`(`!$`?$"3]?[_E^,#%:KB`Q.J]0,`JN$#%ZK@_O^7
MX0,8JN`#%:HY_O^7'P0`,:```%3@2T"Y``P4$A\00''@`@!4X`,5JB7^_Y?@
M`Q6J%_[_E^`#%*KI_O^70/O_M1\@`]7@`Q2J4?[_E^`#$ZKW_?^7\U-!J?5;
M0JGW8T.IO?__%Z`&0#D?N`!QX?K_5*`*0#D@^?\TU/__%^`#%:JH__^7ZO__
M%_-3`:GU6P*I]V,#J2K^_Y<?(`/5_7NTJ<(``)#]`P"10L!'^?,+`/GA@P"1
M\P,`JD,`0/GC7P#Y`P"`T@+^_Y<@`@`UX2M`N0(`E%(A#!02/Q!`<2000GJ@
M`0!4P0``D"'`1_GB7T#Y(P!`^4(``^L#`(#2@0$`5/,+0/G]>\RHP`-?U@``
M@%+U__\7X`,3JD$`@%)L_O^7'P``<>`7GQKO__\7!/[_E__#$-'#``"0X0,`
MJ@*`@-+]>P"I_0,`D6/`1_GS"P#Y\Z,`D60`0/GD%P+Y!`"`TN`#$ZJV_?^7
M``0`T1\`$^NH``!4!P``%!_T'S@?`!/K@```5`$`0#D_O`!Q8/__5.`#$ZKA
M!8!2N?W_ET`"`+0?`!/KR`$`5.`#$ZI<_O^7P0``D"'`1_GB%T+Y(P!`^4(`
M`^L#`(#2@0(`5/U[0*GS"T#Y_\,0D<`#7]8?```Y\O__%\```)``P$?YX1="
M^0(`0/DA``+K`@"`TN$``%3]>T"I````\/,+0/D`(!:1_\,0D4+^_Q?)_?^7
M_P,1T<,``)#B?X#2_7L`J?T#`)%CP$?Y\U,!J?0#`*KU6P*I]>,`D?8#`:I@
M`$#YX!\"^0``@-+A`Q6J````\`"`%Y'/_?^7'P``<<T"`%0#!`!1H<IC.`$%
M`#5#`@`T<WQ`D^`#%:KB`Q.JX06`4AS^_Y>``0"T1-`[U<```)``N$?Y'R`#
MU1\@`]7C`Q.JX@,5JN$#%*J`:&#X'?[_EPH``!1#T#O5P```D`"X1_D?(`/5
M'R`#U>(#%JKA`Q2J8&A@^)_]_Y?```"0`,!'^>$?0OD"`$#Y(0`"ZP(`@-(!
M`0!4_7M`J?-30:GU6T*I_P,1D<`#7];C`P`JV?__%XC]_Y?]>[JIP@``D/T#
M`)'S4P&I\P,`JO0#`:KU6P*I]V,#J4+`1_D@`$`Y00!`^>$O`/D!`(#2'_0`
M<906E)K@`Q2J$_[_E_8#`*KA4P&1X`,3JJ_^_Y=@"`"T]0,`J@S^_Y??`@#K
M"0<`5/=7@+G8``"0]_)]TV`"0#D?]`!Q!!A`>F`,`%3C`Q.J'R`#U6(<0#A?
M]`!Q1!A`>J'__U1C`!/+8WQ`DQB_1_G`"@"1```#BQ8#0/GZ_?^7P&HW^,`"
M`+3C5X"YX`,3J@(#0/D!%$`X0WAC^#_T`'$D&$!Z80``.<```%0"%$`X8AP`
M.%_T`'%$&$!Z@?__5*$'@%(``(#2810`.()J8#AB:"`X``0`D:+__S7```"0
M`,!'^>$O0/D"`$#Y(0`"ZP(`@-*A!P!4\U-!J?5;0JGW8T.I_7O&J,`#7]8!
M`(#2'R`#U8)J83BB:B$X(00`D:+__S7N__\7V```D/EK!*D`OT?Y&@!`^4`#
M0/D`!0"TX`,:JA4`@%(?(`/5`8Q`^.(#%2JU!@`1H?__M4`,`!$`?'V3P0``
ML"(@0[DB`@`U(@"`4B(@`[F[_?^7^0,`JH`#`+2W?GV3X0,:JN(#%ZK)_/^7
M`+]'^1D``/F@!@`1/]L@^/57`+GY:T2II/__%^$#`*K@`QJJN_S_E_D#`*H!
MOT?Y(```^4`!`+2W?GV3\___%P,`@-*E__\7``*`TA4`@%+@__\7^6L$J?[\
M_Y?Y:T2INO__%_U[OZE!T#O5P@``D/T#`)'```"0`+A'^1\@`]4?(`/5_7O!
MJ",``/!"J$?Y8P`;D2!H8/@!``#P(>`7D3#]_Q?]>[ZI0=`[U?T#`)'S"P#Y
MP```D`"X1_D?(`/5'R`#U3,``(L@:&#XYO__E\(``)`$``#P8`)`^83@))'S
M"T#YXP,$JOU[PJ@!``#P0O!'^2%@&)$%`(!2-?W_%P``@-+``U_6'R`#U1\@
M`]7_`Q31P@``D/U[`*G]`P"10L!'^?-3`:GS`P&J]5L"J>%S`9'U`P"J0P!`
M^>-_`OD#`(#2````\`#`&)']_?^7]`,`JN$%@%+@`Q6J@?S_ET`"`+3A`Q6J
M````\```&9$Q__^7X`,5JL$``)`AP$?YXG]"^2,`0/E"``/K`P"`TL$,`%3]
M>T"I\U-!J?5;0JG_`Q21P`-?UN`#$ZHA_?^7%@``\,%"&9&V_/^7\P,`JB#]
M_[3W8P.I&```\!<``/`8(Q:1]V(9D?DC`/D?(`/58`)`.>`&`#2T``"TX0,3
MJN`#%*H[_?^70`4`-.`#$ZHP_?^7`@0`T?D#`*IB`@*+?P("ZZ,``%0*```4
M7_0?.'\"`NN```!400!`.3^\`'%@__]4X`,3JB+]_Y?Y`P"JX`,5JA_]_Y<9
M`!F+.0<`D3__#_$(`P!4Y`,3JN8#%:KE`Q>J`P``\&.`&9'SXP.1`H"`T@$`
M@%+@`Q.JI?S_E^`#$ZKA@P&17OS_EZ```#7@:T"Y``P4$A\@0'%@`@!4P4(9
MD0``@-)Z_/^7\P,`JH#Y_[7W8T.I^2-`^:K__Q?T`@"T@`)`.1^X`'&```!4
M\P,8JCD`@-+:__\7@`9`.0#^_S3[__\7X`,3JB$`@%*\_/^78/W_->$#$ZH`
M``#P```9D<S^_Y?@`Q.JR?S_E_=C0ZGY(T#YF/__%_,#&*HY`(#2Q___%_=C
M`ZGY(P#Y2/S_E_U[OJGA!8!2_0,`D?,+`/GS`P"J$OS_EQ\``/%@!H":\PM`
M^?U[PJBU_/\7``,`M/U[OJG]`P"1\PL`^?,#`*H!`$`Y(0$`--'\_Y<`!`"1
M'P`0\4@!`%3@`Q.J\PM`^?U[PJ@K_O\7\PM`^0```/#]>\*H`"`6D:'\_Q<`
M`(#2\PM`^?U[PJC``U_6````\``@%I&:_/\7'R`#U?U[NJG```"0_0,`D0#`
M1_GU6P*IU0``D/-3`:D4``#P\U,!D?=C`ZF4HAF1^2,`^;6^1_D!`$#YX2\`
M^0$`@-+A`Q.JX`,4JDK]_Y?@`0"TH@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T
M`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J//W_EV#^_[75``"0%```\)3B&9&U
MOD?YX0,3JN`#%*HT_?^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(
M0/D!!`#Y`"``D:'__[7A`Q.JX`,4JB;]_Y=@_O^UU0``D!0``/"4`AJ1M;Y'
M^>$#$ZK@`Q2J'OW_E^`!`+2B`D#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y
M`00`^0`@`)&A__^UX0,3JN`#%*H0_?^78/[_M=4``)`4``#PE$(:D;6^1_GA
M`Q.JX`,4J@C]_Y?@`0"TH@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$
M`/D`(`"1H?__M>$#$ZK@`Q2J^OS_EV#^_[75``"0%```\)9B&I&UOD?YX0,3
MJN`#%JKR_/^7X`$`M*("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y
M`"``D:'__[7A`Q.JX`,6JN3\_Y=@_O^UU@``D!4``/"UHAJ1UKY'^>$#$ZK@
M`Q6JW/S_E^`!`+3"`D#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@
M`)&A__^UX0,3JN`#%:K._/^78/[_M;8``/`5``#0M\(8D=:^1_GA`Q.JX`,7
MJL;\_Y?@`0"TP@)`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1
MH?__M>$#$ZK@`Q>JN/S_EV#^_[6W``#P%@``T-CB&I'WOD?YX0,3JN`#&*JP
M_/^7X`$`M.("0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'_
M_[7A`Q.JX`,8JJ+\_Y=@_O^UN```\!<``-#Y(AN1&+]'^>$#$ZK@`QFJFOS_
ME^`!`+0"`T#YXU>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^U
MX0,3JN`#&:J,_/^78/[_M;D``/`8``#0&&,;D3F_1_GA`Q.JX`,8JH3\_Y?@
M`0"T(@-`^>-7@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#
M$ZK@`QBJ=OS_EV#^_[6Y``#P&```T!@#&9$YOT?YX0,3JN`#&*IN_/^7X`$`
MM"(#0/GC5X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.J
MX`,8JF#\_Y=@_O^UX0,3J@```-``H!N16_S_E^$#`*I@``"TX"(;D93]_Y?A
M`Q.J````T```')%3_/^7X0,`JH```+0```#0`$`6D8O]_Y?A`Q.J````T`!@
M')%*_/^7X0,`JF`"`+2@PAB1@_W_EZ```/``P$?YX2]`^0(`0/DA``+K`@"`
MTF$"`%2`8AJ1`0``T/-30:DA`!V1]5M"J?=C0ZGY(T#Y_7O&J'/]_Q?A`Q.J
M````T`"@')$R_/^7X0,`JF#]_[3`XAJ1:_W_E^C__Q?P^O^7_7N^J:(``/``
M``#0_0,`D4+`1_GA4P"1`.`:D4,`0/GC#P#Y`P"`TB'\_Y?``0"T``!`.1_`
M`'$$&$!ZX`>?&J$``/`AP$?YX@]`^2,`0/E"``/K`P"`TJ$``%3]>\*HP`-?
MU@``@%+V__\7U/K_E_U[KJFH``#P[`:`$OT#`)$*P4?YX!N`/>V#`Y'A'X`]
MZP^`$NA#`)'B(X`]J0``\.,G@#WD*X`]Y2^`/>8S@#WG-X`]X8L.J>$#`*KB
M`PBJXY,/J>6;$*GGCP#Y(.E'^4,!0/GC+P#Y`P"`TN.#!)'CCP.I`,`$D>TG
M`/GL+PHIY)=#J>07`:GDET2I!!4!J2/[_Y?@'X!26?K_E_]#`=&C``#P_T-`
MT0$`@-+D0T"10@"`4OU[`*G]`P"18\!'^?-3`:GT`P`J90!`^84D`/D%`(#2
M(OO_EP`$`-'S`P#K`3Q`DG,^0)(S1)/:$P`3Z\0'`%3U6P*I%@``T!4!@-+6
M8AV1]V,#J1<``-#X`P&1]R(=D34`H/(?(`/5X0,3JN`#%"H"`(!2#?O_EQ\$
M`+$`!@!4X0,8JN(#%:K@`Q0JM_K_EP%\0),?!``Q``0`5#\<`/$)!`!4(2``
MT0$#`:NB`P!4P```D`0``!0A!`#1/P`8ZP,#`%0#"$'Y(@!`^7\``NM!__]4
M(``8R_5;0JD``!.+]V-#J:$``/#B0T"1(<!'^4,D0/DD`$#Y8P`$ZP0`@-)!
M`@!4_7M`J?-30:G_0P&1_T-`D<`#7];@`Q:JC/__EW-"0-%_0D"Q`?K_5/5;
M0JGW8T.I``"`DNO__Q?@`Q>J@___E\___Q?U6P*I]V,#J5+Z_Y<?(`/5_7N_
MJ?T#`)'B^O^7V?K_ET```+0``$#Y_7O!J,`#7];]>[RIH@``\/T#`)%"P$?Y
M\U,!J>'3`)'U6P*I%0``T+:B'9%#`$#YXQ\`^0,`@-+T`P"JX`,6JG;[_Y>`
M``"T\P,`J@``0#G``0`UH```\`#`1_GA'T#Y`@!`^2$``NL"`(#2@00`5.$#
M%*J@HAV1\U-!J?5;0JG]>\2HHOS_%^`#%*K#^O^7]0,`JN`#$ZK`^O^7H`(`
MBP`(`)'!^O^7Y`,3JN(#%*KS`P"J`P``T`$``-!C0!F1(8`9D:GZ_Y>@``#P
M`,!'^>$?0/D"`$#Y(0`"ZP(`@-+A``!4X0,3JN`#%JKS4T&I]5M"J?U[Q*B%
M_/\7"_K_EQ\@`]4?(`/5_7NGJ<(``)"A``#P0D`(D?T#`)$AP$?Y\U,!J>,C
M`I'U6P*I\\,!D?5C`9'W8P.I]P,`JO13`9%&'$&I(`!`^>#'`/D``(#22JQ"
MJ>:?!:D8``#02*1#J>$#%*I$E$2I`,,8D4RT1:EH)`&I3A!`^:X*`/E&-$#Y
M9@H`^604`JGL-P>IZJ\(J2+[_Y>```"T`0!`.?8#`*K!'@`U^6L$J9#__Y?Y
M`P"JH!L`M.`#&:IX^O^7`/A_TP`$`)%Y^O^7(@-`.?8#`*I"`0`T&@``T/,#
M%JI:`QZ1X`,3JN$#&JI@^O^7<PH`D2(?0#AB__\U`0``T/E#`I$@0!:1`P``
M%""'0/A`"`"TX0,4J@'[_Y?S`P"J8/__M`$`0#DA__\TF/O_E^#^_S3@`Q.J
M-?K_E_,#`*K`!@"TX`,3JE7Z_Y?U`P"JX`,6JE+Z_Y>Y`@`+.1,0$1H``-`Y
M?T"3X`,9JE#Z_Y?E`Q:JX@,3JD-C&9$$``#0A"`>D?4#`*H!``#0(4`>D3?Z
M_Y?@`Q6J`3B`4OSY_Y<?!``Q`!(`5.'C`I'@`Q6J@_G_E_,#`"H@!0`TH```
M\`$``-#B`D#YXP,5J@#H1_DA@!^1%`"`T@#`!)&(^?^7H```\`#`1_GAQT#Y
M`@!`^2$``NL"`(#201X`5.`#%*KS4T&I]5M"J?=C0ZGY:T2I_7O9J,`#7]89
M`(!2L]IY^+,!`+1@`D`Y8`$`-#D'`!'@`Q.J5OO_ER#__S3@`Q.J\_G_E_,#
M`*K@]_^UL]IY^+/^_[43`(#2N___%^##0+D`#!02'Q!`<:'Z_U3VUT"Y`_K_
ME]\"`&LA^O]4X,-`N0`@`!(?``=QH?G_5.`#&:H'^O^7X0,4JO0#`*H```#0
M`.`@D9[Z_Y?A`P"J0!``M.`"0/F._/^7^0,`JL`/`+3@`QFJ]OG_E_8#`*H?
M#`!Q[0$`5`!\0),!``#0`!``T2$`(9$@`P"+]?G_EP`!`#7?$@!QH!0`5-86
M`%$@RW8X'[P`<2`4`%0?(`/56?[_E\```#7@`QFJ`0"`4LSY_Y?V`P`JP`P`
M-0$``-````#0(0`=D0#@&I&V^_^76F,9D1<``-#WXB21H?G_EQD``-#D`P`J
MY0,7JN,#&JKB`Q6JX`,4J@$``-`Y`R*1(<`AD;_Y_Y</```4??G_EP``0+D?
M1`!Q`0(`5)'Y_Y=S!@`1Y`,`*N4#$RKF`Q>JXP,:JN(#%:KA`QFJX`,4JK#Y
M_Y?@`Q2J`3B`4G7Y_Y<?!``QP/W_5.`#%:KM^/^7X0,4J@##&)&/^_^7X`,4
MJLS^_Y>`__\78OG_EP``0+D?1`!QP.W_5/,"0/E=^?^7I0``\`0`0+GB`Q.J
MXP,5JJ7H1_D4`(#2`0``T"&`'I&@P`21]_C_EV___Q<6`(!28-IV^(`!`+3A
M`Q2JU@8`$3CZ_Y=@__^T`0!`.2'__S1P^?^7^0,`JB#C_[5@VG;XP/[_M:`!
M@-*2^?^7&0``T/8#`*HYXQV18@J`4AC__Q>D_O^7H```\`#`1_GAQT#Y`@!`
M^2$``NL"`(#280D`5.`#%JKS4T&I]5M"J?=C0ZG]>]FH5_G_%_D"0/F"__\7
M,/[_E_<#`*H@\_^W`2@`T0(`@%+@`Q8J^4,%D5GY_Y?A`QFJX`,6*L(`@-(%
M^?^7'Q@`\8```%0```#0`$`AD?G]_Y?A4T&Y`&"(4B!HJ'(_``!KP?#_5"$+
M0'D`J8A2/P``:T'P_U3AR@#1`@"`4N`#%BKS8P610OG_E^`#%BKA`Q.J`@6`
MTN[X_Y<?H`#Q@```5````-``0"&1XOW_E^0#$ZI#8QF1X@,5JN`#%*H%``#0
M`0``T*7@))$A@"&1_P,&.3OY_Y>0__\7]@9`^1;L_[3@`Q:J0OG_EQ\,`/&)
MZ_]4`!``T0$``-#``@"+(2`AD4/Y_Y<?``!Q.1.6FE3__Q>;^/^7^6L$J9GX
M_Y?]>[NIH@``\`0``-#]`P"1\U,!J5/0.]6@``#P`+A'^1\@`]4?(`/5]5L"
MJ?43`9'A`Q6J]QL`^4+`1_EC:F#X@&`:D44`0/GE)P#Y!0"`TF3`0CEDQ`(Y
MOOG_EX```+0!`$`Y/\0`<8`+`%1=_/^7H```\`"X1_D?(`/5'R`#U78"`(M@
M:F#X`0``T&,`@%(A8"*1(@"`4I[X_Y?T`P"J8`$`M`$(0/D"``#0P`)`^4*`
M(I$A`$#Y<OC_EX`*0/D!`$#Y(`Q`N:`*L#>@``#P`+A'^1\@`]4?(`/5=P(`
MBV!J8/@!``#08P"`4B&@(I$B`(!2U@``D(;X_Y?T`P"J8`$`M.$#%:H```#0
M```9D9#Y_Y?B`P"J0`<`M($*0/G@`D#Y(0!`^5;X_Y>@``#P`+A'^1\@`]4?
M(`/5=`(`BW-J8/@B`(!2`0``T"'`(I'@`Q.J?_C_EP(``-#A`P"J0B`CD>`#
M$ZI&^/^7TX(,D8("0/GA`Q6J````T`#`&)%?Q`(Y<_G_EV`*`/G``@"T`3B`
M4HOX_Y<!!``1/P0`<02(47I!!`!4H```\`#`1_GA)T#Y`@!`^2$``NL"`(#2
M00,`5/-30:GU6T*I]QM`^?U[Q:C``U_6``1`.:#T_S2C__\78`9`^1;^_Y=@
M"@#Y`/W_M0```-``0".16?C_E\""#)&""D#Y`01`^4``0/DA`$#Y2_K_E\7_
M_Q?``D#Y9/C_EZK__Q<.^/^7UH(,D>,#`"H```#0`.`CD<&*0*DA`$#Y1_C_
MEQ\@`]4?(`/5_7N^J?T#`)'S"P#Y\P,`JA']_Y=_`@#Q!!A`>H$``%3S"T#Y
M_7O"J,`#7]9@`D`Y@/__-.`#$ZJZ^_^7KOO_EP$``-""`(#2(2`>D;GW_Y>`
M_O\UX`,3JO,+0/G]>\*H9?G_%P```````````````/^#"M&A``#PXF,`D07@
M`)$(X`&1Y`,"JOU[`*G]`P"1(<!'^2,`0/GC3P'Y`P"`TJ$`0+F$0`"1HX1`
M^"$(P%J!`!_X8?Q@TV?\:-/G''B2)MQXTR$<"%/&''"2X^!#JB$`!JHA``.J
M@8`?^!\!!>L!_O]410`(D4$`0/E"(`"10P1`^40<0/EC``'*03!`^2$`!,HA
M``/*(_Q?TV$$`:HA?$"203P`^5\`!>MA_O]4!RA`J20SC](()$&I1%"K\N6;
M0:DC=)W2XOQ;T^S\0M-"%`>JC'D'JB$!",HCVZWR(0`*BB$`"<JE``2+(D`B
MBT']0M,A>`JJ0@`%BPL00/DM?$"2I0$(RL8`!(M"``L+I0`'BJ4`",HA``S*
M3OQ;TT_\0M.E``F+SA4"JJ4`!HLA``**XGD"JJ4`#@OFOT*I(0`-RJ[\6],A
M``B+SA4%JK#\0M/&``2+[P$$BR$`!HN&`0+*(0`."XQ]0)+&``6*Q@`,RB[\
M6]/&``V+SA4!J@UZ!:K&``^+Q@`."T4`#<KN'T#Y0GQ`DJ4``8HP_$+3I0`"
MRLX!!(NE``R+S_Q;TPQZ`:KO%0:JH0$,RJ4`#HON(T#YI0`/"ZU]0)(A``:*
M(0`-RL_\0M/F>0:J(0`"B\(!!(NN_%O3(0`"BZ_\0M."`0;*SA4%JBX`#@M"
M``6*Y7D%JHQ]0)+AOT2I0@`,RD(`#8O-_5O3K14.JM!\0)(A``2+0@`!B\$`
M!<I-``T+(0`.BB$`$,K&_4+3X@$$B\YX#JHA``R+K_U;TZ9\0)(A``*+[Q4-
MJJ(`#LJE_4+3+P`/"Z5X#:I"``V*X;=%J4(`!LI"`!"+[/U;TXP5#ZKP_4+3
M(0`$BT(``8O!`07*(0`/BLY]0)(A``[*3``,"P]Z#ZJB`02+(0`&BXW]6],A
M``*+H@`/RD(`#(JE?$"20@`%RJT5#*I"``Z++0`-"_"[1JF&_4+3QG@,JJ']
M6],L%`VJX0$&RB$`#8KO?4"2(0`/RA`"!(LA``6+0@`0B[#]0M-,``P+#7H-
MJL(!!(LA``*+P@`-RD(`#(K&?$"20@`&RH[]6]-"``^+SA4,JO"_1ZDN``X+
MA?U"TZ5X#*K!_5O3+!0.JJ$!!<HA``Z*K7U`DA`"!(LA``W*0@`0B]#]0M-,
M``P+#GH.JN(!!(LA``:+(0`"BZ(`#LI"``R*I7Q`DD(`!<J/_5O30@`-B^\5
M#*KPMTBI+P`/"X;]0M/&>`RJX?U;TRP4#ZK!`0;*(0`/BLY]0)(0`@2+(0`.
MRD(`$(OP_4+33``,"P]Z#ZJB`02+(0`%BR$``HO"``_*0@`,BL9\0))"``;*
MC?U;TT(`#HNM%0RJ\+M)J2T`#0N%_4+3I7@,JJ']6],L%`VJX0$%RB$`#8KO
M?4"2$`($BR$`#\I"`!"+L/U"TTP`#`L->@VJP@$$BR$`!HLA``*+H@`-RH[]
M6].E?$"20@`,BLX5#*I"``7*+@`."T(`#XN&_4+3X;]*J<QX#*K&_5O3QA0.
MJB$`!(OD`02+0@`!BZ$!#,HA``Z*K7U`DB$`#<K/_4+3T@`""^YY#JHA``6+
M0D`FB^]?0/DA``2+A`$.RD7^6].$``)*I102JN(!`XN$``V+3?Y"T[```0NR
M>1*JA``"B^)C0/DA0"6+Q0$2R@;^6].E``%*QA00JD$``XL-_D+3HD`LB\\`
M!`NM>1"J0@`!B^%G0/F$0":+1@(-RN7]6]/$``1*I10/JB8``XN!0"Z+L0`"
M"R$`!HOL_4+30D`EB^9K0/F/>0^JI0$/RB3^6].$%!&JI0`"2L(``XLF_D+3
MD``!"]%X$:KF;T#Y(4`DB^0!$<JL0#*+A``!2@7^6].E%!"J@@$"B\$``XL,
M_D+3AD`MBZX``@N0>1"JP0`!B^9S0/E"0"6+)0(0RL3]6].B``)*A!0.JL8`
M`XM"0"^+S/U"TXT``0N.>0ZJ0@`&B^9W0/DA0"2+!`(.RJ7]6].!``%*I10-
MJL8``XLA0#&+KP`""R$`!HNL_4+30D`EB^9[0/F->0VJQ0$-RN3]6].$%`^J
MI0`"2L(``XOF_4+3D0`!"\]X#ZKF?T#Y(4`DBZ0!#\JL0#"+A``!2B7^6].E
M%!&J@@$"B\$``XLL_D+3AD`NB[```@N,>1&JP0`!B^:#0/E"0"6+Y0$,R@3^
M6].B``)*A!00JL8``XM"0"V+#?Y"TXX``0M"``:+IGD0JO"'0/DA0"2+A`$&
MRL7]6].!``%*I10.JA`"`XLA0"^+K0`""R$`$(O/_4+30D`EB_"+0/GN>0ZJ
MQ0`.RJ3]6].$%`VJI0`"2@("`XNP_4+3CP`!"PUZ#:KPCT#Y(4`DB\0!#<JL
M0"R+A``!2N7]6].E%`^JAD`FBP$"`XN"`0*+\/U"TZP``@O!``&+!GH/JO"3
M0/E"0"6+I0$&RH3]6].B``)*A!0,JA`"`XM"0"Z+COU"TX\``0O,>0RJ0@`0
MB_"70/DA0"2+Q``,RN7]6].!``%*I10/JA`"`XLA0"V+[?U"TZX``@NO>0^J
M(0`0B_";0/E"0"6+A0$/RL3]6].B``)*A!0.JD)`)HL%`@.+T/U"TXT``0M%
M``6+`GH.JO"?0/DA0"2+Y`$"RJ;]6].$``%*QA0-J@$"`XN$0"R+L/U"TX0`
M`8L!>@VJS@`%"Z5`)HM&``'*Q0`%2LS]6].E0"^+S_U"TXP5#JKN>0ZJ\#]4
MJ8T!!`N$0"R+IOU;T\84#:H0`@.+I0`0BS``#LK,``4+$`($2@)"(HOD`0.+
ML/U"TX_]6],->@VJ[Q4,JD(`!(NE0":+Y*M`^4(`#PO&`0W*Q0`%2H,``XN&
M_4+3H4`ABT3\6]/,>`RJA!0"JB$``XLA``0+HP$,JN2O0/F%`0T*8P`"BD_\
M0M-C``6JAIN7TF;CL?+B>0*JA``&BV-`+HLE_%O3+OQ"TZ44`:IC``2+A`$"
MJF,`!0N$``&*P7D!JN4[5JE/``P*A``/JF_\0M.$0"V+;?Q;TZT5`ZJE``:+
MS@$&BX0`!8M%``&JA``-"Z4``XHM``(*I0`-JHW\6].E0"R+['D#JJ4`#HNC
M%02J+0`,JJ,``PONNT#YI0$$BHT!`0J/_$+3I0`-JLX!!HNE0"*+;?Q;T^)Y
M!*JM%0.JA`$"JJ4`#HONOT#YI0`-"X0``XI-``P*A``-JF_\0M/.`0:+A$`A
MBZW\6]/A>0.JK14%JH0`#HN$``T+0P`!JNW#0/DN``(*8P`%BJ_\0M-C``ZJ
MY7D%JJT!!HMC0"R+COQ;TRP`!:K.%02J8P`-B^W'0/EC``X+C`$$BJX``0J,
M`0ZJCOQ"TX)!(HNL`0:+;?Q;T\1Y!*JM%0.J0@`,BT(`#0NL``2J[<M`^8X`
M!0J,`0.*;_Q"TXP!#JKC>0.JK0$&BX%!(8M._%O3C``#JLX5`JHA``V+(0`.
M"XP!`HIN``0*C`$.JD[\0M/">0*JA4$EB^V[6:DO_$+3K`$&BRW\6].M%0&J
MI0`,BVP``JJE``T+C`$!BDT``PJ,`0VJX7D!JLX!!HN$022+K?Q;TTP``:JM
M%06JA``.B^[70/F$``T+C`$%BBT``@J,`0VJK_Q"T\X!!HN#02.+C?Q;T^QY
M!:JM%02J)0`,JF,`#HNE``2*[MM`^6,`#0N-`0$*C_Q"TZ4`#:KD>02JS@$&
MBZ)`(HMM_%O3A0$$JJT5`ZI"``Z+[M]`^4(`#0NE``.*C0`,"F_\0M.E``VJ
MS@$&BZ%`(8M-_%O3Y7D#JJT5`JJ#``6J(0`.BT[\0M,A``T+8P`"BL)Y`JJO
M``0*[3M<J6,`#ZIL0"R+(_Q;TV,4`:HO_$+3K0$&BXP!#8NM``*J@P$#"ZP!
M`8I-``4*X7D!JHP!#:IM_%O3A$$DB\P!!HN$``R+3``!JBX``@J,`0.*C`$.
MJJT5`ZJ%026+A``-"^\S7:EM_$+3K7D#JH[\6]/.%02J[P$&BZ,`#XLE``VJ
M8P`."Z4`!(JN`0$*I0`.JH[\0M/$>02JHD`BBX4!!HML_%O3C!4#JD(`!8MN
M_$+3I0$$JD(`#`NE``.*PWD#JH\`#0KL.UZII0`/JJ5`(8M!_%O3(10"JD_\
M0M.,`0:+I0`,BXP``ZJE``$+C`$"BF$`!`KB>0*JC`$!JE```PJ!02V+S0$&
MBZS\6],A``V+C!4%JFT``JKO^T#YC@$!"ZT!!8HL0"R+K0$0JM#]0M.D022+
M[0$&BZ_\0M/&_5O3QA0.JH0`#8OM>06JT@`$"^__0/E!``W*(0`,2L4ZF-)%
M3+GR(4`CB^P!!8M#_EO3#GH.JF,4$JHA``R+A$`FB^P#0?EQ``$+I@$.RB%`
M(XO$``1*A@$%BX)`(HM,_D+3)/Y;TY)Y$JJ$%!&J0@`&B^P'0?F0``(+PP$2
MRB;^0M-C``%*@0$%BVQ`+8L#_EO3T7@1JHP!`8MC%!"J0D`DB^8+0?EO``P+
M1`(1RH-!(XN$``)*X?U;T\(`!8N&0"Z+!/Y"T\8``HN0>!"J(10/JN0/0?DN
M``8+(@(0RNS]0M-"``-*@P`%BT1`,HO"_5O3CWD/JH0``XM"%`ZJP4`AB^,3
M0?E-``0+!@(/RLS]0M/!``%*9@`%BR-`,8NA_5O3CGD.JF,`!HLA%`VJ[`$.
MRN870?DR``,+@D`BBV%`(8N"`0)*Q``%BTQ`,(NF_4+30OY;T\UX#:J$`02+
M0A02JN8;0?E1``0+PP$-RDS^0M-A``%*PP`%BR9`+XLA_EO3PP`#BXQY$JHA
M%!&J@D`BB^8?0?DP``,+I`$,RF%`(8N"``)*Q@`%BT1`+HLN_D+3`OY;TX0`
M!HM"%!"JQGD1JNXC0?E/``0+@P$&RH)`(HMA``%*S@$%BR-`+8L-_D+3X?U;
MT[!Y$*IC``Z+(10/JNXG0?DM``,+Q``0RF%`(8N"``)*Q`$%BTQ`+(ON_4+3
MHOU;T\]Y#ZJ$`02+0A0-JNPK0?E.``0+`P(/RH)`(HMA``%*@P$%BR9`)HNL
M_4+3P?U;T\,``XLA%`ZJAGD-JNPO0?DM``,+Y`$&RF%`(8N"``)*C`$%BT1`
M,(O0_4+3HOU;TPYZ#JJ$``R+0A0-JNPS0?E1``0+PP`.RH)`(HMA``%*C`$%
MBR-`+XNO_4+3(?Y;T^UY#:IC``R+(101JNPW0?DP``,+Q`$-RB_^0M."``)*
MA`$%BTQ`)HL"_EO3A@$$BT(4$*KL>1&J84`AB^0[0?E/``8+HP$,RL)`(HMA
M``%*@P`%BR1`+HL._D+3X?U;TX0``XLA%`^JPWD0JO`_0?DN``0+A@$#RH%`
M(8O"``)*$`(%BT9`+8OM_4+3POU;TZ]Y#ZI"%`ZJQ@`0B^U#0?E0``8+9``/
MRL)`(HN$``%*H0$%BXQ`+(O-_4+3!/Y;TXP!`8NN>0ZJA!00JN%'0?F1``P+
MY@$.RN=`+XO&``)*(@`%B\-`(XLA_EO38P`"BR$4$:IC``$+`?Y"T^U+0?DA
M>!"JAD$DB\(!`<ID_%O3K```T$8`!DJ$%`.JI0$%BR+^0M.$``:+I0`'BX;!
M1_E">!&J(0`)"\L!"PMC``H+A``%"T@`"`L$#`"I"`0!J0L0`/G@3T'YP0!`
M^0```>L!`(#2@0``5/U[0*G_@PJ1P`-?ULCS_Y?]>[^I`!"`TOT#`)%D]/^7
M'Q@`^29@A-(E<972Q)^;TL..BM("/IS2INBL\J7YO?)$%[/R0P:B\D)ZN/(&
M%`"I!`P!J0)\`JD?>`"Y_7O!J,`#7]8?(`/5'R`#U?U[O*E"?$"3_0,`D?-3
M`:D4X`"1\P,!JO5;`JGU`P"J]@,"JO=C`ZD%C$*I`'A`N:0,`@N_``3K8Y2#
MFF)T0HNDB@*IX`(`-!<(@%+W`@!+WP(7:X#"((O7TI<:^'Y`D^(#&*J/\_^7
MH'I`N>`"``N@>@"Y'P`!<<```%3S4T&I]5M"J?=C0ZG]>\2HP`-?UM8"%TMS
M`AB+X`,5JJ'[_Y??_@!Q[0,`5-8"`5'8?@93%P<`$?=J>M-W`A>+'R`#U6`&
M0*F`!@"I<P(!D6(.?:F"#@&IX`,5JF(.?JF"#@*I8@Y_J8(.`ZF.^_^7?P(7
MZX'^_U36&AA+X0,7JL)^0)/@`Q2J9_/_E[9Z`+GS4T&I]5M"J?=C0ZG]>\2H
MP`-?UO<#$ZKU__\7'R`#U1\@`]7]>[RI_0,`D?-3`:GS`P&J]`,`JO5;`JEU
MX@"1X0^`$O<;`/E@%D#Y=BI`N7<R0+D`(`-3U@K`6@($`!'W"L!:H<H@.*#"
M(HM?X`!QC0D`5`$(@%(B``)+`0"`4D)\0)-P\_^7X`,3JF+[_Y=__@.I?_X$
MJ7_^!:E_-@#Y=UH.*>`#$ZI;^_^780)`^>`#$ZHA_%C3@0(`.6$"0/DA_%#3
M@08`.6$"0/DA_$C3@0H`.6$"0/F!#@`Y809`^2'\6-.!$@`Y809`^2'\4-.!
M%@`Y809`^2'\2-.!&@`Y809`^8$>`#EA"D#Y(?Q8TX$B`#EA"D#Y(?Q0TX$F
M`#EA"D#Y(?Q(TX$J`#EA"D#Y@2X`.6$.0/DA_%C3@3(`.6$.0/DA_%#3@38`
M.6$.0/DA_$C3@3H`.6$.0/F!/@`Y81)`^2'\6-.!0@`Y81)`^2'\4-.!1@`Y
M81)`^2'\2-.!2@`Y81)`^8%.`#GS4T&I]5M"J?<;0/G]>\2HVO+_%P$'@%(B
M``)+`0"`4D)\0),E\_^7N___%P```````````````*```-``0`N1`2!!.4$#
M`#6A``"P(@"`4@(@`3D@I$?YH`$`M/U[OZF@``#0_0,`D0`$0?FG\O^7H```
ML`#41_E@`0"T_7O!J"```)``X"&15//_%Z```+``U$?YP```M"```)``X"&1
M3O/_%_U[P:C``U_6P`-?U@``````````H```T`%`"Y$"0$LY0@$`-:(``+`C
M`(!2`T`+.4"T1_F@``"T(2``D2```)``X"&1U/+_%\`#7]8``````````/U[
MNJFC``"P_0,`D6/`1_GS4P&I]5L"J>*'`ZGBXP"1X0,!D>!/`+G@,P&19`!`
M^>0O`/D$`(#2]O+_EZ```+"B``"P`.!'^4+,1_GA`P*J`//_EZ```+``L$?Y
M``!`.8`)`#1!T#O5LP``T*```+``N$?Y'R`#U1\@`]4C:&#XXD]`N6$(4CE`
M$``15`P`$2$$`#)A"!(Y`'Q]D\WR_Y?C`P"JH0``L&*"#)'F(T#Y````D"3(
M1_D`("61Y4]`N0$``)#'`$#Y(0`ED80`0/EG!`"I0P0`^60``:F_!`!Q+0D`
M5"$`@-(@#``1Q'AA^"$$`)%DV"#XOP`!:VS__U2E#``1=8(,D7_8)?CA__^P
MX@,4*B%0*Y$$`(#2H`Y`^>#R_Y<@!0`T=8(,D?93`9&@#D#Y>_+_E_0#`"KA
M`Q:J````D`"@&I&V\_^70`0`M'."#)%@#D#Y7O+_E\'R_Y>@``"P`,!'^>$O
M0/D"`$#Y(0`"ZP(`@-+A!`!4X`,4*O-30:GU6T*I_7O&J,`#7]9T\O^7LP``
MT&*"#)%`#`#Y8`,`M#OR_Y=!T#O5H```L`"X1_D?(`/5'R`#U2!H8/@?&"0Y
MJO__%Z`.0/G9\O^7UO__%Z`*0/E@``"T4?K_E]S__Q?A`Q:J````D`#`&)&,
M\_^7H`H`^<#Z_[1)^O^7U/__%X4`@%*___\7(`"`4O3Q_Y='\O^7<'-?<W1R
M:6YG<R!M:7-S:6YG"@``````+W1M<"\````N`````````%!!4E]435!$25(`
M``````!435!$25(``%1%35!$25(`5$5-4`````!435```````%5315(`````
M55-%4DY!344``````````"5S+R5S````+W!R;V,O8W5R<')O8R]E>&4`````
M````1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R``````!);G1E<FYA;',Z
M.E!!4CHZ0D]/5`````!005)?5$5-4```````````4$%27U!23T=.04U%````
M`#H`````````+P`````````E<R5S)7,``%!%4DPU3$E"``````````!015),
M3$E"`%!%4DPU3U!4``````````!015),24\``%!!4E])3DE424%,25I%1`!0
M05)?4U!!5TY%1```````4$%27T-,14%.`````````%!!4E]$14)51P``````
M``!005)?0T%#2$4`````````4$%27T=,3T)!3%]$14)51P``````````4$%2
M7T=,3T)!3%]435!$25(`````````4$%27T=,3T)!3%]414U0`%!!4E]'3$]"
M04Q?0TQ%04X``````````#$`````````;'-E96L@9F%I;&5D`````')E860@
M9F%I;&5D``````!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````'!A
M<BT`````)7,E<R5S)7,``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E
M('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@
M<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M
M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*````````4$%42```
M``!P87)L`````"YP87(`````<VAO<G0@<F5A9````````"5S)7-C86-H92TE
M<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M)74E<P``````````
M&`````````!P97)L`````#``````````4$%2.CI086-K97(Z.E9%4E-)3TX`
M````,2XP-C,```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``
M````````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T
M;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@``````````+64````````M+0``
M`````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")]
M)PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G
M92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE
M+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN
M4$%2+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").
M(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<
M,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`
M7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S
M:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I
M("4@)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP
M;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F
M:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N
M9W1H*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("10
M05)?34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M
M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@
M)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@
M*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O
M<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U
M=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ
M1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK
M('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?
M=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL
M93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z
M.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G
M+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA
M;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@
M,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?
M=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@
M,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@
M,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N
M=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A
M8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M
M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@
M9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN
M:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S
M>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@
M.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R
M?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/
M(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N
M86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S
M971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.
M5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,
M15]S96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S
M=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@
M>PIO=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[
M("`C(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M
M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC
M*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F
M:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N
M86UE(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A
M="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*
M<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N
M92`B7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@
M8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#
M86-H92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?
M<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D
M1DE,15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B
M=68I.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F
M<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I
M>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L
M;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@
M("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)
M3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@
M-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L
M;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA
M;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD
M77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I
M(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L
M("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*
M:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\
M86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L
M("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M
M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N
M86UE?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL
M;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D
M9G5L;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E
M860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S
M96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@
M(21M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR
M971U<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D
M=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B
M.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ
M;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE
M(")#86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I
M.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*
M<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H
M(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL
M97T@/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA
M;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T
M=7)N("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F
M:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A
M9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD[
M"G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D
M>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL
M96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S
M<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@
M>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL
M96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE
M(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E
M7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I
M('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN
M(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O
M<R`]("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F
M('II<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F
M:"P@)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@
M(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*
M:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N
M<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T
M;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S
M:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=6
M6S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!0
M7U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L
M(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!
M4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0[
M"FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS
M:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E
M<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E
M<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U
M;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@
M/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*
M96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@
M9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF
M("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'
M5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.
M5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0[
M"B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z
M.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF
M="!`24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER
M92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG
M97-T.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*
M:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A
M<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*
M8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^
M;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D
M:6YG('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@
M>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G3454
M02YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@
M<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*
M?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G
M+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@
M)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R
M(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!
M1TE#.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]
M($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)
M3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W
M+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*
M)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII
M9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(
M+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL
M;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6
M>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO
M*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E
M<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN
M="@D;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@
M(B1O=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V
M>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I
M.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!
M3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#
M;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@
M9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y
M;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@
M)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@
M(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H
M(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M
M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C
M:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*
M96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M
M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL
M("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N
M*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F
M:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I
M<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA
M<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I
M9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@
M)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;
M4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA
M;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2![
M"B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]
M('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q
M6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC
M;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT
M96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)
M37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT
M97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE
M+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG
M=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI:
M:7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L
M(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R
M('=R:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ
M<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M
M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/
M2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=
M.PI]"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%
M04XB*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY
M("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*
M)&-T>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA
M;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@
M)&-A8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE
M("X](")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?
M;F%M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*
M;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H
M+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M
M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O
M=70B.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N
M;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q
M=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR
M97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY
M("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@
M("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE
M"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P
M96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^
M<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E
M.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A
M<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I
M>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!
M4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I
M;F<@>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@
M<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB
M97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER
M96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*
M;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z
M)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY
M("1E>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P
M96,M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII
M9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E
M<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B
M('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI]
M(&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@
M(B1D97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M
M960H)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D
M:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N
M86UE(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U
M>"(["GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]
M*2!["F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI
M8BYP87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@
M<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R
M"BX*)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF
M="A`05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\[
M"G)E<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H
M+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G
M7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A
M=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET
M:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*
M<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@
M<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I
M<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*
M<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E
M<75I<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE
M.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q
M=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O
M;7!R97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$
M:6=E<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*
M<F5Q=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T
M<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I
M<F4@871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R
M97%U:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!]
M.PIE=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A
M;"![(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![
M(')E<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V
M86P@>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF
M("AD969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]
M?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H
M(&UY("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U0
M1$E2(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@
M*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@
M86YD("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![
M*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H
M(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F
M5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*
M)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6
M>U5315).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S
M=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L
M("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?
M9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M
M<"TD)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA
M;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@
M*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@
M>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@
M;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F
M+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2![
M"G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L
M96YG=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@
M)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<[
M"B1C86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*
M:68@*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%
M3E9[4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM
M?21C86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?
M5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D
M14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]
M"G-U8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@
M/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S
M<R`H+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N
M)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF
M)B`H<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE
M('%Q6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO
M9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N
M86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R
M;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE
M9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]
M?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D
M,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%
M3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O
M9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*
M?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L
M("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@
M)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O
M9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*
M<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A
M=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6
M>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]
M?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D
M<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US
M("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R
M;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*
M:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F
M:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/
M1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@
M,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*
M;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W
M9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#
M=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH
M/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI
M=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2![
M"G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*
M?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/
M1TY!344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O
M(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@
M/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'
M3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%
M3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE
M=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@
M>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O
M;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#
M;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H
M97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*
M?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF
M4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q
M6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@
M3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N
M86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>
M7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%
M4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T52
M4D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@
M:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS:2YC``````````!#
M04-(10```1L#.P0!```?````:(K__QP!``#(BO__2`$``"R+__]T`0``F(S_
M_^`!``",C?__2`(```B/__^H`@``G(___]`"``"(D/__!`,``(R1__\\`P``
MO)/__Y`#``#\D___L`,``%B4___4`P``:)3__^@#``",EO__2`0``+B6__]L
M!```*)?__Z`$``#LF___W`0``%R<___\!```^)S__Q@%``!HGO__I`4``(B>
M___$!0``B)___P`&``!(I?__<`8``)BG__^D!@``"*C__P`'``",N/__)`<`
M`.BX__]$!P``&+K__X@'``"XN___M`$``#B\__^@`0``>+S__]`&````````
M$``````````!>E(`!'@>`1L,'P`H````&````$2)__]<`````$$.,)T&G@5$
ME0)%E`.3!$K4TT+>W=4.`````"@```!$````>(G__V0`````00XPG0:>!4*3
M!)0#0I4"4@K>W=73U`X`00L`*````'````"PB?__:`$```!!#D"=")X'0I0%
MDP9!E@.5!$&7`@`````````0````G````)"Z__\X`````````"@```"P````
M_+G__W@`````2@X0G0*>`4C>W0X`20X0G0*>`4'>W0X`````9````-P```"P
MBO__]`````!!#D"=")X'0I,&E`5"F`&7`D:6`Y4$5M;50=C70][=T]0.`$$.
M0),&E`65!)8#EP*8`9T(G@=/UM5!V-=!WMW3U`X`00Y`DP:4!9T(G@="EP*8
M`4+8UP!<````1`$``#R+__]X`0```$$.X`&=')X;2)09DQI$U--(WMT.`$$.
MX`&3&I09G1R>&T28%9<61)87E1AP"M330=;50=C700M)T]35UM?80909DQI!
MEA>5&$&8%9<6```D````I`$``%B,__^4`````$$.P`&=&)X71),65@K>W=,.
M`$$+````,````,P!``#$C/__[`````!!#K`(1)V&`9Z%`4.3A`%B"MW>TPX`
M00M."MW>TPX`00L``#0``````@``?(W__P0!````00[`"$.=B`&>AP%#DX8!
ME(4!0I6$`9:#`70*W=[5UM/4#@!!"P``4````#@"``!(CO__,`(```!!#F"=
M#)X+0Y,*E`E,E0B6!Y<&F`4"0`K>W=?8U=;3U`X`00M)F@.9!%\*VME!"TG9
MVD*9!)H#0]G:09H#F01"VMD`'````(P"```DD/__0`````!!#A"=`IX!2-[=
M#@`````@````K`(``$20__]<`````$$.()T$G@-#DP).WMW3#@`````0````
MT`(``'R0__\(`````````%P```#D`@``>)#__R0"````00Z`"D*=H`&>GP%#
MDYX!E)T!0Y6<`9:;`5P*W=[5UM/4#@!!"TB8F0&7F@%%F9@!=PK8UT'900M5
M"MC70=E!"T/7V-E!F)D!EYH!09F8`2````!$`P``/)+__RP`````00X@G02>
M`T.3`D;>W=,.`````#````!H`P``1)+__VP`````0@X@G02>`T*3`DH*WMW3
M#@!!"T,*WMW3#@!""T/>W=,.```X````G`,``("2___$!````$$.8)T,G@M$
ME0B6!T*3"I0)19<&F`69!`,:`0K>W=G7V-76T]0.`$$+```<````V`,```B7
M__]P`````$$.()T$G@-7"M[=#@!!"Q@```#X`P``6)?__YP`````00Z@`ITD
MGB,```"(````%`0``-B7__]L`0```$$.4$(.T(`$1)V*0)Z)0$.3B$"4AT!-
MEH5`E89`1)B#0)>$0&+6U4+8UTK=WM/400Z`@`1!#@!!#M"`!).(0)2'0)6&
M0):%0)>$0)B#0)V*0)Z)0$;6U4'8UT*5AD"6A4"7A$"8@T!#U=;7V$&6A4"5
MAD!!F(-`EX1``!P```"@!```O)C__R``````00X0G0*>`4;>W0X`````.```
M`,`$``"\F/__^`````!!#D"=")X'1I,&E`65!)8#6`K>W=76T]0.`$$+7`K>
MW=76T]0.`$$+````;````/P$``"`F?__P`4```!!#I`#G3*>,4B3,)0OE2Z6
M+4.7+)@K7)HIF2H"60K:V4'>W=?8U=;3U`X`00L"F=G:3=[=U]C5UM/4#@!!
M#I`#DS"4+Y4NEBV7+)@KF2J:*9TRGC$"1-G:09HIF2H``#````!L!0``T)[_
M_T@"````00Y0G0J>"423")0'1I4&E@5#EP0"9PK>W=?5UM/4#@!!"P`H````
MH`4``.R@__]D`````$$.()T$G@-"DP)'"M[=TPX`00M-WMW3#@```"P```#,
M!0``H+7__Q@"````00Y@G0R>"T63"I0)E0B6!P)="M[=U=;3U`X`00L``"``
M``#\!0```*'__X00````00Z@!4:=5)Y3`Q@$"MW>#@!!"QP````@!@``8+'_
M_U0`````00X0G0*>`5/>W0X`````0````$`&``"<L?__*`$```!!#D"=")X'
M0Y,&E`5#E026`T.7`I@!60K>W=?8U=;3U`X`00MC"M[=U]C5UM/4#@!!"P`P
M````A`8``(BR__^4`0```$$.0)T(G@="DP:4!4.5!)8#0Y<"`E4*WMW7U=;3
MU`X`00L`````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````$%,```````"04@``````````
M`````````0`````````!``````````$`````````<0(````````!````````
M`'L"`````````0````````"^`@````````$`````````W0(````````/````
M`````'8$````````&0````````!`K0$``````!L`````````"``````````:
M`````````$BM`0``````'``````````(``````````0`````````2`(`````
M```%`````````#@/````````!@````````!X!0````````H`````````O@0`
M```````+`````````!@`````````%0````````````````````,`````````
MZ*\!```````"```````````&````````%``````````'`````````!<`````
M````$!<````````'`````````/@3````````"``````````8`P````````D`
M````````&`````````#[__]O``````````@`````^?__;P`````7````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````6*T!``````!0
MK0$```````````````````````````!`K0$`````````````````````````
M`````````````$"S`0````````````````!(K0$``````("R`0``````````
M````````````````````````````6*T!````````````````````````````
MR+(!``````````````````"R`0``````(#P`````````````````````````
M```````````````0'0```````!`=````````$!T````````0'0```````!`=
M````````$!T````````0'0```````!`=````````$!T````````0'0``````
M`!`=````````$!T````````0'0```````!`=````````$!T````````0'0``
M`````!`=````````$!T````````0'0```````!`=````````$!T````````0
M'0```````!`=````````$!T````````0'0```````!`=````````$!T`````
M```0'0```````!`=````````$!T````````0'0```````!`=````````$!T`
M```````0'0```````!`=````````$!T````````0'0```````!`=````````
M$!T````````0'0```````!`=````````$!T````````0'0```````!`=````
M````$!T````````0'0```````!`=````````$!T````````0'0```````!`=
M````````$!T````````0'0```````!`=````````$!T````````0'0``````
M`!`=````````$!T````````0'0```````!`=````````$!T````````0'0``
M`````!`=````````$!T````````0'0```````,"R`0``````"+(!```````*
M4$%2+G!M"@``````````@%4```````"(50``````````````````D%4`````
M``"@50```````*A5````````L%4```````"X50``````````````````P%4`
M``````#(50``````````````````4%D```````!'0T,Z("A.971"4T0@;F(S
M(#(P,C,Q,#`X*2`Q,"XU+C``1T-#.B`H;F(S(#(P,C,Q,#`X*2`Q,"XU+C``
M)$YE=$)31#H@8W)T,"Y3+'8@,2XR(#(P,3@O,3$O,C8@,3<Z,S<Z-#0@:F]E
M<F<@17AP("0`)$YE=$)31#H@8W)T,"UC;VUM;VXN8RQV(#$N,C<@,C`R,B\P
M-B\R,2`P-CHU,CHQ-R!S:W)L;"!%>'`@)``D3F5T0E-$.B!C<G1I+E,L=B`Q
M+C$@,C`Q-"\P."\Q,"`P-3HT-SHS-B!M871T($5X<"`D`"1.971"4T0Z(&-R
M=&)E9VEN+F,L=B`Q+C$W(#(P,3@O,3(O,C@@,3@Z,3<Z,3$@8VAR:7-T;W,@
M17AP("0`)$YE=$)31#H@8W)T96YD+E,L=B`Q+C$@,C`Q-"\P."\Q,"`P-3HT
M-SHS-B!M871T($5X<"`D``#QSP(`````````````````"`````@````(````
M"````!L````!```````````J*BH@3F\@3&%B96P@4')O=FED960@*BHJ``#Q
MSP(`````````````````"`````@````(````"````!L````!```````````J
M*BH@3F\@3&%B96P@4')O=FED960@*BHJ```N<VAS=')T86(`+FEN=&5R<``N
M;F]T92YN971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A>``N:&%S:``N9'EN
M<WEM`"YD>6YS='(`+G)E;&$N9'EN`"YR96QA+G!L=``N=&5X=``N<F]D871A
M`"YE:%]F<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R
M<F%Y`"YJ8W(`+F1Y;F%M:6,`+F=O=``N9V]T+G!L=``N9&%T80`N8G-S`"YC
M;VUM96YT`"Y354Y77V-T9@``````````````````````````````````````
M``````````````````````````````````````````````````````L````!
M`````@```````````@`````````"````````%P````````````````````$`
M```````````````````3````!P````(`````````&`(````````8`@``````
M`!@````````````````````$````````````````````)@````<````"````
M`````#`"````````,`(````````4````````````````````!```````````
M`````````#<````%`````@````````!(`@```````$@"````````+`,`````
M```%``````````@`````````!``````````]````"P````(`````````>`4`
M``````!X!0```````,`)````````!@````,````(`````````!@`````````
M10````,````"`````````#@/````````.`\```````"^!```````````````
M`````0```````````````````$T````$`````@````````#X$P```````/@3
M````````&`,````````%``````````@`````````&`````````!7````!```
M`$(`````````$!<````````0%P`````````&````````!0```!,````(````
M`````!@`````````7`````$````&`````````!`=````````$!T````````@
M!```````````````````$``````````0`````````&$````!````!@``````
M```P(0```````#`A````````.#0``````````````````!``````````````
M``````!G`````0````(`````````:%4```````!H50```````&]!````````
M```````````(````````````````````;P````$````"`````````-B6````
M````V)8````````$`0``````````````````!````````````````````'T`
M```!`````@````````#@EP```````."7````````O`8`````````````````
M``@```````````````````"'````#@````,`````````0*T!``````!`K0``
M``````@````````````````````(``````````@`````````DP````\````#
M`````````$BM`0``````2*T````````(````````````````````"```````
M```(`````````)\````!`````P````````!0K0$``````%"M````````"```
M``````````````````@```````````````````"D````!@````,`````````
M6*T!``````!8K0```````.`!````````!@`````````(`````````!``````
M````K0````$````#`````````#BO`0``````.*\```````"P````````````
M````````"``````````(`````````+(````!`````P````````#HKP$`````
M`.BO````````&`(```````````````````@`````````"`````````"[````
M`0````,``````````+(!````````L@```````(@````````````````````(
M````````````````````P0````@````#`````````,"R`0``````B+(`````
M``"(````````````````````0````````````````````,8````!````,```
M`````````````````(BR````````/0````````````````````$`````````
M`0`````````?`````0```#````````````````````#%L@```````!X!````
M```````````````!``````````$`````````SP````$`````````````````
M````````Y+,```````"/````````````````````!```````````````````
M``$````#`````````````````````````'.T````````V0``````````````
4``````$`````````````````````
