/**
 * @file libcomprex/fsiter.h Filesystem iteration API
 * 
 * $Id: fsiter.h,v 1.4 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _LIBCOMPREX_FSITER_H_
#define _LIBCOMPREX_FSITER_H_

#ifdef __cplusplus
extern "C" {
#endif

typedef struct _CxFsIterator CxFsIterator; /**< Filesystem iterator. */

typedef enum
{
	CX_FSITER_FILES = 0,  /**< File iterator.               */
	CX_FSITER_DIRS,       /**< Directory iterator.          */
	CX_FSITER_FILES_DIRS  /**< File and directory iterator. */

} CxFsIteratorType;

#include <libcomprex/directory.h>
#include <libcomprex/fsnode.h>

/**
 * An iterator interface for iterating through files or directories
 * in an archive.
 */
struct _CxFsIterator
{
	CxFsIteratorType type;  /**< The filesystem iterator type. */

	CxArchive   *archive;   /**< The archive.                  */
	CxFsNode    *lastNode;  /**< The last node returned.       */
};

/**************************************************************************/
/** @name Structure (De)allocation Functions                              */
/**************************************************************************/
/*@{*/

/**
 * Creates a filesystem iterator.
 *
 * This will normally only be used within libpackman.
 *
 * @param archive The associated archive.
 * @param type    The filesystem iterator type.
 *
 * @return A new CxFsIterator.
 */
CxFsIterator *cxNewFsIterator(CxArchive *archive, CxFsIteratorType type);

/**
 * Destroys a filesystem iterator.
 *
 * This should be called when done with an iterator.
 *
 * @param iterator The iterator to destroy.
 */
void cxDestroyFsIterator(CxFsIterator *iterator);

/*@}*/


/**************************************************************************/
/** @name Iteration Functions                                             */
/**************************************************************************/
/*@{*/

/**
 * Returns the first item.
 *
 * @param iterator The iterator.
 *
 * @return The first item.
 *
 * @see cxGetIterNext()
 * @see cxGetIterPrev()
 */
CxFsNode *cxGetFsIterFirst(CxFsIterator *iterator);

/**
 * Returns the next item.
 *
 * If this hasn't been called previously, it will return the first item.
 *
 * @param iterator The iterator.
 *
 * @return The next item.
 *
 * @see cxGetIterFirst()
 * @see cxGetIterPrev()
 */
CxFsNode *cxGetFsIterNext(CxFsIterator *iterator);

/**
 * Returns the previous item.
 *
 * If cxGetIterNext() hasn't been called yet, this will return NULL.
 * 
 * @param iterator The iterator.
 * 
 * @return The previous item.
 *
 * @see cxGetIterFirst()
 * @see cxGetIterNext()
 */
CxFsNode *cxGetFsIterPrev(CxFsIterator *iterator);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _LIBCOMPREX_FSITER_H_ */

