/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.trogdor.coordinator;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.utils.Scheduler;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.trogdor.common.Node;
import org.apache.kafka.trogdor.common.Platform;
import org.apache.kafka.trogdor.common.ThreadUtils;
import org.apache.kafka.trogdor.coordinator.NodeManager;
import org.apache.kafka.trogdor.rest.TaskDone;
import org.apache.kafka.trogdor.rest.TaskPending;
import org.apache.kafka.trogdor.rest.TaskRunning;
import org.apache.kafka.trogdor.rest.TaskState;
import org.apache.kafka.trogdor.rest.TaskStopping;
import org.apache.kafka.trogdor.rest.TasksResponse;
import org.apache.kafka.trogdor.task.TaskController;
import org.apache.kafka.trogdor.task.TaskSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class TaskManager {
    private static final Logger log = LoggerFactory.getLogger(TaskManager.class);
    private final Platform platform;
    private final Scheduler scheduler;
    private final Time time;
    private final Map<String, ManagedTask> tasks;
    private final ScheduledExecutorService executor;
    private final Map<String, NodeManager> nodeManagers;
    private AtomicBoolean shutdown = new AtomicBoolean(false);

    TaskManager(Platform platform, Scheduler scheduler) {
        this.platform = platform;
        this.scheduler = scheduler;
        this.time = scheduler.time();
        this.tasks = new HashMap<String, ManagedTask>();
        this.executor = Executors.newSingleThreadScheduledExecutor(ThreadUtils.createThreadFactory("TaskManagerStateThread", false));
        this.nodeManagers = new HashMap<String, NodeManager>();
        for (Node node : platform.topology().nodes().values()) {
            if (Node.Util.getTrogdorAgentPort(node) <= 0) continue;
            this.nodeManagers.put(node.name(), new NodeManager(node, this));
        }
        log.info("Created TaskManager for agent(s) on: {}", (Object)Utils.join(this.nodeManagers.keySet(), (String)", "));
    }

    public TaskSpec createTask(String id, TaskSpec spec) throws ExecutionException, InterruptedException {
        TaskSpec existingSpec = this.executor.submit(new CreateTask(id, spec)).get();
        if (existingSpec != null) {
            log.info("Ignoring request to create task {}, because there is already a task with that id.", (Object)id);
            return existingSpec;
        }
        return spec;
    }

    public TaskSpec stopTask(String id) throws ExecutionException, InterruptedException {
        TaskSpec spec = this.executor.submit(new CancelTask(id)).get();
        return spec;
    }

    public void handleWorkerCompletion(String nodeName, String id, String error) {
        this.executor.submit(new HandleWorkerCompletion(nodeName, id, error));
    }

    public TasksResponse tasks() throws ExecutionException, InterruptedException {
        return this.executor.submit(new GetTasksResponse()).get();
    }

    public void beginShutdown(boolean stopAgents) throws ExecutionException, InterruptedException {
        if (this.shutdown.compareAndSet(false, true)) {
            this.executor.submit(new Shutdown(stopAgents));
        }
    }

    public void waitForShutdown() throws ExecutionException, InterruptedException {
        while (!this.executor.awaitTermination(1L, TimeUnit.DAYS)) {
        }
    }

    class Shutdown
    implements Callable<Void> {
        private final boolean stopAgents;

        Shutdown(boolean stopAgents) {
            this.stopAgents = stopAgents;
        }

        @Override
        public Void call() throws Exception {
            log.info("Shutting down TaskManager{}.", (Object)(this.stopAgents ? " and agents" : ""));
            for (NodeManager nodeManager : TaskManager.this.nodeManagers.values()) {
                nodeManager.beginShutdown(this.stopAgents);
            }
            for (NodeManager nodeManager : TaskManager.this.nodeManagers.values()) {
                nodeManager.waitForShutdown();
            }
            TaskManager.this.executor.shutdown();
            return null;
        }
    }

    class GetTasksResponse
    implements Callable<TasksResponse> {
        GetTasksResponse() {
        }

        @Override
        public TasksResponse call() throws Exception {
            TreeMap<String, TaskState> states = new TreeMap<String, TaskState>();
            for (ManagedTask task : TaskManager.this.tasks.values()) {
                states.put(task.id, task.taskState());
            }
            return new TasksResponse(states);
        }
    }

    class HandleWorkerCompletion
    implements Callable<Void> {
        private final String nodeName;
        private final String id;
        private final String error;

        HandleWorkerCompletion(String nodeName, String id, String error) {
            this.nodeName = nodeName;
            this.id = id;
            this.error = error;
        }

        @Override
        public Void call() throws Exception {
            ManagedTask task = (ManagedTask)TaskManager.this.tasks.get(this.id);
            if (task == null) {
                log.error("Can't handle completion of unknown worker {} on node {}", (Object)this.id, (Object)this.nodeName);
                return null;
            }
            if (task.state == ManagedTaskState.PENDING || task.state == ManagedTaskState.DONE) {
                log.error("Task {} got unexpected worker completion from {} while in {} state.", new Object[]{this.id, this.nodeName, task.state});
                return null;
            }
            boolean broadcastStop = false;
            if (task.state == ManagedTaskState.RUNNING) {
                task.state = ManagedTaskState.STOPPING;
                broadcastStop = true;
            }
            task.maybeSetError(this.error);
            task.activeWorkers.remove(this.nodeName);
            if (task.activeWorkers.size() == 0) {
                task.doneMs = TaskManager.this.time.milliseconds();
                task.state = ManagedTaskState.DONE;
                log.info("Task {} is now complete on {} with error: {}", new Object[]{this.id, Utils.join((Collection)task.workers, (String)", "), task.error.isEmpty() ? "(none)" : task.error});
            } else if (broadcastStop) {
                log.info("Node {} stopped.  Stopping task {} on worker(s): {}", (Object)this.id, (Object)Utils.join((Collection)task.activeWorkers, (String)", "));
                for (String workerName : task.activeWorkers) {
                    ((NodeManager)TaskManager.this.nodeManagers.get(workerName)).stopWorker(this.id);
                }
            }
            return null;
        }
    }

    class CancelTask
    implements Callable<TaskSpec> {
        private final String id;

        CancelTask(String id) {
            this.id = id;
        }

        @Override
        public TaskSpec call() throws Exception {
            ManagedTask task = (ManagedTask)TaskManager.this.tasks.get(this.id);
            if (task == null) {
                log.info("Can't cancel non-existent task {}.", (Object)this.id);
                return null;
            }
            switch (task.state) {
                case PENDING: {
                    task.cancelled = true;
                    task.clearStartFuture();
                    task.doneMs = TaskManager.this.time.milliseconds();
                    task.state = ManagedTaskState.DONE;
                    log.info("Stopped pending task {}.", (Object)this.id);
                    break;
                }
                case RUNNING: {
                    task.cancelled = true;
                    if (task.activeWorkers.size() == 0) {
                        log.info("Task {} is now complete with error: {}", (Object)this.id, (Object)task.error);
                        task.doneMs = TaskManager.this.time.milliseconds();
                        task.state = ManagedTaskState.DONE;
                        break;
                    }
                    for (String workerName : task.activeWorkers) {
                        ((NodeManager)TaskManager.this.nodeManagers.get(workerName)).stopWorker(this.id);
                    }
                    log.info("Cancelling task {} on worker(s): {}", (Object)this.id, (Object)Utils.join((Collection)task.activeWorkers, (String)", "));
                    task.state = ManagedTaskState.STOPPING;
                    break;
                }
                case STOPPING: {
                    log.info("Can't cancel task {} because it is already stopping.", (Object)this.id);
                    break;
                }
                case DONE: {
                    log.info("Can't cancel task {} because it is already done.", (Object)this.id);
                }
            }
            return task.spec;
        }
    }

    class RunTask
    implements Callable<Void> {
        private final ManagedTask task;

        RunTask(ManagedTask task) {
            this.task = task;
        }

        @Override
        public Void call() throws Exception {
            TreeSet<String> nodeNames;
            this.task.clearStartFuture();
            if (this.task.state != ManagedTaskState.PENDING) {
                log.info("Can't start task {}, because it is already in state {}.", (Object)this.task.id, (Object)this.task.state);
                return null;
            }
            try {
                nodeNames = this.task.findNodeNames();
            }
            catch (Exception e) {
                log.error("Unable to find nodes for task {}", (Object)this.task.id, (Object)e);
                this.task.doneMs = TaskManager.this.time.milliseconds();
                this.task.state = ManagedTaskState.DONE;
                this.task.maybeSetError("Unable to find nodes for task: " + e.getMessage());
                return null;
            }
            log.info("Running task {} on node(s): {}", (Object)this.task.id, (Object)Utils.join(nodeNames, (String)", "));
            this.task.state = ManagedTaskState.RUNNING;
            this.task.startedMs = TaskManager.this.time.milliseconds();
            this.task.workers = nodeNames;
            this.task.activeWorkers = new HashSet();
            for (String workerName : this.task.workers) {
                this.task.activeWorkers.add(workerName);
                ((NodeManager)TaskManager.this.nodeManagers.get(workerName)).createWorker(this.task.id, this.task.spec);
            }
            return null;
        }
    }

    class CreateTask
    implements Callable<TaskSpec> {
        private final String id;
        private final TaskSpec spec;

        CreateTask(String id, TaskSpec spec) {
            this.id = id;
            this.spec = spec;
        }

        @Override
        public TaskSpec call() throws Exception {
            ManagedTask task = (ManagedTask)TaskManager.this.tasks.get(this.id);
            if (task != null) {
                log.info("Task ID {} is already in use.", (Object)this.id);
                return task.spec;
            }
            TaskController controller = null;
            String failure = null;
            try {
                controller = this.spec.newController(this.id);
            }
            catch (Throwable t) {
                failure = "Failed to create TaskController: " + t.getMessage();
            }
            if (failure != null) {
                log.info("Failed to create a new task {} with spec {}: {}", new Object[]{this.id, this.spec, failure});
                task = new ManagedTask(this.id, this.spec, null, ManagedTaskState.DONE);
                task.doneMs = TaskManager.this.time.milliseconds();
                task.maybeSetError(failure);
                TaskManager.this.tasks.put(this.id, task);
                return null;
            }
            task = new ManagedTask(this.id, this.spec, controller, ManagedTaskState.PENDING);
            TaskManager.this.tasks.put(this.id, task);
            long delayMs = task.startDelayMs(TaskManager.this.time.milliseconds());
            task.startFuture = TaskManager.this.scheduler.schedule(TaskManager.this.executor, (Callable)new RunTask(task), delayMs);
            log.info("Created a new task {} with spec {}, scheduled to start {} ms from now.", new Object[]{this.id, this.spec, delayMs});
            return null;
        }
    }

    class ManagedTask {
        private final String id;
        private final TaskSpec spec;
        private final TaskController controller;
        private ManagedTaskState state;
        private long startedMs = -1L;
        private long doneMs = -1L;
        boolean cancelled = false;
        private Future<?> startFuture = null;
        private Set<String> workers = null;
        private Set<String> activeWorkers = null;
        private String error = "";

        ManagedTask(String id, TaskSpec spec, TaskController controller, ManagedTaskState state) {
            this.id = id;
            this.spec = spec;
            this.controller = controller;
            this.state = state;
        }

        void clearStartFuture() {
            if (this.startFuture != null) {
                this.startFuture.cancel(false);
                this.startFuture = null;
            }
        }

        long startDelayMs(long now) {
            if (now > this.spec.startMs()) {
                return 0L;
            }
            return this.spec.startMs() - now;
        }

        TreeSet<String> findNodeNames() {
            Set<String> nodeNames = this.controller.targetNodes(TaskManager.this.platform.topology());
            TreeSet<String> validNodeNames = new TreeSet<String>();
            TreeSet<String> nonExistentNodeNames = new TreeSet<String>();
            for (String nodeName : nodeNames) {
                if (TaskManager.this.nodeManagers.containsKey(nodeName)) {
                    validNodeNames.add(nodeName);
                    continue;
                }
                nonExistentNodeNames.add(nodeName);
            }
            if (!nonExistentNodeNames.isEmpty()) {
                throw new KafkaException("Unknown node names: " + Utils.join(nonExistentNodeNames, (String)", "));
            }
            if (validNodeNames.isEmpty()) {
                throw new KafkaException("No node names specified.");
            }
            return validNodeNames;
        }

        void maybeSetError(String newError) {
            if (this.error.isEmpty()) {
                this.error = newError;
            }
        }

        TaskState taskState() {
            switch (this.state) {
                case PENDING: {
                    return new TaskPending(this.spec);
                }
                case RUNNING: {
                    return new TaskRunning(this.spec, this.startedMs);
                }
                case STOPPING: {
                    return new TaskStopping(this.spec, this.startedMs);
                }
                case DONE: {
                    return new TaskDone(this.spec, this.startedMs, this.doneMs, this.error, this.cancelled);
                }
            }
            throw new RuntimeException("unreachable");
        }
    }

    static enum ManagedTaskState {
        PENDING,
        RUNNING,
        STOPPING,
        DONE;

    }
}

