<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * xAPI LRS IRI values generator.
 *
 * @package    core_xapi
 * @since      Moodle 3.9
 * @copyright  2020 Ferran Recio
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
namespace core_xapi;
defined('MOODLE_INTERNAL') || die();

use stdClass;
use moodle_url;

/**
 * Class to translate Moodle objects to xAPI elements.
 *
 * @copyright  2020 Ferran Recio
 * @since      Moodle 3.9
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
class iri {

    /**
     * Generate a valid IRI element from a $value and an optional $type.
     *
     * Verbs and Objects in xAPI are in IRI format. This function could get
     * a valid IRI value (and will return without modifiyng it) or a simple
     * string and a type and generate a fake IRI valir for any xAPI statement.
     *
     * @param string $value a valid IRI value or any string
     * @param string|null $type if none passed $type will be 'element'
     * @return string a valid IRI value
     */
    public static function generate(string $value, ?string $type = null): string {
        if (self::check($value)) {
            return $value;
        }
        if (empty($type)) {
            $type = 'element';
        }
        return (new moodle_url("/xapi/$type/$value"))->out(false);
    }

    /**
     * Try to extract the original value from an IRI.
     *
     * If a real IRI value is passed, it will return it without any change. If a
     * fake IRI is passed (generated by iri::generate)
     * it will try to extract the original value.
     *
     * @param string $value the currewnt IRI value.
     * @param string|null $type if $value is a fake IRI, the $type must be provided.
     * @return string the original value used in iri::generate.
     */
    public static function extract(string $value, ?string $type = null): string {
        if (empty($type)) {
            $type = 'element';
        }
        $xapibase = (new moodle_url("/xapi/$type/"))->out(false);
        if (strpos($value, $xapibase) === 0) {
            return substr($value, strlen($xapibase));
        }
        return $value;
    }

    /**
     * Check if a $value could be a valid IRI or not.
     *
     * @param string $value the currewnt IRI value.
     * @return bool if the $value could be an IRI.
     */
    public static function check(string $value): bool {
        $iri = new moodle_url($value);
        return in_array($iri->get_scheme(), ['http', 'https']);
    }
}
