package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.42.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.42.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@("`0`````````````#`+<````!````````(%``````````0```````
M`02(``````!``#@`"`!``!D`&`````8````$`````````$``````````0```
M``````!``````````<`````````!P``````````(`````P````0````````"
M``````````(``````````@``````````%P`````````7``````````$````!
M````!0```````````````````````````````````````)SL````````G.P`
M``````$```````$````&````````_4````````']0````````?U`````````
M!3`````````&"````````0```````@````8```````#]6````````?U8````
M```!_5@````````!X`````````'@``````````@````$````!`````````(8
M`````````A@````````"&``````````L`````````"P`````````!&1TY5``
M```$````````E:````````"5H````````)6@`````````/P`````````_```
M```````$9'3E4@````0```````#]0````````?U````````!_4`````````"
MP`````````+```````````$O=7-R+VQI8F5X96,O;&0N96QF7W-O```````'
M````!`````%.971"4T0``$&0JP`````$````!`````-085@`````````````
M``!A````90``````````````,`````T``````````````#<````.````/P``
M```````O``````````,```!+````7@`````````?````2@````8````=````
M%`````````!!`````````%H`````````%P```#L```!5``````````````!A
M````8P```"X````2````5````$,````5````60```$0`````````1P```"(`
M```T````6````"8````J``````````<```!-``````````L````*````9```
M`%<```!)````(````$(```!B`````````%P`````````&````"T```!,````
M````````````````$0```#(``````````````%T````K``````````D`````
M````)````#@`````````3@````4```!%````&P```#H`````````````````
M````````2``````````Y````40```%\`````````8```````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````!,`````````````````````````````````
M```````````````````C````(0`````````I`````````"<````H````+```
M````````````#``````````<````&@```!D``````````````!``````````
M,P```#$````V````'@`````````````````````````$``````````````!`
M````)0```````````````````!8``````````````#X````(````````````
M````````/0```%``````````4P``````````````/```````````````3P``
M``````!&````#P````````!2````5@```#4```!;````````````````````
M```````````````````````#```*````````(%`````````````````#```0
M```````!_5```````````````(<2``````````````````````````````$2
M``````````````````````````````D2````````````````````````````
M`U$2```*````````*4``````````"````!$B````````````````````````
M`````"`2`````````````````````````````KH2````````````````````
M`````````RH0```5```````"`H```````````````"82````````````````
M`````````````V(2```*````````,X``````````(````"T2````````````
M`````````````````L<2`````````````````````````````#42````````
M`````````````````````WT2```*````````//0`````````9````'@2````
M`````````````````````````SD0```5```````"`T@``````````````#T2
M`````````````````````````````XD0```4```````"`G``````````````
M`$T2`````````````````````````````%P2````````````````````````
M`````&(2`````````````````````````````&D2````````````````````
M`````````',2`````````````````````````````'T2````````````````
M`````````````S@0```5```````"`T@``````````````(82````````````
M`````````````````?@2`````````````````````````````(P2````````
M`````````````````````),2`````````````````````````````*$2````
M`````````````````````````+(1`````````````````````````````F8B
M`````````````````````````````+\2````````````````````````````
M`Y`2```*````````*Y0`````````+````YT2```*````````*\``````````
M;````ZD2```*````````-*0````````%O````,T6````````````````````
M`````````.`2`````````````````````````````MD2````````````````
M`````````````.L2`````````````````````````````/01```5```````"
M`T``````````"````/P2`````````````````````````````K`2````````
M`````````````````````0@2`````````````````````````````0\2````
M`````````````````````````[82```*````````3C`````````!&````2(1
M`````````````````````````````302````````````````````````````
M`402`````````````````````````````5(2````````````````````````
M`````5<2`````````````````````````````\$1```4```````"`F@`````
M````"````682`````````````````````````````7(2````````````````
M`````````````8<2`````````````````````````````9(2````````````
M`````````````````TP0```5```````"`T@``````````````GP2````````
M`````````````````````:42`````````````````````````````_P2```*
M````````(%``````````"````;,2`````````````````````````````;L2
M`````````````````````````````<<2````````````````````````````
M`T00```5```````"`T@``````````````\L2```*````````+#`````````$
MT````<T2`````````````````````````````=@2````````````````````
M`````````]@2```*````````3U`````````!P````>D2````````````````
M`````````````^(0```5```````"`H```````````````^X2```*````````
M**0`````````0````_82```*````````4=`````````")````?`2````````
M`````````````````````D<B`````````````````````````````_L2```*
M````````(%``````````"```!`,2```*````````*5`````````"1```!!`2
M```*````````,Z`````````!!````?<2````````````````````````````
M!"(2```*````````,0``````````>````@`2````````````````````````
M`````E\2````````````````````````````!#`2```*````````,8``````
M````G````A$2`````````````````````````````AH2````````````````
M`````````````B`2`````````````````````````````J,2````````````
M`````````````````B42`````````````````````````````I81```5````
M```"`H@`````````"````BT2`````````````````````````````LX2````
M`````````````````````````C02````````````````````````````!#@2
M```*````````3?``````````0````CL1````````````````````````````
M`HL1```4```````"`>@`````````"```!$$2```*````````.F`````````"
ME````1L2`````````````````````````````D`2````````````````````
M`````````N`2``````````````````````````!R96%L;&]C`&UE;6UO=F4`
M7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T<FYC;7``<W1R<F-H<@!B
M;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`86)O<G0`=6YL:6YK`%]?
M;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X:70`;65M8W!Y`'!E<FQ?
M9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D;U]U;F1U;7``4&5R;%]S
M=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R;%]A;&QO8P!C;&]S961I
M<@!E;G9I<F]N`%]?;W!E;F1I<C,P`&UE;7-E=`!097)L7W-A9F5S>7-M86QL
M;V,`7U]S=&%C:U]C:&M?9W5A<F0`4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT
M9E]C:&L`<F5A9`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!097)L7V-R
M;V%K7VYO8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%]?97)R;F\`4&5R;%]M9U]S970`;6MD:7(`<&5R;%]P
M87)S90!097)L7VYE=UA37V9L86=S`&%C8V5S<P!G971P:60`=F9P<FEN=&8`
M4&5R;%]A=&9O<FM?;&]C:P!P97)L7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`
M9V5T=6ED`'-T<FQE;@!?7W-&`'-T<F-M<`!?7V1E<F5G:7-T97)?9G)A;65?
M:6YF;P!S=')D=7``7U]R96=I<W1E<E]F<F%M95]I;F9O`'!T:')E861?871F
M;W)K`%]?<')O9VYA;64`7U]P<U]S=')I;F=S`%]?9V5T<'=U:60U,`!?7W-Y
M<V-A;&P`7U]S='!C<'E?8VAK`&%T97AI=`!?;&EB8U]I;FET`'-T<G1O:P!?
M7W)E861D:7(S,`!L:6)P97)L+G-O`&QI8FTN<V\N,`!L:6)C<GEP="YS;RXQ
M`&QI8G!T:')E860N<V\N,0!L:6)C+G-O+C$R`%]?8G-S7W-T87)T7U\`7U]B
M<W-?96YD7U\`7U]E;F1?7P!?96YD`'!A<E]C=7)R96YT7V5X96,`9V5T7W5S
M97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7V5D871A`'!A<E]B
M87-E;F%M90!P87)?9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!M
M>5]P87)?<&P`<&%R7VEN:71?96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X
M<U]I;FET`&UA:6X`7U]S=&%R=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI
M8G!A=&@`<&%R7V5N=E]C;&5A;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R
M;F%L<U]005)?0D]/5``O=7-R+W!K9R]L:6(O<&5R;#4O-2XT,BXP+V%A<F-H
M-C1E8BUN971B<V0M=&AR96%D+6UU;'1I+T-/4D4Z+W5S<B]P:V<O;&EB````
M``````````````']0`````````0#````````49````````']2`````````0#
M````````41````````'_0`````````0#```````!_5````````'_6```````
M``0#```````!_4````````'_>`````````0#```````"`T````````'_B```
M``````0#```````!_4@```````'_D`````````0#```````"`F@```````'_
ML`````````0#```````!_5@```````'_R`````````0#```````"`H@`````
M``'_V`````````0#```````"`>@```````'_X`````````0#````````.F``
M``````(!Z`````````0#```````"`H````````(!\`````````0#```````"
M`?````````(""`````````0#````````5\@```````("$`````````0#````
M````5"````````("(`````````0#````````54@```````("*`````````0#
M````````5Z@```````(",`````````0#````````5[````````(".```````
M``0#````````5[@```````("0`````````0#````````5\````````("4```
M``````0#````````5Y````````("6`````````0#````````5Y@```````("
M:`````````0#````````5]````````'_2`````<```0!````````````````
M``'_4````!,```0!``````````````````'_8````"$```0!````````````
M``````'_:````"(```0!``````````````````'_<````"<```0&````````
M``````````'_@````#$```0!``````````````````'_F````#H```0!````
M``````````````'_J````$P```0!``````````````````'_P````%(```0!
M``````````````````'_T````%\```0!``````````````````(```````,`
M``0"``````````````````(`"`````0```0"``````````````````(`$```
M``4```0"``````````````````(`&`````<```0"``````````````````(`
M(`````@```0"``````````````````(`*`````D```0"````````````````
M``(`,`````L```0"``````````````````(`.`````T```0"````````````
M``````(`0`````X```0"``````````````````(`2`````\```0"````````
M``````````(`4````!$```0"``````````````````(`6````!4```0"````
M``````````````(`8````!8```0"``````````````````(`:````!<```0"
M``````````````````(`<````!@```0"``````````````````(`>````!D`
M``0"``````````````````(`@````!H```0"``````````````````(`B```
M`!P```0"``````````````````(`D````!T```0"``````````````````(`
MF````!X```0"``````````````````(`H````!\```0"````````````````
M``(`J````"````0"``````````````````(`L````"(```0"````````````
M``````(`N````",```0"``````````````````(`P````"@```0"````````
M``````````(`R````"D```0"``````````````````(`T````"H```0"````
M``````````````(`V````"P```0"``````````````````(`X````"T```0"
M``````````````````(`Z````"X```0"``````````````````(`\````"\`
M``0"``````````````````(`^````#(```0"``````````````````(!````
M`#,```0"``````````````````(!"````#0```0"``````````````````(!
M$````#4```0"``````````````````(!&````#<```0"````````````````
M``(!(````#@```0"``````````````````(!*````#D```0"````````````
M``````(!,````#P```0"``````````````````(!.````#T```0"````````
M``````````(!0````#\```0"``````````````````(!2````$````0"````
M``````````````(!4````$$```0"``````````````````(!6````$0```0"
M``````````````````(!8````$4```0"``````````````````(!:````$<`
M``0"``````````````````(!<````$L```0"``````````````````(!>```
M`$P```0"``````````````````(!@````%````0"``````````````````(!
MB````%,```0"``````````````````(!D````%4```0"````````````````
M``(!F````%8```0"``````````````````(!H````%<```0"````````````
M``````(!J````%@```0"``````````````````(!L````%D```0"````````
M``````````(!N````%L```0"``````````````````(!P````%P```0"````
M``````````````(!R````%T```0"``````````````````(!T````&(```0"
M``````````````````(!V````&,```0"``````````````````(!X````&0`
M``0"`````````````````````/![OZGP``#0$?Y'^1#B/Y$@`A_6'R`#U1\@
M`]4?(`/5\```\!$"0/D0`@"1(`(?UO```/`1!D#Y$"(`D2`"'];P``#P$0I`
M^1!"`)$@`A_6\```\!$.0/D08@"1(`(?UO```/`1$D#Y$((`D2`"'];P``#P
M$19`^1"B`)$@`A_6\```\!$:0/D0P@"1(`(?UO```/`1'D#Y$.(`D2`"'];P
M``#P$2)`^1`"`9$@`A_6\```\!$F0/D0(@&1(`(?UO```/`1*D#Y$$(!D2`"
M'];P``#P$2Y`^1!B`9$@`A_6\```\!$R0/D0@@&1(`(?UO```/`1-D#Y$*(!
MD2`"'];P``#P$3I`^1#"`9$@`A_6\```\!$^0/D0X@&1(`(?UO```/`10D#Y
M$`("D2`"'];P``#P$49`^1`B`I$@`A_6\```\!%*0/D00@*1(`(?UO```/`1
M3D#Y$&("D2`"'];P``#P$5)`^1""`I$@`A_6\```\!%60/D0H@*1(`(?UO``
M`/`16D#Y$,("D2`"'];P``#P$5Y`^1#B`I$@`A_6\```\!%B0/D0`@.1(`(?
MUO```/`19D#Y$"(#D2`"'];P``#P$6I`^1!"`Y$@`A_6\```\!%N0/D08@.1
M(`(?UO```/`1<D#Y$((#D2`"'];P``#P$79`^1"B`Y$@`A_6\```\!%Z0/D0
MP@.1(`(?UO```/`1?D#Y$.(#D2`"'];P``#P$8)`^1`"!)$@`A_6\```\!&&
M0/D0(@21(`(?UO```/`1BD#Y$$($D2`"'];P``#P$8Y`^1!B!)$@`A_6\```
M\!&20/D0@@21(`(?UO```/`1ED#Y$*($D2`"'];P``#P$9I`^1#"!)$@`A_6
M\```\!&>0/D0X@21(`(?UO```/`1HD#Y$`(%D2`"'];P``#P$:9`^1`B!9$@
M`A_6\```\!&J0/D00@61(`(?UO```/`1KD#Y$&(%D2`"'];P``#P$;)`^1""
M!9$@`A_6\```\!&V0/D0H@61(`(?UO```/`1ND#Y$,(%D2`"'];P``#P$;Y`
M^1#B!9$@`A_6\```\!'"0/D0`@:1(`(?UO```/`1QD#Y$"(&D2`"'];P``#P
M$<I`^1!"!I$@`A_6\```\!'.0/D08@:1(`(?UO```/`1TD#Y$((&D2`"'];P
M``#P$=9`^1"B!I$@`A_6\```\!':0/D0P@:1(`(?UO```/`1WD#Y$.(&D2`"
M'];P``#0$>)`^1`"!Y$@`A_6\```T!'F0/D0(@>1(`(?UO```-`1ZD#Y$$('
MD2`"'];P``#0$>Y`^1!B!Y$@`A_6\```T!'R0/D0@@>1(`(?UN$#`JHT```4
M``````````#]>[VIX```L`"@1_G]`P"1]1,`^?4``+"U(C61OP(`ZX(!`%3S
M4P&I%`0`T90"%<L3`(#2E/Y#TZ!Z<_@``#_6?P(4ZW,&`)&!__]4\U-!J?43
M0/G]>\.HP`-?UA\@`]7]>[VI_0,`D?-3`:GS``"P<])'^?43`/GU``"PM=Y'
M^7\"%>MB`0!48`Y`N1\@$/&!`0!4=`)`^7-B`)%@@E_X```_UH`"`/E_`A7K
MX_[_5/-30:GU$T#Y_7O#J,`#7]8(__^7_7N\J?T#`)'S4P&I]5L"J?=C`ZE!
M"0"T]`,!JO<#`*HA`$#YX```L`#D1_GU``"PM;Y'^2(`0/D4``#Y@`I`^:`"
M`/G@``"P`.Q'^0('`+0"``#Y0P!`.<,``#1"!`"1?[P`<6#__U1#`$`Y@___
M-9O__Y?@``"P`-A'^0`'`+3@``"P`*Q'^?@``+`8`S61'P,`ZV(!`%06!`#1
M$P"`TM8"&,O6_D/3'R`#U0%[<_@@`#_6?P(6ZW,&`)&!__]4=P``M.`#%ZK%
M_O^7````D`"``9'"_O^7]P``L.```+``Q$?Y]P(UD?\"`.MB`0!4%@0`T1,`
M@-+6`A?+UOY#TQ\@`]7@>G/X```_UM\"$^MS!@"1@?__5($"0/FB`D#Y@`I`
MN>(+`)2-_O^7X0``T"$`"I$!``#YSO__%P(``/!"`!"18P*`TD$`@%*``(#2
M\_[_ER``@%+%_O^7D/__E\C__Q<``````````/U[O*G]`P"1\U,!J:`&`+3W
M8P.I]P,`JN```+``O$?Y&`!`^3@&`+3U6P*I]@,!JA4`@%+@`D`Y'_0`<008
M0'K!`@!4$P-`^=,!`+2U?D"3]`,8JA\@`]7B`Q6JX0,7JN`#$ZI__O^7@```
M-6!J=3@?]`!Q``(`5)..0/CS_O^U]5M"J?=C0ZG@`Q.J\U-!J?U[Q*C``U_6
M]0,7JJ`>0#@?]`!Q!!A`>J'__U2U`A=+Y?__%X`"&,NU!@"1<P(5BP#\0Y/`
M`@"YX`,3JO-30:GU6T*I]V-#J?U[Q*C``U_6$P"`TNG__Q?W8T.I$P"`TN;_
M_Q?_`P/1X@``L$+`1_GA`P"1_7L*J?V#`I'S6P#Y\P,`JD,`0/GC3P#Y`P"`
MTFS^_Y=``@`UX0M`N0(`E%(A#!02/Q!`<2000GK``0!4X0``L"'`1_GC3T#Y
M(@!`^6,``NL"`(#2H0$`5/U[2JGS6T#Y_P,#D<`#7]8``(!2]/__%^`#$ZI!
M`(!2T?[_EQ\``''@%Y\:[O__%VW^_Y?_0P/1X0``L"'`1_G]>PJI_8,"D2(`
M0/GB3P#Y`@"`TL```+3S4PNI\P,`J@$`0#F!`0`U\U-+J>```+``P$?YXD]`
M^0$`0/E"``'K`0"`TH$(`%3]>TJI_T,#D<`#7];U6PRI:_[_E_0#`*H@!@"T
MX`,4JNO^_Y=`!0"T`31`.14T`)$_N`!Q@0``5*$&0#D!__\T'R`#U0`T0#D?
MN`!QX`0`5.`#$ZK2_O^7]@,`JN`#%:K/_O^7P`(`"P`(`!$`?$"3S_[_E^,#
M%:KB`Q.J]0,`J@$``/`A8!"1N?[_E^`#%:KA`P"1'O[_EQ\$`#&@``!4X`M`
MN0`,%!(?$$!QH`(`5.`#%:H*_O^7X`,5JOS]_Y?@`Q2JPO[_EP#[_[7@`Q2J
M-_[_E^`#$ZK=_?^7\U-+J?5;3*G`__\7H`9`.1^X`'$!^_]4H`I`.2#Y_S35
M__\7X`,5JJK__Y?L__\7\U,+J?5;#*D2_O^7'R`#U?U[OJGA`P"J_0,`D>(`
M`+!"P$?Y\PL`^?]#$-'S(P"10`!`^>`'`OD``(#2`H"`TN`#$ZK#_?^7``0`
MT1\`$^NH``!4!P``%!_T'S@?`!/K@```5`$`0#D_O`!Q8/__5.`#$ZKA!8!2
MQOW_ET`"`+0?`!/KR`$`5.`#$ZIA_O^7X0``L"'`1_GC!T+Y(@!`^6,``NL"
M`(#2@0(`5/]#$)'S"T#Y_7O"J,`#7]8?```Y\O__%^```+``P$?YX@="^0$`
M0/E"``'K`0"`TN$``%3_0Q"1````\`"`$)'S"T#Y_7O"J$?^_Q?6_?^7'R`#
MU?^#`=']>P&I_4,`D?-3`JGS`P"J]`,!JO5;`ZGW8P2IX```L`#`1_DA`$`Y
M`@!`^>('`/D"`(#2/_0`<906E)K@`Q2J4_[_E_8#`*KA$P"1X`,3JN_^_Y<@
M"`"T]0,`JDS^_Y??`@#KZ08`5/<'@+GX``"P&+]'^??R?=-@`D`Y'_0`<008
M0'K`"P!4Y`,3JH,<0#A_]`!Q9!A`>J'__U2$`!/+A'Q`DQ4#0/G`"@"1```$
MBSO^_Y>@:C?XH`(`M`$#0/G@`Q.J(VAW^`$40#AA```Y/_0`<2080'K```!4
M`A1`.&(<`#A?]`!Q1!A`>H'__U2A!X!2``"`TF$4`#B":F`X8F@@.``$`)&B
M__\UX```L`#`1_GB!T#Y`0!`^4(``>L!`(#280<`5/U[0:GS4T*I]5M#J?=C
M1*G_@P&1P`-?U@$`@-*":F$XHFHA."$$`)&B__\U[O__%_@``+`8OT?Y^6L%
MJ1H#0/E``T#YP`0`M.`#&JH5`(!2'R`#U0&,0/CB`Q4JM08`$:'__[5`#``1
M`'Q]D^$``-`BZ$*Y`@(`-2(`@%(BZ`*Y_?W_E_D#`*I``P"TMWY]D^$#&JKB
M`Q>J%_W_EQD#`/F@!@`1]0<`N3_;(/CY:T6IJ/__%^$#`*K@`QJJ"OW_EP`#
M`/GY`P"J0`$`M+=^?9/T__\7!`"`TJG__Q<``H#2%0"`4N+__Q?Y:P6I3OW_
ME_EK1:F\__\7_7N_J4'0.]4C``#P_0,`D>```+``N$?Y'R`#U1\@`]7]>\&H
M8P`5D2!H8/@!``#PX@``L$*H1_DAH!"1?/W_%_U[OJE!T#O5_0,`D?,+`/G@
M``"P`+A'^1\@`]4?(`/5,P``BR!H8/CF__^78`)`^00``/#S"T#YA.`=D?U[
MPJCC`P2JX@``L$+P1_D!``#P!0"`4B$@$9&!_?\7``"`TL`#7]8?(`/5'R`#
MU?U[NZGB``"P0L!'^?T#`)'S4P&I\P,!JO5;`JG_PQ+1]0,`JD``0/G@5P+Y
M``"`TN$S`)$```#P`(`1D4'^_Y?A!8!2]`,`JN`#%:K1_/^70`(`M.$#%:H`
M``#P`,`1D33__Y?@`Q6JX0``L"'`1_GC5T+Y(@!`^6,``NL"`(#2@0T`5/_#
M$I'S4T&I]5M"J?U[Q:C``U_6X`,3JFG]_Y<6``#PP0(2D0;]_Y?S`P"J(/W_
MM/=S`OGWHP*1^'<"^1@``/`80Q*1^7L"^1D``/`Y(Q*1^G\"^6`"0#G@!@`T
MM```M.$#$ZK@`Q2J??W_EP`%`#3@`Q.J<OW_EP($`-'Z`P"J8@("BW\"`NNC
M``!4"@``%%_T'SA_`@+K@```5$``0#D?O`!Q8/__5.`#$ZID_?^7^@,`JN`#
M%:IA_?^7```:BP`$`)$?_`_QR`(`5.0#$ZKF`Q6JY0,9JN,#&*H"@(#2`0"`
M4N`#%ZKQ_/^7X4,`D>`#%ZJN_/^7H```->`;0+D`#!02'R!`<<`"`%3!`A*1
M``"`TLK\_Y?S`P"JP/G_M?=S0OGX=T+Y^7M"^?I_0OFH__\75`,`M(`"0#D?
MN`!QH```5!,``/`Z`(#2<X(0D=G__Q>`!D`YH/W_-/K__Q?@`Q>J(0"`4@7]
M_Y<`_?\UX0,7J@```/``P!&1S/[_E^`#%ZH._?^7]W-"^?AW0OGY>T+Y^G]"
M^9/__Q<3``#P.@"`TG."$)'#__\7]W,"^?AW`OGY>P+Y^G\"^9#\_Y?]>[ZI
MX06`4OT#`)'S"P#Y\P,`JEK\_Y<?``#Q8`:`FO,+0/G]>\*H]?S_%P`#`+3]
M>[ZI_0,`D?,+`/GS`P"J`0!`.2$!`#0-_?^7``0`D1\`$/%(`0!4X`,3JO,+
M0/G]>\*H9O[_%_,+0/D```#P_7O"J`"`$)'A_/\7\PM`^0``@-+]>\*HP`-?
MU@```/``@!"1VOS_%Q\@`]7_@P'1X```L`#`1_G]>P&I_4,`D?-3`JD4``#P
M\Q,`D?5;`ZF48A*1]V,$J?DK`/GU``"PM;Y'^0$`0/GA!P#Y`0"`TN$#$ZK@
M`Q2JA?W_EP`"`+2B`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0?(`/5`0A`^0$$
M`/D`(`"1H?__M>$#$ZK@`Q2J=OW_ET#^_[7U``"PM;Y'^10``/"4HA*1X0,3
MJN`#%*IN_?^7X`$`M*("0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y
M`"``D:'__[7A`Q.JX`,4JF#]_Y=@_O^U]0``L+6^1_D4``#PE,(2D>$#$ZK@
M`Q2J6/W_E^`!`+2B`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@
M`)&A__^UX0,3JN`#%*I*_?^78/[_M?4``+"UOD?Y%```\)0"$Y'A`Q.JX`,4
MJD+]_Y?@`0"TH@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1
MH?__M>$#$ZK@`Q2J-/W_EV#^_[7V``"PUKY'^10``/"5(A.1X0,3JN`#%:HL
M_?^7X`$`M,("0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'_
M_[7A`Q.JX`,5JA[]_Y=@_O^U]@``L-:^1_D5``#PM6(3D>$#$ZK@`Q6J%OW_
ME^`!`+3"`D#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^U
MX0,3JN`#%:H(_?^78/[_M?<``+#WOD?Y%0``\+:"$9'A`Q.JX`,6J@#]_Y?@
M`0"TX@)`^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#
M$ZK@`Q:J\OS_EV#^_[7X``"P&+]'^18``/#7HA.1X0,3JN`#%ZKJ_/^7X`$`
MM`(#0/GC!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.J
MX`,7JMS\_Y=@_O^U^0``L#F_1_D7``#P^.(3D>$#$ZK@`QBJU/S_E^`!`+0B
M`T#YXP>`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#
M&*K&_/^78/[_M?D``+`YOT?Y&```\!@C%)'A`Q.JX`,8JK[\_Y?@`0"T(@-`
M^>,'@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`QBJ
ML/S_EV#^_[7Y``"P.;]'^1@``/`8PQ&1X0,3JN`#&*JH_/^7X`$`M"(#0/GC
M!X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,8JIK\
M_Y=@_O^UX0,3J@```-``8!21E?S_E^$#`*I@``"TX.(3D8W]_Y?A`Q.J````
MT`#`%)&-_/^7X0,`JH```+0```#0`"`5D83]_Y?A`Q.J````T`!@%9&$_/^7
MX0,`JH`"`+2@@A&1?/W_E^```)``P$?YX@=`^0$`0/E"``'K`0"`TH$"`%3]
M>T&I@"(3D?-30JD!``#0]5M#J2$`%I'W8T2I^2M`^?^#`9%K_?\7X0,3J@``
M`-``H!61:_S_E^$#`*I`_?^TP*(3D6/]_Y?G__\7-?O_E_^#`-$```#0X@``
MD$+`1_GA$P"1_7L!J?U#`)$`H!.10P!`^>,'`/D#`(#26?S_E^`!`+0``$`Y
M'\``<0080'K@!Y\:X0``D"'`1_GC!T#Y(@!`^6,``NL"`(#2P0``5/U[0:G_
M@P"1P`-?U@``@%+U__\7%_O_EQ\@`]4?(`/5_X,$T>L&@!+I``"0*<%'^>R#
M`Y']>P6I_4,!D>H/@!+H`P"1X!N`/>$?@#WB(X`]XR>`/>0K@#WE+X`]YC.`
M/><W@#WABPZIX0,`JN(#"*KCDP^IY9L0J>>/`/G@``"0`.A'^2,!0/GC)P#Y
M`P"`TN.#!)'CCP*IXZ,`D>P?`/D`P`21ZRL(*6`$0*T`!0"M7/O_E^`?@%*:
M^O^7'R`#U?U[O*E"`(!2_0,`D>$``)`AP$?Y\U,!J?0#`"KU6P*I]V,#J?]#
M`-'_0T#1XT-`D20`0/ED!`#Y!`"`T@$`@-)4^_^7$P0`T>`#$^MA/D"2`#Q`
MDB!$@-IS`@#K1`<`5!<``-`6``#0%0&`TO@#`)'W(A:1UF(6D34`H/(?(`/5
MX0,3JN`#%"H"`(!20?O_EQ\$`+'`!0!4X@,5JN$#&*K@`Q0J\_K_EQ\$`#$@
M!`!4`'Q`DQ\<`/$)!`!4`"``T0$#`*NB`P!4XP``L`0``!0A!`#1/P`8ZP,#
M`%0B`$#Y8/Q`^5\``.M!__]4(0`8RR``$XOA``"0(<!'^>-#0)%D!$#Y(@!`
M^80``NL"`(#200(`5/]#`)'_0T"1\U-!J?5;0JGW8T.I_7O$J,`#7];@`Q:J
MB___EW-"0-%_0D"Q`?K_5```@)+K__\7X`,7JH3__Y?1__\7EOK_EQ\@`]7]
M>[^I_0,`D1K[_Y<1^_^70```M```0/G]>\&HP`-?UO\#`='A``"0(<!'^?U[
M`:G]0P"1\U,"J?0#`*KU6P.I%0``T+:B%I$B`$#YX@<`^0(`@-+@`Q:JX1,`
MD:W[_Y>```"T\P,`J@``0#G@`0`UX```D`#`1_GB!T#Y`0!`^4(``>L!`(#2
MP00`5/U[0:GA`Q2J\U-"J:"B%I'U6T.I_P,!D9?\_Q?@`Q2J^?K_E_4#`*K@
M`Q.J]OK_EZ`"`(L`"`"1]_K_E^0#$ZKB`Q2J\P,`J@,``-`!``#08P`2D2%`
M$I'?^O^7X```D`#`1_GB!T#Y`0!`^4(``>L!`(#2`0$`5/U[0:GA`Q.J\U-"
MJ>`#%JKU6T.I_P,!D7G\_Q=,^O^7_T,&T>$``+`AX`>1XN,`D22@`)']>Q2I
M_0,%D>,``)!CP$?Y\U,5J?.#`)$H)$&I]5L6J?4#`*HFG$6I]V,7J?@C`)&!
M"$"M8`!`^>"?`/D``(#2@`C`/?03`)$D$$#Y%@``T",T0/G`@A&1X0,4JD$(
M`*WHIP"I8PH`^00+`/GF'P*I0`B`/5G[_Y>```"T`0!`.?<#`*JA'0`U^6L8
MJ8___Y?Y`P"J8`T`M+#Z_Y<`^'_3``0`D;'Z_Y<B`T`Y\P,`JO<#`*HB`0`T
M&@``T%H#%Y'@`Q.JX0,:JICZ_Y=S"@"1(A]`.&+__S4```#0^0,!D0`@%9$#
M```4((=`^*`!`+3A`Q2J.?O_E_,#`*I@__^T`0!`.2'__S1Q^_^7X/[_-.`#
M$ZIQ^O^7\P,`JF`!`+43`T#Y,P$`M&`"0#F`#``TX`,3JF;[_Y=@"0`U$P=`
M^1@C`)$S__^UX`,3JH/Z_Y?X`P"JX`,7JH#Z_Y<8`P`+&!,0$1D``-`8?T"3
MX`,8JG[Z_Y?E`Q>JX@,3JB,C$I$$``#0A"`7D?<#`*H!``#0(4`7D67Z_Y?@
M`Q>J`3B`4C+Z_Y<?!``QH`8`5.&C`9'@`Q>JO?G_E_,#`"J@```UX'-`N0`,
M%!(?$$!QP`<`5*("0/D!``#0X```D`#H1_GC`Q>J(8`8D0#`!)&_^?^7%`"`
MTN```)``P$?YXI]`^0$`0/E"``'K`0"`TH$<`%3]>U2IX`,4JO-35:GU6U:I
M]V-7J?EK6*G_0P:1P`-?UBKZ_Y?Y`P"J`//_M7,B`)%@`D#Y@!$`M.$#%*KC
M^O^78/__M`$`0#G!_O\U<R(`D?C__Q?@`Q.J&",`D1OZ_Y?S`P"J8/7_M+3_
M_Q?S^?^7``!`N1]$`'$@^?]4LP)`^>[Y_Y?E``"0I>A'^>,#%ZH$`$"YX@,3
MJJ#`!)$!``#0(8`7D8WY_Y?.__\7$P"`TJ+__Q?ZAT"Y'/K_EU\#`&L!^/]4
MX'-`N0`@`!(?``=Q@??_5.`#&*H@^O^7X0,4JO0#`*H```#0`.`9D;?Z_Y?A
M`P"J@`P`M*`"0/EC_/^7^@,`J@`,`+3@`QJJ#_K_E_@#`*H?#`!QS0$`5`!\
M0),!``#0`!``T2$`&I%``P"+#OK_E^```#4?$P!QP!``5!@7`%%`RW@X'[P`
M<4`0`%0Z_O^7P```->`#&JH!`(!2YOG_E_4#`"H@"0`U`0``T````-`A`!:1
M`*`3D8_[_Y<U(Q*1&```T!CC'9'#^?^7&0``T.0#`"KE`QBJXP,5JN(#%ZK@
M`Q2J`0``T#D#&Y$AP!J1V?G_EP\``!2?^?^7``!`N1]$`'$!`@!4L_G_EW,&
M`!'D`P`JY0,3*N8#&*KC`Q6JX@,7JN$#&:K@`Q2JROG_E^`#%*H!.(!2E_G_
MEQ\$`#'`_?]4X`,7JA/Y_Y?A`Q2JP((1D6C[_Y?@`Q2JKO[_EW#__Q>L_O^7
MX```D`#`1_GBGT#Y`0!`^4(``>L!`(#280H`5/U[5*G@`Q>J\U-5J?5;5JGW
M8U>I_T,&D9KY_Q>@`8#2O/G_EQD``-#S`P"J.>,6D?<#`*IB"H!2"O__%[H"
M0/F@__\7,_[_E_@#`*K`]O^W`2@`T0(`@%+@`Q4J^@,$D9#Y_Y?A`QJJX`,5
M*L(`@-)$^?^7'Q@`\8```%0```#0`$`:D?O]_Y?@`T&Y`-!040`,!7&!]/]4
M00M`>:`(B5(_``!K`?3_5`'+`-$"`(!2X`,5*O,C!)%Z^?^7X`,5*N$#$ZH"
M!8#2+OG_EQ^@`/&```!4````T`!`&I'E_?^7Y`,3JB,C$I'B`Q>JX`,4J@4`
M`-`!``#0I>`=D2&`&I'_PP0Y<_G_EZ[__Q>U!D#YU>__M.`#%:IZ^?^7'PP`
M\4GO_U0`$`#1`0``T*`"`(LA(!J1>_G_EQ\``'%:$Y6:<O__%]_X_Y?Y:QBI
MW?C_E_]#`=']>P&I_4,`D>```)``N$?Y'R`#U1\@`]7A``"0(<!'^?-3`JE3
MT#O5]5L#J?43`)'W8P2I=&I@^"``0/G@!P#Y``"`T@```-#A`Q6J`"`3D8+"
M0CF"Q@(Y]?G_EX```+0!`$`Y/\0`<2`-`%18_/^7X```D`"X1_D?(`/5'R`#
MU71J8/C@`Q2J`0``T&,`@%(A8!N1(@"`4MWX_Y?T`P"J``(`M($*0/G@``"0
M`+A'^1\@`]4?(`/5=@(`BP(``-!@:F#X0H`;D2$`0/FP^/^7@`I`^0$`0/D@
M#$"Y0`NP-^```)``N$?Y'R`#U1\@`]5W`@"+8&I@^`$``-!C`(!2(:`;D2(`
M@%+V``"PP/C_E_0#`*I@`0"TX0,5J@```-``P!&1POG_E^(#`*I@!P"T@0I`
M^>`"0/DA`$#YE/C_E^```)``N$?Y'R`#U1\@`]5T`@"+<VI@^"(`@%(!``#0
M(<`;D>`#$ZJY^/^7`@``T.$#`*I"(!R1X`,3JH3X_Y?3H@N1@@)`^>$#%:H`
M``#0`(`1D5_$`CFE^?^78`H`^8`"`+0!.(!2Q?C_EP$$`!$_!`!Q!(A1>N$%
M`%3@``"0`,!'^>('0/D!`$#Y0@`!ZP$`@-+A!`!4_7M!J?-30JGU6T.I]V-$
MJ?]#`9'``U_68`9`^1/^_Y=@"@#Y0/W_M0```-``0!R1E?C_EP`$0#F@\_\T
MEO__%S'[_Y?X`P"J@`$`M($*0/GB`P"JX`)`^2$`0/E6^/^7X`,8JB#X_Y?`
M__\7P`)`^9GX_Y>E__\7P*(+D8$*0/D"!$#Y(0!`^4(`0/G@`D#Y2?C_E[7_
M_Q<_^/^7UJ(+D>,#`"H```#0`.`<D<&*0*DA`$#Y=/C_E_U[OJG]`P"1\PL`
M^?,#`*K__/^7?P(`\0080'J!``!4\PM`^?U[PJC``U_68`)`.8#__S3@`Q.J
MI?O_EYG[_Y<!``#0@@"`TB$@%Y'L]_^7@/[_->`#$ZKS"T#Y_7O"J+/Y_Q<`
M`````````/U[OZDD``#0`>``D>,``)!CP$?Y_0,`D>7E`&__0PK1X@,`D818
MP3UE`$#YY4<!^04`@-($X`&1XP,"JB($P3Q!'"5.0`1@;R,`!$X"``1.8HR?
M3#\`!.LA__]4\)]`K4*@"9'FCT&MX:,!D>630JWA'\`]\>4`;R*`V3PA0`"1
M(`#?/``<(FX`'"-N`!PP;O`<IT[''*9.9ARC3@)404\`!&%OHQRE3H4<I$XD
M'*%.0!R@3@$<,4XA@(`\/P`"Z\']_U0'*$"I)#./T@@D0:E$4*ORY3]`J2-T
MG=+B_%O3[/Q"TT(4!ZJ,>0>J(0$(RB/;K?(A``J*(0`)RJ4`!(LB0"*+0?U"
MTPL00/DA>`JJ+7Q`DD(`!8NE`0C*0@`+"^9#0:FE``>*I0`(RN\!!(M._%O3
MI0`)B\X5`JHA``S*I0`/BR$``HJE``X+3_Q"TR$`#<KB>0*JQ@`$BZ[\6],A
M``B+SA4%JB$`!HN&`0+*(0`."XQ]0)+&``6*KOQ"T\8`#,H0`@2+Q@`-BR_\
M6]/->06J[Q4!JNX30/E%``W*Q@`0BT)\0)+&``\+I0`!BJ4``LHP_$+3S@$$
MBZ4`#(O/_%O3#'H!JN\5!JJE``Z+I0`/"Z$!#,K/_$+3(0`&BN9Y!JJM?4"2
M[K]"J2$`#<HA``*+L/Q"T\(!!(NN_%O3(0`"BX(!!LI"``6*C'U`DLX5!:I"
M``S*+@`."P5Z!:KA`02+0@`-BT(``8O!``7*T'Q`DB$`#HHA`!#*S?U;TR$`
M#(NM%0ZJ[[-#J4T`#0O&_4+3SG@.JJ9\0)+B`02+K_U;TR$``HNB``[*[Q4-
MJD(`#8HO``\+I?U"TT(`!LJ!`02+I7@-JD(`$(M"``&+[/U;T\$!!<KP_4+3
MC!4/JB$`#XH/>@^J3``,"^W#1*G.?4"2(0`.RB$`!HN&_5O3QA0,JJ(!!(L0
M`@2+(0`"BZ(`#\HM``8+I7Q`DD(`#(J&_4+30@`%RL9X#*JA_5O30@`.BT(`
M$(LL%`VJL/U"T^$!!LHA``V*#7H-JN[#1:E,``P+[WU`DB$`#\HA``6+A?U;
MTZ44#*K"`02+$`($BR$``HO"``W*+@`%"\9\0))"``R*A?U"TT(`!LJE>`RJ
MP?U;TT(`#XM"`!"++!0.JM#]0M.A`07*(0`.B@YZ#JKOPT:I3``,"ZU]0)(A
M``W*(0`&BX;]6]/&%`RJX@$$BQ`"!(LA``*+H@`.RB\`!@NE?$"20@`,BH;]
M0M-"``7*QG@,JN']6]-"``V+0@`0BRP4#ZKP_4+3P0$&RB$`#XH/>@^J[<-'
MJ4P`#`O.?4"2(0`.RB$`!8N%_5O3I10,JJ(!!(L0`@2+(0`"B\(`#\HM``4+
MA?U"TZ5X#*I"``R*H?U;T\9\0)(L%`VJX0$%RB$`#8KO?4"20@`&RB$`#\I"
M``Z+(0`&B^Z;2*E"`!"+3``,"[#]0M,->@VJD/U"T\(!!(N._5O3(0`"B\X5
M#*JB``W*+@`."Z5\0))"``R*0@`%R@QZ#*K!``2+0@`/B\;]6]-"``&+[T]`
M^:$!#,K&%`ZJK7U`DM(``@LA``Z*(0`-RN0!!(LA``6+T/U"TT7^6],.>@ZJ
M[U-`^2$`!(NE%!*J1D`FB[```0N$`0[*A``&2N(!`XN$``V+!OY;TTW^0M.$
M``*+XE=`^;)Y$JK&%!"J)4`EB\\`!`O!`1+*)0`%2D$``XNB0"R+#/Y"T^7]
M6].->1"J0@`!BZ44#ZKA6T#YL0`""X9`)HM$`@W*A``&2NS]0M,F``.+@4`N
MBR3^6]./>0^J(0`&BX04$:KF7T#YD``!"T5`)8NB`0_*10`%2B[^0M/1>1&J
MK$`RB\(``XL%_EO3YF-`^8(!`HNE%!"J)$`DB^$!$<JN``(+)``$2@S^0M/!
M``.+AD`MBY!Y$*K!``&+YF=`^<3]6]-%0"6+A!0.JB("$,J-``$+0@`%2L8`
M`XM"0"^+S/U"TXYY#JI"``:+YFM`^:7]6].E%`VJ)$`DBP$"#LJO``(+(0`$
M2JS]0M.->0VJQ@`#BR%`,8OD_5O3(0`&BT5`)8OF;T#YP@$-RH04#ZI%``5*
MD0`!"ZQ`,(OP_4+3P@`#B^9S0/D/>@^J)?Y;TR1`)(NA`0_*@@$"BZ44$:HD
M``1*L``""\$``XLL_D+3AD`NBXQY$:K!``&+YG=`^45`)8L$_EO3X@$,RH04
M$*I"``5*C@`!"\8``XM"0"V+#?Y"T_%[0/E"``:+IGD0JB1`)(O%_5O3@0$&
MRJ44#JHA``1*K0`""S`"`XLA0"^+S_U"TR$`$(ON>0ZJ\']`^:3]6].$%`VJ
M14`EB[']0M/"``[*CP`!"RUZ#:I%``5*`@(#B_"#0/DD0"2+P0$-RJQ`+(LD
M``1*Y?U;TZ44#ZJ&0":+\?U"TP$"`XO!``&+@@$"BR9Z#ZJL``(+14`EBZ(!
M!LI"``5*A/U;TT)`+HN$%`RJ\+M0J9']0M./``$++'H,JB1`)(O!``S*Y?U;
MTR$`!$KQ_4+3I10/JA`"`XLO>@^J0@`0BR%`+8OMQU&IL``""\X!`XM%0"6+
M!/Y;TR$`#HN$%!"J@@$/RHX``0M"``5*0D`FBZ4!`XL-_D+310`%B\;]6].B
M>1"JQA0.JB1`)(OA`0+*S0`%"R0`!$HA`@.+A$`LB]']0M/PET#YA``!BZS]
M6],A>@ZJC!4-JJ9`)HM%``'*C@$$"Z4`!DH0`@.+I4`OBZ_]0M/O>0VJI0`0
MB_";0/G&_5O3QA0.JHQ`+(LD``_*S0`%"XP`#$K1_4+3!`(#BX)!(HNP_5O3
M0@`$BR1Z#JJF0":+[)]`^1`6#:KE`03*0@`0"Z4`!DJN_4+3H4`AB\5Y#:J&
M`0.+3OQ"TXP`!:I#_%O38Q0"JHP!`HK">0*J(0`&B^T[5*DA``,+HP`$"H:;
ME]*,`0.J9N.Q\H-!+XLO_$+3K`$&BRW\6].M%0&J8P`,BZP``JIC``T+C`$!
MBDT`!0J,`0VJX7D!JLX!!HN$022+;?Q;TTP``:JM%0.JA``.BX0`#0N,`0.*
M+0`""F[\0M.,`0VJC?Q;TX5!)8O,>0.J[SM5J:,5!*HM``RJ[P$&B\X!!HNE
M``^+C_Q"TZ,``PNE`02*C0$!"J4`#:IM_%O3I4`BB^)Y!*JM%0.JA`$"JJ4`
M#HN$``.*I0`-"TT`#`J$``VJ;?Q"TX1`(8NA>0.J[S=6J:[\6]/.%06J0P`!
MJF,`!8KO`0:+K0$&BX0`#XNO_$+3A``."RX``@IC``ZJY7D%JF-`+(N._%O3
MSA4$JBP`!:IC``V+C`$$BNV[0/EC``X+K@`!"H_\0M.,`0ZJY'D$JH)!(HMN
M_%O3K`$&B\X5`ZKMOT#Y0@`,BZP`!*I"``X+C`$#BHX`!0J,`0ZJ;_Q"T^-Y
M`ZJM`0:+@4$ABT[\6]/.%0*JC``#JB$`#8N,`0**(0`."VX`!`J,`0ZJ3OQ"
MT\)Y`JJ%026+[3M8J2_\0M.L`0:++?Q;TZT5`:JE``R+;``"JJ4`#0N,`0&*
M30`#"HP!#:KA>0&JS@$&BX1!)(NM_%O33``!JJT5!:J$``Z+A``-"XP!!8HM
M``(*KOQ"TXP!#:J-_%O3@T$CB\QY!:KO.UFIK14$JB4`#*JE``2*[P$&B\X!
M!HMC``^+C_Q"TV,`#0N-`0$*I0`-JN1Y!*JB0"*+;?Q;TZT5`ZJ%`02J0@`.
MBZ4``XI"``T+C0`,"J4`#:IM_$+3H4`ABZ5Y`ZKO-UJI3OQ;T\X5`JJ#``6J
M8P`"BN\!!HNM`0:+(0`/BZ\`!`HA``X+8P`/JD[\0M-L0"R+PGD"JB/\6].,
M`0V+8Q0!JJT``JHO_$+3@P$#"ZP!`8KA>0&J30`%"NX_6ZF,`0VJA$$DBVW\
M6].M%0.JS`$&BRX``@J$``R+3``!JH0`#0N,`0.*C`$.JFW\0M.M>0.J[P$&
MBX5!)8N._%O3[.-`^:,`#XO.%02J)0`-JF,`#@NE``2*K@$!"H_\0M.E``ZJ
M;OQ;TZ)`(HN%`0:+Y'D$JLX5`ZI"``6+I0$$JD(`#@MN_$+3I0`#BL-Y`ZKL
MNURICP`-"J4`#ZI/_$+3I4`ABT'\6],A%`*JC`$&BZ4`#(N,``.JI0`!"XP!
M`HIA``0*XGD"JHP!`:I0``,*@4$MB\T!!HNL_%O3(0`-B^_O0/EM``*JC!4%
MJJT!!8J.`0$+K0$0JJ1!)(NP_$+3[0$&B\;]6]/O\T#YA``-B\84#JH->@6J
MT0`$"RQ`+(M!``W*Q3J8TB$`#$I%3+GR(4`CB^P!!8O0_4+3(_Y;T^_W0/D.
M>@ZJ8Q01JB$`#(ML``$+AD`FBZ0!#LHC0".+A``&2N8!!8N"0"*++_Y"TX3]
M6]/Q>1&J0@`&BX04#*KF^T#YD@`""\$!$<I$0"2+(P`#2L$`!8MF0"V+C?U"
MTT/^6].P>0RJQ@`!BV,4$JKA_T#Y;P`&"R("$,I,_D+30@`$2B0`!8M"0"Z+
MX?U;TXUY$JI$``2+X@-!^2$4#ZHN``0+PT`CBP8"#<KL_4+3PP`#2D8`!8MC
M0#&+POU;TX]Y#ZIC``:+Y@=!^4(4#JI1``,+@4`ABZ0!#\IB0"*+@0`!2L0`
M!8LL0#"+T/U"TR'^6],.>@ZJY@M!^80!!(LA%!&JXP$.RC``!`MB``)*+/Y"
MT\,`!8M&0"V+`OY;TXQY$:K#``.+Y@]!^4(4$*I-``,+@4`AB\0!#,H1_D+3
M@0`!2L8`!8LD0"^+H?U;T^\30?F$``:+(10-JB9Z$*IB0"*+,``$"X,!!LKO
M`06+8@`"2K']0M-#0"Z+`OY;T^X70?DM>@VJ8P`/BT(4$*I/``,+@4`AB\0`
M#<IB0"*+@0`!2L0!!8LL0"R+#OY"T^']6]/0>1"JA`$$BR$4#ZKL&T'Y,0`$
M"Z,!$,J!0"&+8@`"2H,!!8M&0":+[/U"TR+^6]./>0^JPP`#BT(4$:KF'T'Y
M3@`#"P0"#\IB0"*+@0`!2L8`!8LD0"V++?Y"T\']6].$``:+["-!^:9Y$:HA
M%`ZJXP$&RBT`!`MB``)*C`$%BT-`,(O0_4+3HOU;TPYZ#JIC``R+["=!^4(4
M#:I1``,+@4`AB\0`#LIB0"*+@0`!2H0!!8LL0"^+K_U"TR'^6]/M>0VJA`$$
MBR$4$:KL*T'Y,``$"\,!#<HO_D+38@`"2HP!!8M"0":+`_Y;TT8`#(MC%!"J
MXB]!^>QY$:J!0"&+;P`&"Z0!#,K#0".+A``!2D$`!8N$0"Z+XOU;TP[^0M.$
M``&+\3-!^<%Y$*I"%`^JA@$!RDX`!`O#``-*,`(%BV9`+8OQ_4+3P_U;T^TW
M0?EC%`ZJ+WH/JL8`$(MP``8+@D`BBR0`#\K&0".+A``"2J(!!8N,0"R+S?U"
MTP3^6].,`0*+XCM!^804$*JN>0ZJ@P`,"^T!#LKG0"^+K0$&2D8`!8NA02&+
M8OQ;TT(4`ZHA``:+[3]!^2$``@L"_D+3AD$DBT)X$*HD_%O3S`$"RH04`:J,
M`09*I@$%BV7\0M/&``>+HW@#JH0`#(O'``#PY\!'^2$`"@M"``D+Q0$+"X0`
M!@MC``@+!`0`J0,(`:D%$`#YXD=!^>$`0/E"``'K`0"`TH$``%3_0PJ1_7O!
MJ,`#7];[\_^7'R`#U1\@`]7]>[^I`!"`TOT#`)&)]/^7'Q@`^2(``+`A``"P
M'W@`N4%@P3TB``"P(ES!/4!DP3T"!`"M``B`/?U[P:C``U_6_7N\J4)\0)/]
M`P"1\U,!J13@`)'S`P&J]5L"J?4#`*KV`P*J]V,#J06,0JD`>$"YI`P""[\`
M!.MCE(.:8G1"BZ2*`JG@`@`T%PB`4O<"`$O_`A9K@,(@B_?2EAKX?D"3X@,8
MJL?S_Y>@>D"YX`(`"Z!Z`+D?``%QP```5/-30:GU6T*I]V-#J?U[Q*C``U_6
MU@(72W,"&(O@`Q6JI?O_E]_^`'%M`P!4U@(!4=A^!E,7!P`1]VIZTW<"%XL?
M(`/58`9`K>`#%:J`!@"M8`9!K7,"`9&`!@&MEOO_EW\"%^L!__]4UAH82^$#
M%ZK"?D"3X`,4JJ/S_Y>V>@"Y\U-!J?5;0JGW8T.I_7O$J,`#7];W`Q.J]?__
M%Q\@`]4?(`/5_7N\J>(/@!+]`P"1\U,!J?0#`*KS`P&J]5L"J3;@`)'H&P#]
M-11`^2@80/V@(@-3`00`$<+*(#C`PB&+/^``<4T+`%0""(!20@`!2P$`@%)"
M?$"3J_/_E^`#$ZIM^_^7?S8`^0`$`$_``@"MP`J`/08%:'^@`F>>!05P?P0%
M>'_@`Q.J`P1H?P($<'\!!'A_P`0/;J`$#6Z`!`MN``4);F`$!VY`!`5N(`0#
M;F`Z`/U8^_^780)`^>`#$ZHA_%C3@0(`.6$"0/DA_%#3@08`.6$"0/DA_$C3
M@0H`.6$"0/F!#@`Y809`^2'\6-.!$@`Y809`^2'\4-.!%@`Y809`^2'\2-.!
M&@`Y809`^8$>`#EA"D#Y(?Q8TX$B`#EA"D#Y(?Q0TX$F`#EA"D#Y(?Q(TX$J
M`#EA"D#Y@2X`.6$.0/DA_%C3@3(`.6$.0/DA_%#3@38`.6$.0/DA_$C3@3H`
M.6$.0/F!/@`Y81)`^2'\6-.!0@`Y81)`^2'\4-.!1@`Y81)`^2'\2-.!2@`Y
M81)`^8%.`#GS4T&I]5M"J>@;0/W]>\2H"_/_%P('@%)"``%+`0"`4D)\0)-2
M\_^7K?__%\```/``0`J1`4!!.4$#`#7!``#0(:1'^2(`@%("0`$YH0$`M/U[
MOZG```#P_0,`D0#X0/G;\O^7P```T`#41_E@`0"T_7O!J"```)``X!R1A//_
M%\```-``U$?YP```M"```)``X!R1?O/_%_U[P:C``U_6P`-?U@``````````
MP```\`%`"I$"0$HY0@$`-<(``-!"M$?Y(P"`4@-`"CFB``"T(4``D2```)``
MX!R1"//_%\`#7]8``````````/^#`='#``#08\!'^?U[`ZG]PP"1XH<`J>(C
M`)'A0P"1X!\`N>!S`)'S4P2I]5L%J60`0/GD%P#Y!`"`TB7S_Y?"``#00LQ'
M^<```-``X$?YX0,"JB_S_Y?```#0`+!'^0``0#G`"0`T0=`[U=,``/#```#0
M`+A'^1\@`]4?(`/5]!]`N2)H8/B`$@`1E`X`$0!\?9-!"%(Y(00`,D$($CG\
M\O^7XP,`JN8+0/EBH@N1P```T`#(1_D!``"0QP!`^2$`'I$$`$#Y````D.4?
M0+D`(!Z19P0`J4,$`/ED``&IOP0`<6T)`%0A`(#2'R`#U2`,`!'$>&'X(00`
MD638(/B_``%K;/__5*4,`!%UH@N1?]@E^.'__[#B`Q0J(9`CD00`@-*@#D#Y
M#O/_ET`%`#1UH@N1]I,`D:`.0/FM\O^7]`,`*N$#%JH```"0`&`3D=SS_Y=@
M!`"T<Z(+D6`.0/F0\O^7[_+_E\```-``P$?YXA=`^0$`0/E"``'K`0"`T@$%
M`%3]>T.IX`,4*O-31*GU6T6I_X,!D<`#7]:E\O^7TP``\&*B"Y%`#`#Y8`,`
MM&SR_Y=!T#O5P```T`"X1_D?(`/5'R`#U2!H8/@?&"0YJ/__%Z`.0/G^\O^7
MU?__%Z`*0/E@``"T3_K_E]O__Q?A`Q:J````D`"`$9&Q\_^7H`H`^:#Z_[1'
M^O^7T___%X4`@%*^__\7(`"`4B7R_Y=X\O^7````````````````<'-?<W1R
M:6YG<R!M:7-S:6YG"@``````)7,O)7,````N`````````$1Y;F%,;V%D97(Z
M.F)O;W1?1'EN84QO861E<@``````26YT97)N86QS.CI005(Z.D)/3U0`````
M4$%27U1%35```````````%!!4E]04D]'3D%-10`````Z`````````"\`````
M````)7,E<R5S``!015),-4Q)0@``````````4$523$Q)0@!015),-4]05```
M````````4$523$E/``!005)?24Y)5$E!3$E:140`4$%27U-005=.140`````
M`%!!4E]#3$5!3@````````!005)?1$5"54<`````````4$%27T-!0TA%````
M`````%!!4E]'3$]"04Q?1$5"54<``````````%!!4E]'3$]"04Q?5$U01$E2
M`````````%!!4E]435!$25(```````!005)?1TQ/0D%,7U1%35``4$%27T=,
M3T)!3%]#3$5!3@``````````,0````````!L<V5E:R!F86EL960`````<F5A
M9"!F86EL960``````$Q$7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````
M<&%R+0`````E<R5S)7,E<P``````````)7,Z(&-R96%T:6]N(&]F('!R:79A
M=&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@```````"5S
M.B!P<FEV871E('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R
M96UO=F4@:70@86YD(')E=')Y('EO=7(@;W!E<F%T:6]N*0H```````!0051(
M`````'!A<FP`````+G!A<@````!S:&]R="!R96%D````````)7,E<V-A8VAE
M+25S)7,``"5S)7-T96UP+25U)7,````E<R5S=&5M<"TE=2TE=25S````````
M```8`````````'!E<FP`````,`````````!005(Z.E!A8VME<CHZ5D524TE/
M3@`````Q+C`V,P```%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y
M```````````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E
M8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```````````M90```````"TM
M````````55-%4@````!54T523D%-10``````````5$U01$E2``!414U01$E2
M`%1%35``````5$U0```````O=&UP+P```&5V86P@)V5X96,@+W5S<B]B:6XO
M<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N
M9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'
M24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*0D5'
M24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S971?
M<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`](#0P
M.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?<&%R
M7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V-"`J
M(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S(#T@
M*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@(R!.
M3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI;&4@
M*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L("1B
M=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H*&UY
M("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["G)E='5R
M;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U<FX@
M+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO
M9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U
M:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I
M<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID
M:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@
M(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@
M+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P
M87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G
M*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M
M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D
M:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I
M;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*
M>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E
M;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS
M;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN
M+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD
M(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD
M*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F
M;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN=&5R
M;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z
M.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D
M14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D
M7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H
M)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I
M+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@
M.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M
M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<
M)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A
M<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*;&]C
M86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?1D@L
M("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@9G)O
M;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E(&%S
M(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA9VEC
M7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O<R`\
M(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!4E]-
M04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]"F]U
M=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S(BD[
M"G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2!L
M96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T:"@B
M7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N92`B
M7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD["FQA
M<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O=6YD
M72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4L
M(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@)&]F
M9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O;2!S
M=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G:6-?
M<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A<G1?
M;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0[
M"G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@("`@
M(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%
M(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C
M:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N
M<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F
M=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968[
M"FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N
M*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F
M+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E
M>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`]
M(%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V
M>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE
M879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-E
M('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC
M<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E7VQI
M<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L
M($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR971U
M<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@)$UO
M9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID96QE
M=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]("(O
M;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N
M9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL
M92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F:6QE
M.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[8G5F
M?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*;7D@
M)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I;F9O
M+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@;7D@
M)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E860@
M(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S=')A
M<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2(["GTL
M($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD[
M"G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y
M.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z
M.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U
M:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F
M:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M
M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE
M;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O
M.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II9B`H
M)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@
M869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S
M="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@
M("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@
M*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@
M)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E
M(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D05)'
M5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T
M7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@
M("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI
M.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF
M="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!4D=6.PI]
M"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN
M8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),8G%P
M:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S:"!`
M861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U
M<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R7V%R
M9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]"F5L
M<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I9B`H
M)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@
M)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N
M("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G(&9I
M;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE7VYA
M;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I<W1?
M8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I;F,H
M*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"D@
M=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R($!!
M4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*?0I]
M"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!<F-H
M:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]('-H
M:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N
M(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN9"!P
M87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G*3L*
M)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M1FEL
M94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I
M"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R(ET[
M"GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@
M/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ
M7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>
M("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2![
M"F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R
M96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@
M)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL
M97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E871E
M4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/
M.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)3SHZ
M1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C<F5A
M=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@7T9(
M+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?1DE,
M15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?1DE,
M15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R
M(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L;V%D
M97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z
M.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A
M9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;17)R
M;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N9&QE
M*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN
M86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@
M0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@
M)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!]
M(&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)
M3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,[
M"B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE
M8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF
M:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_
M"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL
M97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]
M("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA
M;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H
M0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]
M?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-
M;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I
M.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@
M/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@
M9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X
M="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%
M)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R
M968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E
M;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69]
M.PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@)SPZ
M<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=
M.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R.CI0
M;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I
M"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M
M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA
M<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B
M1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H
M(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N
M="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N
M=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,12!T
M;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*
M?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I=&EN
M9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE86Y]
M*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D969I
M;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W
M*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS965K
M("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*
M)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H95]N
M86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA;65?
M<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A8VAE
M7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L;"`M
M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F9G-E
M="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O<B!W
M<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L("1O
M=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,[
M"E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I
M<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@
M("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R
M871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I
M+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV
M93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI
M=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@
M)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I92!Q
M<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C
M:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2
M.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]
M("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I;F5D
M("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O
M:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN
M97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@
M/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE
M(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN86UE
M?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*;7D@
M)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?5$5-
M4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M<R!?
M(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q6U-K
M:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI<W1S
M(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A8W1I
M;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D;65M
M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C:&EV
M93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN9R!Z
M:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D97-T
M7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D4$%2
M.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S($!!
M4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M36UO
M9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\+6(@
M72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345](#T@
M)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T95!A
M=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA;64[
M"FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N86UE
M.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE.CI0
M871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D<&%T
M:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E7VUO
M9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR97%U
M:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI;F=S
M.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P.CI(
M96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD[
M"G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@1FEL
M93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U:7)E
M($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E($%R
M8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!005([
M"G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E<75I
M<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL=&5R
M.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*979A;"![(')E
M<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@>R!R
M97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H:79E
M.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H.CI.
M86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U:7)E
M(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*?0IS
M=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]
M(&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP(#T@
M)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6>R1?
M?2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@*2DL
M"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F:6YE
M9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E<FYA
M;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D/BD@
M:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I;DYA
M;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE;'-I
M9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]"F5L
M<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U5315)]
M('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L
M:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP
M9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%04Y]
M*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@;7D@
M)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@
M(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN9%]P
M87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O<PHM
M("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`["FUY
M("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@
M*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@
M)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C86-H
M95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M95]S
M:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M)&)U
M9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C86-H
M95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*)'-T
M;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D<W1M
M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S=#L*
M?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM0
M05)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D;F%M
M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@(B1P
M87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY("1T
M96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F:"P@
M)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT<RD*
M)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T96UP
M;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D
M("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN
M:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?
M<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD
M("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@
M)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N
M9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P
M<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L
M("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%
M?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII
M9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M
M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O
M9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D
M9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@
M>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D
M14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\[
M"B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES
M>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N
M86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD
M<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G
M;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N
M86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[
M7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z
M.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ
M1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W
M9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I
M+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O
M9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.
M251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM0
M05)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@
M0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[
M(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'
M("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?(GT@
M:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.5GM0
M05)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U8B!O
M=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D
M;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@
M:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E
M9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X
M<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C
M:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@
M4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O
M<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L
M('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R
M>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z
M.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I
M92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z
M.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI
M7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?
M7PH`<&5R;'AS:2YC``````````!#04-(10`````````````'!@4$`P(!``\.
M#0P+"@D(`````&=%(P$`````[\VKB0````"8NMS^`````!`R5'8`````P]+A
M\````````````1L#.P```/P````>__^*P````13__XL@```!0/__BX0```%L
M__^,\````=C__XWD```"0/__CH````)H__^/\````KC__Y#@```"\/__DP0`
M``-$__^31````V3__Y.@```#B/__D[````.<__^5]```!`3__Y8@```$*/__
MEI````1<__^;8```!)C__YO@```$O/__G(````38__^=X```!1S__YX````%
M//__GP0```5X__^DP```!>C__Z=4```&(/__I\````:`__^X4```!JS__[B0
M```&S/__N;````<0__^[<````:S__[OP```!F/__O#````9,`````````!``
M`````7I2``1X'@$;#!\`````*````!C__XFD````7`!!#C"=!IX%1)4"190#
MDP1*U--"WMW5#@`````````H````1/__B=@```!D`$$.,)T&G@5"DP24`T.5
M`E$*WMW5T]0.`$$+`````"@```!P__^*$````60`00Y`G0B>!T*4!9,&098#
ME01!F`&7`@``````````$````)S__[I0````.``````````H````L/__N;P`
M``!X`$H.$)T"G@%(WMT.`$D.$)T"G@%!WMT.`````````&0```#<__^+$```
M`/0`00Y`G0B>!T*3!I0%0I@!EP)&E@.5!%;6U4'8UT/>W=/4#@!!#D"3!I0%
ME026`Y<"F`&=")X'3];50=C70=[=T]0.`$$.0),&E`6=")X'0I<"F`%!V-<`
M````)````43__XN<````G`!!#L`!1)T$G@-"DP)6"MW>TPX`00L``````$P`
M``%L__^,$````6P`00[0`4.=!IX%1I0#DP1$U--)W=X.`$$.T`&3!)0#G0:>
M!4&6`94"=`K4TT'6U4$+2=/4U=9!E`.3!$&6`94"````-````;S__XTP````
M[`!!#B"=!)X#1@ZP"),"80H.($+>W=,.`$$+2@H.($3>W=,.`$$+``````!0
M```!]/__C>@```(D`$$.8$&="IX)0I,(E`=-E0:6!9<$F`-_"MW>U]C5UM/4
M#@!!"TF:`9D"70K:V4$+2-G:0ID"F@%#V=I!F@&9`D+:V0`````<```"2/__
MC[@```!``$$.$)T"G@%(WMT.`````````"````)H__^/V````%P`00X@G02>
M`T.3`DS>W=,.`````````!````*,__^0$`````@`````````9````J#__Y`,
M```"1`!!#E"="IX)1),(E`=##H`*E0:6!5H*#E!#WMW5UM/4#@!!"TB7!$*8
M`T.9`D.:`70*UT'80=E!VD$+5@K70=A!V4':00M$U]C9VD&7!$&8`T&9`D&:
M`0`````@```#"/__D>@````L`$$.()T$G@-#DP)&WMW3#@`````````P```#
M+/__D?````!L`$(.()T$G@-"DP)*"M[=TPX`00M#"M[=TPX`0@M#WMW3#@``
M````.````V#__Y(L```$T`!!#F!#G0J>"4*3")0'1I4&E@67!)@#F0(#`1T*
MW=[9U]C5UM/4#@!!"P``````(````YS__Y;`````>`!!#B!%G0*>`50*W=X.
M`$$+````````&````\#__Y<<````G`!!#J`"19T:GAD``````$````/<__^7
MH````5P`00Y`G0B>!T63!I0%1`Y0E026`Y<"F`%!#M"`!'L*#L"`!$$.0$3>
MW=?8U=;3U`X`00L`````'```!"#__YB\````(`!!#A"=`IX!1M[=#@``````
M```X```$0/__F+P```$$`$$.0$.=!IX%0I,$E`-"E0*6`5D*W=[5UM/4#@!!
M"UT*W=[5UM/4#@!!"P````!L```$?/__F80```6\`$$.D`-%G0J>"423")0'
M0Y4&E@5#EP28`UB:`9D"`F<*VME!W=[7V-76T]0.`$$+`H+9VD[=WM?8U=;3
MU`X`00Z0`Y,(E`>5!I8%EP28`YD"F@&="IX)`DO9VD&:`9D"````-```!.S_
M_Y[0```"E`!!#E!!G0B>!TB3!I0%0I4$E@-"EP*8`0)N"MW>U]C5UM/4#@!!
M"P`````H```%)/__H2P```!D`$$.()T$G@-"DP)'"M[=TPX`00M-WMW3#@``
M`````#````50__^UW````B0`00Y@0YT&G@5(DP24`Y4"E@$"6@K=WM76T]0.
M`$$+```````````H```%A/__H3@``!"(`$$.$)T"G@%'#J`%`P07"@X00=[=
M#@!!"P```````!P```6P__^QG````$``00X0G0*>`4[>W0X`````````0```
M!=#__[&\```!&`!!#D"=")X'0Y,&E`5#E026`T.7`I@!60K>W=?8U=;3U`X`
M00M?"M[=U]C5UM/4#@!!"P`````T```&%/__LI@```'``$$.0)T(G@=#DP:4
M!4.5!)8#0@5(`@)@"M[=U=;3U`9(#@!!"P``````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````!1D```````
M`%$0`````````````````````0````````+L``````````$````````"]P``
M```````!`````````P$``````````0````````,/``````````$````````#
M'P`````````/````````!%<`````````&0```````?U``````````!L`````
M````"``````````:```````!_4@`````````'``````````(``````````0`
M```````"2``````````%````````#N``````````!@````````5H````````
M``H````````$H0`````````+`````````!@`````````%0``````````````
M``````,```````'_Z``````````"````````!;@`````````%``````````'
M`````````!<````````6H``````````'````````$X@`````````"```````
M``,8``````````D`````````&`````!O___[``````@`````````;___^0``
M```````7````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``']6````````?U0`````````````````````````````?U`````````````
M```````````````````````````"`T```````````````````?U(```````"
M`F@```````````````````````````````````````']6```````````````
M``````````````("B``````````````````"`>@````````Z8```````````
M`````````````````````````````!Q@````````'&`````````<8```````
M`!Q@````````'&`````````<8````````!Q@````````'&`````````<8```
M`````!Q@````````'&`````````<8````````!Q@````````'&`````````<
M8````````!Q@````````'&`````````<8````````!Q@````````'&``````
M```<8````````!Q@````````'&`````````<8````````!Q@````````'&``
M```````<8````````!Q@````````'&`````````<8````````!Q@````````
M'&`````````<8````````!Q@````````'&`````````<8````````!Q@````
M````'&`````````<8````````!Q@````````'&`````````<8````````!Q@
M````````'&`````````<8````````!Q@````````'&`````````<8```````
M`!Q@````````'&`````````<8````````!Q@````````'&`````````<8```
M`````!Q@````````'&`````````<8````````!Q@````````'&`````````<
M8````````!Q@```````"`H````````(!\`I005(N<&T*````````````````
M``!7R````````%0@``````````````````!52````````%>H````````5[``
M``````!7N````````%?```````````````````!7D````````%>8````````
M``````````!7T$=#0SH@*$YE=$)31"!N8C$@,C`R-3`W,C$I(#$R+C4N,``D
M3F5T0E-$.B!C<G0P+E,L=B`Q+C(@,C`Q."\Q,2\R-B`Q-SHS-SHT-"!J;V5R
M9R!%>'`@)``D3F5T0E-$.B!C<G0P+6-O;6UO;BYC+'8@,2XS,"`R,#(U+S`U
M+S`R(#(S.C`T.C`V(')I87-T<F%D:"!%>'`@)``D3F5T0E-$.B!C<G1I+E,L
M=B`Q+C$@,C`Q-"\P."\Q,"`P-3HT-SHS-B!M871T($5X<"`D`"1.971"4T0Z
M(&-R=&)E9VEN+F,L=B`Q+C$W(#(P,3@O,3(O,C@@,3@Z,3<Z,3$@8VAR:7-T
M;W,@17AP("0`)$YE=$)31#H@8W)T96YD+E,L=B`Q+C$@,C`Q-"\P."\Q,"`P
M-3HT-SHS-B!M871T($5X<"`D```N<VAS=')T86(`+FEN=&5R<``N;F]T92YN
M971B<V0N:61E;G0`+FYO=&4N;F5T8G-D+G!A>``N:&%S:``N9'EN<WEM`"YD
M>6YS='(`+G)E;&$N9'EN`"YR96QA+G!L=``N=&5X=``N<F]D871A`"YE:%]F
M<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YJ
M8W(`+F1Y;F%M:6,`+F=O=``N9V]T+G!L=``N9&%T80`N8G-S`"YC;VUM96YT
M````````````````````````````````````````````````````````````
M``````````````````````````````````````L````!``````````(`````
M```"``````````(``````````!<````````````````````!````````````
M```3````!P`````````"`````````A@````````"&``````````8````````
M````````````!```````````````)@````<``````````@````````(P````
M`````C``````````%`````````````````````0``````````````#<````%
M``````````(````````"2`````````)(`````````R`````%````````````
M```(``````````0````]````"P`````````"````````!6@````````%:```
M``````EX````!@````,`````````"``````````8````10````,`````````
M`@````````[@````````#N`````````$H0````````````````````$`````
M`````````$T````$``````````(````````3B````````!.(`````````Q@`
M```%```````````````(`````````!@```!7````!`````````!"````````
M%J`````````6H`````````6X````!0```!,`````````"``````````8````
M7`````$`````````!@```````!Q@````````'&`````````#\```````````
M`````````!```````````````&$````!``````````8````````@4```````
M`"!0````````,Z0````````````````````0``````````````!G`````0``
M```````"````````5`````````!4`````````$&@````````````````````
M$```````````````;P````$``````````@```````)6@````````E:``````
M````_`````````````````````0``````````````'T````!``````````(`
M``````"6H````````):@````````!DP````````````````````(````````
M``````"'````#@`````````#```````!_4````````#]0``````````(````
M````````````````"``````````(````DP````\``````````P```````?U(
M````````_4@`````````"`````````````````````@`````````"````)\`
M```!``````````,```````']4````````/U0``````````@`````````````
M```````(``````````````"D````!@`````````#```````!_5@```````#]
M6`````````'@````!@``````````````"``````````0````K0````$`````
M`````P```````?\X````````_S@`````````L`````````````````````@`
M````````"````+(````!``````````,```````'_Z````````/_H````````
M`@`````````````````````(``````````@```"[`````0`````````#````
M```"`>@```````$!Z`````````"(````````````````````"```````````
M````P0````@``````````P```````@*````````!`G``````````R```````
M`````````````$```````````````,8````!`````````#``````````````
M`````0)P`````````"(````````````````````!``````````$````?````
M`0`````````P``````````````````$"D@````````$B````````````````
M`````0`````````!`````0````,````````````````````````````!`[0`
?````````SP````````````````````$`````````````
