/*
 * AppleRecords - A DAAP client
 * Copyright (C) 2004  Chris Davies <c.davies@cdavies.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
 
package org.cdavies.applerecords;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;

import org.cdavies.itunes.*;
import javazoom.jl.player.*;
import org.cdavies.applerecords.table.*;

/**
 * 
 * This Panel contains swing controls for controlling playing
 * behaviour of AppleRecords. In addition, this class controls
 * the player thread, and has methods for manipulating it.
 * <p>
 * At present this class is perhaps the least "finalised" in the
 * entire software. It has been evolved, not designed to control
 * what the PlayerThread ought to be doing, and will continue to
 * evolve as the PlayerThread is adapted to handle downloads also
 * <p>
 * It is entirely possible (probable?) that this class will be 
 * entirely rewritten in the next version. Don't bank on its
 * interface.
 * 
 * 
 * @author       Chris Davies (c.davies@cdavies.org)
 * @version      0.5 21/06/2004
 * @see			 PlayerThread
 *
 */
 
public class AppleRecordsControlPanel extends JPanel implements ActionListener {
	
	private ImageIcon _stop;
	private ImageIcon _play;
	private ImageIcon _pause;
	private ImageIcon _download;
	
	private JButton _stateButton;
	private JButton _stopButton;
	private JButton _downloadButton;
	
	private JTable _trackTable;
	private TrackListTableModel _model;
	private AppleRecordsStatusPanel _status;
	private PlayerThread _thread;
	
	private int _playIndex;
	private boolean _playing;
	private boolean _contPlay;
	
	/**
	 *
	 * Constructs a new Control Panel, of which there will be only one in
	 * any given player. Constucts a Swing JPanel, loads various image
	 * resources for the buttons using the image locator, and constructs some
	 * buttons for it.
	 * <p>
	 * In addition, it constructs and runs a new PlayerThread which will be
	 * under the direct and exclusive control o this class, hopefully to 
	 * avoid more significant thread safety issues (it doesn't at present ;)
	 *
	 *
	 * @param	trackTable	The table widget in the main display, from which we extract tracks
	 * @param	model	The table model for the above table
	 * @param	status	The status panel for status related messages
	 * @see	ResourceLocator
	 * @see	PlayerThread
	 *
	 */
	
	public AppleRecordsControlPanel(JTable trackTable, TrackListTableModel model, AppleRecordsStatusPanel status) {
		
		super();
		
		_trackTable = trackTable;
		_model = model;
		_status = status;
		_playIndex = -1;
		
		ResourceLocator _loc = new ResourceLocator();
		_play= new ImageIcon(_loc.loadImageResource("play"));
		_pause = new ImageIcon(_loc.loadImageResource("pause"));
		_stop = new ImageIcon(_loc.loadImageResource("stop"));
		_download = new ImageIcon(_loc.loadImageResource("download"));
		
		_stateButton = new JButton(_play);
		_stateButton.addActionListener(this);
		_stateButton.setToolTipText("Play");
		
		_stopButton = new JButton(_stop);
		_stopButton.addActionListener(this);
		_stopButton.setToolTipText("Stop");
		
		_downloadButton = new JButton(_download);
		_downloadButton.addActionListener(this);
		_downloadButton.setToolTipText("Download");
		
		add(_stateButton);
		add(_stopButton);
		add(_downloadButton);
		
		_thread = new PlayerThread(_status, this);
		_thread.start();
		
		_playing = false;
		
	}
	
   /**
	*
	* One of the various dodgy temporary methods that may or
	* may not dissapear faily soon.
	* <p>
	* This one indicates that the player message should not
	* continue playing after the end of the present track, if
	* indeed it is actually playing at present.
	*
	*/
	
	public synchronized void noNextTrack() {
		
		_contPlay = false;
		
	}
	
   /**
	* 
	* Indicates whether the player thread is playing a track at the momeent.
	* 
	* @return	true if the player thread is playing a track at the moment, false otherwise.
	* 
	*/
	
	public synchronized boolean isPlaying() {
		
		return _playing;
		
	}
	
   /**
	*
	* A dodgy temporary, misnamed method. It was originally supposed to allow
	* the PlayerThread to indicate it had stopped playing for some reason or 
	* other.
	* <p>
	* As with other similar methods, it may or may not go away in the near future.
	*
	*/
	
	public synchronized void stopped() {
		
		_playing = false;
		
	}
	
   /**
	*
	* Returns the next in the sequence of tracks that should be played,
	* as indicated by the the TrackTable model.
	* <p>
	* This method is used by the PlayerThread to get the next track to be
	* played.
	*
	* @return	The next Track to be played, or null if no track is to be played.
	*/
	
	public Track getNextTrack() {
		
		if (_playIndex == -1)
			return null;
		
		if (_playIndex == (_model.getRowCount() - 1))
			_contPlay = false; /* Don't play past the end of the table */
		
		return (Track)_model.getValueAt(_playIndex++, 3);
		
	}
	
   /**
	*
	* This method causes the TrackTable to highlight the currently playing
	* track as inicated by the table model.
	* <p>
	* This method may, or may not be integrated in to the getNextTrack() method
	* in the near future, depending on how the download functionality evolves.
	*
	* @see #getNextTrack()
	*
	*/
	
	public void selectPlayingTrack() {
		
		if (!_playing || (_playIndex == -1))
			return;
		
		_trackTable.setRowSelectionInterval(_playIndex - 1, _playIndex - 1);
		
	}
	
   /**
	*
	* Indicates whether or not the PlayerThread will continue playing after
	* the current track ends.
	*
	* @return	true if the PlayerThread should continue playing after the end of the next track, false otherwise.
	*/
	
	public synchronized boolean continuePlaying() {
		
		return _contPlay;
		
	}
	
   /**
	*
	* Controls the execution of the player thread, if it becomes false
	* the PlayerThread will terminate, which should only happen when the
	* application terminates.
	*
	* @return false if the PlayerThread should continue, true otherwise.
	*
	*/
	
	public boolean closeDownMode() {
		
		return false;
		
	}
	
   /**
	*
	* Indicates the PlayerThread should begin playing the currently selected 
	* track immediately.
	* <p>
	* This happens at the expense of the currently playing track, which is 
	* terminated, no matter what stage of execution it is currently at.
	*
	*/
	
	public synchronized void playSelectedTrack() {
		
		stopTrack();
		_thread.setPlayMode();
		
		_playIndex = _trackTable.getSelectedRow();
		_contPlay = true;
			
		_playing = true;
		
	}
	
	public synchronized void downloadSelectedTrack() {
		
		stopTrack();
		_thread.setDownloadMode();
		
		String _response = (String)JOptionPane.showInputDialog(
			this, "Enter file to save as...",
			"Enter File Name", JOptionPane.PLAIN_MESSAGE,
			null, null, "");
			
		if (_response != null && _response.length() > 0) {
			
			File _file = new File(_response);
			
			try {
				
				if (!_file.exists() && !_file.createNewFile()) {
					
					_status.setTimedText("Could not write file...");
					return;
					
				}
			}
			catch (Exception _e) {
				
				_status.setTimedText("An error occured while writing file...");
				return;
				
			}
			
			
			_thread.setDownloadFile(_file);
			
			_playIndex = _trackTable.getSelectedRow();
			_contPlay = true;
			
			_playing = true;
			
		}
		
		
		
	}
	
   /**
	*
	* Stops the currently playing track and indicates the PlayerThread should
	* wait for a further signal to begin playing once again.
	*
	*/
	
	public synchronized void stopTrack() {
		
		if (!_playing)
			return;
		
		_playing = false;
		
		_contPlay = false;
		_thread.stopTrack();
		
	}
	
   /**
	*
	* This class also acts as its own ActionListener to detect events
	* which occur on this Panel's buttons.
	* <p>
	* Essentially, method controls the actions of the managed PlayerThread.
	*
	* @param	e	The description of the event which has just occured
	*/
	
	public void actionPerformed(ActionEvent e) {
		
		if (e.getSource() == _stopButton)
			stopTrack();
		else if (e.getSource() == _stateButton)
			playSelectedTrack();
		else if (e.getSource() == _downloadButton)
			downloadSelectedTrack();
		
	}
	
	
}
