//
// $Id: SocketAddress.h,v 1.21 2007/03/08 19:59:21 will_mason Exp $
//
// vi: set ft=objc:

/*
 * ObjectiveLib - a library of containers and algorithms for Objective-C
 *
 * Copyright (c) 2004-2007
 * Will Mason
 *
 * Portions:
 *
 * Copyright (c) 1994
 * Hewlett-Packard Company
 *
 * Copyright (c) 1996,1997
 * Silicon Graphics Computer Systems, Inc.
 *
 * Copyright (c) 1997
 * Moscow Center for SPARC Technology
 *
 * Copyright (c) 1999 
 * Boris Fomitchev
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You may contact the author at will_mason@users.sourceforge.net.
 */

#if !defined(SOCKETADDRESS_OL_GUARD)
#define SOCKETADDRESS_OL_GUARD

#include <ObjectiveLib/ObjectBase.h>
#if defined(OL_HAVE_INTTYPES_H)
#include <inttypes.h>
#else
#include <stdint.h>
#endif

@class OLText;
@class OLVector;
struct sockaddr;
struct sockaddr_in;
struct sockaddr_in6;
struct sockaddr_un;

/**
 * @class OLSocketAddress SocketAddress.h ObjectiveLib/SocketAddress.h
 *
 * An address suitable for use with sockets. This base class provides common
 * functionality shared by all addresses.
 *
 * @ingroup Sockets
 */
@interface OLSocketAddress :
#if defined(OL_NO_OPENSTEP)
    Object
#else
    NSObject <NSCopying>
#endif
{
}

#if defined(OL_NO_OPENSTEP)
/**
 * Make a copy of this socket address.
 *
 * @return the copy
 */
- (id) copy;
#else
/**
 * Make a copy of this socket address allocating memory from @a zone.
 *
 * @param zone the zone from which to allocate memory
 * @return the copy
 */
- (id) copyWithZone: (NSZone*)zone;
#endif

/**
 * Return a description of the socket address. The nature of the description will
 * depend on the type of socket address.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return the description
 */
- (OLText*) description;

/**
 * Return the hash code of this socket address. The hash code is computed using
 * OLHash.
 *
 * @return the hash code
 */
- (unsigned) hash;

/**
 * Return a platform-dependent representation of the address. The returned
 * data can be used in low-level socket operations and is intentionally
 * opaque.
 *
 * @return the host representation
 */
- (const struct sockaddr*) hostRepresentation;

/**
 * Return the length of the host representation. The length of the structure
 * returned by #hostRepresentation is returned.
 *
 * @return the length of the host representation
 */
- (unsigned) hostRepresentationLength;

@end

/**
 * @class OLInternetAddress SocketAddress.h ObjectiveLib/SocketAddress.h
 *
 * An address in the Internet family of addresses. Internet address are suitable
 * for contacting Internet hosts.
 *
 * @ingroup Sockets
 */
@interface OLInternetAddress : OLSocketAddress
{
@protected
    /**
     * The canonical name
     */
    char*           canonicalName;

    /**
     * The type of socket for which this address is suitable
     */
    int             socketType;
}

/**
 * Create and return a new address. The address of the current host is determined
 * and associated with the given port.
 *
 * @exception OLSocketException if there is a problem determining the host's address
 * @param port the port that should be associated with the address
 * @return a reference to the new address
 */
+ (OLInternetAddress*) addressWithCurrentHostAndPort: (uint16_t)port;

/**
 * Create and return a new address. The address of the current host is determined
 * and associated with the given service.
 * Service names are those enumerated (usually) in the /etc/services file. The
 * service entry corresponding to the requested service is retrieved and its port
 * is associated with the new address.
 *
 * @exception OLSocketException if there is a problem determining the host's address or
 * if the service cannot be found
 * @param service the service to look up
 * @return a reference to the new address
 */
+ (OLInternetAddress*) addressWithCurrentHostAndService: (const char*)service;

/**
 * Create and return a new address. The address of the named host is looked up
 * and associated with the given port.
 *
 * @exception OLSocketException if the is a problem determining the host's address
 * @param name the name of the host
 * @param port the port that should be associated with the address
 * @return a reference to the new address
 */
+ (OLInternetAddress*) addressWithHost: (const char*)name port: (uint16_t)port;

/**
 * Create and return a new address. The address of the named host is determined
 * and associated with the given service.
 * Service names are those enumerated (usually) in the /etc/services file. The
 * service entry corresponding to the requested service is retrieved and its port
 * is associated with the new address.
 *
 * @exception OLSocketException if the is a problem determining the host's address or
 * if the service cannot be found
 * @param name the name of the host
 * @param service the service to look up
 * @return a reference to the new address
 */
+ (OLInternetAddress*) addressWithHost: (const char*)name service: (const char*)service;

/**
 * Return all addresses available from the given port of the given host. The vector
 * will contain objects of type OLInternetAddress.
 *
 * @param name the name of the host to look up
 * @param port the port number of the desired host
 */
+ (OLVector*) allAddressesWithHost: (const char*)name port: (uint16_t)port;
/**
 * Return all addresses available from the given service of the given host. The vector
 * will contain objects of type OLInternetAddress.
 *
 * @param name the name of the host to look up
 * @param service the name of the service to look up
 */
+ (OLVector*) allAddressesWithHost: (const char*)name service: (const char*)service;

/**
 * Initialize the class when it is loaded into the runtime system. Please don't call
 * this method.
 */
#if defined(OL_NO_OPENSTEP)
+ (id) initialize;
#else
+ (void) initialize;
#endif

/**
 * Return the loopback address for this machine associated with the given port.
 *
 * @param port the port to use at the loopback address
 * @return the address
 */
+ (OLInternetAddress*) loopbackWithPort: (uint16_t)port;

/**
 * Return the loopback address for this machine associated with the given service.
 *
 * @param service the service to use at the loopback address
 * @return the address
 */
+ (OLInternetAddress*) loopbackWithService: (const char*)service;

/**
 * Return whether the system prefers to use IPv6 addresses.
 *
 * @return YES if IPv6 addresses are preferred, NO otherwise
 */
+ (BOOL) preferIPv6Addresses;

/**
 * Set whether the system should prefer IPv6 addresses.
 *
 * @param state should be YES if IPv6 addresses are preferred, NO otherwise
 */
+ (void) setPreferIPv6Addresses: (BOOL)state;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Finalize the address and deallocate any allocated memory.
 */
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
/* @} */

/**
 * Return the canonical name of the host. The host to which this address points is
 * looked up to determine its name. The name is returned.
 *
 * @exception OLSocketException if there is a problem looking up the name
 * @return the canonical name
 */
- (const char*) canonicalName;

/**
 * Return whether this address is a local wildcard. A local wildcard address is one that
 * can refer to any of the current host's IP addresses without specifying exactly which
 * one.
 *
 * @return YES if this is a wildcard address, NO if not
 */
- (BOOL) isLocalWildcard;

/**
 * Return the port.
 *
 * @return the port
 */
- (uint16_t) port;

/**
 * Return the type of socket with which this address may be used. This value is a
 * platform-dependent number that will correspond to a defined value such as
 * SOCK_STREAM or SOCK_DGRAM.
 *
 * @sa OLSocket#socketType
 *
 * @return the type of socket
 */
- (int) socketType;

@end

/**
 * @class OLInternet4Address SocketAddress.h ObjectiveLib/SocketAddress.h
 *
 * An internet address for protocol version 4. This type of address can be used
 * to communicate with a host via Internet protocol version 4.
 *
 * @note This class will not be terribly useful if instantiated manually. The
 * normal way to obtain an Internet address is to use one of the class
 * methods from OLInternetAddress.
 *
 * @ingroup Sockets
 */
@interface OLInternet4Address : OLInternetAddress
{
@protected
    /**
     * The host representation of the address
     */
    struct sockaddr_in* hostRep;
}

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the address. A local wildcard address is initialized with a port number
 * of zero.
 *
 * @return a reference to this address
 */
- (id) init;

/**
 * Initialize the address. A local wildcard address is created with the given port number.
 *
 * @param port the port of the address
 * @return a reference to this address
 */
- (id) initWithPort: (uint16_t)port;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
/* @} */

#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif

/**
 * Return a description of this address. A human-readable version of the address
 * is returned. If a canonical name exists, then that name will be returned. Otherwise,
 * a textual representation of the IP address and port number is returned.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @return the description of this address
 */
- (OLText*) description;
- (const struct sockaddr*) hostRepresentation;
- (unsigned) hostRepresentationLength;

/**
 * Return whether this address is equal to another one.
 *
 * @param object the object with which to compare this one
 * @return YES if @a object is equal to this address, NO otherwise
 */
- (BOOL) isEqual: (id)object;
- (BOOL) isLocalWildcard;
- (uint16_t) port;

@end

#if defined(OL_HAVE_INET6_SOCKETS)

/**
 * @class OLInternet6Address SocketAddress.h ObjectiveLib/SocketAddress.h
 *
 * An internet address for protocol version 6. This type of address can be used
 * to communicate with a host via Internet protocol version 6.
 *
 * @note This class will not be terribly useful if instantiated manually. The
 * normal way to obtain an Internet address is to use one of the class
 * methods from OLInternetAddress.
 *
 * @ingroup Sockets
 */
@interface OLInternet6Address : OLInternetAddress
{
@protected
    /**
     * The host representation of the address
     */
    struct sockaddr_in6* hostRep;
}

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the address.
 * @copydoc OLInternet4Address::init
 */
- (id) init;

/**
 * Initialize the address.
 * @copydoc OLInternet4Address::initWithPort:
 */
- (id) initWithPort: (uint16_t)port;
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
/* @} */
#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif

/**
 * Return a description of this address.
 * @copydoc OLInternet4Address::description
 */
- (OLText*) description;
- (const struct sockaddr*) hostRepresentation;
- (unsigned) hostRepresentationLength;

/**
 * Return whether this address is equal to another one.
 *
 * @param object the object with which to compare this one
 * @return YES if @a object is equal to this address, NO otherwise
 */
- (BOOL) isEqual: (id)object;
- (BOOL) isLocalWildcard;
- (uint16_t) port;

@end

#endif

#if defined(OL_HAVE_UNIX_SOCKETS)

/**
 * @class OLUnixAddress SocketAddress.h ObjectiveLib/SocketAddress.h
 *
 * An address for Unix domain sockets. Unix domain socket addresses
 * occupy a position in the file system, so they are referred to by
 * their path name.
 *
 * @sa OLUnixServerSocket, OLUnixClientSocket
 *
 * @ingroup Sockets
 */
@interface OLUnixAddress : OLSocketAddress
{
@protected
    /**
     * The host representation of the address
     */
    struct sockaddr_un* hostRep;
}

/**
 * Create and return a new socket address. The address is created with the
 * given path.
 *
 * @exception NSGenericException if @a pth cannot be converted to a valid file
 * name or if the full path name cannot be resolved
 * @param pth the path in the file system for the address
 * @return a reference to the newly created socket address
 */
+ (id) addressWithPath: (const char*)pth;

/**
 * @name Initializers and Deallocators
 */
/* @{ */
/**
 * Initialize the socket address. The address is created with the
 * given path.
 *
 * @exception NSGenericException if @a pth cannot be converted to a valid file
 * name or if the full path name cannot be resolved
 * @param pth the path in the file system for the address
 * @return a reference to this socket address
 */
- (id) initWithPath: (const char*)pth;

/**
 * Finalize the address and deallocate any allocated memory.
 */
#if defined(OL_NO_OPENSTEP)
- (id) free;
#else
- (void) dealloc;
#endif
/* @} */

#if defined(OL_NO_OPENSTEP)
- (id) copy;
#else
- (id) copyWithZone: (NSZone*)zone;
#endif

/**
 * Return a description of this address. The description is the same as the
 * path name.
 *
 * @note If OpenStep is present the returned object will be autoreleased
 * before being returned.
 *
 * @sa #path
 *
 * @return the description
 */
- (OLText*) description;
- (const struct sockaddr*) hostRepresentation;
- (unsigned) hostRepresentationLength;

/**
 * Return whether this address is equal to another one.
 *
 * @param object the object with which to compare this one
 * @return YES if @a object is equal to this address, NO otherwise
 */
- (BOOL) isEqual: (id)object;

/**
 * Return the path name occupied by this socket address.
 *
 * @return the path name
 */
- (const char*) path;

@end

#endif

#endif
