/*
 * Copyright (c) 2002, 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef _ST_HANDLER_API_H
#define _ST_HANDLER_API_H

#include <gtk/gtk.h>
#include <st-category-api.h>
#include <st-handler-field-api.h>
#include <st-plugin-api.h>
#include <st-stream-api.h>

G_BEGIN_DECLS

#define ST_TYPE_HANDLER			(st_handler_get_type())
#define ST_HANDLER(obj)			(G_TYPE_CHECK_INSTANCE_CAST((obj), ST_TYPE_HANDLER, STHandler))
#define ST_HANDLER_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST((klass), ST_TYPE_HANDLER, STHandlerClass))
#define ST_IS_HANDLER(obj)		(G_TYPE_CHECK_INSTANCE_TYPE((obj), ST_TYPE_HANDLER))
#define ST_IS_HANDLER_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE((klass), ST_TYPE_HANDLER))
#define ST_HANDLER_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS((obj), ST_TYPE_HANDLER, STHandlerClass))

typedef struct _STHandler		STHandler;
typedef struct _STHandlerClass		STHandlerClass;
typedef struct _STHandlerPrivate	STHandlerPrivate;

struct _STHandler
{
  GObject		object;

  STHandlerPrivate	*priv;
};
  
struct _STHandlerClass
{
  GObjectClass		parent_class;
};
  
typedef enum
{
  ST_HANDLER_NO_CATEGORIES		= 1 << 0,
  ST_HANDLER_CONFIRM_DELETION		= 1 << 1
} STHandlerFlags;

typedef enum
{
  ST_HANDLER_STOCK_FIELD_NAME,
  ST_HANDLER_STOCK_FIELD_GENRE,
  ST_HANDLER_STOCK_FIELD_DESCRIPTION,
  ST_HANDLER_STOCK_FIELD_HOMEPAGE,
  ST_HANDLER_STOCK_FIELD_URI_LIST
} STHandlerStockField;

typedef enum
{
  ST_HANDLER_EVENT_RELOAD,
  
  ST_HANDLER_EVENT_STREAM_NEW,
  ST_HANDLER_EVENT_STREAM_FIELD_GET,
  ST_HANDLER_EVENT_STREAM_FIELD_SET,
  ST_HANDLER_EVENT_STREAM_FREE,

  ST_HANDLER_EVENT_STREAM_TUNE_IN,
  ST_HANDLER_EVENT_STREAM_RECORD,
  ST_HANDLER_EVENT_STREAM_BROWSE,

  ST_HANDLER_EVENT_CATEGORY_NEW,
  ST_HANDLER_EVENT_CATEGORY_FREE,

  ST_HANDLER_EVENT_THREAD_BEGIN,
  ST_HANDLER_EVENT_THREAD_END,

  ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE,
  ST_HANDLER_EVENT_RELOAD_MULTIPLE,

  ST_HANDLER_EVENT_STREAM_MODIFY,
  ST_HANDLER_EVENT_STREAM_DELETE,

  ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET,
  ST_HANDLER_EVENT_STREAM_RESOLVE,

  ST_HANDLER_EVENT_PREFERENCES_WIDGET_NEW,

  ST_HANDLER_N_EVENTS
} STHandlerEvent;

/**
 * STHandlerReloadCallback:
 * @category: the category to reload.
 * @categories: a location to store a #GNode of categories.
 * @streams: a location to store a #GList of streams for @category.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to #ST_HANDLER_EVENT_RELOAD.
 *
 * The function will be called to reload the list of categories and
 * the contents of a specific category, and should store a tree of
 * categories at @categories, and a list of streams for @category at
 * @streams (you are giving the ownership of @categories and @streams
 * to streamtuner, which will free them when no longer needed).
 *
 * Note that if #ST_HANDLER_EVENT_RELOAD_MULTIPLE is bound, this
 * function will not be called, since
 * #ST_HANDLER_EVENT_RELOAD_MULTIPLE has priority over
 * #ST_HANDLER_EVENT_RELOAD.
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the reload was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STHandlerReloadCallback)  (STCategory	*category,
						GNode		**categories,
						GList		**streams,
						gpointer	data,
						GError		**err);

/**
 * STHandlerReloadMultipleCallback:
 * @categories: a location to store a #GNode of categories.
 * @streams: a location to store a #GHashTable containing all the
 * streams of the directory. The hash table must contain string/#GList
 * key/value pairs, where the string is the name of a category, and
 * the #GList is the list of streams for that category. The ownership
 * of the hash table is given to streamtuner, which will destroy it
 * using g_hash_table_destroy() when no longer needed. If you intend
 * to use allocated strings as keys, create the hash table using
 * g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL).
 * However, never supply a destructor for the values, as streamtuner
 * will take care of them.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_RELOAD_MULTIPLE.
 *
 * The function will be called to reload the list of categories and
 * streams of a directory, and should store a tree of categories at
 * @categories, and a hash table of streams at @streams (you are
 * giving the ownership of @categories and @streams to streamtuner,
 * which will free them when no longer needed).
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the reload was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STHandlerReloadMultipleCallback) (GNode      **categories,
						       GHashTable **streams,
						       gpointer   data,
						       GError     **err);

/**
 * STCategoryNewCallback:
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_CATEGORY_NEW.
 *
 * The function is used to subclass #STCategory.
 *
 * Return value: the function should return a new #STCategory.
 **/
typedef STCategory*(*STCategoryNewCallback)    (gpointer	data);

/**
 * STCategoryFreeCallback:
 * @category: a category to free.
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_CATEGORY_FREE.
 *
 * This function will be called to destroy a category.
 **/
typedef void       (*STCategoryFreeCallback)   (STCategory	*category,
						gpointer	data);

/**
 * STStreamNewCallback:
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_NEW.
 *
 * The function is used to subclass #STStream.
 *
 * Return value: the function should return a new #STStream.
 **/
typedef STStream*  (*STStreamNewCallback)      (gpointer	data);

/**
 * STStreamFieldGetCallback:
 * @stream: a stream.
 * @field: a field to get the value of.
 * @value: a value (initialized to the type of @field) to store the
 * field value in.
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_FIELD_GET.
 *
 * The function will be called when streamtuner needs to get the value
 * of a stream field.
 **/
typedef void	   (*STStreamFieldGetCallback) (STStream	*stream,
						STHandlerField	*field,
						GValue		*value,
						gpointer	data);

/**
 * STStreamFieldSetCallback:
 * @stream: a stream.
 * @field: a field to set the value of.
 * @value: a value to set the value field from. The type of the value
 * is the one passed to st_handler_field_new().
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_FIELD_SET.
 *
 * The function will be called when streamtuner needs to set the value
 * of a stream field.
 **/
typedef void	   (*STStreamFieldSetCallback) (STStream	*stream,
						STHandlerField	*field,
						const GValue	*value,
						gpointer	data);

/**
 * STStreamModifyCallback:
 * @stream: a stream.
 * @fields: a list of #STHandlerField fields to modify.
 * @values: a list of #GValue values for @fields.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_MODIFY.
 *
 * The function will be called when the user modifies a stream. The
 * first element of @values is the value of the first element of
 * @fields, and so on.
 *
 * Return value: the function should return %FALSE if there was an
 * error. In such case it should also set @err.
 **/
typedef gboolean   (*STStreamModifyCallback)   (STStream	*stream,
						GSList		*fields,
						GSList		*values,
						gpointer	data,
						GError		**err);

/**
 * STStreamDeleteCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_DELETE.
 *
 * The function will be called when the user deletes a stream. If the
 * #ST_HANDLER_CONFIRM_DELETION flag is set, the function will only
 * be called if the user confirms the stream deletion.
 *
 * Return value: the function should return %FALSE if there was an
 * error. In such case it should also set @err.
 **/
typedef gboolean   (*STStreamDeleteCallback)   (STStream	*stream,
						gpointer	data,
						GError		**err);


/**
 * STStreamFreeCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_FREE.
 *
 * This function will be called to destroy a stream.
 **/
typedef void	   (*STStreamFreeCallback)     (STStream	*stream,
						gpointer	data);


/**
 * STStreamResolveCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_RESOLVE.
 *
 * The function will be called when streamtuner needs to resolve
 * @stream (streams should be resolved before they are bookmarked).
 *
 * The function should resolve the stream in a handler-specific way
 * (for instance, it should connect to the stream directory server and
 * retrieve the stream URL).
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the task was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STStreamResolveCallback)  (STStream	*stream,
						gpointer	data,
						GError		**err);

/**
 * STStreamTuneInCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_TUNE_IN.
 *
 * The function will be called when the user tunes into @stream, and
 * should perform the action needed to tune into @stream.
 *
 * Note that if #ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE is bound,
 * this function will not be called, since
 * #ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE has priority over
 * #ST_HANDLER_EVENT_STREAM_TUNE_IN.
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the task was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STStreamTuneInCallback)   (STStream	*stream,
						gpointer	data,
						GError		**err);

/**
 * STStreamRecordCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_RECORD.
 *
 * The function will be called when the user records @stream, and
 * should perform the action needed to record @stream.
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the task was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STStreamRecordCallback)   (STStream	*stream,
						gpointer	data,
						GError		**err);

/**
 * STStreamBrowseCallback:
 * @stream: a stream.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_BROWSE.
 *
 * The function will be called when the user browses the web page of
 * @stream, and should perform the action needed to browse the web
 * page of @stream.
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the task was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STStreamBrowseCallback)   (STStream	*stream,
						gpointer	data,
						GError		**err);

/**
 * STThreadBeginCallback:
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_THREAD_BEGIN.
 *
 * The function will be called after a new thread of execution has
 * been spawned.
 *
 * Return value: the returned value will be passed to
 * STThreadEndCallback() when the thread is about to exit.
 **/
typedef gpointer   (*STThreadBeginCallback)    (gpointer	data);

/**
 * STThreadEndCallback:
 * @thread_data: data returned by STThreadBeginCallback().
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_THREAD_END.
 *
 * The function will be called when a thread of execution is about to
 * exit.
 **/
typedef void       (*STThreadEndCallback)      (gpointer	thread_data,
						gpointer	data);

/**
 * STStreamTuneInMultipleCallback:
 * @streams: a list of #STStream streams.
 * @data: data passed to st_handler_bind().
 * @err: a location to store errors.
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_TUNE_IN_MULTIPLE.
 *
 * The function will be called when the user tunes into @streams, and
 * should perform the action needed to tune into @streams.
 *
 * Return value: the function should return %FALSE if there was an
 * error, or if the task was cancelled by the user. Additionally, if
 * there was an error, @err must be set.
 **/
typedef gboolean   (*STStreamTuneInMultipleCallback)(GSList	*streams,
						     gpointer	data,
						     GError	**err);


/**
 * STStreamStockFieldGetCallback:
 * @stream: a stream.
 * @stock_field: a stock field to get the value of.
 * @value: a value (initialized to the type of @stock_field) to store
 * the stock field value in.
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_STREAM_STOCK_FIELD_GET.
 *
 * The function will be called when streamtuner needs to get the value
 * of a stream stock field.
 **/
typedef void       (*STStreamStockFieldGetCallback) (STStream	*stream,
						     STHandlerStockField stock_field,
						     GValue	*value,
						     gpointer	data);

/**
 * STHandlerPreferencesWidgetNewCallback:
 * @data: data passed to st_handler_bind().
 *
 * Specifies the type of function to bind to
 * #ST_HANDLER_EVENT_PREFERENCES_WIDGET_NEW.
 *
 * If a function is bound to the
 * #ST_HANDLER_EVENT_PREFERENCES_WIDGET_NEW event, a page containing
 * the returned widget will be available in the streamtuner
 * preferences.
 *
 * Return value: the function should return a new #GtkWidget.
 **/
typedef GtkWidget* (*STHandlerPreferencesWidgetNewCallback) (gpointer data);

GType      st_handler_get_type                  (void);

STHandler  *st_handler_new			(const char	*name);
STHandler  *st_handler_new_from_plugin		(STPlugin	*plugin);

void       st_handler_set_label			(STHandler	*handler,
						 const char	*label);
void       st_handler_set_description		(STHandler	*handler,
						 const char	*description);
void       st_handler_set_home			(STHandler	*handler,
						 const char	*home);
void       st_handler_set_icon_from_pixbuf	(STHandler	*handler,
						 GdkPixbuf	*pixbuf);
void       st_handler_set_stock_categories	(STHandler	*handler,
						 GNode		*categories);
void       st_handler_set_flags			(STHandler	*handler,
						 STHandlerFlags	flags);
void       st_handler_set_stream_version	(STHandler	*handler,
						 int		version);
void       st_handler_add_field			(STHandler	*handler,
						 STHandlerField	*field);
void       st_handler_bind			(STHandler	*handler,
						 STHandlerEvent	event,
						 gpointer	cb,
						 gpointer	data);
void       st_handler_notice			(STHandler	*handler,
						 const char	*format,
						 ...) G_GNUC_PRINTF(2, 3);
void       st_handler_config_register		(STHandler	*handler,
						 GParamSpec	*pspec);
GParamSpec *st_handler_config_lookup		(STHandler	*handler,
						 const char	*key);

/**
 * STHandlerConfigForeachCallback:
 * @pspec: the param spec of the configuration key.
 * @value: the value of the configuration key.
 * @data: data passed to st_handler_config_foreach().
 *
 * Specifies the type of function to pass to
 * st_handler_config_foreach(). The function will be called for each
 * key/value pair in the handler configuration database.
 **/
typedef void (*STHandlerConfigForeachCallback)	(GParamSpec	*pspec,
						 const GValue	*value,
						 gpointer	data);

void       st_handler_config_foreach		(STHandler	*handler,
						 STHandlerConfigForeachCallback cb,
						 gpointer	data);
void       st_handler_config_get_value		(STHandler	*handler,
						 const char	*key,
						 GValue		*value);
void       st_handler_config_set_value		(STHandler	*handler,
						 const char	*key,
						 const GValue	*value);

/**
 * st_handler_config_get_boolean:
 * @handler: a handler.
 * @key: a G_TYPE_BOOLEAN configuration key previously registered with
 * st_handler_config_register().
 *
 * Gets a boolean configuration value.
 *
 * Return value: the value of @key.
 **/
gboolean   st_handler_config_get_boolean	(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_boolean:
 * @handler: a handler.
 * @key: a G_TYPE_BOOLEAN configuration key previously registered with
 * st_handler_config_register().
 * @value: the value to set.
 *
 * Sets a boolean configuration value.
 **/
void       st_handler_config_set_boolean	(STHandler	*handler,
						 const char	*key,
						 gboolean	value);

/**
 * st_handler_config_get_int:
 * @handler: a handler.
 * @key: a G_TYPE_INT configuration key previously registered with
 * st_handler_config_register().
 *
 * Gets a signed integer configuration value.
 *
 * Return value: the value of @key.
 **/
int        st_handler_config_get_int		(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_int:
 * @handler: a handler.
 * @key: a G_TYPE_INT configuration key previously registered with
 * st_handler_config_register().
 * @value: the value to set.
 *
 * Sets a signed integer configuration value.
 **/
void       st_handler_config_set_int		(STHandler	*handler,
						 const char	*key,
						 int		value);

/**
 * st_handler_config_get_uint:
 * @handler: a handler.
 * @key: a G_TYPE_UINT configuration key previously registered with
 * st_handler_config_register().
 *
 * Gets an unsigned integer configuration value.
 *
 * Return value: the value of @key.
 **/
unsigned int st_handler_config_get_uint		(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_uint:
 * @handler: a handler.
 * @key: a G_TYPE_UINT configuration key previously registered with
 * st_handler_config_register().
 * @value: the value to set.
 *
 * Sets an unsigned integer configuration value.
 **/
void       st_handler_config_set_uint		(STHandler	*handler,
						 const char	*key,
						 unsigned int	value);

/**
 * st_handler_config_get_double:
 * @handler: a handler.
 * @key: a G_TYPE_DOUBLE configuration key previously registered with
 * st_handler_config_register().
 *
 * Gets a double configuration value.
 *
 * Return value: the value of @key.
 **/
double     st_handler_config_get_double		(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_double:
 * @handler: a handler.
 * @key: a G_TYPE_DOUBLE configuration key previously registered with
 * st_handler_config_register().
 * @value: the value to set.
 *
 * Sets a double configuration value.
 **/
void       st_handler_config_set_double		(STHandler	*handler,
						 const char	*key,
						 double		value);

/**
 * st_handler_config_get_string:
 * @handler: a handler.
 * @key: a G_TYPE_STRING configuration key previously registered with
 * st_handler_config_register().
 *
 * Gets a string configuration value.
 *
 * Return value: the value of @key. The value should be freed when no
 * longer needed.
 **/
char       *st_handler_config_get_string	(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_string:
 * @handler: a handler.
 * @key: a G_TYPE_STRING configuration key previously registered with
 * st_handler_config_register().
 * @value: the value to set.
 *
 * Sets a string configuration value.
 **/
void       st_handler_config_set_string		(STHandler	*handler,
						 const char	*key,
						 const char	*value);

/**
 * st_handler_config_get_value_array:
 * @handler: a handler.
 * @key: a G_TYPE_VALUE_ARRAY configuration key previously registered
 * with st_handler_config_register().
 *
 * Gets a value array configuration value.
 *
 * Return value: the value of @key. The value should be freed with
 * g_value_array_free() when no longer needed.
 **/
GValueArray *st_handler_config_get_value_array	(STHandler	*handler,
						 const char	*key);

/**
 * st_handler_config_set_value_array:
 * @handler: a handler.
 * @key: a G_TYPE_VALUE_ARRAY configuration key previously registered
 * with st_handler_config_register().
 * @value: the value to set.
 *
 * Sets a value array configuration value.
 **/
void       st_handler_config_set_value_array	(STHandler	*handler,
						 const char	*key,
						 const GValueArray *value);

#ifndef ST_DISABLE_DEPRECATED

void       st_handler_set_copyright		(STHandler	*handler,
						 const char	*copyright);
void       st_handler_set_info			(STHandler	*handler,
						 const char	*label,
						 const char	*copyright);
void       st_handler_set_icon			(STHandler	*handler,
						 int		size,
						 const guint8	*data);
void	   st_handler_set_icon_from_inline	(STHandler	*handler,
						 int		size,
						 const guint8	*data);
gboolean   st_handler_set_icon_from_file	(STHandler	*handler,
						 const char	*filename,
						 GError		**err);

#define ST_HANDLER_EVENT_REFRESH		ST_HANDLER_EVENT_RELOAD
#define ST_HANDLER_EVENT_REFRESH_MULTIPLE	ST_HANDLER_EVENT_RELOAD_MULTIPLE
#define N_ST_HANDLER_EVENTS			ST_HANDLER_N_EVENTS

typedef STHandlerReloadCallback			STHandlerRefreshCallback;
typedef STHandlerReloadMultipleCallback		STHandlerRefreshMultipleCallback;

#endif /* ! ST_DISABLE_DEPRECATED */

G_END_DECLS

#endif /* _ST_HANDLER_API_H */
