;; emacs support for rsltc tool
;; Chris George, UNU/IIST, March 1999
;; Based heavily on compile.el

(require 'compile)

;; include directory where rsltc is located if not on path
(defvar rsltc-command "rsltc"
  "RSL type checker")

;; set this to vcg command (probably no "x" in Windows, and may need path)
;; like "c:\\raise\\vcg"
(defvar vcg-command "xvcg"
  "Command to run vcg")

(defvar RSL-entity-regexp "\\<\\(scheme\\|object\\|theory\\|devt_relation\\|hint\\)\\>"
  "Regular expression for keyword starting an RSL entity.")

(defvar RCS-regexp "[ \t]*--[ \t]*\\$Id:"
  "Regular expression for start of RCS data.")

(defvar rsl-line-comment-regexp "[ \t]*--"
  "Start of line comment")

(defvar rsl-block-comment-start-regexp "/\\*"
  "Start of block comment")

(defvar rsl-block-comment-end-regexp "\\*/"
  "End of block comment")

(defvar rsl-context-regexp "[-a-zA-Z0-9_./]+"
  "Context entry.")

(defvar RSL-basename nil
  "Base name of RSL file")

(defvar RSL-process nil
  "Process created by rsl command, or nil if none exists now.
Note that the process may have been \"deleted\" and still
be the value of this variable.")

(defvar RSL-buffer nil
  "Buffer containing RSL.")

(defvar rsltc-error-regexp
  "\\.rsl:\\([0-9]+\\):\\([0-9]+\\):"
  "Regular expression for line and column in rsltc error output.")

(defun rsltc (params)
  "Run RSL type checker"
  (interactive "sCommand line arguments: ")
  (if (buffer-modified-p) (save-buffer))
  (setq RSL-basename
	(file-name-sans-extension (file-name-nondirectory (buffer-file-name))))
  (compile (concat rsltc-command " " params " " RSL-basename)))

(defun rslpp (len-str)
  "Run RSL pretty printer on file in current buffer"
  (interactive "sLine length (default 60): ")
  (let ((len
	 (if (string-equal len-str "") 60 (string-to-int len-str))))
    (delete-other-windows) ; prevent old error messages still being visible
    (rsl-do
     (format "%s -p -pl %d " rsltc-command len)
     'end-pp-ok)))

(defun end-pp-ok (buff)
  (save-excursion
    (set-buffer buff)
    (copy-to-buffer RSL-buffer (point-min) (point-max))
    (message "Pretty printed: C-x u to undo")))

(defun rslgg ()
  "Make dependency graph of file in current buffer and display"
  (interactive)
  (rsl-do
   (concat rsltc-command " -g ")
   'end-gg-ok))

(defun end-gg-ok (buff)
  (call-process
   shell-file-name nil 0 nil "-c"
   (concat vcg-command " " RSL-basename ".vcg")))

(defun rsl-do (command post-fun)
  "Run COMMAND (which must end with a space) on RSL-basename
and if successful then run post-fun with shell output buffer name as argument."
  (if (buffer-modified-p) (save-buffer))
  (setq RSL-basename
	(file-name-sans-extension (file-name-nondirectory (buffer-file-name))))
  (setq RSL-buffer (current-buffer))
  (if (get-buffer " *rslbuff*") (kill-buffer " *rslbuff*"))
  (let ((status (call-process
		 shell-file-name nil
		 (get-buffer-create " *rslbuff*") nil "-c"
		 (concat command RSL-basename))))
    (if (and (numberp status)
	     (or (zerop status)
		 ;; strange status returned with Cygwin bash
		 (and (string= shell-file-name "bash")
		      (= status 131072))))
	(apply post-fun (list " *rslbuff*"))
      (end-RSL-error " *rslbuff*"))))

(defun end-RSL-error (buff)
  (set-buffer buff)
  (goto-char (point-min))
  (if (re-search-forward rsltc-error-regexp nil t)
      ;; Extract line and column number from error message.
      (let
	  ((linenum
	    (string-to-int
	     (buffer-substring (match-beginning 1) (match-end 1))))
	   (colnum
	    (string-to-int
	     (buffer-substring (match-beginning 2) (match-end 2)))))
	;; Go to RSL buffer and find the erring line.
	(set-buffer RSL-buffer)
	(goto-char (point-min))
	(forward-line (1- linenum))
	(forward-char (1- colnum))
	(message
	 (format "Error (probably parse) here: line %d, column %d" linenum colnum)))
    ;; something else
    ;; just show rsltc output
    (switch-to-buffer-other-window buff)))

;; support error reports in RSL comments 
(setq compilation-error-regexp-alist
      (cons
       '(
	 "[ \t]*(*-- \\([a-zA-Z0-9_/.]+\\):\\([0-9]+\\):\\([0-9]+\\):" 1 2 3)
	    compilation-error-regexp-alist))

;; additions for emacs-based mkdoc ------------------

(defun mkdoc ()
  "Creates .tex files from \RAISEIN, \RAISEINFRAME, and \RAISEINBOX commands"
  (interactive)
  (let ((save-case-fold-search case-fold-search)
	(save-point (point)))
    (goto-char (point-min))
    (setq case-fold-search nil)
    (while (re-search-forward
	    "^\\\\\\(RAISEIN\\|RAISEINFRAME{[^}]*}\\|RAISEINBOX\\){\\([^}]*\\)}" nil t)
      (let ((rslfile (concat (buffer-substring (match-beginning 2) (match-end 2)) ".rsl"))
	    (texfile (concat (buffer-substring (match-beginning 2) (match-end 2)) ".tex")))
	(if (file-readable-p rslfile)
	    (if (not (up-to-date texfile rslfile))
		(save-excursion
		  ;; uses a special buffer with a leading space in name
		  ;; to avoid font-lock starting and slowing things down
		  (if (get-buffer " *rslbuff*") (kill-buffer " *rslbuff*"))
		  (setq rslfile (expand-file-name rslfile))
		  (setq texfile (expand-file-name texfile))
		  (let ((texbuffer (get-buffer-create " *rslbuff*")))
		    (set-buffer texbuffer)
		    (insert-file rslfile)
		    (goto-char (point-min))
		    (insert "\\RSLatex\n")
		    (setq case-fold-search nil)
		    (mark-context)
		    (goto-char (point-max))
		    (backward-char 1)
		    (if (looking-at "\n")
			(forward-char 1)
		      (progn
			(forward-char 1)
			(insert "\n")))
		    (insert "\\endRSLatex\n")
		    (do-latex)
		    (write-region (point-min) (point-max) texfile)
		    (kill-buffer texbuffer))))
	  (progn
	    (setq case-fold-search save-case-fold-search)
	    (error "RSL file %s cannot be found" rslfile)))))
    (setq case-fold-search save-case-fold-search)
    (goto-char save-point))
  (message "mkdoc completed"))

(defun up-to-date (texfile rslfile)
  "Returns t if TEXFILE exists and modified no earlier than RSLFILE."
  (and (file-readable-p texfile)
      (let ((tex-mod-time (nth 5 (file-attributes texfile)))
	    (rsl-mod-time (nth 5 (file-attributes rslfile))))
	(or (> (car tex-mod-time) (car rsl-mod-time))
	    (and (= (car tex-mod-time) (car rsl-mod-time))
		 (>= (car (cdr tex-mod-time)) (car (cdr rsl-mod-time))))))))

(defun mark-context ()
  (re-search-forward
   (concat rsl-block-comment-start-regexp "\\|"
	   rsl-line-comment-regexp "\\|" rsl-context-regexp) nil t)
  (if (string-match rsl-block-comment-start-regexp
		    (buffer-substring (match-beginning 0) (match-end 0)))
      (progn
	(skip-block-comment 1)
	(mark-context))
    (beginning-of-line)
    (if (looking-at rsl-line-comment-regexp)
	(progn
	  (if (looking-at RCS-regexp)
	      ;; do not include RCS data
	      (let ((start (point)))
		(search-forward "$" nil nil 2)
		(forward-line 1)
		(delete-region start (point)))
	    (forward-line 1))
	  (mark-context))
      (if (not (looking-at (concat "[ \t]*" RSL-entity-regexp)))
	  (let ((end-found nil))
	    (insert "\\CONTTWO{")
	    (re-search-forward "[^ \t]")
	    (while (not end-found)
	      (re-search-forward "[ \t\n]+" nil t)
	      (let ((p1 (match-beginning 0))
		    (p2 (match-end 0)))
		(if (looking-at ",") ()
		  (goto-char (1- p1))
		  (setq end-found (not (looking-at ",")))
		  (goto-char p2))))
	    (backward-char 1)
	    (insert "}"))))))
	    

(defun skip-block-comment (count)
  (while (> count 0)
    (if (re-search-forward
	 (concat rsl-block-comment-start-regexp "\\|"
		 rsl-block-comment-end-regexp) nil t)
	(progn
	  (goto-char (match-beginning 0))
	  (if (looking-at rsl-block-comment-start-regexp)
	      (setq count (1+ count))
	    (setq count (1- count)))
	  (forward-char 2))
      (error "Perhaps opening comment not closed")))
  (if (looking-at rsl-line-comment-regexp)
      (forward-line)))

;; menu functions needed for rsl-mode
(defun rsltc-only () ""
  (interactive)
  (rsltc ""))

(defun rslpp-dflt () ""
  (interactive)
  (rslpp ""))

(defun rsltc-c () ""
  (interactive)
  (rsltc "-c"))
    
(defun rsltc-cc () ""
  (interactive)
  (rsltc "-cc"))
    
(defun rsltc-d () ""
  (interactive)
  (rsltc "-d"))
    
(defun rsltc-m () ""
  (interactive)
  (rsltc "-m"))
    
(defun rsltc-cpp () ""
  (interactive)
  (rsltc "-c++"))
    
(defun rsltc-cppv () ""
  (interactive)
  (rsltc "-cpp"))
    
(defun rsltc-sql () ""
  (interactive)
  (rsltc "-c++ -sql"))

(defun rsltc-sqlv () ""
  (interactive)
  (rsltc "-cpp -sql"))

(defun rsltc-pvs () ""
  (interactive)
  (rsltc "-pvs"))

(defun rsltc-pc () ""
  (interactive)
  (rsltc "-pc"))


    
(provide 'rsltc)


