import logging

from multiqc.base_module import BaseMultiqcModule, ModuleNoSamplesFound
from multiqc.modules.spaceranger.count import parse_count_html

log = logging.getLogger(__name__)


class MultiqcModule(BaseMultiqcModule):
    """
    The module parses the quality reports generated by 10x Genomics Space Ranger
    (tested on Cell Ranger 2.1). Space Ranger has 2 main modules: `count` and `vdj`. The module summarises
    the main information useful for QC, including:

    - sequencing metrics
    - mapping metrics
    - estimated number of spots, and reads per spot.
    - detected genes per spot

    Note that clustering results, differentially expressed genes and the images themselves are not reported.

    The input files are web summaries generated by Space Ranger. Expected file names are `*web_summary.html`.
    Sample IDs are parsed directly from the reports.

    If present in the original report, any warning is reported as well.
    """

    def __init__(self):
        super(MultiqcModule, self).__init__(
            name="Space Ranger",
            anchor="spaceranger",
            href="https://support.10xgenomics.com/spatial-gene-expression/software/pipelines/latest/what-is-space-ranger",
            info="Tool to analyze 10x Genomics spatial transcriptomics data.",
            doi=[],
        )

        n = dict()
        # Call submodule functions
        n["count"] = parse_count_html(module=self)
        if n["count"] > 0:
            log.info("Found {} Space Ranger count reports".format(n["count"]))

        # Exit if we didn't find anything
        if sum(n.values()) == 0:
            raise ModuleNoSamplesFound
