package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$"`0`````````````"`!0````!`8`2\````#0``0.X```````T`"``
M"``H`"``'P````8````T`8``-`&``#0```$````!``````0````$`````P``
M`30!@`$T`8`!-````!<````7````!`````$````!``````&````!@`````"6
MV```EM@````%``$```````$``/Z$`8'^A`&!_H0```*4```$$`````8``0``
M`````@``_Q`!@?\0`8'_$````.````#@````!@````0````$```!3`&``4P!
M@`%,````+````"P````$````!&1TY5```(WX`8"-^`&`C?@```$,```!#```
M``0````$9'3E4@``_H0!@?Z$`8'^A````7P```%\````!`````$O=7-R+VQI
M8F5X96,O;&0N96QF7W-O```````'````!`````%.971"4T0``#6DZ0`````$
M````!`````-085@``````````&$```!D``````````````!#````&P``````
M````````"0```$\````#`````````$0`````````/@```!````!*````````
M`#8````&````%P```$D```!2````#````$T`````````'``````````6````
M7````$$``````````````#$```!B````8P````````!'````3@```$8```!;
M````'@`````````+````0````$4````U````(0```%D``````````````#<`
M````````+0````````!:````)````"@```!@````!P```%X`````````5P``
M``````!6````-````"X```````````````````!1````*P`````````F````
M7P```#D`````````/0`````````Z````80````````!3````&@```!4`````
M````5`````````````````````````!"`````````%@````P````.```````
M``!=``````````````````````````````````````````$`````````````
M``````````````````````````````````````````````````4`````````
M``````T````````````````````````````````````````````````````1
M```````````````````````````````3`````````",`````````%`````0`
M````````````````````````*0`````````/`````````!@````J````)0``
M```````L````````````````````"@```!D`````````````````````````
M)P`````````R`````````#,````B```````````````_````'0```!(`````
M````````````````````"```````````````2````%````!,````````````
M````````(``````````"````.P```$L````\````50`````````.````'P``
M`"\````````````````````````#/@&`120```)L$@``"@```A@`````````
M`!(```````&Z```````````2````````J@&"`2@````!$0``&P```:P`````
M`````!(```````!?```````````2````````#```````````$@```````(\`
M`````````!(```````%M```````````2```````#0P&`)J0````X$@``"@``
M`?```````````!(```````"``8!,X``````2```+```!WP``````````$@``
M`````7D``````````!(```````(H```````````2```````!]P``````````
M$@```````DX``````````!(```````+,`8(!$`````01```9```"JP``````
M````$@```````0```````````!(```````'"```````````2````````90``
M````````$@```````UX!@!X0````$!(```H```"&```````````2```````!
M$P``````````$@```````!0``````````!(````````P```````````2````
M```"UP&"`1@````$$0``&@```#@``````````!(```````'4```````````2
M```````#;P&`)MP```$8$@``"@```.,``````````!(```````.!`8`G]```
M!M`2```*```$`0&`$L@`````$@``"0```KH``````````!(```````(N````
M```````2```````"Y```````````$@```````"(``````````!(```````'^
M```````````2```````#C@&`'=@````X$@``"@```,4``````````!8`````
M``#8```````````2```````!+```````````$@```````Y8!@!+P````'!(`
M``H````I```````````2```````";0``````````(@```````+<`````````
M`!(```````.>`8`CF````(P2```*```#K`&`+L0```)H$@``"@```\(!@$"0
M```!E!(```H```%/`8'^F````!,1```2```"\0``````````$@```````\T!
M@@$4````!!$``!D```"<```````````2```````#EP&`$O`````<$@``"@``
M`DD!@@$P```!"!$``!L```#L`8(!(`````01```:```#UP&`(/0```!`$@``
M"@```1H!@@(X````(!$``!L```%>```````````2```````"^P``````````
M$@```````)```````````!(```````'_```````````2```````"A0``````
M````(@```````A\``````````!(```````/D`8(!&``````0```:```#\`&"
M`I0`````$```&P```0<``````````!(```````%*```````````2````````
M4```````````$@```````_8!@"0D````G!(```H````<```````````2````
M````0`&`2G``````$@```````_X!@$`@````<!(```H```('`8!*@``````2
M```````$!P&`(30```!X$@``"@```<X``````````!(```````03`8`AK```
M`>P2```*```#"```````````$@```````3P``````````!(```````![````
M```````2```````$(`&"`1@`````$```&0``!"<!@!X@```"U!(```H```&9
M`8!*T``````2```````$-`&`0B0```&@$@``"@```&P``````````!(`````
M``,/```````````2```````!C@``````````$@```````QH``````````!(`
M``````,A```````````2```````",P``````````$@```````_$!@@*4````
M`!```!L```"5```````````2````````=@``````````$@```````D(`````
M`````!(```````,M```````````2```````$/@&`,2P```"$$@``"@```CL`
M`````````!(```````#T```````````2`````&QI8G!E<FPN<V\`<F5A;&QO
M8P!M96UM;W9E`')M9&ER`'-T<G-T<@!S=')C:'(`<W1R;F-M<`!S=')R8VAR
M`&)O;W1?1'EN84QO861E<@!P97)L7V-O;G-T<G5C=`!A8F]R=`!U;FQI;FL`
M7U]L<W1A=#4P`'!E<FQ?9G)E90!?9FEN:0!?7W-T870U,`!?97AI=`!M96UC
M<'D`<&5R;%]D97-T<G5C=`!03%]D;U]U;F1U;7``4&5R;%]S=E]S971P=@!0
M3%]C=7)R96YT7V-O;G1E>'0`<&5R;%]A;&QO8P!C;&]S961I<@!E;G9I<F]N
M`%]?;W!E;F1I<C,P`&UE;7-E=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C
M:U]C:&M?9W5A<F0`4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A
M9`!03%]M96UO<GE?=W)A<`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!0
M97)L7V-R;V%K7VYO8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N
M;&]C:P!097)L7W-Y<U]T97)M`%]?97)R;F\`4&5R;%]M9U]S970`;6MD:7(`
M<&5R;%]P87)S90!097)L7VYE=UA37V9L86=S`&%C8V5S<P!G971P:60`=F9P
M<FEN=&8`4&5R;%]A=&9O<FM?;&]C:P!M96UC;7``<&5R;%]R=6X`;'-E96L`
M;W!E;@!S<')I;G1F`&=E='5I9`!S=')L96X`7U]S1@!S=')C;7``;&EB;2YS
M;RXP`&QI8F-R>7!T+G-O+C$`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`7U]R
M96=I<W1E<E]F<F%M95]I;F9O`&QI8G!T:')E860N<V\N,0!P=&AR96%D7V%T
M9F]R:P!S=')D=7``;&EB8RYS;RXQ,@!?7W!R;V=N86UE`%]?<'-?<W1R:6YG
M<P!?7V=E='!W=6ED-3``7U]S>7-C86QL`%]?<W1P8W!Y7V-H:P!A=&5X:70`
M7VQI8F-?:6YI=`!S=')T;VL`7U]R96%D9&ER,S``7U]S=&%C:U]C:&M?9F%I
M;`!M86EN`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C=7)R96YT
M7V5X96,`<&%R7W-E='5P7VQI8G!A=&@`<&%R7VUK=&UP9&ER`'AS7VEN:70`
M7U]S=&%R=`!P87)?96YV7V-L96%N`%A37TEN=&5R;F%L<U]005)?0D]/5`!S
M:&%?=7!D871E`&UY7W!A<E]P;`!P87)?8F%S96YA;64`7U]B<W-?<W1A<G0`
M7U]E;F0`<&%R7V1I90!S:&%?:6YI=`!P87)?9&ER;F%M90!P87)?:6YI=%]E
M;G8`7V5D871A`'!A<E]F:6YD<')O9P!S:&%?9FEN86P`<&%R7V-L96%N=7``
M+W5S<B]P:V<O;&EB+W!E<FPU+S4N,S@N,"]P;W=E<G!C+6YE=&)S9"UT:')E
M860M;75L=&DO0T]213HO=7-R+W!K9R]L:6(````!@?[X``!``0`````!@?\,
M```N`0`````!@?_P```I20`````!@?Z8```S$P`````!@@$H```$$P`````!
M@@$P```X$P`````!@@(X```[$P`````!@@`````"%0`````!@@`$```#%0``
M```!@@`(```%%0`````!@@`,```&%0`````!@@`0```'%0`````!@@`4```(
M%0`````!@@`8```)%0`````!@@`<```+%0`````!@@`@```-%0`````!@@`D
M```.%0`````!@@`H```/%0`````!@@`L```0%0`````!@@`P```1%0`````!
M@@`T```3%0`````!@@`X```4%0`````!@@`\```5%0`````!@@!````6%0``
M```!@@!$```8%0`````!@@!(```9%0`````!@@!,```:%0`````!@@!0```;
M%0`````!@@!4```=%0`````!@@!8```>%0`````!@@!<```@%0`````!@@!@
M```C%0`````!@@!D```D%0`````!@@!H```E%0`````!@@!L```F%0`````!
M@@!P```G%0`````!@@!T```J%0`````!@@!X```K%0`````!@@!\```M%0``
M```!@@"````N%0`````!@@"$```O%0`````!@@"(```T%0`````!@@",```V
M%0`````!@@"0```\%0`````!@@"4```]%0`````!@@"8```^%0`````!@@"<
M```_%0`````!@@"@``!`%0`````!@@"D``!!%0`````!@@"H``!$%0`````!
M@@"L``!%%0`````!@@"P``!&%0`````!@@"T``!(%0`````!@@"X``!)%0``
M```!@@"\``!+%0`````!@@#```!-%0`````!@@#$``!/%0`````!@@#(``!0
M%0`````!@@#,``!1%0`````!@@#0``!4%0`````!@@#4``!6%0`````!@@#8
M``!7%0`````!@@#<``!8%0`````!@@#@``!9%0`````!@@#D``!:%0`````!
M@@#H``!;%0`````!@@#L``!=%0`````!@@#P``!>%0`````!@@#T``!?%0``
M```!@@#X``!@%0`````!@@#\``!B%0`````!@@$```!C%0````!\"`*FD`$`
M!)0A__!(`#&Y."$`$(`!``1\"`.F3H``(```````````0I\`!7UH`J8]JP`"
M.:UN''SC.WA]!$-X2``!#90A_^!\"`*F0I\`!9/!`!A_R`*F/]X``I.A`!0[
MWFN8D^$`'(.^@`2#_H``D`$`)'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?
MZ$!!G/_L@`$`)(.A`!2#P0`8?`@#IH/A`!PX(0`@3H``()0A_^!\"`*F0I\`
M!9/!`!A_R`*F/]X``I.!`!`[WFLLD^$`'(.>@`R#_H`(D`$`)'^?X$!`G`!(
MB3\`!Y.A`!0OB0#X0;X`%$@``%")/P`'+XD`^$">`$2!/P`(._\`#(.___1]
M*0.F3H`$(7^?X$"0?0``09S_V(.A`!2``0`D@X$`$(/!`!A\"`.F@^$`'#@A
M`"!.@``@2``SL90A_^!\"`*F0I\`!9.!`!!\G"-YD\$`&'_(`J8_W@`"D`$`
M)#O>:I"380`,DZ$`%)/A`!Q!@@$`@3P``(#^@!R!20``@WZ`("^*``"!'``(
MDX<``)$;``!!G@#,@1Z`))%(``"!20``B2H``"^)``!!G@`@*XD`+SE*``%`
MGO_LD4@``(DJ```OB0``0)[_Z"^#``!!G@`(2``U]4@`-D&!/H`L+XD``$&>
M`+"#_H`0@[Z`,'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?Z$!!G/_L@'Z`
M-$@`-;6#_H`4@[Z`.'^?Z$!`G``<@3\``#O_``1]*0.F3H`$(7^?Z$!!G/_L
M@'Z`/$@`-85+__VI@+L``("<``"`?``$2``O]4@`-+V!/H`D@5Z`*)%)``!+
M__]@@-Z`&#C@`!,XH``".&```#B```1,QC&"2``T43A@``%(`#)Y2__^#4O_
M_U`L`P``E"'_T)/A`"Q!@@$(/2`!@I-A`!R#:0$@+YL``$&>`/R3P0`HB2,`
M`"^)``!!G@#<*XD`/3O```!`G@",@_L``"^?``!!G@"X?`@"II-!`!B0`0`T
M?)HC>).!`"!\?!MXDZ$`)']]VWA_Q?-X?X3C>'_C^WA(`#+E+X,``$">`!!]
M/_"N+XD`/4&>`%R'_0`$+Y\``$">_]2``0`T@T$`&(-A`!Q\"`.F@X$`((.A
M`"2#P0`H?^/[>(/A`"PX(0`P3H``('Q^&WB-/@`!+HD``"N)`#U!E@`(0+[_
M\'_#\%!+__]<?[OH4#O>``%_O19P?__R%).Z``!+__^@@V$`'(/!`"A+__^P
M.\```$O__S`[X```2___H#O@``"#80`<2___E)0A_]!\"`*FDZ$`)#^@`8([
MO0(XD`$`-)/!`"B!/0``D2$`'#D@``"3X0`L/\`!@GQ_&W@X@0`8?^/[>$S&
M,8)+__Z9+X,``$&>`$R!00`8@1X!(%5*$#I]*%(4@.D`!"^'``!\Z%$N0;[_
MS(%)``@OB@``E4D`!$">__0X@0`8?^/[>$S&,8)+__Y1+X,``$">_[R!(0`<
M@5T``'TI4GDY0```0((`((`!`#2#H0`D@\$`*'P(`Z:#X0`L."$`,$Z``"!(
M`#01E"'_,).A`,1\?1MYD^$`S#_@`8([_P(X@3\``)$A`*PY(```08(`$(D]
M```OB0``0)X`*($A`*R!7P``?2E2>3E```!`@@%0@Z$`Q(/A`,PX(0#03H``
M('P(`J:3@0#`D`$`U$@`,\E\?!MY08(`X),A`+0_(`&`DT$`N#]``8"380"\
M/V`!@)/!`,@[>TU8.UI-7#LY36!_@^-X2``S(2P#``!!@@"X.\,`#7]DVWA_
MP_-X2``P.2^#``!!OO_<?T33>'_#\WA(`#`E+X,``$&>_\A_H^MXDP$`L$@`
M,R%\>!MX?\/S>$@`,Q5\>!H4.&,``D@`,%E_QO-X?'@;>'^EZWA_),MX3,8Q
M@D@`,L$X@0`0?P/#>$@`,"4O@___?P/#>$&>`!2!(0`852D$)B^)0`!!G@!8
M2``O]7\#PWA(`#(=@P$`L$O__U"``0#4@X$`P'P(`Z9+__[<?X/C>$@`,#U_
MH^MX2``QE8`!`-2#(0"T@T$`N'P(`Z:#80"\@X$`P(/!`,A+__ZL2__^=4O_
M_ZQ\"`*FDP$`L)`!`-23(0"TDT$`N)-A`+R3@0#`D\$`R$@`,EF4(?_@?`@"
MICS``8`\H`&`D^$`'#_@`8([___TD`$`)(/___P\@`&`?_\2%#C&312`?P``
M.*5*<#B$36A(`#&I@`$`)#S@`8"`?P``..=04#R@`8`\@`&`@^$`''SF.W@X
MI2[$.(1-A#D```!\"`.F."$`($@`+H"4(?^P?`@"II`!`%2380`\DX$`0'Q\
M&WB3H0!$D\$`2#_``8*3X0!,.]X".),A`#2!7@``D4$`'#E```"300`XB20`
M`&DI`#TA*0``?^0!E'_C^WA(`#%Y.($`&'Q[&WA_@^-X3,8Q@DO_^WE\?1MY
M08(!)$@`,5E_@]A`0)P`_(,A`!@_0`&"5SD0.HD\```OB0``09X!P"N)`#U!
MG@&X?X/C>$@```PKB0`]09X`$(TC``$OB0``0)[_\'Q\&%`[>P`"@[H!('Q[
M&A1(`"Y1+X,``'Q]R2Y!G@&8B3P``(%!`!B!&@$@+XD``%5*$#I]2%`NF2H`
M`$"^`!A(```<C3P``2^)``"=*@`!09X`#"N)`#U`GO_L.2``/3O___^9*@``
MC3\``2^)``"=*@`!0)[_]#A@``"!(0`<@5X``'TI4GDY0```0((!,(`!`%2#
M(0`T@T$`.'P(`Z:#80`\@X$`0(.A`$2#P0!(@^$`3#@A`%!.@``@.____SN]
M__^-/P`!+XD``)T]``%`GO_T2___I#]``8*3`0`PDN$`+(,:`2"!.```+XD`
M`$&>`+!_",-X.4```$@```A_JNMXA2@`!#NJ``$OB0``0)[_\#E*``-51!`Z
M/4`!@H$J`H0OB0``0)X`4#D@``%\@R-XD2H"A$@`+2E\=QMY08(`;%>Y$#I_
M!,-X?R7+>$@`+2&2^@$@.3T``3E```!5*1`Z?5=)+I.A`!B"X0`L@P$`,$O_
M_EQ_`\-X2``L!2^#``!\=QMXD'H!($&>`"!7N1`Z2___Q#A@``!+__YH.(``
M"#N@``!+__]T@N$`+(,!`#`X8/__2__^Q)+A`"R3`0`P2``O790A_T!\"`*F
MD\$`N#B!`!"3X0"\/^`!@I`!`,0[_P(X?'X;>($_``"1(0"L.2```$@`+IDL
M`P``0((`9($A`!A5*00F*XE``$&>`#QM*O__+XJ@`$&>`#"!(0"L@5\``'TI
M4GDY0```0((`/(`!`,2#P0"X@^$`O'P(`Z8X(0#`3H``('_#\W@X@``"2``K
M77QC`#148]E^2___P#A@``!+__^X2``NM90A^]!\"`*F?&0;>#B@!`"3P00H
M/\`!@I/A!"P[X0`<.]X".'_C^WB!/@``D2$$'#D@``"0`00T2``LZ7T_&%`Y
M*?__?5]*%'^*^$!`G0`X?3](KB^)`"]`G@`L.2$`'3E```!]*1A0?2D#ID@`
M`!")(___+XD`+T">``R=0___0@#_\#B``"]_X_MX2``KE2P#``!!@@!,?X/X
M0$&=`#A_X_MX2``KK8$A!!R!7@``?2E2>3E```!`@@!<@`$$-(/!!"B#X00L
M?`@#IC@A!#!.@``@.4```)E#``!+___$@2$$'(%>``!]*5)Y.4```$""`"2`
M`00T/&`!@(/!!"@X8TU8@^$$+'P(`Z8X(00P2``K0$@`+9T](`&"E"'_\#DI
M__0\P`&`@2G__#R@`8!]*1(4/(`!@(!I```XQDT4.*5*<#B$36@X(0`02``L
M])0A__`X8```."$`$$Z``""4(?L0?8``)CT@`8!\"`*FDT$$V#]``8*3@03@
M.UH".)/A!.Q\?!MX?)\C>#AI39PX@0`<D`$$]),A!-2!.@``D2$$O#D@``"1
M@03`3,8Q@DO_]P5\>1MX.(``+W^#XWA(`"L!+X,``$&>`%@\8`&`?X3C>#AC
M3:A,QC&"2__Z^7^#XWB!(02\@5H``'TI4GDY0```0(("((`!!/2!@03`@R$$
MU'P(`Z:#0038?8"!((.!!."#X03L."$$\$Z``""3803</V`!@'_C^W@[>TVX
M2``J(7]DVWA(`"P9?'\;>4&"`<R2H03$+AD``)+!!,@ZH```DN$$S#[``8"3
M`030/N`!@#\``8"3P03HDZ$$Y#K635@Z]U"$.QA-O&*U@``[H```B3\``"^)
M``!!G@$,09(`&'_D^WA_(\MX2``H]2^#``!!G@#`?^/[>$@`*_4Y0___?'X;
M>'T_4A1_GTA`0)P`1'T_4*XOB0`O0)X`.'Q_&A0Y7P`!?2H84'TI`Z9(```0
MB4/__R^*`"]`G@`,GZ/__T(`__!_X_MX2``KI7Q^&WA_@^-X2``KF7QC\A0[
MPP`!*YX#_T&=`&!_Y_MX?XGC>'[HNWA_!L-X.*`$`#B````X80"\3,8Q@D@`
M*K4X@0`@.&$`O$@`*)DO@P``0)X`%($A`"A5*00F?XFH`$&>`%Q_9-MX.&``
M`$@`*N5\?QMY0(+_$(*A!,2"P03(@N$$S(,!!-"#803<@Z$$Y(/!!.A+__XX
M09(`&'[$LWA_(\MX2``G[2^#``!!GO^X.\```7[?LWA+__],.(```3AA`+Q(
M`"=]+X,``$">_Y@\8`&`.($`O#AC3:A,QC&"2__X]3AA`+Q(`"AI@J$$Q(+!
M!,B"X03,@P$$T(-A!-R#H03D@\$$Z$O__=R#803<2__]O)*A!,22P03(DN$$
MS),!!-"3803<DZ$$Y)/!!.A(`"J!E"'_X'P(`J8X@``OD^$`''Q_&WB0`0`D
M2``GU2P#``!!@@`(.^,``8`!`"1_X_MX@^$`'#@A`"!\"`.F2``GX)0A_^"3
MX0`<?'\;>4&"`$")/P``+XD``$&>`#1\"`*FD`$`)$@`*@DX8P`!*X,$`$&=
M`#"``0`D?^/[>(/A`!PX(0`@?`@#IDO_^T0\8`&`@^$`'#AC35@X(0`@2``G
M@(`!`"0X8```@^$`'#@A`"!\"`.F3H``()0A_\!\"`*F/&`!@)`!`$0X8TW$
MDV$`+#]@`8"3@0`P.WM.").A`#0_H`&`D\$`.#_``8"3X0`\/^`!@CO_`C@[
MWDWL@3\``)$A`!PY(```3,8Q@DO_])T\8`&`.&--T$S&,8)+__2-/&`!@#AC
M3=@[O4V<3,8Q@DO_]'D\8`&`.&--Y#^``8!,QC&"2__T97_#\WA,QC&"2__T
M63Q@`8`X8TW\.YQ.%$S&,8)+__1%?Z/K>$S&,8)+__0Y?V/;>$S&,8)+__0M
M?X/C>$S&,8)+__0A/&`!@#AC3B!,QC&"2__T$3Q@`8`X8TVH3,8Q@DO_]`$\
M8`&`.($`&#AC3BQ,QC&"2__RQ7QD&WE!@@`0?X/C>$S&,8)+__;1/&`!@#B!
M`!@X8TY`3,8Q@DO_\IU\9!MY08(`%#Q@`8`X8TY43,8Q@DO_]J4\8`&`.($`
M&#AC3F!,QC&"2__R<7QD&WE!@@!<?Z/K>$S&,8)+__9]/(`!@'_#\W@XA$_H
M3,8Q@DO_]FF!(0`<@5\``'TI4GDY0```0((`5(`!`$2#80`L@X$`,'P(`Z:#
MH0`T@\$`.(/A`#PX(0!`3H``(#Q@`8`X@0`8.&-.<$S&,8)+__']?&0;>4&B
M_YQ_8]MX3,8Q@DO_]@E+__^,2``GW90A_]!\"`*F/&`!@)/A`"P_X`&"D`$`
M-#O_`C@X8TX(.($`&($_``"1(0`<.2```$S&,8)+__&I+`,``#D@``!!@@`<
MB&,``"^#``!!G@`0:&,`,#$C__]]*1D0@4$`'($?``!]2D)Y.0```'TC2WA`
M@@`8@`$`-(/A`"PX(0`P?`@#IDZ``"!(`"=1E"'_@'P(`J:0`0"$D($`%)"A
M`!B0P0`<D.$`()$!`"21(0`HD4$`+$"&`"38(0`PV$$`.-AA`$#8@0!(V*$`
M4-C!`%C8X0!@V0$`:#U``8(](`&"?&0;>#DI`C@X:@'@.,```3C@```Y`0"(
M.4$`$#BA`'"!:0``D6$`?#E@``"8P0!PF.$`<9$!`'2100!X2``DG3A@`/](
M`"4U/`#__GPL"WA@`/^P/2```7PA`6Y\"`*F82D`'#C@``*3#/_@DTS_Z#\`
M`8*3;/_L.Q@".)/,__@XH```D`P`!#C```"2[/_<?'H;>),L_^23C/_PDZS_
M])/L__R!6```?4D)+CE```!(`",),23__WQC`=1\>_YP?3[:>'_;\!!7W@0^
M?][:>'_;\!!_>]D0?]Y($'][&1`OFP``09P!$#[@``$_@`&".Z$`%&+W``@[
MG`$$/R#__SC@``!_9=MX?\;S>']#TWA(`"*I+X/__T&>`/A^Y;MX?Z3K>']#
MTWA(`"2A+X/__T&>`-`K@P`'0)T`I#AC__A__1H4?Y_H0$"\`!1(``"0.___
M_W^?Z$!!G`"$.*``"'^$XWA_X_MX2``AL2^#``!`GO_@.($`%'R$^%!\@_YP
M?(3P%'QCV10](``!82D`''U)""Z!.```?4I*>3D@``!`@@"(@6$``(`+``2"
MZ__<@PO_X'P(`Z:#*__D@TO_Z(-K_^R#B__P@ZO_](/+__B#Z__\?6%;>$Z`
M`"!_WL@4?WL!U"^;__]`GO\8?Y[(`$">_Q`X8/__.(#__TO__XP\8`&`.&-.
ME$S&,8)+__VA+X3__T">_P@\8`&`.&-.A$S&,8)+__V)2``DT90A__!\"`*F
MD`$`%$@`)-%(`")]+`,``$&"`!B`8P``@`$`%#@A`!!\"`.F3H``(#A@``!+
M___LE"'_P'P(`J:3P0`X/\`!@).!`#`[WDZ@DZ$`-#B!`!B3X0`\/^`!@I`!
M`$1\?1MX._\".'_#\WB!/P``D2$`'#D@``!,QC&"2__N47Q\&WE_I.MX08(`
M?(D\```OB0``09X`<$@`(@$O@P``0+X`<'^CZWB380`L2``D#7Q[&WA_@^-X
M2``D`7Q[&A0X8P`"2``A13S``8`\@`&`?'L;>'^'XWA_I>MX.,9-N#B$3;Q,
MQC&"2``CH7]DVWA_P_-X3,8Q@DO_\?6#80`L2```$'_#\WA,QC&"2__QX8$A
M`!R!7P``?2E2>3E```!`@@`D@`$`1(.!`#"#H0`T?`@#IH/!`#B#X0`\."$`
M0$Z``""380`L2``C@90A_K!\"`*FD\$!2#_``8`[WDT4DX$!0($>`!`_@`&"
M@5X`%#N<`CB!/@`8.($`'(#^`!R1`0`D@1P``)$!`1PY````D4$`*($>`""!
M7@`DD2$`.($^`"B380$\/V`!@(#>``P[>TV<D.$`/)$!`$"`_@`L@1X`,)%!
M`$21(0!(@5X`-($^`#B300$X?'H;>']CVWB0`0%4D^$!3)#!`""0X0!,D0$`
M+)%!`#"1(0`T3,8Q@DO_[,U\?QMY08(`$(D_```OB0``0)X"H).A`422X0$L
MDP$!,),A`31+__W5?'T;>4&"`IA_H^MX2``B@51C"#PX8P`!2``?Q8B]``!\
M>1MX+X4``$&>`#`_``&`?S_+>#L83KA_X_MX?P3#>$S&,8)(`"(9C+T``3O_
M``(OA0``0)[_Y(!A`#@O@P``09X!4#OA`#A(```0A'\`!"^#``!!G@$\.($`
M'$S&,8)+_^PA?'T;>4&B_^2)/0``+XD``$&>_]A+__*M+X,``$&>_\Q_H^MX
M2``?E7Q]&WE!@@$`?Z/K>#[@`8!(`"'1.O=0A'Q_&WA_(\MX2``AP7Q_&A0[
MXP0$?^/[>$@`'P$\X`&`/(`!@'QX&WA_*,MX?Z7K>'[FNW@XYT[`.(1.R$S&
M,8)(`"%9?P/#>#B``<!(`""M+X/__T&>`2@X@0!0?P/#>$@`(.E\?1MY0((`
M%($A`%A5*00F+XE``$&>`;0\@`&`/&`!@H"Z``!_!L-X.(1/$#AC`>!,QC&"
M2``?T8+A`2P[X```@P$!,(,A`32#H0%$@2$!'(%<``!]*5)Y.4```'_C^WA`
M@@0\@`$!5(-!`3B#80$\?`@#IH.!`4"#P0%(@^$!3#@A`5!.@``@.N```%;I
M$#HY00$@?2I*%(/I_P`OGP``09X`N(D_```OB0``09X`K%;X$#HY(0`@?^/[
M>'\)PA1+__%=.[<``2^#``!_M^MX0)X`-(?X``0OGP``09X`?(D_```OB0``
M09X`<'_C^W@[O0`!2__Q*7^WZW@O@P``09[_U'_C^WA(`!X-?'T;>4&"_WQ+
M__YX2``<C8$C```OB0`109[^T(/Z``!(`!QY@.,``#R``8`\8`&"?^7[>'\&
MPW@XA$[4.&,!X$S&,8)(`!ZU2__^Y#N@``!+__XP2__[@7_C^WA(`!VM?'\;
M>$O__MP[(```5S\0.CDA`"P[___\?^GZ%(1_``0O@P``09X`;#B!`!P[.0`!
M3,8Q@DO_Z=E\:1MY0:+_X(DI```OB0``09[_U$@`'5U\?1MY0(+]($O__[2#
M(0!L2``?N7^9&`!`GOY$@2$`6)*A`215*07^DL$!*"^)`<!!G@`L@J$!)(+!
M`2A+__X@.&``#3^@`8!(`!RQ.[U.L'QY&W@XH`!32__\\'_C^WA(`!R9.($`
M''Q_&W@\8`&`.&-/9$S&,8)+_^E%?&0;>4&"`6"`>@``2__QX7QY&WE!@@%0
M?R/+>$@`'Q$K@P`#?'8;>$"=`#@\@`&`.&/__#B$3VQ\>1H42``;X2^#``!`
MG@`<*Y8`!$&>`=A^V;(4B3;_^R^)`"]!G@'(2__W!7QZ&WE`@@`<?R/+>#B`
M``!,QC&"2``<=7QY&WE`@@#L/(`!@#Q@`8`XA$_H.&-."#_``8!,QC&"2__L
MQ3O>4%!(`!MI/(`!@'QG&WA_R/-X.(1/F'[FNWA_!<-X?^/[>#]``8!,QC&"
M2``>,3M:3ZA(``!`2``:E8$C```OB0`10)X`1$@`&R4[O0`!?&<;>'^HZWA_
MR?-X?N:[>'\%PWA_1--X?^/[>$S&,8)(`!WM.(`!P'_C^WA(`!U!+X/__T&>
M_[1_`\-X2``=87_D^WA_8]MX3,8Q@DO_["5_X_MX2__Y98*A`22"P0$H@N$!
M+(,!`3"#(0$T@Z$!1$O__+2#.@``2__^L$O_]R%\=1MY?)8C>$&@_PPPQ/_V
M..```'RU`=1_(\MX2``:<3B!`.@XH``&?R/+>$@`''$O@P`&0)X`W($A`.B!
M7@`\?XE0`$">_M"A(0#LH5X`0'^)4`!`GO[`,-;_SCC@``!\M0'4?R/+>$@`
M&B5_(\MX.($`\#B@`"A(`!PE+X,`*$">`)`]``&`/(`!@#CA`/!^YKMX.0A0
M4#B$3XA_!<-X?^/[>)M!`1A,QC&"2``<X4O__PB#6@`$+YH``$&^_C1_0]-X
M2``<^2N#``-`O?XD.&/__#R``8!\>AH4.(1/=$@`&<TO@P``0+[^"']9TWA+
M__X`DJ$!))+!`2B2X0$LDP$!,),A`323H0%$2``<P3Q@`8`X8T]\3,8Q@DO_
M]664(?_`?`@"ICQ@`8"3X0`\/^`!@CO___23H0`TD`$`1($___R380`L?2D2
M%).!`#`_H`&"D\$`.#N]`CB!70``D4$`'#E````X@0`8@2D``#AC3>R)20!D
MF4D`94S&,8)+_^91+`,``$&"`!")(P``+XD`,4&>`6Q+__)Q@]___#R``8!_
MWA(4.(1/O(!^```XP``#.*```4@`&A5\?!MY08(`,($\``P\H`&`@'X``#BE
M3\"`B0``2``:)8$\``R`B0``@20`"'4I`$!`@@%L@]___#R``8!_WA(4.(1/
MR(!^```XP``#.*```4@`&<%\>QMY08(`,#Q@`8`X@0`8.&--J$S&,8)+_^6I
M?&4;>4&"`02!.P`,@'X``(")``!(`!F]/X`!@CN<`H2#___\/(`!@'__$A0X
MA$_,@]\``#B@``%_P_-X2``7Y3R@`8!\9!MX.*5/Y'_#\WA(`!F!@3\``#E`
M```\8`&`F4D`93B!`!@X8TV<3,8Q@DO_Y34O@P``D'P`"$&>`&@X@`'`2``:
M/2^#`!%!G@`0.2,``2N)``%!G0"@@2$`'(%=``!]*5)Y.4```$""`*R``0!$
M@V$`+(.!`#!\"`.F@Z$`-(/!`#B#X0`\."$`0$Z``"")(P`!+XD``$&^_I1+
M__Z,@'P`!$O_]STO@P``D'P`"$">_XP\8`&`.&-/[$S&,8)(`!=-/X`!@H%;
M``P[G`*$@'X``($\``2`B@``@*D``$@`&*U+__[X@'X``$@`%X%+__Z0@5P`
M!#T@`8"`O``(?&8;>("*```X:5`03,8Q@D@`%OU(`!I)E"'_X'P(`J:3X0`<
M?'\;>)`!`"1+__)9+X,``$&>`!@OGP``09X`$(D_```OB0``0)X`&(`!`"2#
MX0`<."$`('P(`Z9.@``@?^/[>$O_[[U+_^]Y/(`!@#B$3L`XH``$2``712^#
M``!`GO_,@`$`)'_C^WB#X0`<."$`('P(`Z9+_^6XE"'^@#R``8(XA`(X..``
M$)-!`6@Y00`8DV$!;#D#`!B3@0%P?.D#II.A`723P0%XD^$!?($D``"1(0%<
M.2```'U)4WB$Z``$E.D`!$(`__@XH0$8?4A3>(3J``2!*``X@,@`)'^%4$"!
M"``,?2DR>'TI0GA]*3IX52D(/I$J`$!`GO_4@2$`'(`#``B`PP`,/2E:@H%C
M``0Y*7F9@8,``'P?,GB`HP`057WP/H%!`"!5GB@^?_]8.'P(ZG@\ZEJ"?2DJ
M%(%!`"1__S)X?2GR%'T(8#A5GO`^..=YF7TI^A1\YS(4/^I:@GT(`GA_JO)X
M?0@Z%'U*2#B`X0`H._]YF54\*#Y]2NIX52GP/G__`A1]".(4/V=:@GU*^A1_
MQTIX@^$`+%4<*#Y\YT`X.WMYF54(\#Y]2N(4?WOJ%'SG\GA]/4)X/Y]:@GSG
MVA157R@^?[U0.#N<>9E\Y_H4?YSR%(/A`#!_O4IX54KP/G^]XA14_"@^?[WB
M%'T>4GB#@0`T/_]:@CO_>9E_WC@X5.?P/GT_2A1_WD)X?5\Z>#^<6H)_WDH4
M.YQYF5>I*#Y__^@X?]Y*%'T<0A2!(0`X?_]2>%>]\#Y__T(45\@H/G__0A0_
MB5J"@0$`/'SIZGA]*?`X.YQYF5?>\#Y_G%(4/0A:@GTJ.GA_J?)X5_LH/GU*
MXA0Y"'F9@X$`0'TI^#A]2MH45__P/GSH.A1]*>IX?\CZ>#]\6H)]*3H455PH
M/GTIXA0[>WF9?0A0.(.!`$151_`^?[OJ%'T(\GA_ZCIX?0CJ%#^<6H)5/2@^
M?4I(.'T(ZA0[G'F9@Z$`2%4I\#Y_G/(4?4KZ>'S^2G@_?5J"?4KB%%4=*#Y]
M2NH4?]Y`.(.A`$P[>WF950CP/G][^A1_WCIX?3]">#^=6H)_WMH455TH/G__
M4#A_WNH4.YQYF8.A`%!52O`^?YPZ%'__2GA]!U)X/WU:@G__XA17W2@^?__J
M%'SG\#B#H0!4.WMYF7][2A1\Z4)X5][P/C^=6H)]*=H45_TH/GTIZA1]1_)X
M@Z$`6'SG^#@[G'F95__P/G^<0A1\YU)X?\CZ>#]=6H)]"$@X53TH/GSGXA0[
M6GF9?.?J%(.!`%Q_6E(4?0KR>%4I\#Y]2M(45.@H/G_]2GA]"D(4/WQ:@H%!
M`&!_O3@X.WMYF53G\#Y_>_(4/XI:@G^^^GB!00!D?3TZ>'_>VA1_O4`X51LH
M/CN<>9E5"/`^?][:%'^<^A0]2EJ"?[]*>'S]0GA7VR@^?__B%#E*>9E_O?`X
M@X$`:%?>\#Y__]H4?2I*%'^].GA]"O)X?[U*%#^<6H)7Z2@^?[U*%'U*^#B!
M(0!L.YQYF7^<.A1]1T)X5__P/CU);MI\Y^(45[PH/GSGXA2!(0!P?]SZ>#E*
MZZ%_G.IX?4I"%%>]\#X]"6[:?4KB%%3\*#Y]2N(4@2$`='_\ZG@Y".NA?YPZ
M>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`'A_O/)X..?KH7^<4GA\Y_H4
M?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0!\?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15
M'?`^5/PH/CT);MI]2N(4@2$`@'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<
M*#X\Z6[:?0CB%($A`(1_O/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4EN
MVGSGXA2!(0"(?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4
M@2$`C'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`)!_
MO/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"4?]SZ>#E*
MZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4@2$`F'_\ZG@Y".NA?YPZ
M>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`)Q_O/)X..?KH7^<4GA\Y_H4
M?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"@?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15
M'?`^5/PH/CT);MI]2N(4@2$`I'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<
M*#X\Z6[:?0CB%($A`*A_O/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4EN
MVGSGXA2!(0"L?]SZ>#E*ZZ%_G$)X?4KJ%'U*XA15'?`^5/PH/CT);MI]2N(4
M@2$`L'_\ZG@Y".NA?YPZ>'T(\A1]".(45/[P/E5<*#X\Z6[:?0CB%($A`+1_
MO/)X..?KH7^<4GA\Y_H4?.?B%%5?\#Y5'"@^/4ENVGSGXA2!(0"X?]SZ>#E*
MZZ%]2NH4?YU">(.!`+Q5"/`^?4KJ%#TI;MI4_2@^?_M">'U*ZA0Y*>NA5/WP
M/C^<CQQ_9SIX?2GR%#N<O-Q]'NMX?YSZ%'TI.A1]'^@X54<H/G_>4#A]*3H4
M?][[>(#A`,!52O`^53LH/G_>XA1_OU-X?][:%#SGCQR#80#$?[Q0.#CGO-Q_
M_T@X?0="%%4I\#Y__^-X/YN/''U'2WA__T(4.YR\W%?(*#Y__T(4?YSJ%'SG
M\#A]74@X@0$`R%?>\#Y\Y^MX5_LH/GT]\WA\Y^(4/0B/''SGVA1]//`X@V$`
MS#D(O-Q_O?@X?4A2%'^]XWA7__`^/YN/''^]4A14ZB@^?[U2%'_(^WB!00#0
M.YR\W'T\2A1]"#@X?]SX.%3G\#Y]".-X/XJ/'#N<O-Q_ZCMX5[LH/G^<\A1]
M"$H4?_XX.%>I\#Y]2N@X@Z$`U'T(VA1]2O-X?/Y+>#]]CQQ]2N(451TH/CM[
MO-Q]2NH4@Z$`V'_>0#A_>_H4?/](.%4(\#Y_W_MX/YV/''T^0WA572@^?__:
M%#N<O-Q__^H4?]Y0.(.A`-Q_G#H4?2=`.%5*\#Y_QSMX/WV/''SGXA17_2@^
M?.?J%'T>4WB#H0#@.WN\W'_>^#A_>TH4?0E0.%?_\#Y_R4MX/YV/''U>^WA4
M_2@^?2G:%#N<O-Q]*>H4?]XX.(.A`.1_G$(4?4CX.%3G\#Y_R$-X/WV/''T(
MXA15/2@^?0CJ%'_^.WB#H0#H.WN\W'_>2#A_>U(4?^HX.%4I\#Y_RE-X/YV/
M''S^2WA5'2@^?4K:%#N<O-Q]2NH4?YSZ%(.A`.Q\_T@X?]Y`.%4(\#Y_WOMX
M/WV/''_>XA1572@^?][J%'T_0WB#H0#P.WN\W'][.A1__U`X?2=`.%5*\#Y_
M_SMX/YV/''T'4WA7W2@^?__:%#N<O-Q__^H4?.?P.(.A`/1_G$H4?0E0.%?>
M\#Y\Z4MX/WV/''TIXA17_2@^?2GJ%'U'\WB#H0#X.WN\W'][0A1\Y_@X?4CP
M.%?_\#Y\YT-X/YV/''_(^WA5/2@^?.?:%#N<O-Q\Y^H4?YQ2%'T(2#A_RO@X
M@Z$`_%4I\#Y]"%-X5/LH/G_J2WA]".(4/[V/''T(VA1__$@X@V$!`#N]O-Q]
M2C@X?]WR%'U*XWA4Y_`^/YN/''U*\A15'B@^?4KR%'T].WB#P0$$.YR\W'_\
M^A1_O4`X?3PX.%4(\#Y_O>-X/YZ/''S^0WA_O?H455LH/E5?\#X[G+S<?]Y0
M.(%!`0A]/$H4?/Q`.'^]VA1_WN-X/XJ/''_>2A1]"OMX5ZDH/CN<O-Q_WDH4
M?4KH.'^<.A2!(0$,?0?X.%>]\#Y]1SMX/4G*8WSGXA17W"@^?.?B%($A`1!_
M_.IX.4K!UG^<\GA]2D(45][P/CT)RF-]2N(45/PH/GU*XA2!(0$4?[SR>#D(
MP=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4@2$!&'_<^G@XY\'6?YQ2
M>'SGZA1\Y^(455WP/E4<*#X]2<IC?.?B%($A`1Q__.IX.4K!UG^<0GA]2O(4
M?4KB%%4>\#Y4_"@^/0G*8WU*XA2!(0$@?[SR>#D(P=9_G#IX?0CZ%'T(XA14
M__`^55PH/CSIRF-]".(4@2$!)'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<
M*#X]2<IC?.?B%($A`2A__.IX.4K!UG^<0GA]2O(4?4KB%%4>\#Y4_"@^/0G*
M8WU*XA2!(0$L?[SR>#D(P=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4
M@2$!,'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<*#X]2<IC?.?B%($A`31_
M_.IX.4K!UG^<0GA]2O(4?4KB%%4>\#Y4_"@^/0G*8WU*XA2!(0$X?[SR>#D(
MP=9_G#IX?0CZ%'T(XA14__`^55PH/CSIRF-]".(4@2$!/'_<^G@XY\'6?YQ2
M>'SGZA1\Y^(455WP/E4<*#X]2<IC?.?B%($A`4!__.IX.4K!UG^<0GA]2O(4
M?4KB%%4>\#Y4_"@^/0G*8WU*XA2!(0%$?[SR>#D(P=9_G#IX?0CZ%'T(XA14
M__`^55PH/CSIRF-]".(4@2$!2'_<^G@XY\'6?YQ2>'SGZA1\Y^(455WP/E4<
M*#X]2<IC@2$!3'SGXA0Y2L'6?_SJ>'^<0GA]2O(451[P/CT)RF.!(0%0?4KB
M%%3\*#XY",'6/2G*8WU*XA0Y*<'6?[SR>'^<.GA]"/H4?2GJ%(.A`514__`^
M?0CB%%5'*#Y_W/IX?0@Z%#^]RF.`X0%8?YQ2>#N]P=952O`^?2GB%'_=\A15
M'"@^?_U2>#SGRF-]*>(4?[U">#CGP=95"/`^?][J%%4\*#Y]74)X?.=B%'^]
M2GA_GN(4?.?Z%'SGZA17GB@^?.?R%%4I\#Z0XP``?09"%'U%4A1]:^(4?2D"
M%(#A`5R`Q```?.<R>3C```"10P`0D0,`#)%C``21(P`(0((`)(-!`6B#80%L
M@X$!<(.A`72#P0%X@^$!?#@A`8!.@``@?`@"II`!`81(``M5E"'_\'P(`J8X
M8`!@D`$`%$@`"($\@&=%/*#OS3S`F+H\X!`R/0##TCE```!@A",!8*6KB6#&
MW/Y@YU1V80CA\)"#``"0HP`$D,,`")#C``R1`P`0D4,`%)%#`!B10P!<@`$`
M%#@A`!!\"`.F3H``()0A_]!\"`*F5*D8.%2H'WZ300`8?)HC>)-A`!R3@0`@
M?+PK>)/A`"Q\?QMXD`$`-#M_`!R3P0`H@4,`%(!C`%Q]*5(4@/\`&'U*2!`O
M@P``?4I1$)$_`!1\ZCA0?0@Z%)$?`!A!G@!<(2,`0'R^*WA_A4@`09T`X'Q[
M&A1_Q?-X?T33>$@`"BV!/P!<?3Y*%"^)`$"1/P!<09X`Y(`!`#2#00`8@V$`
M''P(`Z:#@0`@@\$`*(/A`"PX(0`P3H``().A`"0OG``_?YWC>']>TWA`G0!(
M?\3S>#B@`$!_8]MX.[W_P$@`"<U_X_MX2__P-2^=`#\[W@!`09W_W#N<_\!7
MB=&^'4G_P#DI``%5*3`R?XKB%']:2A1_1--X?V/;>'^%XWA(``F-DY\`7(`!
M`#2#H0`D@T$`&'P(`Z:#80`<@X$`((/!`"B#X0`L."$`,$Z``"!]/DMX?'L:
M%'_%\WA_1--X2``)38$_`%Q]/DH4+XD`0)$_`%Q`GO\D?^/[>).A`"1_GN!0
M?UKR%$O_[Y5+__\TE"'_T'P(`J8Y0/^`DV$`'#MD`!R3P0`H?'X;>)/A`"Q\
MGR-XD`$`-).!`""3H0`D@Z0`%(.$`!A7J>Z^.*D``7U;2:XOA0`X?'LJ%$"=
M`40@I0!`.(```$@`!?5_X_MX2__O+7]CVW@XH``X.(```$@`!=U7J<(^5X5&
M/E>&A#Y7A\(^5ZA&/E>JA#Z;GP!7?^/[>)N_`%N9/P!:F+\`5)C?`%68_P!6
MF1\`6)E?`%E+_^[=B3\``'_C^WB9/@``H3\``)D^``&!/P``52G"/ID^``*!
M/P``F3X``XD_``29/@`$H3\`!)D^``6!/P`$52G"/ID^``:!/P`$F3X`!XD_
M``B9/@`(H3\`")D^``F!/P`(52G"/ID^``J!/P`(F3X`"XD_``R9/@`,H3\`
M#)D^``V!/P`,52G"/ID^``Z!/P`,F3X`#XD_`!"9/@`0H3\`$)D^`!&!/P`0
M52G"/ID^`!*!/P`0F3X`$X`!`#2#80`<@X$`('P(`Z:#H0`D@\$`*(/A`"PX
M(0`P2``'$""E`#@X@```2``$M4O__MB4(?_@?`@"ID*?``63P0`8?\@"IC_>
M``*0`0`D.]X[((%>@`2)*@`D+XD``$">`&R3@0`0.2```9/A`!R#GH`8@_Z`
M%)DJ`"1_G^!`0)P`,).A`!0[___\@3\`!#N_``0[_P`(?2D#IDZ`!"%_G^!`
M?[_K>$&<_^2#H0`4@3Z`'"^)``!!G@`H@'Z`#$@`!4&#@0`0@^$`'(`!`"2#
MP0`8."$`('P(`Z9.@``@@`$`)(.!`!"#X0`<?`@#IH/!`!@X(0`@3H``()0A
M_^!\"`*F0I\`!9/!`!A_R`*F/]X``I`!`"0[WCI8@)Z`!(DD```OB0``0)X`
M6($^@`@Y0``!DZ$`%"^)``"3X0`<F40``$&>`!"`?H`,.(0`!$@`!3&#_H``
M@[Z`$'^?Z$!`G0`8A3___'TI`Z9.@`0A?Y_H0$&=__"#H0`4@^$`'(`!`"2#
MP0`8."$`('P(`Z9.@``@E"'_P'P(`J:3P0`X/\`!@I!A`!P[W@(XD($`&#AA
M`!R0H0`4.($`&#BA`!2!/@``D2$`+#D@``"0`0!$D^$`/$@`!&T\H`&`/&`!
M@#BE2M`X8TJ`?*0K>$@``N4](`&"B^D!*"^?``!!G@%8/2`!@I.A`#0Y*?_T
M@Z$`'($I__Q]*1(4/4`__X$)```X?0`$84K__SN]``-_@U!`B2@#UF$I``.9
M*`/609T!H%1C$#H_X`&"2``$68#A`!R!00`8/2`!@B^'``&`R0$4@*H``#T`
M`8`](`&`.0A05#DI4%@[_P*$?&L;>)"C``"1`P`$D,,`")$C``R0?P`$0)T!
M1%3I$#HY`P`,.2G_^%4I\+XY*0`!?2D#IH4J``25*``$0@#_^#CG``-4Z1`Z
M/(`!@(!_``PY0```?Z7K>#B$&1Q]9EMX..```'U+22Y(``*!+X,``$&>`+B`
M?P`,2``#03B!`"A\?1MX/&`!@#AC3?Q,QC&"2__.W2^#``!!G@!\@'\`#$@`
M!*E(``$%@2$`+(%>``!]*5)Y.4```'^CZWA`@@#`@`$`1(.A`#2#P0`X?`@#
MIH/A`#PX(0!`3H``($@``GT]0`&"+X,``)!J`I"3H0`T09X`D$@``U4](`&"
M.2G_](%)__Q]2A(4@4H``)OJ!G9+__Z$@'\`""^#``!!G@`82__J!4O__WB`
M?P`,2``"[4O__T0\8`&`.($`*#AC39Q,QC&"2__.*2^#``"0?P`(09[_3$O_
MZ=%+__]$.2``$$O__N@\@`&"/&`!@#B$_I@X8T^43,8Q@D@``*%(``/M.&``
M`4@``F4]8`&"@6L``'UI`Z9.@`0@/6`!@H%K``1]:0.F3H`$(#U@`8*!:P`(
M?6D#IDZ`!"`]8`&"@6L`#'UI`Z9.@`0@/6`!@H%K`!!]:0.F3H`$(#U@`8*!
M:P`4?6D#IDZ`!"`]8`&"@6L`&'UI`Z9.@`0@/6`!@H%K`!Q]:0.F3H`$(#U@
M`8*!:P`@?6D#IDZ`!"`]8`&"@6L`)'UI`Z9.@`0@/6`!@H%K`"A]:0.F3H`$
M(#U@`8*!:P`L?6D#IDZ`!"`]8`&"@6L`,'UI`Z9.@`0@/6`!@H%K`#1]:0.F
M3H`$(#U@`8*!:P`X?6D#IDZ`!"`]8`&"@6L`/'UI`Z9.@`0@/6`!@H%K`$!]
M:0.F3H`$(#U@`8*!:P!$?6D#IDZ`!"`]8`&"@6L`2'UI`Z9.@`0@/6`!@H%K
M`$Q]:0.F3H`$(#U@`8*!:P!0?6D#IDZ`!"`]8`&"@6L`5'UI`Z9.@`0@/6`!
M@H%K`%A]:0.F3H`$(#U@`8*!:P!<?6D#IDZ`!"`]8`&"@6L`8'UI`Z9.@`0@
M/6`!@H%K`&1]:0.F3H`$(#U@`8*!:P!H?6D#IDZ`!"`]8`&"@6L`;'UI`Z9.
M@`0@/6`!@H%K`'!]:0.F3H`$(#U@`8*!:P!T?6D#IDZ`!"`]8`&"@6L`>'UI
M`Z9.@`0@/6`!@H%K`'Q]:0.F3H`$(#U@`8*!:P"`?6D#IDZ`!"`]8`&"@6L`
MA'UI`Z9.@`0@/6`!@H%K`(A]:0.F3H`$(#U@`8*!:P",?6D#IDZ`!"`]8`&"
M@6L`D'UI`Z9.@`0@/6`!@H%K`)1]:0.F3H`$(#U@`8*!:P"8?6D#IDZ`!"`]
M8`&"@6L`G'UI`Z9.@`0@/6`!@H%K`*!]:0.F3H`$(#U@`8*!:P"D?6D#IDZ`
M!"`]8`&"@6L`J'UI`Z9.@`0@/6`!@H%K`*Q]:0.F3H`$(#U@`8*!:P"P?6D#
MIDZ`!"`]8`&"@6L`M'UI`Z9.@`0@/6`!@H%K`+A]:0.F3H`$(#U@`8*!:P"\
M?6D#IDZ`!"`]8`&"@6L`P'UI`Z9.@`0@/6`!@H%K`,1]:0.F3H`$(#U@`8*!
M:P#(?6D#IDZ`!"`]8`&"@6L`S'UI`Z9.@`0@/6`!@H%K`-!]:0.F3H`$(#U@
M`8*!:P#4?6D#IDZ`!"`]8`&"@6L`V'UI`Z9.@`0@/6`!@H%K`-Q]:0.F3H`$
M(#U@`8*!:P#@?6D#IDZ`!"`]8`&"@6L`Y'UI`Z9.@`0@/6`!@H%K`.A]:0.F
M3H`$(#U@`8*!:P#L?6D#IDZ`!"`]8`&"@6L`\'UI`Z9.@`0@/6`!@H%K`/1]
M:0.F3H`$(#U@`8*!:P#X?6D#IDZ`!"`]8`&"@6L`_'UI`Z9.@`0@/6`!@H%K
M`0!]:0.F3H`$($@``0!(``#\2```^$@``/1(``#P2```[$@``.A(``#D2```
MX$@``-Q(``#82```U$@``-!(``#,2```R$@``,1(``#`2```O$@``+A(``"T
M2```L$@``*Q(``"H2```I$@``*!(``"<2```F$@``)1(``"02```C$@``(A(
M``"$2```@$@``'Q(``!X2```=$@``'!(``!L2```:$@``&1(``!@2```7$@`
M`%A(``!42```4$@``$Q(``!(2```1$@``$!(```\2```.$@``#1(```P2```
M+$@``"A(```D8````&````!@````8````&````!@````8````&`````]@`&"
M/6O^@(`,__@Y:[1@?`D#IGP+6A2!C/_\?6!:%$Z`!"!@````8````&````!@
M````8````&````!@````?`@"II`!``24(?_P2__VV3@A`!"``0`$?`@#IDZ`
M`"!P<U]S=')I;F=S(&UI<W-I;F<*`'!E<FQX<VDN8P````&`4'P!@$U8````
M``&`3E0!@$Y8`8!0<`&`3:`!@%!X``````&`4%P!@%!D``````!#04-(10``
M+@```"XN```E<R\E<P```$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!)
M;G1E<FYA;',Z.E!!4CHZ0D]/5`````!005)?5$5-4`````!005)?4%)/1TY!
M344`````.@```"5S)7,E<P``4$523#5,24(`````4$523$Q)0@!015),-4]0
M5`````!015),24\``%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1`!005)?
M0TQ%04X```!005)?1$5"54<```!005)?0T%#2$4```!005)?1TQ/0D%,7T1%
M0E5'`````%!!4E]'3$]"04Q?5$U01$E2````4$%27U1-4$1)4@``4$%27T=,
M3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%04X`````;'-E96L@9F%I;&5D````
M`')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````
M<&%R+0`````E<R5S)7,E<P`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S
M=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET
M(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!0051(`````'!A<FP`````
M+G!A<@````!S:&]R="!R96%D```E<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M
M)74E<P```"5S)7-T96UP+25U+25U)7,`````&````'!E<FP`````,````%!!
M4CHZ4&%C:V5R.CI615)324].`````#$N,#8Q````56YA8FQE('1O(&-R96%T
M92!C86-H92!D:7)E8W1O<GD`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@
M8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@`````M
M90``+2T``%5315(`````55-%4DY!344`````5$5-4$1)4@!435``+W9A<B]T
M;7`O````979A;"`G97AE8R`O=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`
M(GTG"FEF(#`[(",@;F]T(')U;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K
M86=E(%]?<&%R7W!L.PIM>2`H)%!!4E]-04=)0RP@)$9)3$5?;V9F<V5T7W-I
M>F4L("1C86-H95]N86UE7W-I>F4I.PI"14=)3B!["B1005)?34%'24,@/2`B
M7&Y005(N<&U<;B(["B1&24Q%7V]F9G-E=%]S:7IE(#T@-#L@("`C('!A8VLH
M(DXB*0HD8V%C:&5?;F%M95]S:7IE(#T@-#`["B102UI)4%]-04=)0R`](")0
M2UPP,#-<,#`T(CL*?0IS=6(@9FEN9%]P87)?;6%G:6,*>PIM>2`H)&9H*2`]
M($!?.PIM>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@
M)'-I>F4@/2`M<R`D9F@["FUY("1P;W,@/2`H)'-I>F4M,2D@+2`H)'-I>F4M
M,2D@)2`D8VAU;FM?<VEZ93L@("`@("`C($Y/5$4Z("1P;W,@:7,@82!M=6QT
M:7!L92!O9B`D8VAU;FM?<VEZ90IW:&EL92`H)'!O<R`^/2`P*2!["G-E96L@
M)&9H+"`D<&]S+"`P.PIR96%D("1F:"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L
M96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@
M)%!!4E]-04=)0RDI(#X](#`I('L*<F5T=7)N("1P;W,@*R`D:3L*?0HD<&]S
M("T]("1C:'5N:U]S:7IE.PI]"G)E='5R;B`M,3L*?0IM>2`H)'!A<E]T96UP
M+"`D<')O9VYA;64L($!T;7!F:6QE+"`E36]D=6QE0V%C:&4I.PI%3D0@>R!I
M9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q
M=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D
M=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*
M;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I;B`B)'!A<E]T96UP(ETI.PI&:6QE
M.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI
M;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A
M<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&
M:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL
M93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M
M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D9FEL96YA;64@/2`D=&UP+3YF:6QE
M;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB.PIX/3$[('=H:6QE(%L@7"1X("UL
M="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*:68@6R!<(2`M9"`G)'!A
M<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I"G-L965P(#$*>#U@97AP<B!<)'@@
M*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G"BXN+@IC;&]S92`D=&UP.PIC:&UO
M9"`P-S`P+"`D9FEL96YA;64["FUY("1C;60@/2`B)&9I;&5N86UE(#XO9&5V
M+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C;60I.PIO=71S*'%Q6U-P87=N960@
M8F%C:V=R;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@)&9I;&5N
M86UE72D["GT*?2!]"D)%1TE.('L*26YT97)N86QS.CI005(Z.D)/3U0H*2!I
M9B!D969I;F5D("9);G1E<FYA;',Z.E!!4CHZ0D]/5#L*979A;"!["E]P87)?
M:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@
M)7-Y<R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_("<[
M)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I;GQ/
M4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^("@D
M7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?<')O
M9VYA;64H*3L*;W5T<RAQ<5M<)'!R;V=N86UE(#T@(B1P<F]G;F%M92)=*3L*
M7W-E=%]P87)?=&5M<"@I.PIO=71S*'%Q6UPD14Y6>U!!4E]414U0?2`]("(D
M14Y6>U!!4E]414U0?2)=*3L*;7D@*"1S=&%R=%]P;W,L("1S=&%R=%]O9E]&
M24Q%7W-E8W1I;VXI.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]
M('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE
M*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=
M*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R;F\N
M<&T*;&%S="!-04=)0SL*?0IM>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G
M:6,H*E]&2"D["FEF("@D;6%G:6-?<&]S(#P@,"D@>PIO=71S*'%Q6T-A;B=T
M(&9I;F0@;6%G:6,@<W1R:6YG("(D4$%27TU!1TE#(B!I;B!F:6QE("(D<')O
M9VYA;64B72D["FQA<W0@34%'24,["GT*;W5T<R@B1F]U;F0@4$%2(&UA9VEC
M(&%T('!O<VET:6]N("1M86=I8U]P;W,B*3L*<V5E:R!?1D@L("1M86=I8U]P
M;W,@+2`D1DE,15]O9F9S971?<VEZ92`M(&QE;F=T:"@B7#!#04-(12(I+"`P
M.PIR96%D(%]&2"P@)&)U9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F
M(&YE(")<,$-!0TA%(B`F)B`D8G5F(&YE(")<,$-,14%.(BD@>PIO=71S*").
M;R!C86-H92!M87)K97(@9F]U;F0B*3L*;&%S="!-04=)0SL*?0IO=71S*'%Q
M6T-A8VAE(&UA<FME<B`B)&)U9B(@9F]U;F1=*3L*<V5E:R!?1D@L("1M86=I
M8U]P;W,@+2`D1DE,15]O9F9S971?<VEZ92P@,#L*<F5A9"!?1D@L("1B=68L
M("1&24Q%7V]F9G-E=%]S:7IE.PIM>2`D;V9F<V5T(#T@=6YP86-K*").(BP@
M)&)U9BD["F]U=',H(D]F9G-E="!F<F]M('-T87)T(&]F($9)3$5S(&ES("1O
M9F9S970B*3L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`D1DE,15]O9F9S971?
M<VEZ92`M("1O9F9S970L(#`["B1S=&%R=%]O9E]&24Q%7W-E8W1I;VX@/2!T
M96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[
M("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B
M1DE,12(*=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F
M+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F
M=6QL;F%M92`]("1B=68["F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL
M;F%M92(N+BY=*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF
M7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X
M="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&
M2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I
M.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI
M>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT
M(BP@)&)U9BP@,#<U-2D["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN
M86UE?2`]("1F:6QE;F%M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL
M96YA;65](#T@)&9U;&QN86UE.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U
M;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^
M("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*
M<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H
M)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O
M<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T[
M"G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M
M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L
M92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE
M.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D
M:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E
M*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I
M.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A
M<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF
M:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE
M;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR
M971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T
M(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,
M;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V
M>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET
M7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F
M:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL
M97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO
M*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F
M:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA
M;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I
M<F5?;&ES='LD9FEL96YA;65].PI]"FEF("@D8G5F(&YE("102UI)4%]-04=)
M0RD@>PIO=71S*'%Q6TYO('II<"!F;W5N9"!A9G1E<B!&24Q%('-E8W1I;VX@
M:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#(#L*?0HD<W1A<G1?
M<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@
M;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO
M9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5]
M.R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M
M*0II9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I
M;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB
M7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@
M('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%2
M1U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!
M4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A
M<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*
M)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S
M=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE
M;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q
M(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q
M(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD
M8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L
M92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*
M?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O
M<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS
M:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!
M4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D
M14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES
M=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!
M4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H
M:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U
M:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$
M:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP
M+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E
M861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E;65T85]P87(@/2!D
M;R!["FEF("@D>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-
M151!+GEM;"<I*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]
M?B!S+UY<4RXJ+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG
M.PI]"GT["FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A
M=R<L("1P87(@;W(@9&EE('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B
M.B`D(5T["FUY("1B=68["G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P
M87(B(&ES(&YO="!A('!A<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`D4$M:25!?
M34%'24,["F-L;W-E("1P:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H
M(#T@24\Z.D9I;&4M/FYE=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\
M($E/.CI&:6QE.CI/7U)$5U(H*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W
M-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=
M.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97([
M"FEF("AD969I;F5D("1S=&%R=%]O9E]&24Q%7W-E8W1I;VXI('L*<F5A9"!?
M1D@L("1L;V%D97(L("1S=&%R=%]O9E]&24Q%7W-E8W1I;VX["GT@96QS92![
M"FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II9B`H(21%
M3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`<F5M
M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ1FEL
M=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0HD9F@M/G!R
M:6YT*"1L;V%D97(I"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG(&QO861E<B!T
M;R`B)&]U="(Z("0A73L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE
M879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC
M*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$
M4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D
M0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z
M.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\
M?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`
M1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM
M>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@
M("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A
M8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N
M86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#
M86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*
M?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^
M(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME
M>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO
M*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH
M)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D
M9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*
M;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT
M(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N
M9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ
M<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[
M;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI
M('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O
M(#T@=6YD968["F]P96X@;7D@)'1H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@
M<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1T:#X[
M"F-L;W-E("1T:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O
M;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!
M5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I
M;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I
M;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE
M;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z
M.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.
M)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*0IO<B!D:64@<7%;17)R
M;W(@=W)I=&EN9R!E;6)E9&1E9"!&24Q%('1O("(D;W5T(CH@)"%=.PIO=71S
M*'%Q6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*:68@*"1Z:7`I('L*)'II
M<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2`]/2!!<F-H:79E.CI::7`Z.D%:
M7T]+"F]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('II<"!P87)T(&]F("(D;W5T
M(ET["GT*:68@*"1M971A7W!A<GMC;&5A;GTI('L*)&9H+3YP<FEN="@B7#!#
M3$5!3B(I.PI]"F5L<V4@>PII9B`H(61E9FEN960@)&-A8VAE7VYA;64I('L*
M;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["G-E96L@)&9H+"`P+"`P
M.PHD8W1X+3YA9&1F:6QE*"1F:"D["G-E96L@)&9H+"`P+"`R.PHD8V%C:&5?
M;F%M92`]("1C='@M/FAE>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`]('-U8G-T
M<B`D8V%C:&5?;F%M92P@,"P@)&-A8VAE7VYA;65?<VEZ93L*)&-A8VAE7VYA
M;64@+CT@(EPP(B!X("@D8V%C:&5?;F%M95]S:7IE("T@;&5N9W1H("1C86-H
M95]N86UE*3L*)&9H+3YP<FEN="@D8V%C:&5?;F%M92P*(EPP0T%#2$4B*3L*
M?0IM>2`D;V9F<V5T(#T@)&9H+3YT96QL("T@;&5N9W1H*"1L;V%D97(I.PHD
M9F@M/G!R:6YT*'!A8VLH)TXG+"`D;V9F<V5T*2P*)%!!4E]-04=)0RD["B1F
M:"T^8VQO<V4@;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@=')A:6QE<B!O9B`B
M)&]U="(Z("0A73L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@
M=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR
M97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI[
M"G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*
M;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@
M("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD
M;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D
M;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ
M92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@D>FEP
M+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]($%R8VAI
M=F4Z.EII<#HZ05I?3TLH*2D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP
M(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK
M4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD
M4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*;W5T<R@B17AT<F%C
M=&EN9R!Z:7`N+BXB*3L*:68@*&1E9FEN960@)$5.5GM005)?5$5-4'TI('L@
M(R!S:&]U;&0@8F4@<V5T(&%T('1H:7,@<&]I;G0A"F9O<F5A8V@@;7D@)&UE
M;6)E<B`H("1Z:7`M/FUE;6)E<G,@*2!["FYE>'0@:68@)&UE;6)E<BT^:7-$
M:7)E8W1O<GD["FUY("1M96UB97)?;F%M92`]("1M96UB97(M/F9I;&5.86UE
M.PIN97AT('5N;&5S<R`D;65M8F5R7VYA;64@/7X@;7L*7@HO/W-H;&EB+PHH
M/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*
M;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ
M4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D[
M"FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP
M<F5S<V5D4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M
M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D[
M"GT@96QS92!["F]U=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T
M;R`B)&1E<W1?;F%M92(N+BY=*3L**"1M96UB97(M/F5X=')A8W14;T9I;&5.
M86UE9"@D9&5S=%]N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R
M(&1I92!Q<5M%<G)O<B!E>'1R86-T:6YG('II<"!M96UB97(@=&\@(B1D97-T
M7VYA;64B73L*8VAM;V0H,#4U-2P@)&1E<W1?;F%M92D@:68@)%Y/(&5Q(")H
M<'5X(CL*?0I]"GT*?0IU;FQE<W,@*"1005(Z.DQI8D-A8VAE>R1P<F]G;F%M
M97TI('L*9&EE(#P\("(N(B!U;FQE<W,@0$%21U8["E5S86=E.B`D,"!;("U!
M;&EB+G!A<B!=(%L@+4ED:7(@72!;("U-;6]D=6QE(%T@6R!S<F,N<&%R(%T@
M6R!P<F]G<F%M+G!L(%T*)#`@6R`M0GPM8B!=(%LM3V]U=&9I;&5=('-R8RYP
M87(*+@HD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64@/2`D,"`]('-H
M:69T*$!!4D=6*3L*?0IS=6(@0W)E871E4&%T:"!["FUY("@D;F%M92D@/2!`
M7SL*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*;7D@*"1B87-E;F%M92P@)'!A
M=&@L("1E>'0I(#T@1FEL93HZ0F%S96YA;64Z.F9I;&5P87)S92@D;F%M92P@
M*"=<+BXJ)RDI.PIR97%U:7)E($9I;&4Z.E!A=&@["D9I;&4Z.E!A=&@Z.FUK
M<&%T:"@D<&%T:"D@=6YL97-S*"UE("1P871H*3L@(R!M:W!A=&@@9&EE<R!W
M:71H(&5R<F]R"GT*<W5B(')E<75I<F5?;6]D=6QE<R!["G)E<75I<F4@;&EB
M.PIR97%U:7)E($1Y;F%,;V%D97(["G)E<75I<F4@:6YT96=E<CL*<F5Q=6ER
M92!S=')I8W0["G)E<75I<F4@=V%R;FEN9W,["G)E<75I<F4@=F%R<SL*<F5Q
M=6ER92!#87)P.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5R<FYO
M.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CL*
M<F5Q=6ER92!&8VYT;#L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U:7)E($9I
M;&4Z.E-P96,["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@0V]N9FEG.PIR
M97%U:7)E($E/.CI(86YD;&4["G)E<75I<F4@24\Z.D9I;&4["G)E<75I<F4@
M0V]M<')E<W,Z.EIL:6(["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E
M($1I9V5S=#HZ4TA!.PIR97%U:7)E(%!!4CL*<F5Q=6ER92!005(Z.DAE879Y
M.PIR97%U:7)E(%!!4CHZ1&ES=#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N=#L*<F5Q
M=6ER92!A='1R:6)U=&5S.PIE=F%L('L@<F5Q=6ER92!#=V0@?3L*979A;"![
M(')E<75I<F4@5VEN,S(@?3L*979A;"![(')E<75I<F4@4V-A;&%R.CI5=&EL
M('T["F5V86P@>R!R97%U:7)E($%R8VAI=F4Z.E5N>FEP.CI"=7)S="!].PIE
M=F%L('L@<F5Q=6ER92!4:64Z.DAA<V@Z.DYA;65D0V%P='5R92!].PIE=F%L
M('L@<F5Q=6ER92!097)L24\[(')E<75I<F4@4&5R;$E/.CIS8V%L87(@?3L*
M979A;"![(')E<75I<F4@=71F."!].PI]"G-U8B!?<V5T7W!A<E]T96UP('L*
M:68@*&1E9FEN960@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!]
M(#U^("\H+BLI+RD@>PHD<&%R7W1E;7`@/2`D,3L*<F5T=7)N.PI]"F9O<F5A
M8V@@;7D@)'!A=&@@*`HH;6%P("1%3E9[)%]]+"!Q=R@@4$%27U1-4$1)4B!4
M35!$25(@5$5-4$1)4B!414U0(%1-4"`I*2P*<7<H($,Z7%Q414U0("]T;7`@
M+B`I"BD@>PIN97AT('5N;&5S<R!D969I;F5D("1P871H(&%N9"`M9"`D<&%T
M:"!A;F0@+7<@)'!A=&@["FUY("1U<V5R;F%M93L*;7D@)'!W=6ED.PIE=F%L
M('LH)'!W=6ED*2`](&=E='!W=6ED*"0^*2!I9B!D969I;F5D("0^.WT["FEF
M("@@9&5F:6YE9"@F5VEN,S(Z.DQO9VEN3F%M92D@*2!["B1U<V5R;F%M92`]
M("97:6XS,CHZ3&]G:6Y.86UE.PI]"F5L<VEF("AD969I;F5D("1P=W5I9"D@
M>PHD=7-E<FYA;64@/2`D<'=U:60["GT*96QS92!["B1U<V5R;F%M92`]("1%
M3E9[55-%4DY!345]('Q\("1%3E9[55-%4GT@?'P@)U-94U1%32<["GT*;7D@
M)'-T;7!D:7(@/2`B)'!A=&@D<WES>U]D96QI;7UP87(M(BYU;G!A8VLH(D@J
M(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D:7(L(#`W-34["FUY("1C86-H
M95]D:7(["FEF("@D14Y6>U!!4E]#3$5!3GTI('L*)&-A8VAE7V1I<B`](")T
M96UP+20D(CL*?0IE;'-E('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G
M;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PII
M9B`H*&UY("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M86=I8R@D9F@I*2`^/2`P
M*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*
M+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*;7D@)&)U9CL*<F5A9"`D9F@L("1B
M=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!E<2`B7#!#04-(12(I
M('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM
M(&QE;F=T:"@B7#!#04-(12(I"BT@)&-A8VAE7VYA;65?<VEZ92P@,#L*<F5A
M9"`D9F@L("1B=68L("1C86-H95]N86UE7W-I>F4["B1B=68@/7X@<R]<,"\O
M9SL*)&-A8VAE7V1I<B`](")C86-H92TD8G5F(CL*?0I]"F-L;W-E("1F:#L*
M?0II9B`H(21C86-H95]D:7(I('L*)&-A8VAE7V1I<B`](")T96UP+20D(CL*
M)$5.5GM005)?0TQ%04Y](#T@,3L*?0HD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L
M:6U])&-A8VAE7V1I<B(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PHD14Y6>U!!
M4E]414U0?2`]("1S=&UP9&ER.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF
M("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\[
M"GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E
M*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL
M97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M
M92XD)"(["FUY("1F:#L**&]P96X@)&9H+"`G/CIR87<G+"`D=&5M<&YA;64I
M"B8F("AP<FEN="`D9F@@)&-O;G1E;G1S*0HF)B`H8VQO<V4@)&9H*0IO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R`B)'1E;7!N86UE(CH@)"%=.PIC:&UO9"`D
M;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M
M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T
M=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I
M;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345]
M(#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]
M("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@
M)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P
M<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`K(#$I
M.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M
M92P@)'-Y<WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG
M+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P
M<F]G;F%M921S>7-[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D<WES>U]E>&5]
M.PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421S>7-[
M<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI<W1S("1%
M3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!].PHD9&ER
M(#U^(',O7%$D<WES>U]D96QI;7U<120O+SL**"@D<')O9VYA;64@/2`B)&1I
M<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](CL**"@D
M<')O9VYA;64@/2`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T
M*0II9B`M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@
M7V9I>%]P<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]0
M4D]'3D%-17T["FEF("AI;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M/"`P*2!["B1P<F]G;F%M92`]("(N)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*
M?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T
M8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E
M=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O
M7B@_/5PN7"X_7%$D<WES>U]D96QI;7U<12DO)'!W9"1S>7-[7V1E;&EM?2\[
M"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I
M;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I
M('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R
M.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!0
M4D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLB4$%27R1?(GT["GT*9F]R("AQ
M=R\@5$U01$E2(%1%35`@0TQ%04X@1$5"54<@+RD@>PHD14Y6>R)005)?)%\B
M?2`]("1%3E9[(E!!4E]'3$]"04Q?)%\B?2!I9B!E>&ES=',@)$5.5GLB4$%2
M7T=,3T)!3%\D7R)].PI]"FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@
M)$5.5GM005)?0TQ%04Y].PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U
M:65T.PII9B`H)&QO9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS
M92!["G!R:6YT(")`7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@
M0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z
M.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R
M8VAE>'`@<VET96QI8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI
M.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@
M7"9005(Z.F9I;F1?<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@
M<7%;<&%R+G!L.B!#86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B
M.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T;W)Y7&Y="G5N;&5S<R`M92`D<')O
M9VYA;64["F1O("1P<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^
M+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?
M7T524D]2(#T@)$`@:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?
M15)23U(@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/
M4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.1%]?"@`!&P,[```!"````"#__X44
M```!(/__A8````%0__^&'````9#__X=\```"3/__B*0```*T__^);````N3_
M_XLD```#?/__BYP```.D__^.(````_S__X[(```$*/__C^````1@__^0&```
M!'C__Y`H```$D/__DOP```4<__^3/```!43__Y.T```%A/__E:````6X__^6
M+```!>#__Y;(```%_/__F*P```9`__^8Y```!F3__YG\```&H/__H,P```=,
M__^C-```!X3__Z.X```'\/__LB@```@H__^RF```"$C__[0L```(I/__M<P`
M``(`__^VE````=#__[<L```'M/__N9@```&\````$``````!>E(`!'Q!`1L,
M`0`````L````&/__@^P```!L`$$.($$)00!"G@)%G0.?`4,107],!D%"W][=
M#@`````````\````2/__A"@```"<`$$.($$)00!"G@)%G`2?`40107]#G0-/
MW40&04+?WMP.`$$.()P$G0.>`I\!$4%_````*````(C__X2$```!8`!!#B!!
M"4$`0IP$0IX"19L%$4%_09T#09\!```````0````M/__M]0```50````````
M`"P```#(__^TO````)@`00X@00E!`$*>`D<107]$G0-"GP%/W4'?0]X.`$$&
M00```$@```#X__^SQ````,@`00X@00E!`$*>`D<107]"G`1"GP%&G0-*W4;<
M0=]#W@X`009!00X@G`2>`I\!$4%_0MQ!WT$&04+>#@````!D```!1/__A2@`
M``$H`$(.,$&?`4.;!42>`DH)00!!F@9!$4%_0IP$0IT#3]I!VT$&04'<0=U!
MWD/?#@!!#C";!9X"GP%(F@:<!)T#$4%_1MK<W09!00K;0=Y!"T+;WD*;!4+;
M`````"P```&L__^%Z````,@`00XP0@E!`)T#2!%!?YX"GP%C"@9!0M_>W0X`
M00L``````)0```'<__^&@````;@`00[0`4&=`T*?`5$*W]T.`$$+00E!`$&<
M!$$107]$F0="F@9"FP5"G@)4F`A;V$'9VMO>0MQ!!D%!F0>:!IL%G`2>`A%!
M?T;90=I!!D%!VT'<0=Y!F`B9!YH&FP6<!)X"$4%_0MC9VMO<W@9!00E!`$&8
M"$$107]!F0=!F@9!FP5!G`1!G@(`````)````G3__X>@````>`!!#B!$"4$`
MGP%#$4%_5`9!0=\.`````````%0```*<__^'\````H0`00Y01!%!?YL%G`1#
MG0.>`DN?`9D'F@8"1`H&04;?WMW<V]K9#@!!"TF8"$&7"6,*UT'800M(U]A"
MEPF8"$370=A#EPE!F`@````H```"]/__BAP```"H`$$.P`%$"4$`G@*?`4(1
M07]8"@9!0=_>#@!!"P```#0```,@__^*F````1@`00ZP"$0)00">`D*?`4<1
M07]E"@9!0=_>#@!!"TX*!D%!W]X.`$$+````%````UC__XMX````.`!"#A!+
M#@``````%````W#__XN8````$`!!#A!"#@``````B````XC__XN0```"U`!!
M#O`)1`E!`)H&1)P$GP%*$4%_F0<%1@Q5"@E&#$(&04(&1D/?W-K9#@!!"T&;
M!4F5"T*6"D*7"4*8"$.>`D&=`P)#"M5!UD'70=A!VT'=0=Y!"U;50=9!UT'8
M0=M!W4'>09L%0=M"E0M!E@I!EPE!F`A!FP5!G0-!G@(````D```$%/__C=@`
M``!``$$.($,)00"?`4(107](WPX`009!````````/```!#S__XWP````>`!!
M#B!!GP%&"4$`01%!?TC?#@!!!D%!#B"?`43?#@!!#B"?`1%!?T3?#@!!!D$`
M`````#````1\__^.*````>P`00Y`11%!?YL%1)P$G0-"G@)"GP$"7`H&043?
MWMW<VPX`00L````D```$L/__C^````",`$$.,$,)00"?`4(107]:"M\.`$$&
M04$+````&```!-C__Y!$````G`!!#H`!0A%!?P```````$````3T__^0Q```
M`>0`10[0@`1%"4$`F`B:!DV;!9X"$4%_EPF9!YP$G0.?`0)%"@9!2-_>W=S;
MVMG8UPX`00L`````(```!3C__Y)D````.`!!#A!"$4%_1PH.`$$&04$+````
M````.```!5S__Y)X```!&`!!#D!""4$`G@)&G`2=`Y\!2!%!?TV;!57;30H&
M04/?WMW<#@!!"T&;!0``````J```!9C__Y-4```&T`!!#M`"0@E!`)X"1)P$
M4)L%3)H&21%!?Y\!2)T#09<)09@(09D'`EK70MA!V4'=2@9!1-_>W-O:#@!!
M#M`"EPF8")D'F@:;!9P$G0.>`I\!$4%_=M?8V=U%EPF8")D'G0-;E0M"E@I#
MU4'62)4+E@H"6PK50=9!UT'80=E!W4$+`D#5UM?8V=U!E0M!E@I!EPE!F`A!
MF0=!G0,``````#0```9$__^9>````F@`00Y`0PE!`)\!2)T#$4%_FP6<!$*>
M`@)E"@9!1-_>W=S;#@!!"P``````+```!GS__YNH````A`!!#B!""4$`GP%"
M$4%_2PK?#@!!!D%!"TW?#@!!!D$`````.```!JS__Z]P```";`!!#D!""4$`
MG@)-$4%_GP%-G0,"3MU"!D%"W]X.`$$.0)X"GP$107]%G0,`````-```!NC_
M_YO````.<`!!#H`#4)H&FP6<!)T#G@*?`0,#APK?WMW<V]H.`$$+00E!`$$1
M07\````<```'(/__J?@```!P`$$.$$,107]6#@!!!D$``````%@```=`__^J
M2````90`00XP1`E!`)H&0YL%G`1"GP%/$4%_G@)2"@9!1-_>W-O:#@!!"T&=
M`US=0@9!1=_>W-O:#@!!#C":!IL%G`2>`I\!$4%_3)T#````````-```!YS_
M_ZN````!H`!!#C!#"4$`FP5"G@)"GP%$$4%_G`2=`P)3"@9!1-_>W=S;#@!!
M"P``````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````#_____
M`````/____\``````````````````````````````````````8(!$`&!_H0!
M@?Z$`8`/I`&`#Z0!@?Z$`8'^A`&`30`!@@$8`8(!(`&"`1`!@@)8`8'_$`&!
M_H0!@!,,`8'^A`&`3.`!@?Z(`8("7``````!@(_(`8'^B`&!_I`!@?Z0````
M``````$````!`````0```E4````!```"7P````$```*;`````0```L$````/
M```$2@````P!@!+(````#0&`3.`````$`8`!>`````4!@`K4````!@&`!)0`
M```*```$D@````L````0````%0`````````#`8(```````(```,,````%```
M``<````7`8`/O'`````!@?_T````!P&`#V@````(```#8`````D````,````
M````````````````````````````````````````````````````````````
M``````&!_Q````````````&`2Z`!@$ND`8!+J`&`2ZP!@$NP`8!+M`&`2[@!
M@$N\`8!+P`&`2\0!@$O(`8!+S`&`2]`!@$O4`8!+V`&`2]P!@$O@`8!+Y`&`
M2^@!@$OL`8!+\`&`2_0!@$OX`8!+_`&`3``!@$P$`8!,"`&`3`P!@$P0`8!,
M%`&`3!@!@$P<`8!,(`&`3"0!@$PH`8!,+`&`3#`!@$PT`8!,.`&`3#P!@$Q`
M`8!,1`&`3$@!@$Q,`8!,4`&`3%0!@$Q8`8!,7`&`3&`!@$QD`8!,:`&`3&P!
M@$QP`8!,=`&`3'@!@$Q\`8!,@`&`3(0!@$R(`8!,C`&`3)`!@$R4`8!,F`&`
M3)P!@$R@"E!!4BYP;0H``````8("6`&`4(A'0T,Z("A.971"4T0@;F(S(#(P
M,3DP,S$Y*2`W+C0N,`!'0T,Z("AN8C,@,C`Q.3`S,3DI(#<N-"XP`"1.971"
M4T0Z(&-R=#`N4RQV(#$N-R`R,#$X+S$Q+S(V(#$W.C,W.C0U(&IO97)G($5X
M<"`D`"1.971"4T0Z(&-R=#`M8V]M;6]N+F,L=B`Q+C(S(#(P,3@O,3(O,C@@
M,C`Z,3(Z,S4@8VAR:7-T;W,@17AP("0`)$YE=$)31#H@8W)T:2Y3+'8@,2XT
M(#(P,30O,#,O,#8@,3DZ,#4Z-3D@;6%T="!%>'`@)``D3F5T0E-$.B!C<G1B
M96=I;BYC+'8@,2XQ-R`R,#$X+S$R+S(X(#$X.C$W.C$Q(&-H<FES=&]S($5X
M<"`D`"1.971"4T0Z(&-R=&5N9"Y3+'8@,2XS(#(P,30O,#(O,C<@,38Z-#<Z
M-#@@;6%T="!%>'`@)``D3F5T0E-$.B!C<G1N+E,L=B`Q+C(@,C`Q-"\P,B\R
M-R`Q-CHT-SHT."!M871T($5X<"`D`$$````/9VYU``$````'!`$`+G-H<W1R
M=&%B`"YI;G1E<G``+FYO=&4N;F5T8G-D+FED96YT`"YN;W1E+FYE=&)S9"YP
M87@`+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YR96QA+F1Y;@`N<F5L82YP;'0`
M+FEN:70`+G1E>'0`+F9I;FD`+G)O9&%T80`N96A?9G)A;65?:&1R`"YE:%]F
M<F%M90`N8W1O<G,`+F1T;W)S`"YJ8W(`+F1A=&$N<F5L+G)O`"YF:7AU<``N
M9V]T,@`N9'EN86UI8P`N9V]T`"YD871A`"YS9&%T80`N<V)S<P`N8G-S`"YC
M;VUM96YT`"YG;G4N871T<FEB=71E<P``````````````````````````````
M````````````````````````````"P````$````"`8`!-````30````7````
M```````````!`````````!,````'`````@&``4P```%,````&```````````
M````!``````````F````!P````(!@`%D```!9````!0```````````````0`
M````````-P````4````"`8`!>````7@```,<````!0`````````$````!```
M`#T````+`````@&`!)0```24```&0`````8````!````!````!````!%````
M`P````(!@`K4```*U```!)(```````````````$`````````30````0````"
M`8`/:```#V@```!4````!0`````````$````#````%<````$````0@&`#[P`
M``^\```##`````4````7````!`````P```!A`````0````8!@!+(```2R```
M`"````````````````0`````````9P````$````&`8`2\```$O```#GP````
M```````````0`````````&T````!````!@&`3.```$S@````(```````````
M````!`````````!S`````0````(!@$T```!-````0/@```````````````0`
M````````>P````$````"`8"-^```C?@```$,```````````````$````````
M`(D````!`````@&`CP0``(\$```'U```````````````!`````````"3````
M`0````,!@?Z$``#^A`````@```````````````0`````````F@````$````#
M`8'^C```_HP````(```````````````$`````````*$````!`````P&!_I0`
M`/Z4````!```````````````!`````````"F`````0````,!@?Z8``#^F```
M`!,```````````````@`````````LP````$````#`8'^K```_JP````$````
M```````````$`````````+H````!`````P&!_K```/ZP````8```````````
M````!`````````#`````!@````,!@?\0``#_$````.`````&``````````0`
M```(````R0````$````#`8'_\```__`````0```````````````$````!```
M`%P````!`````P&"`````0`````!!```````````````!`````````#.````
M`0````,!@@$$``$!!`````D```````````````0`````````U`````$````#
M`8(!$``!`1`````(```````````````$`````````-L````(`````P&"`1@`
M`0$8````$```````````````"`````````#A````"`````,!@@$H``$!&```
M`6P```````````````@`````````Y@````$````P```````!`1@````[````
M```````````!`````0```!\````!`````````````0%3```!5@``````````
M`````0````````#O;___]0````````````$"J0```!````````````````$`
K`````````0````,````````````!`KD```#_```````````````!````````
