#!/usr/pkg/bin/perl
# -*- indent-tabs-mode: nil; -*-
# vim:ft=perl:et:sw=4
# $Id$

# L. Marcotte has written a version of alias_manager.pl that is LDAP enabled
# check the contrib. page for more information :
# http://sympa.org/contrib.html

# Sympa - SYsteme de Multi-Postage Automatique
#
# Copyright (c) 1997, 1998, 1999 Institut Pasteur & Christophe Wolfhugel
# Copyright (c) 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2005,
# 2006, 2007, 2008, 2009, 2010, 2011 Comite Reseau des Universites
# Copyright (c) 2011, 2012, 2013, 2014, 2015, 2016, 2017 GIP RENATER
# Copyright 2017, 2018 The Sympa Community. See the AUTHORS.md file at the
# top-level directory of this distribution and at
# <https://github.com/sympa-community/sympa.git>.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

use lib split(/:/, $ENV{SYMPALIB} || ''), '/usr/pkg/sympa/bin';

use strict;
use warnings;
use Getopt::Long;
use Pod::Usage;

use Sympa::Aliases;
use Conf;
use Sympa::Crash;    # Show traceback.
use Sympa::Language;
use Sympa::List;
use Sympa::Log;

my %options;
GetOptions(\%main::options, 'help|h');

if ($main::options{'help'}) {
    pod2usage(0);
}

## Load Sympa.conf
unless (defined Conf::load()) {
    printf STDERR
        "Unable to load sympa configuration, file %s or one of the vhost robot.conf files contain errors. Exiting.\n",
        Conf::get_sympa_conf();
    exit 1;
}

my $log = Sympa::Log->instance;
$log->{level} = $Conf::Conf{'log_level'};
$log->openlog($Conf::Conf{'syslog'}, $Conf::Conf{'log_socket_type'});

my ($operation, $listname, $domain, $file) = @ARGV;

if (($operation !~ /^(add|del)$/) || ($#ARGV < 2)) {
    printf STDERR "Usage: $0 <add|del> <listname> <robot> [<file>]\n";
    exit 2;
}

my $list = Sympa::List->new($listname, $domain, {just_try => 1});
unless ($list) {
    printf STDERR "Unknown list %s@%s\n", $listname, $domain;
    exit 2;
}

my $language = Sympa::Language->instance;
$language->set_lang(Conf::get_robot_conf($domain, 'lang'),
    $Conf::Conf{'lang'}, 'en');

my $aliases = Sympa::Aliases->new('Template', file => $file);
if ($operation eq 'add') {
    exit 127 unless $aliases and defined $aliases->add($list);
} elsif ($operation eq 'del') {
    exit 127 unless $aliases and defined $aliases->del($list);
} else {
    print STDERR "Action $operation not implemented yet\n";
    exit 2;
}
exit 0;

# Moved to: Sympa::Aliases::Tempalte::_already_defined().
#sub already_defined;

__END__

=encoding utf-8

=head1 NAME

alias_manager, alias_manager.pl - Manage Sympa aliases (Obsoleted)

=head1 OBSOLETED

F<alias_manager.pl> was obsoleted as of Sympa 6.2.24, and will be removed
in the future.
To manage aliases based on template, setting "C<alias_manager Template>" in
F<sympa.conf> or F<robot.conf> is recommended.

=head1 SYNOPSIS

S<C<alias_manager.pl> C<add> I<listname> I<domain> [ I<file> ]>

S<C<alias_manager.pl> C<del> I<listname> I<domain> [ I<file> ]>

=head1 DESCRIPTION

Alias_manager is a program that helps in installing aliases for newly
created lists and deleting aliases for closed lists. 

Alias management is performed only if it was setup in F</usr/pkg/etc/sympa/sympa.conf>
(C<sendmail_aliases> configuration parameter).

Administrators using MTA functionalities to manage aliases (i.e.
virtual_regexp and transport_regexp with postfix) can disable alias
management by setting
C<sendmail_aliases> configuration parameter to C<none>.

=head1 OPTIONS

=over 4

=item C<add> I<listname> I<domain> [ I<file> ]

Add the set of aliases for the mailing list I<listname> in the
domain I<domain>.

=item C<del> I<listname> I<domain> [ I<file> ]

Remove the set of aliases for the mailing list I<listname> in the
domain I<domain>.

=back

=head1 FILES

F<$SENDMAIL_ALIASES> sendmail aliases file.

=head1 DOCUMENTATION

The full documentation in HTML formats can be
found in L<https://sympa-community.github.io/manual/>. 

The mailing lists (with web archives) can be accessed at
https://listes.renater.fr/sympa/lists/informatique/sympa.

=head1 HISTORY

This program was originally written by:

=over 4

=item Serge Aumont

ComitE<233> RE<233>seau des UniversitE<233>s

=item Olivier SalaE<252>n

ComitE<233> RE<233>seau des UniversitE<233>s

=back

This manual page was initially written by
JE<233>rE<244>me Marant <jerome.marant@IDEALX.org>
for the Debian GNU/Linux system.

=head1 LICENSE

You may distribute this software under the terms of the GNU General
Public License Version 2.  For more details see F<README> file.

Permission is granted to copy, distribute and/or modify this document
under the terms of the GNU Free Documentation License, Version 1.1 or
any later version published by the Free Software Foundation; with no
Invariant Sections, no Front-Cover Texts and no Back-Cover Texts.  A
copy of the license can be found under
L<http://www.gnu.org/licenses/fdl.html>.

=head1 BUGS

Report bugs to Sympa bug tracker.
See L<http://www.sympa.org/tracking>.

=head1 SEE ALSO

L<Sympa::Aliases::Template>.

=cut
