package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.40.2 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.40.2) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````F!8``#0````PE0````0`!30`(``'
M`"@`'``;``8````T````-````#0```#@````X`````0````$`````P```!0!
M```4`0``%`$``!<````7````!`````$````!````````````````````:(\`
M`&B/```%```````!``$```!HCP``:(\!`&B/`0"T`@``'`,```8```````$`
M`@```'2/``!TCP$`=(\!`/````#P````!@````0````$````+`$``"P!```L
M`0``3````$P````$````!````%#E=&20B0``D(D``)")```$`0``!`$```0`
M```$````+W5S<B]L:6)E>&5C+VQD+F5L9E]S;P``!P````0````!````3F5T
M0E-$``!`#*H[!`````0````#````4&%8```````'````"0````4```!.971"
M4T0``&5A<FUV-FAF`````&$```!K```````````````P````#0``````````
M````.`````X```!``````````"\``````````P```$X```!C`````````!\`
M``!-````!@```!T````4`````````$,```!8````7P`````````7````/```
M`%H```!D`````````&<```!I````+@```!(```!9````10```!4```!>````
M1P````````!*````1@```#4```!=````)0```"D`````````!P```$\`````
M````"P```"L```!J````7````$P````@````4````&@`````````80``````
M```8````+0```$$````````````````````1````,@``````````````8@``
M`"H`````````"0`````````C````.0````````!1````!0```$@````;````
M.P````````````````````````!+`````````#H```!4````90````````!F
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````$P``````````````
M````````````````````````````````(@```"$`````````*``````````*
M````)@```"<````L```````````````,`````````!P````:````-````!D`
M`````````````!``````````,P```#$````W````'@``````````````````
M````````````!````````````````````$(````D````````````````````
M%@`````````_````1`````@````````````````````^`````````%,`````
M`````````%<``````````````#T```!6`````````%(`````````20````\`
M`````````````%4```!;````-@```&``````````````````````````````
M``````"8%@````````,`"P``````<(\!```````#`!$`D@``````````````
M$@````P``````````````!(````4```````````````2````<P,``&0A```(
M````$@`+`!P``````````````"(````K```````````````2````/`,`````
M````````$@```(0#```<D@$``````!``%0`Q```````````````2````D@,`
M`#`L```8````$@`+`#@``````````````!(```!)`P`````````````2````
M0```````````````$@```*T#``#H-@``6````!(`"P"#```````````````2
M````P0,``(22`0``````$``5`$@``````````````!(```"Y`P``')(!````
M```0`!0`6```````````````$@```&<``````````````!(```!M````````
M```````2````=```````````````$@```'X``````````````!(```"(````
M```````````2````P`,``(22`0``````$``5`)$``````````````!(```!0
M`@`````````````2````EP``````````````$@```)X``````````````!(`
M``"L```````````````2````O0``````````````$0```,H`````````````
M`!(```#,`P``Y",``"0````2``L`V0,```@D``!0````$@`+`.4#``!D+0``
MW`8``!(`"P#8```````````````6````ZP``````````````$@```%L#````
M`````````!(```#V```````````````2````_P```("2`0`$````$0`5``<!
M`````````````!(````0`0`````````````2````,@,`````````````$@``
M`!P!`````````````!(````C`0`````````````2````\@,``/1"```$`0``
M$@`+`#8!`````````````!$```!(`0`````````````2````6`$`````````
M````$@```&8!`````````````!(```!K`0`````````````1````>@$`````
M````````$@```/T#```8D@$`!````!$`%`")`0`````````````2````E0$`
M````````````$@```*H!`````````````!(```"U`0`````````````2````
M;@,``(22`0``````$``5`/,"`````````````!(```#(`0`````````````2
M````0`0``)@6````````$@`+`-8!`````````````!(```#>`0``````````
M```B````]@$`````````````$@````("`````````````!(````'!```A)(!
M```````0`!4`#P0``%@D```X!0``$@`+``@"`````````````"(````>`@``
M```````````2````*0(`````````````$@```!P$``#X0P``0`$``!(`"P`Z
M`@`````````````2````)@0``!R2`0``````$``5`#($``"0(```7````!(`
M"P`Z!```"$8``+`"```2``L`00(`````````````$@```#\$``"8%@``````
M`!(`"P!(`@`````````````2````1P0``&PA``!X`@``$@`+`%0$``!(+```
M'`$``!(`"P!/`@`````````````2````9@0``)`I``"4````$@`+`%@"````
M`````````!(```!I`@`````````````2````W`(`````````````$@```'`"
M`````````````!(```!T!```)"H``&0````2``L`?P(`````````````$@``
M`(@"`````````````!(```".`@`````````````2````)0,`````````````
M$@```),"`````````````!(````8`P``1)(!``0````1`!4`FP(`````````
M````$@```%`#`````````````!(```"B`@`````````````2````?`0``)A"
M``!<````$@`+`*D"`````````````!(```"X`@`````````````1````#0,`
M`-21`0`$````$0`4`(4$``!`-```J`(``!(`"P`O`0`````````````2````
MO0(`````````````$@```&(#`````````````!(`````;&EB<&5R;"YS;P!R
M96%L;&]C`&UE;6UO=F4`7U]C>&%?9FEN86QI>F4`<FUD:7(`<W1R8VAR`'-T
M<FYC;7``<W1R<F-H<@!B;V]T7T1Y;F%,;V%D97(`<&5R;%]C;VYS=')U8W0`
M86)O<G0`=6YL:6YK`%]?;'-T870U,`!P97)L7V9R964`7U]S=&%T-3``7V5X
M:70`;65M8W!Y`'!E<FQ?9&5S=')U8W0`7U]S=&%C:U]C:&M?9F%I;`!03%]D
M;U]U;F1U;7``4&5R;%]S=E]S971P=@!03%]C=7)R96YT7V-O;G1E>'0`<&5R
M;%]A;&QO8P!C;&]S961I<@!E;G9I<F]N`')E861L:6YK`%]?;W!E;F1I<C,P
M`&UE;7-E=`!097)L7W-A9F5S>7-M86QL;V,`7U]S=&%C:U]C:&M?9W5A<F0`
M4&5R;%]G=E]F971C:'!V`%]?<W!R:6YT9E]C:&L`<F5A9`!03%]M96UO<GE?
M=W)A<`!097)L7W-Y<U]I;FET,P!097)L7V=E=%]S=@!097)L7V-R;V%K7VYO
M8V]N=&5X=`!097)L7VYE=UA3`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y
M<U]T97)M`%]?97)R;F\`7U]D97)E9VES=&5R7V9R86UE7VEN9F\`4&5R;%]M
M9U]S970`;6MD:7(`7U]R96=I<W1E<E]F<F%M95]I;F9O`'!E<FQ?<&%R<V4`
M4&5R;%]N97=84U]F;&%G<P!A8V-E<W,`9V5T<&ED`&UE;6-H<@!V9G!R:6YT
M9@!097)L7V%T9F]R:U]L;V-K`&UE;6-M<`!097)L7W-V7W-E='!V;@!P97)L
M7W)U;@!L<V5E:P!O<&5N`'-P<FEN=&8`9V5T=6ED`'-T<FQE;@!?7W1L<U]G
M971?861D<@!?7W-&`'-T<F-M<`!L:6)M+G-O+C``;&EB8W)Y<'0N<V\N,0!S
M=')D=7``;&EB<'1H<F5A9"YS;RXQ`'!T:')E861?871F;W)K`&QI8F,N<V\N
M,3(`7U]P<F]G;F%M90!?7W!S7W-T<FEN9W,`7U]G971P=W5I9#4P`%]?<WES
M8V%L;`!?7W-T<&-P>5]C:&L`871E>&ET`%]L:6)C7VEN:70`<W1R=&]K`%]?
M<F5A9&1I<C,P`%]E;F0`<&%R7V-U<G)E;G1?97AE8P!?7V)S<U]S=&%R=%]?
M`&=E=%]U<V5R;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%]E9&%T
M80!?7V)S<U]E;F1?7P!P87)?8F%S96YA;64`<&%R7V1I<FYA;64`<&%R7VUK
M=&UP9&ER`'-H85]U<&1A=&4`;7E?<&%R7W!L`%]?96YD7U\`<&%R7VEN:71?
M96YV`'-H85]F:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`&UA:6X`7U]S=&%R
M=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A
M;@!P87)?9&EE`'-H85]I;FET`%A37TEN=&5R;F%L<U]005)?0D]/5``O=7-R
M+W!K9R]L:6(O<&5R;#4O-2XT,"XP+V5A<FUV-FAF+6YE=&)S9"UT:')E860M
M;75L=&DO0T]213HO=7-R+W!K9R]L:6(`:(\!`!<```!LCP$`%P```'B1`0`7
M````B)$!`!<```"8D0$`%P```*21`0`7````K)$!`!<```"\D0$`%P```,21
M`0`7````S)$!`!<```#0D0$`%P```-21`0`7````V)$!`!<```#HD0$`%P``
M`.R1`0`7````])$!`!<```#XD0$`%P```/R1`0`7`````)(!`!<````$D@$`
M%P````R2`0`7````$)(!`!<````8D@$`%P```'R1`0`5!P``A)$!`!43``",
MD0$`%2$``)"1`0`1)@``E)$!`!(F``"@D0$`%3$``*B1`0`5-0``L)$!`!4[
M``"TD0$`%4$``+B1`0`51@``P)$!`!55``#(D0$`%64``'"0`0`6`P``=)`!
M`!8$``!XD`$`%@4``'R0`0`6!P``@)`!`!8(``"$D`$`%@D``(B0`0`6"P``
MC)`!`!8-``"0D`$`%@X``)20`0`6#P``F)`!`!81``"<D`$`%A4``*"0`0`6
M%@``I)`!`!87``"HD`$`%A@``*R0`0`6&0``L)`!`!8:``"TD`$`%AP``+B0
M`0`6'0``O)`!`!8>``#`D`$`%A\``,20`0`6(```R)`!`!8B``#,D`$`%B<`
M`-"0`0`6*```U)`!`!8I``#8D`$`%BL``-R0`0`6+```X)`!`!8M``#DD`$`
M%BX``.B0`0`6+P``[)`!`!8R``#PD`$`%C,``/20`0`6-```^)`!`!8V``#\
MD`$`%C@```"1`0`6.0``!)$!`!8Z```(D0$`%CT```R1`0`6/@``$)$!`!9`
M```4D0$`%D$``!B1`0`60@``')$!`!9#```@D0$`%D8``"21`0`61P``*)$!
M`!9(```LD0$`%DH``#"1`0`63@``-)$!`!90```XD0$`%E,``#R1`0`65@``
M0)$!`!97``!$D0$`%E@``$B1`0`66@``3)$!`!9;``!0D0$`%EP``%21`0`6
M70``6)$!`!9>``!<D0$`%F```&"1`0`680``9)$!`!9B``!HD0$`%F0``&R1
M`0`6:```<)$!`!9I``!TD0$`%FH```3@+>4$X)_E#N"/X`CPON7H?`$``,:/
MXA?*C.+H_+SE`,:/XA?*C.+@_+SE`,:/XA?*C.+8_+SE`,:/XA?*C.+0_+SE
M`,:/XA?*C.+(_+SE`,:/XA?*C.+`_+SE`,:/XA?*C.*X_+SE`,:/XA?*C.*P
M_+SE`,:/XA?*C.*H_+SE`,:/XA?*C.*@_+SE`,:/XA?*C.*8_+SE`,:/XA?*
MC.*0_+SE`,:/XA?*C.*(_+SE`,:/XA?*C.*`_+SE`,:/XA?*C.)X_+SE`,:/
MXA?*C.)P_+SE`,:/XA?*C.)H_+SE`,:/XA?*C.)@_+SE`,:/XA?*C.)8_+SE
M`,:/XA?*C.)0_+SE`,:/XA?*C.)(_+SE`,:/XA?*C.)`_+SE`,:/XA?*C.(X
M_+SE`,:/XA?*C.(P_+SE`,:/XA?*C.(H_+SE`,:/XA?*C.(@_+SE`,:/XA?*
MC.(8_+SE`,:/XA?*C.(0_+SE`,:/XA?*C.((_+SE`,:/XA?*C.(`_+SE`,:/
MXA?*C.+X^[SE`,:/XA?*C.+P^[SE`,:/XA?*C.+H^[SE`,:/XA?*C.+@^[SE
M`,:/XA?*C.+8^[SE`,:/XA?*C.+0^[SE`,:/XA?*C.+(^[SE`,:/XA?*C.+`
M^[SE`,:/XA?*C.*X^[SE`,:/XA?*C.*P^[SE`,:/XA?*C.*H^[SE`,:/XA?*
MC.*@^[SE`,:/XA?*C.*8^[SE`,:/XA?*C.*0^[SE`,:/XA?*C.*(^[SE`,:/
MXA?*C.*`^[SE`,:/XA?*C.)X^[SE`,:/XA?*C.)P^[SE`,:/XA?*C.)H^[SE
M`,:/XA?*C.)@^[SE`,:/XA?*C.)8^[SE`,:/XA?*C.)0^[SE`,:/XA?*C.)(
M^[SE`,:/XA?*C.)`^[SE`,:/XA?*C.(X^[SE`,:/XA?*C.(P^[SE`,:/XA?*
MC.(H^[SE`,:/XA?*C.(@^[SE`,:/XA?*C.(8^[SE`,:/XA?*C.(0^[SE`,:/
MXA?*C.((^[SE`,:/XA?*C.(`^[SE`,:/XA?*C.+X^KSE`,:/XA?*C.+P^KSE
M`,:/XA?*C.+H^KSE`,:/XA?*C.+@^KSE`!"@X0(`H.$'T,WC-```ZD@@G^5(
M$)_E`B"/X$0`G^4!,)+G``"/X`,`4.$>_R\A<$`MZ0!`H.,`4*#A`6!#X@!@
M1N`F8:#A!#"5Y#/_+^$&`%3A`4"$XOK__QIP@+WHK'D!`!0!``"H>`$`\$$M
MZ6`PG^5@$)_E8""?Y0,PC^`!4)/G`G"3YP<`5>'P@;TH`7!'X@5P1^``0*#C
M"%"%XJ=QH.$'``#J"&`5Y0A0A>(`,);E,_\OX0<`5.$``(;E`4"$XO"!O0@$
M,%7EH`!3X_3__PHG___K2'D!`#@!```<`0``@$`MZ6R!G^4`4%'B"("/X`C0
M3>)+```*7"&?Y5PQG^4"()CG`%""Y0-@F.<(()7E`#"5Y0`@AN4`()/E``!2
MXSH```HX$9_E`1"8YP`@@>4`()/E`##2Y0``4^,%```*`2""XB\`4^,`(($%
M`3#2Y```4^/Z__\:``!0XP````KW_O_KDO__Z_@PG^4#,)CG``!3XS4```KL
M,)_E['"?Y0.0F.<'<(_@"0!7X0&023('D$DP`$"@,RF1H#$$```J!#"7Y#/_
M+^$)`%3A`4"$XOK__QJX`)_EN'"?Y0``C^#?_O_KL#"?Y0=PC^`#@)CG"`!7
MX0&`2#('@$@P`$"@,RB!H#$$```J!#"7Y#/_+^$$`%CA`4"$XOK__QH`();E
M`!"5Y00`E>57"P#KM/[_ZTPPG^5D()_E`S"8YP(@C^``((/ER___ZA/`H.-0
M,)_E`B"@XP0`H.,`$*#C`S"/X`#`C>7Z_O_K`0"@X]?^_^N$___KQ___ZM1X
M`0!@`0``-`$``&@!``!8`0``)`$``$!W`0!(_O___'8!`$`!``"`>0$`V"\`
M`,PPG^7P1RWI`)!0X@,PC^`K```*O""?Y0(PD^<`@)/E``!8XR@```H`4-GE
M`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"8Y0``5.,,```*"&"@X00`H.$%
M(*#A"1"@X9+^_^L``%#C`@``&@4PU.<]`%/C"P``"@1`MN4``%3C\___&@0`
MH.'PA[WH"5"@X0$P]>4``%/C/0!3$_O__QH)4$7@YO__Z@A@1N`!4(7B1F&@
MX05`A.`$`*#A`&"'Y?"'O>@)0*#A[?__Z@A`H.'K___J+'<!`#0!``#P0RWI
M7"&?Y5PQG^4"((_@`S"2YZ303>(`4%#B`#"3Y9PPC>4`,*#C`@``"@`PU>4`
M`%/C"@``&C`AG^4H,9_E`B"/X`,PDN<`()/EG#"=Y0(@,^``,*#C00``&J30
MC>+P@[WHEO[_ZP!`4.+Q__\*#7"@X?B`G^4(@(_@!`"@X0'__^L``%#C)P``
M"@TPT.4-8(#B+@!3XP(``!H!,-;E``!3X_3__PH-,-#E+@!3XR(```H%`*#A
MY_[_ZP"0H.$&`*#AY/[_ZP``B>`"`(#BY_[_ZP8PH.$`8*#A!2"@X0@0H.'3
M_O_K!Q"@X08`H.%2_O_K`0!PXP8`H.$#```*"#"=Y0\Z`^(!"5/C$@``"D'^
M_^L&`*#A-O[_ZP0`H.'9_O_K``!0X]?__QH$`*#A7?[_ZP4`H.$<_O_KO?__
MZ@$PUN4N`%/CV?__&@(PUN4``%/CR/__"M7__^JG___KZ___ZD3^_^M0=@$`
M/`$``!QV`0"$+@``$$`MZ8P@G^6,,)_E`B"/X*#03>(#,)+G#1"@X0`PD^6<
M,(WE`#"@XP!`H.$?_O_K``!0XP``H!,$```:"#"=Y0\Z`^(*"E/C`0E3$PH`
M``I(()_E0#"?Y0(@C^`#,)+G`""3Y9PPG>4"(#/@`#"@XP<``!J@T(WB$("]
MZ`(0H.,$`*#A;/[_ZQ`/;^&@`J#A[O__ZAK^_^O8=`$`/`$``(QT`0`00"WI
M],"?Y?0PG^4!VTWB"-!-X@S`C^`#,)SG!$"-X@`0H.$!*Z#C!`"@X0`PD^4$
M-(WE`#"@X]C]_^L!`$#B!`!0X0<``)H`(*#C`@``Z@0`4.$!(,#E`@``"@$P
M4.0O`%/C^?__"B\0H.,$`*#AUOW_ZP``4.,0```*!`!0X0`PH(,`,,"%!`"@
MX5#^_^MP()_E:#"?Y0(@C^`#,)+G`""3Y00TG>4"(#/@`#"@XQ$``!H!VXWB
M"-"-XA"`O>A$()_E.#"?Y0(@C^`#,)+G`""3Y00TG>4"(#/@`#"@XP4``!HD
M`)_E``"/X`';C>((T(WB$$"]Z#7^_^K7_?_K+'0!`#P!``"H<P$`>',!`,@K
M``#P0"WI`&"@X?@`G^7X,)_E``"/X`';3>(#,)#G#-!-XN@`G^4$<(WB`5"@
MX>`@G^4'$*#A``"/X``PD^4$-(WE`#"@X\_]_^L`P%#B%P``V@%`3.(!.X3B
M"#"#X@,PC>`$-%/E``!3XR,``!H``%3C#@``"@0@H.$O$*#C!P"@X07^_^L`
M`%#C"```"H0`G^4``(_@)_[_ZP0PH.$'(*#A!A"@X0``D.4'_O_K!@``ZF0`
MG^4``(_@'O[_ZP4@H.$&$*#A``"0Y:+]_^M,()_E-#"?Y0(@C^`#,)+G`""3
MY00TG>4"(#/@`#"@XP0``!H!VXWB#-"-XO"`O>@,0*#AV___ZI#]_^L<<P$`
M/`$``*`K``#_`P``R',!`*1S`0!8<@$`\$<MZ0%`H.$4(I_E%#*?Y0(@C^`#
M,)+G"-!-X@`PD^4$,(WE`#"@XP`PT>4`4*#A/0!3XP%`@0($`*#A\?W_ZPT0
MH.$`8*#A!0"@X9_^_^O4<9_E`)!0X@=PC^!!```*Z/W_ZP``5N$X``":O#&?
MY0"`G>4#<)?G"(&@X0`PU>4``%/C/0!3$P$PH!,`,*`#!0``"@4PH.$!(//E
M``!2XST`4A/[__\:!3!#X`(`AN(#`(#@`&"7Y=C]_^L``%#C"`"&YQ8```H%
M`*#A`#"7Y0`@G>4"$9/G`3#0Y```4^,]`%,3`##!Y08```H!(('B`3#0Y`(0
MH.$``%/C/0!3$P$PPN3Y__\:/3"@XP$@H.$`,,'E`3#4Y```4^,!,.+E^___
M&@@AG^7X,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C-0``&@C0C>+PA[WH
M`9!)X@$PU.0``%/C`3#IY?O__QKN___JP#"?Y0-PE^<`@)?E`#"8Y0``4^,D
M```*"#"@X00@L^4)$*#A``!2XP&0B>+Z__\:`Q"!X@$1H.&0,)_E`S"/X``@
MD^4``%+C#P``&@$@H.,!`*#A`""#Y9;]_^L`H%#BU?__"@@0H.$)@:#A"""@
MX=G\_^L`H(?E`""@XP$PB>(#(8KG`)"-Y:'__^H(`*#ASOS_ZP``4.,`H*#A
M``"'Y<7__PH)@:#A\O__Z@.0H.$($*#CW___Z@#]_^OX<0$`/`$``+!Q`0`T
M`0``W'`!`'!R`0!P0"WI/""?Y3P`G^4\,)_E`B"/X```C^`#8)+G;?W_ZRQ0
MG^4L0)_E!5"/X`1`C^`%,*#A!B"@X000H.%P0+WH``"0Y1C]_^J\;P$`Y'`!
M`"`!``"\:```7"@``#!`+>E8`)_E%-!-X@``C^!9_?_K`$"@X0``D.7@___K
M`,"@XSQ0G^4`$)3E.""?Y050C^`T,)_E`B"5YPP0C>4L$)_E`S"/X`P`G>4!
M$(_@`#"-Y03`C>47_?_K%-"-XC"`O>B0<`$`0&\!`&P!``#(*@``_"<`````
MH.,>_R_A\$\MZ3PBG^4\,I_E2]Y-X@S03>("((_@`%"@X2P"G^4#,)+G&&"-
MX@%PH.$``(_@!!!&X@`PD^6T-(WE`#"@X]S]_^LO$*#C`$"@X04`H.&!_/_K
M``!0XQ````KP`9_E!1"@X0``C^`>___K!0"@X>`AG^70,9_E`B"/X`,PDN<`
M()/EM#2=Y0(@,^``,*#C:P``&DO>C>(,T(WB\(^]Z`<`H.'V_/_KK!&?Y0$0
MC^"?_/_K`*!0XN?__PH`,-KEF)&?Y9B!G^68<9_E``!3XPF0C^`(@(_@!W"/
MX#T```H``%3C!```"@H0H.$$`*#A!_W_ZP``4.,M```*"@"@X?K\_^L`,*#A
M`2!`X@(@BN`"`%KA`+"@X0,PBC```*`S`1"*,@,``#H(``#J`P!1X0``P^4"
M```*`2!SY2\`4N/Y__\*"@"@X>C\_^L`L*#A!0"@X>7\_^L+`(#@`0"`X@$+
M4..\__\J_!"?Y;2PC>(!$(_@!S"@X0$KH.,B`(WI"P"@X0`0H.,`H(WE@/S_
MZP80H.$+`*#A3?S_ZP``4.,#```:(#"=Y0\Z`^(""5/C%```"K00G^4``*#C
M`1"/X%O\_^L`H%#BH___"@`PVN4``%/CP?__&@``5.,5```*`##4Y2X`4^,"
M```*`;"@XPF@H.'5___J`3#4Y0``4^/K__\*^/__Z@$0H.,+`*#AC/S_ZP``
M4./E__\:4`"?Y0L0H.$``(_@KO[_ZPL`H.&3_/_KC?__Z@&PH.,(H*#APO__
MZC'\_^O<;@$`/`$``+0G``"()P``>&X!`%`G``"0)@``C"8``"@G``#L)0``
M0"8``,@E```00"WI+Q"@XP!`H.']^__K``!0XP%`@!($`*#A$$"]Z'G\_^H0
M0"WI`$!0X@D```H`,-3E``!3XP8```J-_/_K`0"`X@$+4.,&``"*!`"@X1!`
MO>CW_?_J$`"?Y1!`O>@``(_@:/S_Z@``H.,0@+WHB"0``'!`+>G$))_EQ#2?
MY0(@C^#`5)_E`S"2Y[QDG^4(T$WB!5"/X``PD^4$,(WE`#"@XP9@C^`-0*#A
M!!"@X08`H.$A_?_K``!0XPX```J,-)_E`!"=Y0,PE><`()/E`3&"X`0`D^4`
M`%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^I49)_E!F"/X`00
MH.$&`*#A"_W_ZP``4.,.```*-#2?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0
MXP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J`&2?Y09@C^`$$*#A
M!@"@X?7\_^L``%#C#@``"MPSG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!
M`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZJQCG^4&8(_@!!"@X08`
MH.'?_/_K``!0XPX```J$,Y_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"
MY_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^I88Y_E!F"/X`00H.$&`*#A
MR?S_ZP``4.,.```*+#.?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q
M__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J!&.?Y09@C^`$$*#A!@"@X;/\
M_^L``%#C#@``"M0RG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__
M"@0P@^($()/E``!2XP0@@^3[__\:Z___ZK!BG^4&8(_@!!"@X08`H.&=_/_K
M``!0XPX```I\,I_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$
M,(/B!""3Y0``4N,$((/D^___&NO__^I<8I_E!F"/X`00H.$&`*#AA_S_ZP``
M4.,.```*)#*?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#
MX@0@D^4``%+C!""#Y/O__QKK___J"&*?Y09@C^`$$*#A!@"@X7'\_^L``%#C
M#@``"LPQG^4`$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($
M()/E``!2XP0@@^3[__\:Z___ZK1AG^4&8(_@!!"@X08`H.%;_/_K``!0XPX`
M``IT,9_E`!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3
MY0``4N,$((/D^___&NO__^I@89_E!F"/X`00H.$&`*#A1?S_ZP``4.,.```*
M'#&?Y0`0G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4`
M`%+C!""#Y/O__QKK___J#`&?Y000H.$``(_@,/S_ZP`04.("```*^`"?Y0``
MC^!W_?_K\`"?Y000H.$``(_@)_S_ZP`04.("```*W`"?Y0``C^!N_?_KU`"?
MY000H.$``(_@'OS_ZP`04.(2```*P`"?Y0``C^!E_?_KN""?Y60PG^4"((_@
M`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?Y9@`G^4!$(_@``"/X`C0C>)P
M0+WH5?W_ZH0`G^4$$*#A``"/X`7\_^L`$%#BZ?__"G``G^4``(_@3/W_Z^7_
M_^K4^O_K^&L!`#P!``#D:P$`["0``#0!``"<)```3"0````D``"P(P``:",`
M`+`B``#$(@``>"(``"PB``!<(0``A"$``%@A``!T(0``0"```&0A``"8(```
MF&<!`$`A``"P(```$"$``*`@```$X"WE>""?Y7@PG^4"((_@=`"?Y0S03>(#
M,)+G#1"@X0``C^``,)/E!#"-Y0`PH./7^__K``!0XP4```H``-#E``!0XS``
M4!,!`*`#``"@$P$`(.(T()_E*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXP$``!H,T(WB!/"=Y)GZ_^O`9@$`/`$```@@``!P9@$`#P`MZ03@+>5$()_E
M1#"?Y0(@C^`#,)+G#-!-X@`PD^4$,(WE`#"@XRPPG^44((WB*`"?Y0,PC^``
M((WE``"3YQ`0G>6P`(#BV?K_Z_\`H.-!^O_K*&8!`#P!```$9@$`9`$```(0
MH./P3RWI?$&?Y7SAG^4!V$WB'-!-X@1`C^`.X)3G`<B-X@`@H.,`,*#C`.">
MY13@C.4`X*#C`!"-Y13`C.(`D*#AT/K_ZP%@4.(`$,'BP2^@X<$_)N`",%/@
M<S#_Y@(P(^`",%/@`B#"X`-@5N`"@,'@``!8XSP``+H0L9_E$'&?Y1"AG^4+
ML(_@#%"-X@=PC^``$*#C!B"@X0@PH.$)`*#A`!"-Y;?Z_^L!`''C`0!P`S``
M``H*(*#A!1"@X0D`H.%N^O_K`0!PXQ\```H'`%#C(```F@@`0.(`0)7@`P``
M.AP``.H!0$3B!0!4X1D``#H((*#C!Q"@X00`H.&5^O_K``!0X_;__QH%$$3@
M!@"1X,$?J."`P)_E;""?Y0S`C^`!.(WB`B"<YQ0P@^(`P)+E`""3Y0S`,N``
M(*#C$```&@'8C>(<T(WB\(^]Z$P`G^4``(_@C___ZP%H5N(`@,CB`0!XXP$(
M=@/(__\:``#@XP`0H.'F___J"P"@X87__^O+___J&?K_Z[QE`0`\`0``'!\`
M`+QF`0`(``$`M&0!`&`>```00"WI@OK_ZWOZ_^L``%#C``"0%1"`O>CP0"WI
M\""?Y?!@G^7P,)_E`B"/X!303>(#,)+G!F"/X`!0H.$&`*#A"!"-X@`PD^4,
M,(WE`#"@XR?[_^L`0%#B`@``"@`PU.4``%/C#@``&K`@G^6H,)_E`B"/X`,P
MDN<`()/E##"=Y0(@,^``,*#C(```&@40H.&,`)_E``"/X!30C>+P0+WH7_S_
MZ@4`H.%?^O_K`'"@X00`H.%<^O_K``"'X`(`@.)?^O_K7#"?Y5P0G^4%(*#A
M`$"-Y0,PC^`!$(_@`$"@X4CZ_^M$()_E+#"?Y0(@C^`#,)+G`""3Y0PPG>4"
M(#/@`#"@XP00H`$&`*`!X?__"LWY_^L$9`$`Y!T``#P!``"\8P$`A!T``&0<
M``!D'```0&,!`/!/+>E$%I_E1,:?Y40FG^4!$(_@`B"1YTG?3>(,P(_@##",
MXAA0C.(@<(WB.."-X@`@DN4<(8WE`""@XP"0H.$'`)/H!P"'Z`\`M>@/`*[H
M`P"5Z##`C.(L0(WB`P".Z`<`G.@'`(3H[`6?Y1Q@C>(&$*#A``"/X-#Z_^O<
MI9_E`%!0X@J@C^`"```*`##5Y0``4^,-`0`:B___ZP!04.+Q```*!0"@X1+Z
M_^N``*#A`0"`XA7Z_^L`(-7E`("@X0``4N,)```*"$"@X9"UG^4+L(_@!`"@
MX0L0H.'\^?_K`2#UY0)`A.(``%+C^/__&G`%G^4\4(WB``"/X`(``.H$`)7D
M``!0XPT```H&$*#AJOK_ZP!`4.+X__\*`##4Y0``4^/U__\*.?O_ZP``4./R
M__\*!`"@X=/Y_^L`0%#B$P``&@!0H.,2/HWB!3&#X`!!$^4``%3C#```"@&P
MA>(+L8?@`##4Y0``4^,'```*!`"@X2;[_^L``%#C`5"%XC<``!H$0)OD``!4
MX_3__QH`0*#C!`"@X=;Y_^L`4*#A"`"@X=/Y_^L`4(7@`5N%X@10A>(%`*#A
MU/G_ZP!PH.&D-)_EI!2?Y0,PC^``,(WEG#2?Y00@H.$#,(_@`1"/X`2`C>6Z
M^?_K!QV@XP<`H.&*^?_K`0!PXX<```H'`*#A4!"-XB[Y_^L`0%#B&0``"F`D
MG^5@%)_E`@":YP<PH.$`()GE`1"/X+``@.(P^?_K`%"@XT0DG^48-)_E`B"/
MX`,PDN<`()/E'#&=Y0(@,^``,*#C_```&@4`H.%)WXWB\(^]Z`0`H.&'^?_K
M`$!0XK3__PK&___J6#"=Y0\Z`^(!"5/CX?__&FR`G>63^?_K``!8X=W__QI8
M,)WE@SN@X:,[H.$'#5/CV/__&@4`H.&6^?_K`%"@X<0#G^4&$*#A``"/X#WZ
M_^L`$%#BE```"@``F>5+_/_K`(!0XI````H(`*#A@_G_ZP,`4.,`8*#A#```
MFHP3G^4$`$#B``"(X`$0C^"$^?_K``!0XP4``!H$`%;CO0``"@9@B.`%,%;E
M+P!3X[D```H__O_K`&!0X@0``!H(`*#A!A"@X5[Y_^L`@%#B>```&CP3G^4\
M`Y_E`1"/X```C^!D^__K/OG_ZP#`H.$H,Y_E*!.?Y0,PC^`$,(WE(#.?Y2"3
MG^4@@Y_E(&.?Y0<@H.$%`*#A`S"/X`$0C^``P(WE"9"/X`B`C^`&8(_@2OG_
MZPT``.H2^?_K`#"0Y1$`4^,.```:)OG_ZP#`H.$!0(3B"3"@X0<@H.$($*#A
M!0"@X5``C>D`P(WE._G_ZP<=H.,%`*#A"_G_ZP$`<./L__\*!P"@X:3X_^ND
M`I_E!1"@X0``C^`W^__K!0"@X;+^_^N%___J]OC_ZP`PD.41`%/C<___"@`@
MF>4`4*#C%""-Y>_X_^L`P*#A,!*?Y0#`G.4!`)KG7!*?Y0<PH.$4()WE`1"/
MX`#`C>6P`(#BHOC_ZW'__^H%L*#A$CZ-X@LQ@^#T`!/E``!0XR````H!@(OB
M"%&$X`80H.'*^?_K`#!0X@BPH.$"```*`##3Y0``4^,$```:!`"5Y```4.,3
M```*`8"(XO+__^KQ^/_K`%!0XNC__PKT_O_J@_[_Z]@AG^5X,9_E`B"/X`,P
MDN<`()/E'#&=Y0(@,^``,*#C5```&@4`H.%)WXWB\$^]Z.#X_^H-`*#C__C_
MZZ!1G^53(*#C`("@X050C^#H_O_J`("9Y6S__^K\_?_K`*!1X@"0H.&"__^Z
M"B!0X@`PRN(`8(WE"`"@X>A@C>+6^/_K!B"@XP80H.$(`*#AD/C_ZP8`4.,"
M```*2`&?Y0``C^#1_?_K0#&?Y>@@G>4#`%+A;___&K0@UN$P,9_E`P!2X6O_
M_QH`$*#C,B!9X@`PRN((`*#A`!"-Y?!`C>*^^/_K*""@XP@`H.$$$*#A>/C_
MZR@`4.,"```*]`"?Y0``C^"Y_?_K`,"@X^@@G^7H,)_EZ!"?Y0(@C^`$((WE
M!0"@X0<@H.$`0(WE`S"/X`$0C^`8P<WEL?C_ZWG__^H$8)GE``!6XT+__PH&
M`*#AM/C_ZP,`4.,^__^:I!"?Y00`0.(``(;@`1"/X+;X_^L``%#C!H"@`3;_
M_^HQ^/_KZ&(!`%1D`0`\`0``=!L``'1B`0`L'```=!H``$@;```H&P``C!D`
M`&0!```P&P``O&`!`.@:``"P&@``=!D````9```H&P``3!H``-@7``#,%P``
M4!H``/0:``#@%P``P!@``!Q>`0#8%P``4!@```!#04-(10``\!<``*P8``!H
M%0``S!<``(`7``#P0"WI3"*?Y4PRG^4"((_@2`*?Y0,PDN<,T$WB``"/X``P
MD^4$,(WE`#"@XWWX_^L`,*#A#6"@X0`PD^4@`I_E9"#3Y080H.$``(_@92##
MY2/Y_^L``%#C`@``"@`PT.4Q`%/C8P``"NK[_^OT`9_E]$&?Y0``C^!J^/_K
M!$"/X`00H.$#,*#C`2"@XP!0H.$``)#E!OC_ZP!`4.(*```*##"4Y<0AG^4`
M$)/E``"5Y0(@C^#C]__K##"4Y0`0D^4(,)'E`043XU8``!J@`9_EH$&?Y0``
MC^!2^/_K!$"/X`00H.$#,*#C`2"@XP!0H.$``)#E[O?_ZP!`4.()```*=`&?
MY080H.$``(_@]/C_ZP`@4.([```*##"4Y0``E>4`$)/EQ_?_ZU`!G^50<9_E
M``"/X#OX_^M($9_E`%"0Y0$@H.,`0*#A`1"/X`4`H.'C]__K,"&?Y0`0H.$"
M((_@!0"@X;?W_^L`(*#C`#"4Y1@!G^4&$*#A``"/X&4@P^77^/_K!W"/X```
M4.,(`(?E$```"@<=H./I]__K`3"`XA$`4.,!`%,3(@``BN`@G^6D,)_E`B"/
MX`,PDN<`()/E!#"=Y0(@,^``,*#C&```&@S0C>+P@+WH!`"7Y=#]_^L``%#C
M"`"'Y>G__QJD`)_E``"/X+WW_^L!,-#E``!3XYG__PJ7___JC#"?Y0P@E.4#
M,(_@!#"3Y0``DN4`$)/ELOG_Z[___^H``)7EP??_ZZ7__^J`]__K8""?Y0`P
MH.%<`)_E`B"/X`8`DND``(_@`!"1Y:7W_^L07`$`/`$``"Q=`0`4%0``U%P!
M`*@6``!X%@``=%P!`%06```0%```&%P!`*1<`0#L%0``\!4``)03``!H6@$`
M9!4``!A<`0#D6P$`+!4``!!`+>D`0*#AIOS_ZP``5.,``%`3$("]"``PU.4`
M`%/C$("]"`0`H.$\^__K,OO_ZQP0G^4$(*#C`1"/X"KW_^L``%#C$("]&`0`
MH.$00+WHL/C_ZE03``#P3RWI4]]-X@T@H.$H&Y_E*#N?Y0$0C^`#,)'G`L"@
MX0`PD^5$,8WE`#"@XQP0@.)<X(#B!#"1Y#,_O^8!`%[A!#"LY?K__QH!3(WB
M`A"@X3@PD>4DX)'E#!"1Y03`LN4.,"/@`3`CX`PP(^#C/Z#A!`!2X4`P@N7S
M__\:P#J?Y0@@G>4,8)#E!!"=Y1!0D.4#((+@`%"0Z0!`D.4&<(+@`Q"!X`8@
M+N`%$('@#"`"X`8@(N`,@)WEY!V!X`(0@>!L(2[@9)&@X0.`B.`$(`+@#K"(
MX`X@(N!L@2G@!R""X`&`".`0<)WE;($HX`-PA^`+@(C@%+"=Y>$M@N!LH8?@
M87$IX.*-B.`"<`?@`["+X&(AH.$)L(O@"7`GX&&1(N`(D`G@"G"'X&&1*>`8
MH)WEZ'V'X`N0B>!HL2+@`Z"*X`>P"^#GG8G@8:&*X&<1H.$"<"O@"G"'X!R@
MG>5HL2'@`Z"*X`(@BN`@H)WE";`+X`.@BN#I?8?@:*&*X&B!*^!IL2'@`H"(
MX`>P"^#GC8C@9R&@X0%P*^`*<(?@)*"=Y6FQ(N`#H(K@`:"*X"@0G>4(L`O@
M`Q"!X&D1@>!ID2O@Z'V'X`J0B>!HH2+@!Z`*X`*@*N`!$(K@+*"=Y>>=B>`#
MH(K@9W&@X0*@BN`P()WE:+$GX`FP"^`#((+@:"&"X&B!*^#I'8'@"H"(X&FA
M)^`!H`K@!Z`JX`(@BN`TH)WEX8V(X`.@BN!A$:#A!["*X#APG>5IH2'@`W"'
MX`B@"N#H+8+@:7&'X&FA*N!HD2'@`I`)X`&0*>`'<(G@/)"=Y0N@BN`#D(G@
MXJV*X`&PB>!B(:#A0!"=Y6B1(N`#$('@"I`)X.I]A^!H$8'@:)$IX&J!(N`'
M@`C@`H`HX`&`B.!$$)WE"Y")X`,0@>#GG8G@`K"!X&=QH.%(()WE:A$GX`,@
M@N`)$`'@Z8V(X&HA@N!J$2'@::$GX`B@"N`'H"K@`B"*X$R@G>4+$('@`Z"*
MX`=PBN!0H)WEZ!V!X&B!H.$#,(K@::$HX`&@"N!IH2K@X2V"X`>@BN!A<2C@
M`G`'X&DQ@^`(<"?@6+"=Y0-PA^`0.)_E5)"=Y>*MBN`#L(O@8B&@X0.0B>!A
ML8O@81$BX`H0(>`(@(G@ZGV'X`&`B.#GG8C@7("=Y0.`B.`"@(C@:B$BX`<@
M(N`"((O@Z1V"X&`@G>5G<:#A`R""X&HA@N!JH2?@":`JX`J`B.!DH)WEX8V(
MX`.@BN`'H(K@:7$GX`%P)^`'<(+@:""=Y6$1H.$#((+@:2&"X&F1(>`(D"G@
M"9"*X&R@G>7H?8?@`Z"*X`&@BN!H$2'@!Q`AX`$0@N!P()WEYYV)X`,@@N!G
M<:#A:"&"X&B!)^`)@"C@"("*X'2@G>7I'8'@`Z"*X`>@BN!I<2?@`7`GX`=P
M@N!X()WEX8V(X`,@@N!A$:#A:2&"X&F1(>`(D"G@"9"*X'R@G>7H?8?@`Z"*
MX`&@BN!H$2'@!Q`AX`$0@N"`()WEYYV)X`,@@N!G<:#A:"&"X&B!)^`)@"C@
M"("*X(2@G>7I'8'@`Z"*X`>@BN!I<2?@8;&@X0%P)^#AC8C@!W""X&D1*^"(
M()WE"!`AX.A]A^`!$(K@`R""X&BA*^!I(8+@!Z`JX`J@@N"0()WEYYV!X(P0
MG>5G<:#A`R""X&@A@N`#$('@:($GX`L0@>`)@"C@"("!X)00G>7IK8K@`Q"!
MX`<0@>!I<2?@"G`GX`=P@N"8()WEZHV(X`,@@N!JH:#A:2&"X&F1*N`(D"G@
M"1"!X)R0G>7H?8?@`Y")X`J0B>!HH2K@!Z`JX`H@@N"@H)WEYQV!X`,PBN!G
M<:#A:#&#X&B!)^`!@"C@X2V"X`B`B>!AD2?@`I`IX.*-B.`),(/@Z)V#X*2P
MG>6P-9_EJ*"=Y6(AH.$#L(O@!["+X`.@BN!A<8+A8:&*X`AP!^!A$0+@`1"'
MX0L0@>!HL8+AZ1V!X&EQH.$)L`O@:)$"X`F0B^&LL)WE"I")X+"@G>4#L(O@
M`K"+X`.@BN!H(8?A:*&*X`$@`N!H@0?@X9V)X`@@@N%A@8?A"R""X`F`".!A
ML0?@"X"(X;2PG>7I+8+@`["+X`>PB^"X<)WE:9&@X0J`B.`#<(?@8:&)X0*@
M"N!A<8?@81$)X.*-B.`!$(KA8J&)X0L0@>`(H`K@8K$)X`N@BN&\L)WEZ!V!
MX`.PB^`)L(O@P)"=Y6B!H.$'<(K@`Y")X&*AB.$!H`K@8I&)X&(A".#A?8?@
M`B"*X6&AB.$+((+@!Z`*X&&Q".`+H(KAQ+"=Y><M@N`#L(O@"+"+X,B`G>5G
M<:#A"9"*X`.`B.!AH8?A`J`*X&&!B.!A$0?@XIV)X`$0BN%BH8?A"Q"!X`F@
M"N!BL0?@"Z"*X<RPG>7I'8'@`["+X`>PB^#0<)WE:9&@X0B`BN`#<(?@8J&)
MX0&@"N!B<8?@8B$)X.&-B.`"((KA8:&)X0L@@N`(H`K@8;$)X`N@BN'4L)WE
MZ"V"X`.PB^`)L(O@V)"=Y6B!H.$'<(K@`Y")X&&AB.$"H`K@89&)X&$1".#B
M?8?@`1"*X6*AB.$+$('@!Z`*X&*Q".`+H(KAW+"=Y0F0BN#@H)WEYQV!X`.P
MB^!G<:#A"+"+X`.@BN!B@8?A8J&*X`&`".!B(0?@X9V)X`*`B.%A(8?A"X"(
MX`D@`N!AL0?@"R""X>2PG>4*((+@Z*"=Y>F-B.`#L(O@:9&@X0>PB^`#H(K@
M87&)X6&ABN`(<`?@81$)X.@M@N`!<(?A:!&)X0MPA^`"$`'@:+$)X`L0@>$*
M$('@[*"=Y>)]A^`#H(K@"9"*X/"@G>5B(:#A`S"*X&BA@^!H,8+A!S`#X&B!
M`N`(@(/A"9"(X-PRG^7T@)WEYQV!X&>Q@N$#@(C@`;`+X`*`B.!G(0+@`B"+
MX0H@@N#XH)WEX9V)X`.@BN!A$:#A9Z&*X&=Q(>`)<"?@!X"(X/QPG>4`L9WE
M`W"'X.DM@N`!<(?@:1$AX.*-B.`"$"'@`["+X&(AH.$!$(K@:;&+X`2AG>5I
MD2+@")`IX`EPA^`#H(K@")&=Y>@=@>`"H(K@:"$BX.%]A^`!("+@`Y")X&$1
MH.%HD8G@:($AX`>`*.`(@(K@#*&=Y0(@B^`#H(K@YRV"X`&@BN!G$2'@`A`A
MX`$0B>`0D9WEXHV(X`.0B>!B(:#A9Y&)X&=Q(N`(<"?@!W"*X!2AG>7H'8'@
M`Z"*X`*@BN!H(2+@`2`BX`(@B>`8D9WEX7V'X`.0B>!A$:#A:)&)X&B!(>`'
M@"C@"("*X!RAG>7G+8+@`Z"*X`&@BN!G$2'@`A`AX`$0B>`@D9WEXHV(X`.0
MB>!B(:#A9Y&)X&=Q(N`(<"?@!W"*X"2AG>7H'8'@`Z"*X`*@BN!H(2+@`2`B
MX`(@B>`HD9WEX7V'X`.0B>!A$:#A:)&)X&B!(>`'@"C@"("*X"RAG>7G+8+@
M`Z"*X`&@BN!G$2'@`A`AX`$0B>`PD9WEXHV(X`.0B>!B(:#A9Y&)X&=Q(N`(
M<"?@!W"*X#2AG>7H'8'@`Z"*X`*@BN!H(2+@`2`BX`(@B>`XD9WEX7V'X`.0
MB>!A$:#A:)&)X&B!(>`'@"C@YRV"X`B`BN#BC8C@:&&&X`Q@@.4\89WE`V"&
MX`%@AN!G$2'@`A`AX`$0B>#H'8'@8>&.X`C@@.5`X9WE8B&@X0,PCN!GX2+@
M`E"%X`0P@^`(X"[@:"$BX`$@(N!G,8/@#N"&X.$=CN`",(/@X3V#X`'`C.`(
M$(#H1""?Y2PPG^4"((_@$%"`Y0,PDN<`()/E1#&=Y0(@,^``,*#C`0``&E/?
MC>+PC[WH?_3_ZPA9`0`\`0``F7F"6J'KV6[<O!N/UL%BR@Q.`0!P0"WI8`"@
MX_/T_^L`(*#C,,"?Y3`0G^4P4)_E,$"?Y3#@G^4(4(#E#$"`Y1#@@.4`P(#E
M!!"`Y10@@.48((#E7""`Y7"`O>@!(T5GB:O-[_[<NIAV5#(0\.'2P_!'+>D"
MD*#A`%"@X10PD.6"(:#A`C"3X!@@D.44,(#E7`"0Y0$@@B*I+H+@``!0XP%`
MH.$8((7E'("%X@X```I`<&#B!P!9X0EPH+$``(C@!R"@X47T_^M<,)7E`S"'
MX$``4^-<,(7E\(>]&`4`H.$'D$G@!T"$X//\_^L_`%GC'0``VD"02>(I8Z#A
M`7"&X@=SA.`(,*#A0."$X@#`E.4$`)3E"!"4Y0P@E.400(3B#@!4X0#`@^4$
M`(/E"!"#Y0P@@^40,(/B\___&@4`H.'=_/_K!P!4X>W__QH&;6;@!I.)X`D@
MH.$'$*#A"`"@X1[T_^M<D(7E\(>]Z`1PH.'W___J?\#@X_!!+>D4,)'E&&"1
MY3-_O^:C,:#A/S`#X@$@@^(<@('B.`!2XP!0H.$!0*#A-F^_Y@/`R.<"`(C@
M.P``V@`0H.-`(&+B)?3_ZP0`H.&\_/_K"`"@X3@@H.,`$*#C'_3_ZP0`H.'T
M9<3AM?S_ZP,PU.4$`*#A`##%Y;(PU.$!,,7E`#"4Y2,TH.$",,7E`#"4Y0,P
MQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+,-3E"##%
MY;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XPU.$-,,7E
M##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4Y2,TH.$2
M,,7E$#"4Y1,PQ>7P0;WHL_/_Z@`0H.,X(&+BZ?/_Z\C__^H00"WI6#"?Y5A`
MG^4#,(_@)"#3Y01`C^```%+C$("]&`$0H.-`()_E)!##Y0(PE.<``%/C`@``
M"C`PG^4#`)_GB?/_ZR@PG^4#,)3G``!3XQ"`O0@<`)_E$$"]Z```C^#S\__J
M_$P!`!!+`0`8`0``7$P!`%`!``!D10``/!"?Y3PPG^4!$(_@`"#1Y0,PC^``
M`%+C'O\O$0$`H.,D()_E``#!Y0(PD^<``%/C'O\O`10`G^4$$('B``"/X.7S
M_^J$3`$`F$H!`%0!```(10``\$`MZ4S"G^4DT$WB2#*?Y10`C>40$(WE#""-
MY0S`C^`#,)SG$!"-X@P@C>(4`(WB*$*?Y0`PD^4<,(WE`#"@X[+S_^L8(I_E
M!$"/X!0RG^4"()3G`P"4YP(0H.&W\__K!#*?Y0,PE.<`4-/E``!5XU$```KT
M`9_E``"/X/?S_^L`()#E%&"=Y48STN4$$(;B`S"#XP<!<>,#8(;B1C/"Y6(`
M`(H!`:#AC//_ZP`PH.$0$)WEN'&?Y0``D>6TX9_E!W"/X($`@^BLP9_E#N"/
MX*@AG^4$,([E#`"4YP(@C^`44)WE#""#Y0`@D.4!`%7C"""#Y5(``-H$`$'B
M#,"#X@4!@.`$X+'E`0!0X03@K.7[__\:`U"%X@#`H.-@09_E8!&?Y01`C^`,
M`)3E!B"@X0#`C>4!$(_@!<&#YY3S_^L``%#C*0``"CQ1G^488(WB!5"/X`P`
ME>5"\__K`$"@X2@!G^4&$*#A``"/X&KT_^L``%#C(```"A0QG^4#,(_@#`"3
MY2CS_^MO\__K!"&?Y<0PG^4"((_@`S"2YP`@D^4<,)WE`B`SX``PH.,H```:
M!`"@X230C>+P@+WH,O/_Z]0@G^4``%#C`B"/X`P`@N4=```*"//_Z\``G^4`
M`(_@GO/_ZP`PD.7N5</EH/__Z@P`E.6!\__KTO__Z@@`E>4``%#C`0``"K3[
M_^O9___JC`"?Y080H.$``(_@/O3_ZP``4.,(`(7ETO__"O7__^IP,)_E<`"?
MY0,0E.<``(_@.//_ZP10H..R___J`0"@X\CR_^L&\__K.$H!`#P!```,2@$`
M3`$``%P!```H`0``"$L!`#0%``"42P$`2`$``!P%```\2P$`J-G__Q!+`0`\
M`@``Y$H!`,!(`0"82@$`I$D!`"P!``!$`0``^`(``'!S7W-T<FEN9W,@;6ES
M<VEN9PH`+W1M<"\````N````4$%27U1-4$1)4@``5$5-4$1)4@!435``55-%
M4@````!54T523D%-10`````E<R\E<P```"]P<F]C+V-U<G!R;V,O97AE````
M1'EN84QO861E<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"
M3T]4`````%!!4E]414U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S
M``!015),-4Q)0@````!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%2
M7TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)5
M1P```%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!
M3%]435!$25(```!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``
M```Q````;'-E96L@9F%I;&5D`````')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?
M4$%42`!365-414T``"4P,G@`````)7,E<R5S)7,`````<&%R+0`````E<SH@
M8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE
M<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS
M869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO
M;BD*``!0051(`````'!A<FP`````+G!A<@````!S:&]R="!R96%D```E<R5S
M8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`
M````&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)324].`````#$N
M,#8S````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z
M(&-R96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A
M:6QE9"`H97)R;F\])6DI"@`````M90``+2T``&5V86P@)V5X96,@+W5S<B]B
M:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG
M('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?
M34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S:7IE*3L*
M0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,15]O9F9S
M971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?<VEZ92`]
M(#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B(&9I;F1?
M<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I>F4@/2`V
M-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM>2`D<&]S
M(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[("`@("`@
M(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I>F4*=VAI
M;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A9"`D9F@L
M("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I.PII9B`H
M*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P*2!["G)E
M='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*?0IR971U
M<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@
M)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR
M97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E
M<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE
M.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@
M:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![
M("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER
M("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN
M,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@
M)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T
M;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@
M)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN
M(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R
M7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF
M:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE
M)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D
M8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D
M8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P
M97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["DEN
M=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI0
M05(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]
M("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^
M("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]
M/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@
M)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<
M7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G
M;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ
M<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D
M<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'24,Z('L*
M;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H;W!E;B!?
M1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T(')E860@
M9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D(2!H97)E
M(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*;7D@)&UA
M9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA9VEC7W!O
M<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN9R`B)%!!
M4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!1TE#.PI]
M"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G:6-?<&]S
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L(&QE;F=T
M:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@)&)U9B!N
M92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O=6YD(BD[
M"FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B=68B(&9O
M=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I
M>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ93L*;7D@
M)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S970@9G)O
M;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P.PHD<W1A
M<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI
M<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@
M("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&
M24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N
M<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q
M6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H
M("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD
M968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I
M/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D
M8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD
M("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z
M.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE
M;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F
M+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R97%U:7)E
M7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T.PI]"FQO
M8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]($!?.PIR
M971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I;F9O(#T@
M)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I;F9O.PID
M96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D=6QE?2`]
M("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y]
M(&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ
M1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@=&5M<"!F
M:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I;F9O+3Y[
M8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE;'-E('L*
M;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP;2(L("1I
M;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64["F]P96X@
M;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@(D)O;W1S
M=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D=6QE(2([
M"GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A
M=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE
M879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@
M24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R
M97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R
M("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<
M+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F
M:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP
M;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0II
M9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U
M;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*
M;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@
M("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*
M;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE
M=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y
M(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S(&]R("@D
M05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2!["FUY("5D
M:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A;&Q?<&%R
M"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R:69Y7W!A
M<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E)RD@>PIS
M:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T($!!4D=6
M.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`861D7W1O
M7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;04E-3T),
M8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I('L*<'5S
M:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@>PIE=F%L
M(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF="!`<&%R
M7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]("0R.PI]
M"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*?0IE;'-I
M9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I9B`H)#$@
M97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,)RD@>PIO
M<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P96X@;&]G
M(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A8VAE
M7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@)&1I
M<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI=%]I
M;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M9")]
M*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@9F]R
M($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN8SL*
M?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!!
M<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A<B`]
M('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@>PIO
M<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@9FEN
M9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z.D9I;&4G
M*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&<F]M
M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH
M*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<&%R
M(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE
M=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S
M+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^
M("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R
M*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG
M="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P
M:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@
M=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@["GT*0W)E
M871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L
M"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2*"D@?"!)
M3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#86XG="!C
M<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D["G-E96L@
M7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A<G1?;V9?
M1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A<G1?;V9?
M1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A
M9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD("1L
M;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD
M;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D:64@<7%;
M17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF("@D8G5N
M9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?
M9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*
M;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B=6YD;&4@
M;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E
M>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]
M($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL
M97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O
M8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#
M;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V<R!D;&5X
M="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E
M9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D*&UY("1F
M8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@
M)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@;7D@)&1I
M<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI)'QI*2![
M"B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME
M>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]
M("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H)')E9BP@
M)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M
M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M97TK*SL*
M;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X
M=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=
M+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O
M;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB
M=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY("1T:"P@
M)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE(CH@
M)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ1FEL=&5R
M.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M
M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_
M.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W
M+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN
M="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL"G-P<FEN
M=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N
M=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD
M8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D960@1DE,
M12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=
M*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F
M:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R;W(@=W)I
M=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P87)[8VQE
M86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*:68@*"%D
M969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^
M;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D9F@I.PIS
M965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0[
M"GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L("1C86-H
M95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A8VAE7VYA
M;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H)&-A
M8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F:"T^=&5L
M;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&]F
M9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q<5M%<G)O
M<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D(#`W-34L
M("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S=&%R=%]P
M;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ
M.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E
M<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE+3YN97<[
M("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z.EII<"!O
M<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?
M1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A73L*07)C
M:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R
M8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F
M:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI"F]R(&1I
M92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*
M07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`
M4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE
M?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES
M('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@
M>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA
M;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N
M86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>V%R8VAN
M86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@/2`D,3L*
M;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.5GM005)?
M5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M92`F)B`M
M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO=71S*'%Q
M6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A9'D@97AI
M<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q6T5X=')A
M8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN72D["B@D
M;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT<F%C=&EN
M9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U-34L("1D
M97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL97-S("@D
M4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@=6YL97-S
M($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER(%T@6R`M
M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P(%L@+4)\
M+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/1TY!345]
M(#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B($-R96%T
M95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ0F%S96YA
M;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z.D)A<V5N
M86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER92!&:6QE
M.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S<R@M92`D
M<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R97%U:7)E
M7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A9&5R.PIR
M97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E('=A<FYI
M;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER92!#87)P
M.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A
M=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E<75I<F4@
M1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E(%A33&]A
M9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE.PIR97%U
M:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR97%U:7)E
M($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q=6ER92!0
M05(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I<W0["G)E
M<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!4CHZ1FEL
M=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*979A;"![
M(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T["F5V86P@
M>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER92!!<F-H
M:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE.CI(87-H
M.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/.R!R97%U
M:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T9C@@?3L*
M?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[4$%27U1%
M35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A<E]T96UP
M(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA<"`D14Y6
M>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-4"!435`@
M*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE<W,@9&5F
M:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM>2`D=7-E
M<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P=W5I9"@D
M/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R.CI,;V=I
M;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M93L*?0IE
M;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W=6ED.PI]
M"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D14Y6>U53
M15)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?
M9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S
M=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM005)?0TQ%
M04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92!["F]P96X@
M;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E
M860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S(#T@9FEN
M9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA9VEC7W!O
M<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%(BDL(#`[
M"FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*
M:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G:6-?<&]S
M"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B*0HM("1C
M86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C:&5?;F%M
M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B8V%C:&4M
M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER*2!["B1C
M86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`](#$["GT*
M)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM:V1I<B`D
M<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*;&%S
M=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A;F0@)$5.
M5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R!["FUY("@D
M;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN86UE(#T@
M(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE*2!["FUY
M("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO<&5N("1F
M:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C;VYT96YT
M<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@(B1T
M96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I
M;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N
M;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S
M971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@
M86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE
M(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!]
M(&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2![
M"B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA
M;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.
M04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@
M>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I
M9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD
M<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M
M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(
M?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E
M<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D
M+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D
M<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R
M;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P
M<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R
M;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S
M>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#
M=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS
M,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H
M)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]
M7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D
M<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%2
M7TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.
M5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%
M35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%
M3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%
M0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/0D%,7R1?
M(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II9B`H)$5.
M5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*?0I]"G-U
M8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN
M="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS
M=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P
M(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI
M8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R
M87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I
M<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI
M;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P
M97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C
M=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/
M4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD[
M"F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/
M4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<
M9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%
M3D1?7PH`<&5R;'AS:2YC`````$-!0TA%```!&P,[``$``!\````8C?__&`$`
M`'2-__\T`0``[(W__U0!``"4C___I`$``'"0___(`0``Z)'___0!``",DO__
M&`(``*"3__],`@``Q)3__WP"````E___K`(``%R7___0`@``U)?___`"``#<
ME___!`,``%2:__\\`P``>)K__UP#``#(FO__A`,```"@__^T`P``E*#__]0#
M``#XH/__^`,``*"B__\P!```N*+__T@$``#4H___>`0``+"J___`!```6*W_
M_^@$``"PK?__,`4```BY__]D!0``9+G__X`%``!HNO__I`4``*B[__^$`0``
M)+S__W`!``!XO/__"`4``!```````````7I2``)\#@$;#`T`&````!@```#X
MB___7`````!2#A"$!(4#A@*.`1P````T````.(S__W@`````0@X8A`:%!88$
MAP.(`HX!&````%0```"0C/__J`$```!"#@B'`HX!2`X0`!````!P````K+K_
M_U0`````````'````(0````<NO__?`````!"#@B$`HX!;,[$#@`````@````
MI````.B-___<`````$0.((0(A0>&!H<%B`2)`XH"C@$H````R````*".__]X
M`0```$(.'(0'A0:&!8<$B`.)`HX!2@[``60*#AQ""R````#T````[(___Z0`
M````0@X(A`*.`4@.J`%R"@X(0@L``#`````8`0``;)#__Q0!````0@X(A`*.
M`48.B`A"#I`(`E0*#@A""UH*#@A"SL0.`$(+```L````3`$``$R1__\D`0``
M`$(.%(0%A02&`X<"C@%*#I0(1`Z@"`)L"@X40@L````L````?`$``$"2__\\
M`@```$(.((0(A0>&!H<%B`2)`XH"C@%,#B@"F`H.($(+```@````K`$``$R4
M__]<`````$(.$(0$A0.&`HX!7L[&Q<0.```<````T`$``(24__]X`````$(.
M#(0#A0*.`40.(&H.#!````#P`0``W)3__P@`````````-`````0"``#0E/__
M>`(```!"#B2$"84(A@>'!H@%B02*`XL"C@%&#M0)0@[@"0)$"@XD0@L````<
M````/`(``!"7__\D`````$(."(0"C@%.SL0.`````"0```!<`@``%)?__U``
M````0@X(A`*.`58*SL0.`$(+1`K.Q`X`1`LL````A`(``#R7__\X!0```$(.
M$(0$A0.&`HX!3@X8`SX""@X00L[&Q<0.`$(+```<````M`(``$2<__^4````
M`$(.!(X!2@X0<@H.!$(+`"````#4`@``N)S__V0`````0@X0@`2!`X("@P%"
M#A2.!4H.(#0```#X`@``^)S__Z@!````1`XDA`F%"(8'AP:(!8D$B@.+`HX!
M1@ZD@`1"#L"`!`*:"@XD0@L`%````#`#``!HGO__&`````!"#@B$`HX!+```
M`$@#``!HGO__'`$```!"#A2$!84$A@.'`HX!2@XH=@H.%$+.Q\;%Q`X`0@L`
M1````'@#``!4G___W`8```!"#B2$"84(A@>'!H@%B02*`XL"C@%,#L@"`R(!
M"@XD0@L#3@$*#B1"SLO*R<C'QL7$#@!""P``)````,`#``#HI?__J`(```!"
M#A2$!84$A@.'`HX!3`X@`MP*#A1""QP```#H`P``:*C__U@`````0@X(A`*.
M`6;.Q`X`````)`````@$``!HM___L`(```!"#A2$!84$A@.'`HX!1`XX`M8*
M#A1""S`````P!```>*C__U@+````0@XDA`F%"(8'AP:(!8D$B@.+`HX!0@[P
M`@.6!0H.)$(+```8````9`0``)RS__]<`````$(.$(0$A0.&`HX!(````(`$
M``#<L___!`$```!"#B"$"(4'A@:'!8@$B0.*`HX!+````*0$``"\M/__0`$`
M``!$#AB$!H4%A@2'`X@"C@$"D@K.R,?&Q<0.`$(+`````````+1%```X10``
M``````$````!`````0```,0"```!````S@(```$```#C`@```0````(#```/
M````FP0``!D```!HCP$`&P````0````:````;(\!`!P````$````!````'@!
M```%````8`L```8```"P!```"@```.0$```+````$````!4``````````P``
M`&20`0`"````$`(``!0````1````%P```%P1```1````1!```!(````8`0``
M$P````@```#[__]O````"/K__V\7````````````````````````````````
M`````````````````````````'2/`0```````````&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``
M;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L
M$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3
M``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,``&P3``!L$P``;!,`
M`&P3``!L$P``;!,``&P3``!L$P``;!,``'"/`0````````````````!HCP$`
M````````````````@)(!````````````;(\!```````8D@$`````````````
M````=(\!``````!$D@$``````-21`0!`-```0)(!`-B1`0`*4$%2+G!M"@``
M``#,2```U$@```````#82```W$@``.1(``!820``[$@```````#P2```^$@`
M```````,3```1T-#.B`H3F5T0E-$(&YB,R`R,#(S,3`P."D@,3`N-2XP`$=#
M0SH@*&YB,R`R,#(S,3`P."D@,3`N-2XP`"1.971"4T0Z(&-R=#`N4RQV(#$N
M-"`R,#$X+S$Q+S(V(#$W.C,W.C0T(&IO97)G($5X<"`D`"1.971"4T0Z(&-R
M=#`M8V]M;6]N+F,L=B`Q+C(W(#(P,C(O,#8O,C$@,#8Z-3(Z,3<@<VMR;&P@
M17AP("0`)$YE=$)31#H@8W)T:2Y3+'8@,2XQ(#(P,3(O,#@O,3,@,#(Z-#DZ
M,#0@;6%T="!%>'`@)``D3F5T0E-$.B!C<G1B96=I;BYC+'8@,2XQ-R`R,#$X
M+S$R+S(X(#$X.C$W.C$Q(&-H<FES=&]S($5X<"`D`"1.971"4T0Z(&-R=&5N
M9"Y3+'8@,2XR(#(P,3,O,3(O,C`@,34Z,S4Z-#@@:F]E<F<@17AP("0`\<\"
M``````````````````@````(````"`````@````;`````0``````````*BHJ
M($YO($QA8F5L(%!R;W9I9&5D("HJ*@``\<\"``````````````````@````(
M````"`````@````;`````0``````````*BHJ($YO($QA8F5L(%!R;W9I9&5D
M("HJ*@!!+@```&%E86)I``$D````!39+6@`&!P@!"0$*`A($%`$5`1<#&`$:
M`AP!(@%$`0`N<VAS=')T86(`+FEN=&5R<``N;F]T92YN971B<V0N:61E;G0`
M+FYO=&4N;F5T8G-D+G!A>``N;F]T92YN971B<V0N;6%R8V@`+FAA<V@`+F1Y
M;G-Y;0`N9'EN<W1R`"YR96PN9'EN`"YR96PN<&QT`"YT97AT`"YR;V1A=&$`
M+F5H7V9R86UE7VAD<@`N96A?9G)A;64`+FEN:71?87)R87D`+F9I;FE?87)R
M87D`+FIC<@`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+E-5
M3E=?8W1F`"Y!4DTN871T<FEB=71E<P`N;F]I;FET````````````````````
M````````````````````````````````````"P````$````"````%`$``!0!
M```7```````````````!`````````!,````'`````@```"P!```L`0``&```
M````````````!``````````F````!P````(```!$`0``1`$``!0`````````
M``````0`````````-P````<````"````6`$``%@!```@```````````````$
M`````````$H````%`````@```'@!``!X`0``.`,```8`````````!`````0`
M``!0````"P````(```"P!```L`0``+`&```'`````P````0````0````6```
M``,````"````8`L``&`+``#D!``````````````!`````````&`````)````
M`@```$00``!$$```&`$```8`````````!`````@```!I````"0```$(```!<
M$0``7!$``!`"```&````$P````0````(````;0````$````&````;!,``&P3
M```L`P`````````````$````!````'(````!````!@```)@6``"8%@``(#(`
M````````````!`````````!X`````0````(```"X2```N$@``-=`````````
M``````0`````````@`````$````"````D(D``)")```$`0`````````````$
M`````````(X````!`````@```)2*``"4B@``U`0`````````````!```````
M``"8````#@````,```!HCP$`:(\```0```````````````0````$````I```
M``\````#````;(\!`&R/```$```````````````$````!````+`````!````
M`P```'"/`0!PCP``!```````````````!`````````"U````!@````,```!T
MCP$`=(\``/`````'``````````0````(````O@````$````#````9)`!`&20
M``!P`0`````````````$````!````,,````!`````P```-21`0#4D0``2```
M````````````!`````````#)````"`````,```!`D@$`')(``$0`````````
M`````$``````````S@````$````P`````````!R2```]```````````````!
M`````0```!\````!````,`````````!9D@``'P$``````````````0````$`
M``#7`````0``````````````>),``(\```````````````0`````````X0``
M``,``'````````````>4```O```````````````!`````````/$````!````
M`0```(22`0`VE````````````````````0`````````!`````P``````````
;````-I0``/D```````````````$`````````
