/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.util.concurrent.RateLimiter;
import java.io.File;
import java.net.InetAddress;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BooleanSupplier;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.cassandra.concurrent.JMXEnabledThreadPoolExecutor;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.hints.HintsCatalog;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.HintsDispatcher;
import org.apache.cassandra.hints.HintsReader;
import org.apache.cassandra.hints.HintsService;
import org.apache.cassandra.hints.HintsStore;
import org.apache.cassandra.hints.InputPosition;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.service.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDispatchExecutor {
    private static final Logger logger = LoggerFactory.getLogger(HintsDispatchExecutor.class);
    private final File hintsDirectory;
    private final ExecutorService executor;
    private final AtomicBoolean isPaused;
    private final Function<InetAddress, Boolean> isAlive;
    private final Map<UUID, Future> scheduledDispatches;

    HintsDispatchExecutor(File hintsDirectory, int maxThreads, AtomicBoolean isPaused, Function<InetAddress, Boolean> isAlive) {
        this.hintsDirectory = hintsDirectory;
        this.isPaused = isPaused;
        this.isAlive = isAlive;
        this.scheduledDispatches = new ConcurrentHashMap<UUID, Future>();
        this.executor = new JMXEnabledThreadPoolExecutor(maxThreads, 1L, TimeUnit.MINUTES, new LinkedBlockingQueue<Runnable>(), new NamedThreadFactory("HintsDispatcher", 1), "internal");
    }

    void shutdownBlocking() {
        this.scheduledDispatches.clear();
        this.executor.shutdownNow();
        try {
            this.executor.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            throw new AssertionError((Object)e);
        }
    }

    boolean isScheduled(HintsStore store) {
        return this.scheduledDispatches.containsKey(store.hostId);
    }

    Future dispatch(HintsStore store) {
        return this.dispatch(store, store.hostId);
    }

    Future dispatch(HintsStore store, UUID hostId) {
        return this.scheduledDispatches.computeIfAbsent(hostId, uuid -> this.executor.submit(new DispatchHintsTask(store, hostId)));
    }

    Future transfer(HintsCatalog catalog, Supplier<UUID> hostIdSupplier) {
        return this.executor.submit(new TransferHintsTask(catalog, hostIdSupplier));
    }

    void completeDispatchBlockingly(HintsStore store) {
        Future future = this.scheduledDispatches.get(store.hostId);
        try {
            if (future != null) {
                future.get();
            }
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    void interruptDispatch(UUID hostId) {
        Future future = this.scheduledDispatches.remove(hostId);
        if (null != future) {
            future.cancel(true);
        }
    }

    private final class DispatchHintsTask
    implements Runnable {
        private final HintsStore store;
        private final UUID hostId;
        private final RateLimiter rateLimiter;

        DispatchHintsTask(HintsStore store, UUID hostId) {
            this.store = store;
            this.hostId = hostId;
            int nodesCount = Math.max(1, StorageService.instance.getTokenMetadata().getAllEndpoints().size() - 1);
            int throttleInKB = DatabaseDescriptor.getHintedHandoffThrottleInKB() / nodesCount;
            this.rateLimiter = RateLimiter.create((double)(throttleInKB == 0 ? Double.MAX_VALUE : (double)(throttleInKB * 1024)));
        }

        @Override
        public void run() {
            try {
                this.dispatch();
            }
            finally {
                HintsDispatchExecutor.this.scheduledDispatches.remove(this.hostId);
            }
        }

        private void dispatch() {
            HintsDescriptor descriptor;
            while (!HintsDispatchExecutor.this.isPaused.get() && (descriptor = this.store.poll()) != null) {
                try {
                    if (this.dispatch(descriptor)) continue;
                    break;
                }
                catch (FSReadError e) {
                    logger.error("Failed to dispatch hints file {}: file is corrupted ({})", (Object)descriptor.fileName(), (Object)e);
                    this.store.cleanUp(descriptor);
                    this.store.blacklist(descriptor);
                    throw e;
                }
            }
        }

        private boolean dispatch(HintsDescriptor descriptor) {
            logger.trace("Dispatching hints file {}", (Object)descriptor.fileName());
            InetAddress address = StorageService.instance.getEndpointForHostId(this.hostId);
            if (address != null) {
                return this.deliver(descriptor, address);
            }
            this.convert(descriptor);
            return true;
        }

        private boolean deliver(HintsDescriptor descriptor, InetAddress address) {
            File file = new File(HintsDispatchExecutor.this.hintsDirectory, descriptor.fileName());
            InputPosition offset = this.store.getDispatchOffset(descriptor);
            BooleanSupplier shouldAbort = () -> (Boolean)HintsDispatchExecutor.this.isAlive.apply(address) == false || HintsDispatchExecutor.this.isPaused.get();
            try (HintsDispatcher dispatcher = HintsDispatcher.create(file, this.rateLimiter, address, descriptor.hostId, shouldAbort);){
                if (offset != null) {
                    dispatcher.seek(offset);
                }
                if (dispatcher.dispatch()) {
                    this.store.delete(descriptor);
                    this.store.cleanUp(descriptor);
                    logger.info("Finished hinted handoff of file {} to endpoint {}: {}", new Object[]{descriptor.fileName(), address, this.hostId});
                    boolean bl = true;
                    return bl;
                }
                this.store.markDispatchOffset(descriptor, dispatcher.dispatchPosition());
                this.store.offerFirst(descriptor);
                logger.info("Finished hinted handoff of file {} to endpoint {}: {}, partially", new Object[]{descriptor.fileName(), address, this.hostId});
                boolean bl = false;
                return bl;
            }
        }

        private void convert(HintsDescriptor descriptor) {
            File file = new File(HintsDispatchExecutor.this.hintsDirectory, descriptor.fileName());
            try (HintsReader reader = HintsReader.open(file, this.rateLimiter);){
                reader.forEach(page -> page.hintsIterator().forEachRemaining(HintsService.instance::writeForAllReplicas));
                this.store.delete(descriptor);
                this.store.cleanUp(descriptor);
                logger.info("Finished converting hints file {}", (Object)descriptor.fileName());
            }
        }
    }

    private final class TransferHintsTask
    implements Runnable {
        private final HintsCatalog catalog;
        private final Supplier<UUID> hostIdSupplier;

        private TransferHintsTask(HintsCatalog catalog, Supplier<UUID> hostIdSupplier) {
            this.catalog = catalog;
            this.hostIdSupplier = hostIdSupplier;
        }

        @Override
        public void run() {
            UUID hostId = this.hostIdSupplier.get();
            InetAddress address = StorageService.instance.getEndpointForHostId(hostId);
            logger.info("Transferring all hints to {}: {}", (Object)address, (Object)hostId);
            if (this.transfer(hostId)) {
                return;
            }
            logger.warn("Failed to transfer all hints to {}: {}; will retry in {} seconds", new Object[]{address, hostId, 10});
            try {
                TimeUnit.SECONDS.sleep(10L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            hostId = this.hostIdSupplier.get();
            logger.info("Transferring all hints to {}: {}", (Object)address, (Object)hostId);
            if (!this.transfer(hostId)) {
                logger.error("Failed to transfer all hints to {}: {}", (Object)address, (Object)hostId);
                throw new RuntimeException("Failed to transfer all hints to " + hostId);
            }
        }

        private boolean transfer(UUID hostId) {
            this.catalog.stores().map(store -> new DispatchHintsTask((HintsStore)store, hostId)).forEach(Runnable::run);
            return !this.catalog.hasFiles();
        }
    }
}

